////////////////////////////////////////////////////////
// File  : deterministic.search.cc  
// Desc. : nondeterministic search algorithm implementations 
// Author: Rune M. Jensen CS, CMU
// Date  : 8/28/02
////////////////////////////////////////////////////////

#include <bdd.h>
#include <math.h>
#include <queue>
#include <list>
#include "main.hpp"
#include "numDomain.hpp"
#include "bddLayout.hpp"
#include "partition.hpp"
#include "deterministic.search.hpp"
#include "timer.hpp"



////////////////////////////////////////////////////////
//  Aux. functions   
////////////////////////////////////////////////////////

bdd pruneStates(bdd preImage, bdd v) {
  return preImage & !v;
}



// IN
//  T    : partitioned  transition relation
//  v    : currently visited states/ last preImage (encoded in current state variables)
// OUT
//  preImage of V encoded in current state vars 
//  calculation equal to Clark et. al. p. 80
bdd preImage(TRel& T, bdd v) {
  bdd res;
  bdd vr;
  int i;

  res = bddfalse;

  for (i=0; i<T.p.size(); i++)
    {
      vr = bdd_replace(v,T.p[i].subVarSetb);
      res |= bdd_appex(vr,T.p[i].exp,bddop_and,T.p[i].varSetb);
    }
  
  return res;
}



// IN
//  T    : partitioned transition relation
//  v    : currently visited states/ last image (encoded in current state variables)
// OUT
//  image of V encoded in current state vars
//  calculation equal to research journal 8/16-99 p. 60.
bdd image(TRel& T, bdd v) {
  bdd res;
  bdd u;
  int i;
  
  res = bddfalse;

  for (i=0; i<T.p.size(); i++)
    {
      u = bdd_appex(v,T.p[i].exp,bddop_and,T.p[i].varSetf);
      res |= bdd_replace(u,T.p[i].subVarSetf);
    }
  
  return res;
}



// IN
//  Tact     : disjunctive partitioned transition relation where the each partition 
//             corresponds to an action 
//  set1     : set of state the action leads from
//  set2     : set of states (only one state) the action leads to 
//  act      : action number of the applied action
// OUT
//  predecessor within set1 of the single state in set2 when applying act  
//  backwards (calculation equal to Clark et. al. p. 80)
bdd predecessorOfAct(TRel& Tact,bdd set1,bdd set2,int act) {

  bdd res;
  bdd vr;
  
  vr = bdd_replace(set2,Tact.p[act].subVarSetb);
  res = bdd_appex(vr,Tact.p[act].exp,bddop_and,Tact.p[act].varSetb);
  res &= set1;
  return res;
}




// IN
//  Tact  : disjunctive partitioned transition relation where the each partition 
//             corresponds to an action
//  set1     : set of states the action leads from
//  set2     : set of states (only one state) the action leads to 
//  act      : action number of the applied action
// OUT
//  successor within set2 of the single state in set1 when applying act  
//  forward (calculation equal to Clark et. al. p. 80)
bdd successorOfAct(TRel& Tact,bdd set1,bdd set2,int act) {

  bdd res;
  bdd u;

  u = bdd_appex(set1,Tact.p[act].exp,bddop_and,Tact.p[act].varSetf);
  res = bdd_replace(u,Tact.p[act].subVarSetf);
  res &= set2;
  return res;
}











////////////////////////////////////////////////////////
//   Bidiresctional Search
// 
//   1) Bidirectional image/preimage calculation
// 
//   Preconditions:
//   1) The initial state is assumed to be totally defined 
//	(otherwise the forward search is inconsistent)
////////////////////////////////////////////////////////


//IN
// Tact     : disjunctive partitioning where each partition corresponds to
//               a single action (for action extraction)
// T        : disjunctive backward/forward partitioning  (search)
// init        : initial state
// goal        : goal states
//OUT
// solution (empty if no result)
list<string> bidir(TRel& Tact,TRel& T,bdd init, bdd goal, bool frontierSimpl) {

  int i;
  stateSet preImageSet(MAXSOLLENGTH);
  stateSet imageSet(MAXSOLLENGTH);  
  int stepf;
  int stepb;
  bdd initPoint;
  bdd vf,vb;  // forward and backward visited states
  bdd vfOld,vbOld;
  timer imageTimer, preImageTimer,searchClk,extractClk;
  list<string> solution;
  bdd frontier;
  vector<int> fringeSize;
  bdd simpl;


  //  bidirectional search
  stepf=0;
  stepb=0;
  vf = init;
  vb = goal;
  vfOld = bddfalse;
  vbOld = bddfalse;
  preImageSet.state[0] = goal;
  imageSet.state[0] = init;
  double lastImageCPUtime = -10.0;
  double lastPreImageCPUtime = -10.0;

  if (verbosity > 0)
    searchClk.start();


  while ((imageSet.state[stepf] & preImageSet.state[stepb]) == bddfalse) 
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      
      //  Heuristc: expand the one that took least time at last expansion
      if (lastImageCPUtime < lastPreImageCPUtime)
	{
	  // make image
	  imageTimer.start();
	  stepf++;

	  // frontier set simplification
	  if (frontierSimpl) {
	    if (stepf == 1)
	      frontier = vf;
	    else
	      {
		simpl = bdd_simplify(imageSet.state[stepf-1],!vfOld);
		if (bdd_nodecount(simpl) < bdd_nodecount(imageSet.state[stepf-1]))		  
		    frontier = simpl;
		else
		  frontier = imageSet.state[stepf-1];
	      }
	  }
	  else
	    frontier = imageSet.state[stepf-1];

	  imageSet.state[stepf] = image(T,frontier);
	  imageSet.state[stepf] = pruneStates(imageSet.state[stepf],vf);
	  if (imageSet.state[stepf] != bddfalse)
	    {
	      vfOld = vf;
	      vf |= imageSet.state[stepf];
	      lastImageCPUtime = imageTimer.stop();
	      if (verbosity > 0) 
		{
		  cout << "Step " << stepf + stepb << "  Image " << stepf << "  Time "
		       << lastImageCPUtime << "  Fringe " << bdd_nodecount(imageSet.state[stepf]) 
		       << "/" << bdd_nodecount(vf) << "\n";
		  fringeSize.push_back(bdd_nodecount(imageSet.state[stepf]));
		}
	    }
	  else
	    {
	      cout << "Empty Image: No solution exists!\n";
	      return solution; //empty
	    }
	}
      else
	{
	  // Make preImage
	  preImageTimer.start();
	  stepb++;	  

	  // frontier set simplification
	  if (frontierSimpl) {
	    if (stepb == 1)
	      frontier = vb;
	    else
	      {
		simpl = bdd_simplify(preImageSet.state[stepb-1],!vbOld);
		if (bdd_nodecount(simpl) < bdd_nodecount(preImageSet.state[stepb-1]))
		  frontier = simpl;
		else
		  frontier = preImageSet.state[stepb-1];
	      }
	  }
	  else
	    frontier = preImageSet.state[stepb-1];



	  preImageSet.state[stepb] = preImage(T,frontier);
	    
	  preImageSet.state[stepb] = pruneStates(preImageSet.state[stepb],vb);
	  if (preImageSet.state[stepb] != bddfalse)
	    {
	      vbOld = vb;
	      vb |= preImageSet.state[stepb];
	      lastPreImageCPUtime = preImageTimer.stop();
	      if (verbosity > 0) {
		cout << "Step " << stepf + stepb << "  PreIm " << stepb << "  Time " 
		     << lastPreImageCPUtime  << "  Fringe " 
		     << bdd_nodecount(preImageSet.state[stepb]) << "/" << bdd_nodecount(vb) << "\n";
		fringeSize.push_back(bdd_nodecount(preImageSet.state[stepb]));
	      }
	    }
	  else
	    {
	      cout << "Empty PreImage: No solution exists!\n";
	      return solution; // empty
	    } 
	}
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      forwardExpansionNum = stepf;
      backwardExpansionNum = stepb;
      iterationNum = stepf + stepb;
    }

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  // post: the last preImage and image overlaps 
  //       collapse it to one state in the intersection

  //select one state in overlap
  bdd ints;
  ints = imageSet.state[stepf] & preImageSet.state[stepb];
  ints = bdd_satone(ints);
  imageSet.state[stepf] = ints;
  preImageSet.state[stepb] = ints;

  
  bdd nextState;
  // Extract solution from forward traversal of preImages
  for (i= stepb-1; i >=0; i--)
    {
      int j=-1;
      nextState = bddfalse;
      while (nextState == bddfalse)
	{
	  j++;      
	  nextState = successorOfAct(Tact,preImageSet.state[i+1],preImageSet.state[i],j);
	}
      if (verbosity > 1) cout << "Extracted forward: " << Tact.p[j].actName << endl;
      cout.flush();
      preImageSet.state[i] = nextState;
      solution.push_back(Tact.p[j].actName);      
    }

  bdd previousState;
  // Extract solution from backward traversal of images
  for (i = stepf-1; i >=0; i--)
    {
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = predecessorOfAct(Tact,imageSet.state[i],imageSet.state[i+1],j);
	}
      if (verbosity > 1) cout << "Extracted backward: " << Tact.p[j].actName << endl;
      cout.flush();
      imageSet.state[i] = previousState;
      solution.push_front(Tact.p[j].actName);      
    }

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in forward fringe: " 
	   << aveFringeSize << endl;
    }


  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " << Textract << " seconds\n";
    }
  
  return solution;
}




////////////////////////////////////////////////////////
// FORWARD SEARCH
//
// 1) image calculation
//    using a forward dis partitioning and a regular
//    action backward partitioning
//
// Preconditions:
// 1) The initial state is assumed to be totally defined 
//   (otherwise the forward search is inconsistent)
////////////////////////////////////////////////////////


//IN
// Tact     : disjunctive partitioning where each partition corresponds to
//               a single action (for action extraction)
// T        : disjunctive backward/forward partitioning  (search)
// init        : initial state
// goal        : goal states
// OUT
// solution (empty if no result)
list<string> forward(TRel& Tact,TRel& T,bdd init, bdd goal, bool frontierSimpl) {

  int i;
  stateSet preImageSet(MAXSOLLENGTH);
  stateSet imageSet(MAXSOLLENGTH);  
  int stepf;
  int stepb;
  bdd initpoint;
  bdd vf,vb;  // forward and backward visited states
  bdd vfOld,vbOld;
  timer imageTimer, preImageTimer,searchClk,extractClk;
  list<string> solution;
  bdd frontier;
  vector<int> fringeSize;


  //  bidirectional search
  stepf=0;
  stepb=0;
  vf = init;
  vb = goal;
  vfOld = bddfalse;
  vbOld = bddfalse;
  preImageSet.state[0] = goal;
  imageSet.state[0] = init;
  double lastImageCPUtime = -10.0;
  double lastPreImageCPUtime = -10.0;

  if (verbosity > 0)
    searchClk.start();

  while ((imageSet.state[stepf] & preImageSet.state[stepb]) == bddfalse) 
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      // make image
      imageTimer.start();
      stepf++;
      
      // frontier set simplification
      if (frontierSimpl) {
	if (stepf == 1)
	  frontier = vf;
	else
	  frontier = bdd_simplify(imageSet.state[stepf-1],!vfOld);
      }
      else
	frontier = imageSet.state[stepf-1];
      
      imageSet.state[stepf] = image(T,frontier);
      imageSet.state[stepf] = pruneStates(imageSet.state[stepf],vf);
      if (imageSet.state[stepf] != bddfalse)
	{
	  vfOld = vf;
	  vf |= imageSet.state[stepf];
	  lastImageCPUtime = imageTimer.stop();

	  if (verbosity > 0)
	    {
	      cout << "Step " << stepf + stepb << "  Image " << stepf << "  Time " 
		   << lastImageCPUtime << "  Fringe " << bdd_nodecount(imageSet.state[stepf]) 
		   << "/" << bdd_nodecount(vf)<< "\n";
	      fringeSize.push_back(bdd_nodecount(imageSet.state[stepf]));
	    }
	}
      else
	{
	  cout << "Empty Image: No solution exists!\n";
	  return solution; //empty
	}
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = stepf + stepb;
    }


  // post: the last image and goal overlaps 
  //       collapse it to one state in the intersection

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  //select one state in overlap
  bdd ints;
  ints = imageSet.state[stepf] & preImageSet.state[stepb];
  ints = bdd_satone(ints);
  imageSet.state[stepf] = ints;
  preImageSet.state[stepb] = ints;
  

  bdd previousState;
  // Extract solution from backward traversal of images
  for (i = stepf-1; i >=0; i--)
    {
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = predecessorOfAct(Tact,imageSet.state[i],imageSet.state[i+1],j);
	}
      if (verbosity > 1) cout << "Extracted backward: " << Tact.p[j].actName << endl;
      cout.flush();
      imageSet.state[i] = previousState;
      solution.push_front(Tact.p[j].actName);      
    }
   

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in backward/forward fringe: " 
	   << aveFringeSize << endl;
    }

  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " << Textract << " seconds\n";
    }
  
  return solution;
}





////////////////////////////////////////////////////////
//  DISJUNCTIVE BACKWARDSOLUTION
// 
//  1) Backward preImage calculation
//     using a Dis partitioning 
// 
//  Preconditions:
//  1) The initial state is assumed to be totally defined 
//     (otherwise the forward search is inconsistent)
////////////////////////////////////////////////////////


//IN
// Tact     : disjunctive partitioning where each partition corresponds to
//               a single action (for action extraction)
// T        : disjunctive backward/forward partitioning  (search)
// init        : initial state
// goal        : goal states
// OUT
// solution (empty if no result)
list<string> backward(TRel& Tact,TRel& T,bdd init, bdd goal, bool frontierSimpl) {

  int i;
  stateSet preImageSet(MAXSOLLENGTH);
  stateSet imageSet(MAXSOLLENGTH);  
  int stepf;
  int stepb;
  bdd initPoint;
  bdd vf,vb;  // forward and backward visited states
  bdd vfOld,vbOld;
  timer imageTimer, preImageTimer,searchClk,extractClk;
  list<string> solution;
  bdd frontier;
  vector<int> fringeSize;


  //  bidirectional search
  stepf=0;
  stepb=0;
  vf = init;
  vb = goal;
  vfOld = bddfalse;
  vbOld = bddfalse;
  preImageSet.state[0] = goal;
  imageSet.state[0] = init;
  double lastImageCPUtime = -10.0;
  double lastPreImageCPUtime = -10.0;

  if (verbosity > 0)
    searchClk.start();

  while ((imageSet.state[stepf] & preImageSet.state[stepb]) == bddfalse) 
    {

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      
      // Make preImage
      preImageTimer.start();
      stepb++;	  
      
      // frontier set simplification
      if (frontierSimpl) {
	if (stepb == 1)
	  frontier = vb;
	else
	    frontier = bdd_simplify(preImageSet.state[stepb-1],!vbOld);
      }	    
      else
	frontier = preImageSet.state[stepb-1];
	 
       
      preImageSet.state[stepb] = preImage(T,frontier);
      
      preImageSet.state[stepb] = pruneStates(preImageSet.state[stepb],vb);
      if (preImageSet.state[stepb] != bddfalse)
	{
	  vbOld = vb;
	  vb |= preImageSet.state[stepb];
	  lastPreImageCPUtime = preImageTimer.stop();
	  if (verbosity > 0)
	    {
	      cout << " Step " << stepf + stepb << "  PreIm " << stepb 
		   << "  Time " << lastPreImageCPUtime  << "  Fringe " 
		   << bdd_nodecount(preImageSet.state[stepb]) << "/" 
		   << bdd_nodecount(vb) << "\n";
	      fringeSize.push_back(bdd_nodecount(preImageSet.state[stepb]));
	    }
	}
      else
	{
	  cout << "Empty PreImage: No solution exists!\n";
	  return solution; // empty
	} 
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = stepf + stepb;
    }


  // post: the last preImage and image overlaps 
  //       collapse it to one state in the intersection

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;



  //select one state in overlap
  bdd ints;
  ints = imageSet.state[stepf] & preImageSet.state[stepb];
  ints = bdd_satone(ints);
  imageSet.state[stepf] = ints;
  preImageSet.state[stepb] = ints;
  

  bdd nextState;
  // Extract solution from forward traversal of preImages
  for (i= stepb-1; i >=0; i--)
    {
      int j=-1;
      nextState = bddfalse;
      while (nextState == bddfalse)
	{
	  j++;      
	  nextState = successorOfAct(Tact,preImageSet.state[i+1],preImageSet.state[i],j);
	}
      if (verbosity > 1) cout << "Extracted forward: " << Tact.p[j].actName << endl;
      cout.flush();
      preImageSet.state[i] = nextState;
      solution.push_back(Tact.p[j].actName);      
    }

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in backward fringe: " 
	   << aveFringeSize << endl;
    }

  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " << Textract << " seconds\n";
    }
    
  return solution;
}




























////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
// 
//	     ---- HEURISTIC SEARCH ----
//   
////////////////////////////////////////////////////////
////////////////////////////////////////////////////////








////////////////////////////////////////////////////////
//
// G-H Priority Queue datastructure
// implementation
//
////////////////////////////////////////////////////////


//IN
// (implicit outputs)
// f  :  returns f of top
// g  :  returns g of top
// h  :  returns h of top
//OUT
// a bdd of top states 
bdd ghQueue::popTop(double& f,int& g,int& h) {


  bdd res;

  if (Q.empty())
    return bddfalse; // the queue is empty
  else
    {

      // get g and h of node with highest priority
      Qvalue Qtop = Q.top();
      bddNum--;
      
      g = Qtop.g;
      h = Qtop.h;
      f = Qtop.f;
      
      pair<int,int> gh(g,h);
      // lookup bdd of node
      
      Mvalue Mentry = M[gh];

      // pop the top bdd node of the BDD queue 
      bddNode topBddNode(Mentry.q.top()); 
      res = topBddNode.b;
      Mentry.q.pop();

      // update Q and M
      if (Mentry.q.empty())
	{ 
	  // the BDD queue has been emptied so pop the (g,h) of Mentry from Q
	  Q.pop();
	  // and erase Mentry from M
	  M.erase(gh);
	}
      else
	{ 
	  // update Mentry in M (don't change Q since the entry still exists)
	  M[gh] = Mentry;
	}
      
      return res;
    }
}



void ghQueue::insert(bdd set,bool goalOverlap,int g, int h, double wg, double wh) {

  pair<int,int> gh(g,h);
  
  if (M.count(gh) > 0)
    {
      // key exists in M : update it's BDD queue with "set"
      Mvalue Mentry = M[gh];

      // try to merge it with the top node, otherwise insert "set" as an independent
      // node in the BDD queue  
      bddNode topBddNode(Mentry.q.top()); 
      if (bdd_nodecount(set) + topBddNode.size < upperBound)
	{
	  // coalecs set and topbddnode
	  Mentry.q.pop();
	  topBddNode.b |= set;
	  topBddNode.size = bdd_nodecount(topBddNode.b);
	  Mentry.q.push(topBddNode);
	}
      else
	{ // insert "set" as a new node in the BDD queue 
	  // (obs reusing topnode to create the new node)
	  bddNum++;
	  topBddNode.size = bdd_nodecount(set);
	  topBddNode.b = set;
	  Mentry.q.push(topBddNode);
	}

      
      bool NewGoalOverLap = ( Mentry.goalOverlap || goalOverlap );
      // you cannot make this assignment directly, it won't work
      Mentry.goalOverlap = NewGoalOverLap; 
            
      // finally update value in M
      M[gh] = Mentry;           
    }
  else
    {
      // key does not exist: create it both in Q and M
      bddNum++;
      
      // update M
      priority_queue<bddNode,vector<bddNode>,bddNodeCmp> q;
      q.push(bddNode(bdd_nodecount(set),set));
      Mvalue Mentry(q,goalOverlap);
      M[gh] = Mentry;
    
      // update Q
      Q.push(Qvalue(wg*g + wh*h,g,h));
    }
}


bool ghQueue::topHasGoalOverlap() {

  // get g and h of node with highest priority
  Qvalue Qtop = Q.top();
    
  pair<int,int> gh(Qtop.g,Qtop.h);
  
  // lookup entry in M
  
  Mvalue Mentry = M[gh];

  return Mentry.goalOverlap;
}
  

int ghQueue::gOfTop() { 
  Qvalue Qtop = Q.top();
  return Qtop.g;
}


void ghQueue::print() {
  cout << "SetA* ghQueue structure\n";
  cout << " Upper bound=" << upperBound << endl;
  cout << " Queue content:\n";
  
  priority_queue<Qvalue,vector<Qvalue>,QvalueCmp> q = Q;
  
  // print each entry in Q 
  while (!q.empty()) 
    {
      Qvalue qtop = q.top();
      q.pop();
      
      pair<int,int> gh(qtop.g,qtop.h);
      Mvalue Mentry = M[gh];
      cout << "  (f=" << qtop.f << ",g=" << qtop.g << ",h=" << qtop.h << ") : goalOverLap=" << Mentry.goalOverlap;
      
      priority_queue<bddNode,vector<bddNode>,bddNodeCmp> bddq = Mentry.q;
      
      //print info for each BDD node 
      while (!bddq.empty())
	{
	  bddNode node = bddq.top();
	  bddq.pop();
	  cout << " (size=" << node.size << ")";
	}
      
      cout << endl;
    }      
}













////////////////////////////////////////////////////////
//
// Reachedstates datatype implementation
//
////////////////////////////////////////////////////////


//IN
// states      : the set of new states that should be inserted into the set of reached states
// g           : the g value=depth of=image number of the states
// OUT
// the reached states data structure updated with the new states
void reachedStates::update(bdd states,int g) {

  if (g > seenDepth) {
    // update maxDepth
    for (int i=seenDepth+1; i<=g; i++)
      withinSet[i] |= withinSet[seenDepth];
    seenDepth = g;
  }

  // update reachablestates with new states
  for (int i=g; i<=seenDepth; i++)
  withinSet[i] |= states;
}
  


//IN
// states      : states from which states already reached with lower or same cost should be pruned
// g           : the g value=depth of=image number of the states
// OUT
// a bdd representing the subset of states from states that has not been 
// reached earlier with a lower or equal g-value
bdd reachedStates::prune(bdd states,int g) {
  if (g > seenDepth)
    return (states & !withinSet[seenDepth]);
  else
    return (states & !withinSet[g]);
}
      

//IN
// def           : domain definition
// goal          : bdd of goal set
// glevel        : the g level the goal overlap is found at
// OUT
// a solution given as a list of action names
// OBS: this function desctructs the within sets !!
list<string> reachedStates::extractSolution(TRel& Tact, bdd goal, int gLevel) {

  list<string> solution;
  
  // generate a set of explored images from the within set
  for (int i=gLevel; i > 0; i--)
      withinSet[i] != withinSet[i-1];
  
  // constrain goal set to goal states and choose one
  // OBS we may not extract the exact path that was found by the last node of A*
  // but the generated path is as least as cheap
  withinSet[gLevel] &= goal;
  withinSet[gLevel] = bdd_satone(withinSet[gLevel]);
  bdd previousState;

  for (int i = gLevel; i > 0; i--)
    { 
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = predecessorOfAct(Tact,withinSet[i-1],withinSet[i],j);	  
	}
      if (verbosity > 1) cout << "Extracted action: " << Tact.p[j].actName << endl;
      cout.flush();
      withinSet[i-1] = previousState;
      solution.push_front(Tact.p[j].actName);
    }

  return solution;
}




//IN
// def           : domain definition
// goal          : bdd of goal set
// glevel        : the g level the goal overlap is found at
// OUT
// a solution given as a list of action names
// OBS: this function desctructs the within sets !!
list<string> reachedStates::extractSolutionBackward(TRel& Tact, bdd goal, int gLevel) {

  list<string> solution;
  
  // generate a set of explored images from the within set
  for (int i=gLevel; i > 0; i--)
      withinSet[i] != withinSet[i-1];
  
  // constrain goal set to goal states and choose one
  // OBS we may not extract the exact path that was found by the last node of A*
  // but the generated path is as least as cheap
  withinSet[gLevel] &= goal;
  withinSet[gLevel] = bdd_satone(withinSet[gLevel]);
  bdd previousState;

  for (int i = gLevel; i > 0; i--)
    { 
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = successorOfAct(Tact,withinSet[i],withinSet[i-1],j);	  
	}
      if (verbosity > 1) cout << "Extracted action: " << Tact.p[j].actName << endl;
      cout.flush();
      withinSet[i-1] = previousState;
      solution.push_back(Tact.p[j].actName);      
    }

  return solution;
}



  
void reachedStates::print() {
  
  cout << "depth:bddNode#";
  for (int i=0; i <= seenDepth; i++)
    cout << i << ":" << bdd_nodecount(withinSet[i]);
}


////////////////////////////////////////////////////////
// Other aux. functions
////////////////////////////////////////////////////////




// IN
//  T    : partitioned transition relation
//  v    : currently visited states/ last image (encoded in current state variables)
//  i    : partition image should be made of
// OUT 
//  Image of parition i of disjunctive partitioned transition relation T
bdd partitionImage(TRel& T, bdd v, int i) {
  bdd vr;

  vr = bdd_appex(v,T.p[i].exp,bddop_and,T.p[i].varSetf);
  return bdd_replace(vr,T.p[i].subVarSetf);
}


// IN
//  T    : partitioned transition relation
//  v    : currently visited states/ last image (encoded in current state variables)
//  i    : partition image should be made of
// OUT 
//  preImage of parition i of disjunctive partitioned transition relation T
bdd partitionPreImage(TRel &T, bdd v, int i) {
  bdd vr;
  
  vr = bdd_replace(v,T.p[i].subVarSetb);
  return bdd_appex(vr,T.p[i].exp,bddop_and,T.p[i].varSetb);
}







////////////////////////////////////////////////////////
//  SetA*
//  				
//  Weighted A* search algorithm generalized to sets 
//  of states 
// 
//
////////////////////////////////////////////////////////



//IN
// Tact     : disjunctive partitioning where each partition corresponds
//               to an action (for action extraction)
// T        : disjunctive split partitioning where each partition corresponds to a 
//               splitted action fraction 
// init        : initial states (= PDDL goal states)
// goal        : goal states (= PDDl init state)
// l           : lower bound on BDDs in queue structure (currently not used for anything)
// u           : upper bound on BDDs in queue structure
// wg           : 
// wh           : f = wg * g + wh*h
//OUT
// solution (empty if no result)
list<string> SetAstar(TRel& Tact,TRel& T,bdd init, bdd goal,int hInit,
		      int upperBound, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;

  ghQueue ghQ(upperBound);
  reachedStates RS(MAXDEPTH);  
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = hInit;
  g = 0;
  goalOverLap = (init & goal) != bddfalse;
  ghQ.insert(init,goalOverLap,g,h,wg,wh);
  RS.update(init,g);


  if (verbosity > 0)
      searchClk.start();  
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    { 

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

     
      currentSet = ghQ.popTop(f,g,h);
      it++;
      if (verbosity > 0)
	{
	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionImage(T,currentSet,i);
	  
	  // prune child according to reached states
	  child = RS.prune(child,g + T.p[i].dg);
	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		  {
		    if (T.p[i].dh < 0)
		      cout << "+";
		    else if (T.p[i].dh > 0)
		      cout << "-";
		    else
		      cout << "0";		      
		    cout.flush();
		  }
	      
	      goalOverLap = (child & goal) != bddfalse;
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,h + T.p[i].dh,wg,wh);
	      RS.update(child,g + T.p[i].dg);
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      
      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      iterationNum = it;
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  
  
  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
      solution =  RS.extractSolution(Tact,goal,ghQ.gOfTop());      

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  return solution;
}







////////////////////////////////////////////////////////
//    SetAstarBackward
//  				
//  Weighted A* search algorithm generalized to sets 
//  of states. This algorithm simulates a backward search.
//  Thus init = PDDL goal states and goal = PDDL init states 
// 
//  Comments :
//         Used for the HSPr heuristic
//
//  Assumptions :
// 
//  1) init is a single state
//  2) goal is a single state (otherwise it's the heuristic 
//     value for backward search is ill defined)
//
////////////////////////////////////////////////////////



//IN
// Tact         : disjunctive partitioning where each partition corresponds
//                to an action (for action extraction)
// T            : disjunctive split partitioning where each partition corresponds to a 
//                splitted action fraction 
// init         : initial states paired with their H-value (= PDDL goal state)
// goal         : goal states (= PDDL init state)
// upperBound   : upper bound on BDDs in queue structure
// wg           : 
// wh           : f = wg * g + wh*h
//OUT
// solution (empty if no result)
list<string> SetAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,
			      int upperBound, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;

  ghQueue ghQ(upperBound);
  reachedStates RS(MAXDEPTH);  
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = hInit;
  g = 0;
  goalOverLap = (init & goal) != bddfalse;
  ghQ.insert(init,goalOverLap,g,h,wg,wh);
  RS.update(init,g);


  if (verbosity > 0)
      searchClk.start();  
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      currentSet = ghQ.popTop(f,g,h);
      it++;
      if (verbosity > 0)
	{
	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionPreImage(T,currentSet,i);
	  
	  // prune child according to reached states
	  child = RS.prune(child,g + T.p[i].dg);
	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dh > 0)
		    cout << "+";
		  else if (T.p[i].dh < 0)
		    cout << "-";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = (child & goal) != bddfalse;
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,h - T.p[i].dh,wg,wh);
	      RS.update(child,g + T.p[i].dg);
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      
      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
      solution =  RS.extractSolutionBackward(Tact,goal,ghQ.gOfTop());      

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  return solution;
}













////////////////////////////////////////////////////////
//  
//  FminAstar
//  				
//  A*  search. Expands the 
//  whole fmin set in each iteration (thus, its purpose
//  is to simulate BDDA* with the SetA*
//  approach). 
//
//  The branching parition has been changed such that dg 
//  actually corresponds to df (actually RESOLUTION times df 
//  Since dg must be an integer)
//
//  df = RESOLUTION*wg*dg + RESOLUTION*wh*dh     <=>
//
//  Obs notice that this changes Finit from
//  Finit = wh*Hinit to
//  Finit = RESOLTION*wh*Hinit 
//
//  ==============================================
//  Simulates BDDA* when wg = 1  and wh = 1 and 
//  the upperbound is infinit
//  ==============================================
//
//
//  Assumptions :
// 
//  1) init is a single state
//
////////////////////////////////////////////////////////



//IN
// Tact       : disjunctive partitioning where each partition corresponds
//          	to an action (for action extraction). In this partition dg
//          	corresponds to df for weighted Astar scaled, to allow integer
//              representation (dg = RESOLUTION*wg*dg + RESOLUTION*wh*dh) 
// T          : disjunctive split partitioning search partition
// init       : initial state
// goal       : goal states 
// hInit      : h value in init        
// u            : upper bound (should be inifinit to simulate BDDA*)
// wg           : 
// wh           : f = RESOLTUION*wg*g + RESOLTUION*wh*h
//OUT
// solution (empty if no result)
list<string> FminAstar(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,int u, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;
  vector< pair<int,bdd> > fringe;

  ghQueue ghQ(u); 
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = 0;
  g = int(WEIGHTRESOLUTION*wh*hInit);
  goalOverLap = ( (init & goal) != bddfalse );
  ghQ.insert(init,goalOverLap,g,h,1.0,0.0); // wg = 1.0, wh = 0.0 such that f = g


  if (verbosity > 0)
      searchClk.start();  

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      currentSet = ghQ.popTop(f,g,h);

      pair<int,bdd> elem(g,currentSet);
      fringe.push_back(elem);

	   	
      it++;
      if (verbosity > 0)
	{
	  if ( (currentSet & goal) != bddfalse ) cout << "T";

	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionImage(T,currentSet,i);
	  	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dg > 0)
		    cout << "-";
		  else if (T.p[i].dg < 0)
		    cout << "+";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = ( (child & goal) != bddfalse );
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,0,1.0,0.0);	      
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      

      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
    {
      // extract the solution in a similar way to BDDA*

      bdd currentState = ghQ.popTop(f,g,h);
      currentState = bdd_satone(goal & currentState);
      int currentF = g; 

      // find the fringe of the currentState
      int currentFringe = fringe.size();

      int nextFringe;
      bool nextFringeFound;

      // Extract solution from backward traversal of fringe
      while ( (currentState & init) == bddfalse )
	{
	  nextFringeFound = false;
	  nextFringe = currentFringe;
	  while ( !nextFringeFound )
	    {
	      nextFringe--;
	      if (nextFringe < 0)
		{
		  cout << "deterministic.search.cc : FminAstar : No more" 
		       << " fringes left\nexiting\n";
		  exit(1);
		}
	      
	      // go through each action in order to see if it could lead 
	      // to the fringe (break if solution found
	      for (int j = 0; j < Tact.p.size(); j++)
		{
		  bdd nextState = predecessorOfAct(Tact,fringe[nextFringe].second,currentState,j); 

		  if (nextState != bddfalse && fringe[nextFringe].first == (currentF - Tact.p[j].dg) )
		    {
		      // nextFringe found, update currentState
		      nextFringeFound = true;
		      currentF = fringe[nextFringe].first;
		      currentState = bdd_satone(nextState);
		      currentFringe = nextFringe;
		      if (verbosity > 1) 
			cout << "Extracted backward: " << Tact.p[j].actName << endl;
		      solution.push_front(Tact.p[j].actName);      
		      break;
		    }
		}
	    }
	}
    }


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }


  return solution;
}






////////////////////////////////////////////////////////
//  
//  FminAstarBackward
//  				
//  A*  search. Expands the 
//  whole fmin set in each iteration (thus, its purpose
//  is to simulate BDDA* with the SetA*
//  approach). This algorithm simulates a backward search.
//  Thus init = PDDL goal states and goal = PDDL init states 
//
//  The branching parition has been changed such that dg 
//  actually corresponds to df (actually RESOLUTION times df 
//  Since dg must be an integer)
//
//  df = RESOLUTION*wg*dg + RESOLUTION*wh*dh     <=>
//
//  Obs notice that this changes Finit from
//  Finit = wh*Hinit to
//  Finit = RESOLTION*wh*Hinit 
//
//  ============================================
//  Simulates BDDA* when wg = 1.0, wh = 1.0 and  
//  the upperbound is infinit
//  ============================================
//
//  Comments :
//         Used for the HSPr heuristic 
//
//  Assumptions :
// 
//  1) init is a single state
//  2) goal is a single state (otherwise it's the heuristic 
//     value for backward search is ill defined)
//
////////////////////////////////////////////////////////



//IN
// Tact       : disjunctive partitioning where each partition corresponds
//          	to an action (for action extraction). In this partition dg
//          	corresponds to df for weighted Astar scaled, to allow integer
//              representation (dg = RESOLUTION*wg*dg + RESOLUTION*wh*dh) 
// T          : disjunctive split partitioning search partition
// init       : initial states paired with their H-value (= PDDL goal state)
// goal       : goal states (= PDDL init state)
// hInit      : h value in init        
// u            : upper bound (should be inifinit to simulate BDDA*)
// wg           : 
// wh           : f = RESOLTUION*wg*g + RESOLTUION*wh*h
//OUT
// solution (empty if no result)
list<string> FminAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,int u, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;
  vector< pair<int,bdd> > fringe;

  ghQueue ghQ(u); 
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = 0;
  g = int(WEIGHTRESOLUTION*wh*hInit);
  goalOverLap = ( (init & goal) != bddfalse );
  ghQ.insert(init,goalOverLap,g,h,1.0,0.0); // wg = 1.0, wh = 0.0 such that f = g


  if (verbosity > 0)
      searchClk.start();  

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      currentSet = ghQ.popTop(f,g,h);

      pair<int,bdd> elem(g,currentSet);
      fringe.push_back(elem);

	   	
      it++;
      if (verbosity > 0)
	{
	  if ( (currentSet & goal) != bddfalse ) cout << "T";

	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionPreImage(T,currentSet,i);
	  	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dg > 0)
		    cout << "-";
		  else if (T.p[i].dg < 0)
		    cout << "+";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = ( (child & goal) != bddfalse );
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,0,1.0,0.0);	      
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      

      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
    {
      // extract the solution in a similar way to BDDA*

      bdd currentState = ghQ.popTop(f,g,h);
      currentState = bdd_satone(goal & currentState);
      int currentF = g; 

      // find the fringe of the currentState
      int currentFringe = fringe.size();

      int nextFringe;
      bool nextFringeFound;

      // Extract solution from forward traversal of fringe
      while ( (currentState & init) == bddfalse )
	{
	  nextFringeFound = false;
	  nextFringe = currentFringe;
	  while ( !nextFringeFound )
	    {
	      nextFringe--;
	      if (nextFringe < 0)
		{
		  cout << "deterministic.search.cc : FminAstar : No more" 
		       << " fringes left\nexiting\n";
		  exit(1);
		}
	      
	      // go through each action in order to see if it could lead 
	      // to the fringe (break if solution found
	      for (int j = 0; j < Tact.p.size(); j++)
		{
		  bdd nextState = successorOfAct(Tact,currentState,fringe[nextFringe].second,j);
		  if (nextState != bddfalse && fringe[nextFringe].first == (currentF - Tact.p[j].dg) )
		    {
		      // nextFringe found, update currentState
		      nextFringeFound = true;
		      currentF = fringe[nextFringe].first;
		      currentState = bdd_satone(nextState);
		      currentFringe = nextFringe;
		      if (verbosity > 1) 
			cout << "Extracted forward: " << Tact.p[j].actName << endl;
		      solution.push_back(Tact.p[j].actName);      
		      break;
		    }
		}
	    }
	}
    }


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }


  return solution;
}






////////////////////////////////////////////////////////
//  
//  PureFminAstarBackward
//  				
//  best-first  search. Expands the 
//  whole fmin set in each iteration (thus, its purpose
//  is to simulate pure-BDDA* the SetA*
//  approach). This algorithm simulates a backward search.
//  Thus init = PDDL goal states and goal = PDDL init states 
// 
//  Comments :
//         Used for the HSPr heuristic 
//
//  Assumptions :
// 
//  1) init is a single state
//  2) goal is a single state (otherwise it's the heuristic 
//     value for backward search is ill defined)
//
////////////////////////////////////////////////////////



//IN
// Tact       : disjunctive partitioning where each partition corresponds
//          	to an action (for action extraction). In this partition dg
//          	corresponds to df (e.g. dg = dg + dh for Astar and dg = dh for
//          	pure Astar)
// T          : disjunctive split partitioning search partition
// init       : initial states paired with their H-value (= PDDL goal state)
// goal       : goal states (= PDDL init state)
//OUT
// solution (empty if no result)
list<string> pureFminAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;
  vector< pair<int,bdd> > fringe;
  bdd reach = bddfalse; // reach set

  ghQueue ghQ(100000000); // upper bound is infinit
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = 0;
  g = hInit;
  goalOverLap = (init & goal) != bddfalse;
  ghQ.insert(init,goalOverLap,g,h,1.0,0.0); // wg = 1.0 and wh = 0.0 such that f = g
  reach |= init;


  if (verbosity > 0)
      searchClk.start();  

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      currentSet = ghQ.popTop(f,g,h);
      it++;

      // insert fmin set on fringe
      pair<int,bdd> elem(g,currentSet);
      fringe.push_back(elem);


      if (verbosity > 0)
	{
	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionPreImage(T,currentSet,i);
	  // prune reached children
	  child = child & !reach;
	  	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dh > 0)
		    cout << "+";
		  else if (T.p[i].dh < 0)
		    cout << "-";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = (child & goal) != bddfalse;
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,0,1.0,0.0);
	      reach |= child;
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      
      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
    {
      // extract the solution in a similar way to BDDA*

      bdd currentState = ghQ.popTop(f,g,h);
      currentState = bdd_satone(goal & currentState);
      int currentF = g; 
				    
      // find the fringe of the currentState
      int currentFringe = fringe.size();
      int nextFringe;
      bool nextFringeFound;

      // Extract solution from forward traversal of fringe
      while ( (currentState & init) == bddfalse )
	{
	  nextFringeFound = false;
	  nextFringe = currentFringe;
	  while ( !nextFringeFound )
	    {
	      nextFringe--;
	      if (nextFringe < 0)
		{
		  cout << "deterministic.search.cc : FminAstar : No more" 
		       << " fringes left\nexiting\n";
		  exit(1);
		}
	      
	      // go through each action in order to see if it could lead 
	      // to the fringe (break if solution found
	      for (int j = 0; j < Tact.p.size(); j++)
		{
		  bdd nextState = successorOfAct(Tact,currentState,fringe[nextFringe].second,j);
		  if (nextState != bddfalse && fringe[nextFringe].first == currentF - Tact.p[j].dg)
		    {
		      // nextFringe found, update currentState
		      nextFringeFound = true;
		      currentF = fringe[nextFringe].first;
		      currentFringe = nextFringe;
		      currentState = bdd_satone(nextState);
		      if (verbosity > 1) 
			cout << "Extracted forward: " << Tact.p[j].actName << endl;
		      solution.push_back(Tact.p[j].actName);      
		      break;
		    }
		}
	    }
	}
    }


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  return solution;
}






