////////////////////////////////////////////////////////
// File  : nonDetSearchAlg.cc  
// Desc. : search algorithm implementations for non-
//         deterministic search domains
//         (Universal Planning) 
// Author: Rune M. Jensen CS, CMU
// Date  : 10/28/02
////////////////////////////////////////////////////////

#include <bdd.h>
#include <math.h>
#include <queue>
#include <list>
#include "main.hpp"
#include "numDomain.hpp"
#include "bddLayout.hpp"
#include "partition.hpp"
#include "universal.hpp"
#include "searchQueue.hpp"
#include "searchAlgAuxFunc.hpp"
#include "nonDetSearchAlg.hpp"
#include "timer.hpp"



////////////////////////////////////////////////////////
//  
// Non-deterministic search algorithms
// 8/28/02, 10/28/02 (for ICAPS-03 paper)
//   
////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////
//
// Strong search algortihms
//
///////////////////////////////////////////////////////////




// IN
//  T             : disjunctive partitioning with variables identifying actions
//  init          : initial state (there may be more than one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
// OUT
//  1: success, 0: failure
int strongSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan) {
  
  // reached states at level 0, frontier at level 0  
  bdd reached = goal;
  
  // frontier states encoded in current variables
  bdd frontier;          
  bdd frontierStates;          
    
  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize;
  int it;

  if (verbosity > 0)
      searchClk.start();  


  // make universal plan by backward chaining from goal
  it = 0; // iteration count
  
  universalPlan = bddfalse;
  //main loop
  while ((reached & init) != init)
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      if (verbosity > 0)
	{
	  it++;
	  cout <<  " it=" << it << " Backward expanding size=" << bdd_nodecount(reached);
	  fringeSize.push_back(bdd_nodecount(reached));
	  cout.flush();
	  expandClk.start();	  
	}

      frontier = strongPreImage(T,reached); // act*current bdd vars.
      frontier = frontier & !reached;       // prune frontier from states already reached
      frontierStates = bdd_exist(frontier,Ainfo.Avars); // current vars.

      if (verbosity > 0)
	{
	  cout << " Texpand=" << expandClk.stop() << endl;
	}
      

      if (frontier != bddfalse)
	{
	  universalPlan |= frontier;
	  reached |= frontierStates;
	}
      else
	{
	  if (verbosity > 0)
	    {
	      Tsearch = searchClk.stop();
	      cout << "Total search time: " << Tsearch << " seconds\n";
	      iterationNum = it;
	      double aveFringe = 0.0;
	      for (int i = 0; i < fringeSize.size(); i++)
		aveFringe += fringeSize[i];
	      aveFringeSize = aveFringe / double(fringeSize.size()); 
	      cout << "Average number of nodes in fringe: " 
		   << aveFringeSize << endl;
	    }
	  return 0;
	}
    }


  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = it;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  return 1;
}












// IN
//  T             : disjunctive branching partitioning where transitions are 
//                  associated with action identifiers
//  Ainfo         : action identifier BDD variable info 
//  init          : initial state (max one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
//  hgoal         : heuristic value of the goal state (recall the search is backward)
//  u             : upperLimit of nodes in the search queue
// OUT
//  1: success, 0: failure
int strongHeuristicSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan,
			  int hGoal,int u) {

  double f; // dummy 
  int g;    // dummy
  int h;
  ghQueue searchQueue(u,0.0,1.0); 

  // statistics vars
  vector<int> fringeSize;
  timer expandClk,searchClk;
  int maxQsize = 0;
  int it = 0;


  // initialize gsQ and RS 
  // since we are searching backwards from the goal
  // states, insert goal states and not init in the searchQueue
  h = hGoal;
  universalPlan = bddfalse;
  bdd coveredStates = goal; // states covered by the plan 
  map<int,bdd> coveredStatesPart; // states covered by the plan, partitioned wrt. h-value
  coveredStatesPart[hGoal] = goal;

  if (verbosity > 0)
      searchClk.start();  


  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while ( (coveredStates & init) == bddfalse)  
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      if (verbosity > 0)
	{
	  it++;
	  cout << "It=" << it << " Children=[";
	}
      
      // add covered state to queue
      // compute the SAs that can reach states outside the states covered by the plan  
      // (these SAs cannot be a part of any strong preimage wrt. the covered states)
      bdd nonStrongSAs = preImage(T,!coveredStates);
      

      // go through each partition in plan and add children to queue
      for (map<int,bdd>::iterator mb = coveredStatesPart.begin(); mb != coveredStatesPart.end(); ++mb)
	for (int i = 0; i < T.p.size(); i++)
	  {
	    // compute child 
	    // 1: compute partion preImage
	    bdd childSA = partitionPreImage(T,mb->second,i);
	    
	    // 2:prune child for states already covered by the universal plan  
	    childSA = childSA & !coveredStates;
	    
	    // 2:prune child for non strong SAs
	    childSA = childSA & !nonStrongSAs;
	    
	    // post: child consists of strong SAs with respect to the coverd states
	    	    
	    // add child to queue if nonempty
	    if (childSA == bddfalse)
	      {
		// child should not be added, but write a dot to tell that a child was processed
		if (verbosity > 0)
		  {
		    cout << ".";	   
		    cout.flush();
		  }
	      }
	    else
	      {
		// write child summary
		if (verbosity > 0)
		  {
		    if (T.p[i].dh > 0)
		      cout << "+";
		    else if (T.p[i].dh < 0)
		      cout << "-";
		    else
		      cout << "0";		      
		    cout.flush();
		  }
	      
		// add child to search queue
		searchQueue.insert(childSA,false,0,mb->first + T.p[i].dh);
	      }
	  }  
      if (verbosity > 0)
	cout << "]"; 
  

      // exit with failure if search queue is empty
      if (searchQueue.empty())
	return 0;
      
            
      // pop top node and add to plan 
      // (obs: g,h,and f value written to g,h,and f as a side effect)
      // (obs: g and f not used for anything)
      bdd topSA = searchQueue.popTop(f,g,h);
      bdd topS = bdd_exist(topSA,Ainfo.Avars);
      universalPlan |= topSA;
      coveredStates |= topS;
	if (coveredStatesPart.count(h) == 1)
	  coveredStatesPart[h] |= topS;	      
	else
	  coveredStatesPart[h] = topS;
     
      

      if (verbosity > 0)
	{
	  cout << " |Q|=" << searchQueue.size() << " top (h=" 
	       << h << ",size=" << bdd_nodecount(topS) << ") Texp=" << expandClk.stop() << endl;;
	  if (searchQueue.size() > maxQsize) maxQsize = searchQueue.size();
	  fringeSize.push_back(bdd_nodecount(topS));
	  cout.flush();
	  expandClk.start();	  
	}
      
      // empty the search queue
      searchQueue.clear();
    }
  
  // main loop ended either with success, failure, or timeout
  
  // however, always compute search statistics
  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      iterationNum = it;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      if (Tsearch > timeout)
	{
	  cout << "TIME OUT !\n";
	  return 0;
	}
      else
	cout << "Total search time: " << Tsearch << " seconds\n";
    }
  
  // case solution found
  if ((coveredStates & init) != bddfalse)
    return 1;
  else return 0;
}



///////////////////////////////////////////////////////////
//
// Strong cyclic search AIJ-01
//
///////////////////////////////////////////////////////////
// 
// This algorithm has been re-written from the algorithm
// in version 0.5 to match the algorithm descibed in Traverso
// et. al AIJ paper 2001 figure 8 (ignoring the strong 
// cyclic plan extension.). Correctness and completeness of 
// this algorithm has been proven formally
//  
///////////////////////////////////////////////////////////


// IN
//  T             : disjunctive branching partitioning where transitions are 
//                  associated with action identifiers
//  Ainfo         : action identifier BDD variable info 
//  init          : initial state (max one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
// OUT
//  1: success, 0: failure
int strongCyclicSearch(TRel& T, ActInfo& Ainfo,bdd init,bdd goal,bdd& universalPlan) {
  
  // statistic vars
  timer expandClk,searchClk;
  int preCompNum = 0;

  
  if (verbosity > 0)
    searchClk.start();  
  

  ///////////////////////////////
  // instantiate plan and search 
  ///////////////////////////////  
  
  bdd coveredStates = goal; // states covered by the plan
  universalPlan = bddfalse;
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while ( (coveredStates & init) != init )  
    {

      // initialize data structures for next
      // strong cyclic plan precomponent
      bdd preCompSA      = bddfalse;
      bdd preCompS       = bddfalse;
      bdd preCompCandSA  = bddfalse;
      bdd preCompCandS   = bddfalse;
      

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;
      
      if (verbosity > 0)
	{
	  preCompNum++;
	  if (verbosity > 0) expandClk.start();
	  cout << "PreCompNo : " << preCompNum << endl;
	}


      // find next precomponent
      while (preCompS == bddfalse)
	{
	  // statistic vars
	  int preimageNum = 0;
	  

	  // 1) compute weak preimage SAs to precomponent candidate SAs
	  bdd weakPreImageSA = preImage(T,coveredStates | preCompCandS);     

	  // prune preimage for states already covered by states 
	  // in the plan 
	  weakPreImageSA &= !coveredStates;

	  
	  // Add weak preimage to precomponent candidates
	  bdd OldpreCompCandSA = preCompCandSA;
	  preCompCandSA |= weakPreImageSA;
	  preCompCandS |= bdd_exist(weakPreImageSA,Ainfo.Avars);

	  
	  // 2) Add preimage to candidate precomponent if not empty
	  if (preCompCandSA == OldpreCompCandSA)
	    {
	      // no SC plan exists !
	      if (verbosity > 0) cout << "  Could not extend preCompCandSA!\n"; 	    
	      return 0;
	    }
	  else
	    {
	      if (verbosity > 0)
		{
		  preimageNum++;
		  cout << "  Weak preimageSA : " << preimageNum << " size=" 
		       << bdd_nodecount(weakPreImageSA);
		}
	      

	      
	      preCompSA = preCompCandSA;
	      bdd OldpreCompSA = !preCompCandSA; // using negation for bottom element !
	      
	      while ( OldpreCompSA != preCompSA )
		{
		  OldpreCompSA = preCompSA;
		  preCompSA = PruneUnconnected(T,Ainfo,coveredStates,
					       PruneOutgoing(T,Ainfo,coveredStates,preCompSA));
		}
	      
	      preCompS = bdd_exist(preCompSA,Ainfo.Avars);
	      
	    }
	}
      
      //////////////////////////////////
      // non-empty precomponent exists
      // add it to strong cyclic plan
      //////////////////////////////////
      
      if (verbosity > 0)
	cout << "  Found non-empty preCompSA size=" << bdd_nodecount(preCompSA) 
	     << " Texp=" << expandClk.stop() << endl;

      // add to covered states and plan
      coveredStates |= preCompS;
      universalPlan |= preCompSA;      
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = preCompNum;
    }

  
  return 1;
}
  









// IN
//  T             : disjunctive branching partitioning where transitions are 
//                  associated with action identifiers
//  Ainfo         : action identifier BDD variable info 
//  init          : initial state (max one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
//  hgoal         : heuristic value of the goal state (recall the search is backward)
//  u             : upperLimit of nodes in the search queue
// OUT
//  1: success, 0: failure
int strongCyclicHeuristicSearch(TRel& T, ActInfo& Ainfo, bdd init,bdd goal, 
				bdd& universalPlan,int hGoal,int u) {
  
  // statistic vars
  timer expandClk,searchClk;
  int preCompNum = 0;
 
  
  if (verbosity > 0)
    searchClk.start();  
  

  ///////////////////////////////
  // instantiate plan and search 
  // queue for initial SC precomponent
  ///////////////////////////////  
  
  bdd coveredStates = goal; // states covered by the plan
  map<int,bdd> coveredStatesPart; // states covered by the plan, partitioned wrt. h-value
  coveredStatesPart[hGoal] = goal;

  universalPlan = bddfalse;
  // obs: ties in the f-value of the search queue are broken according
  // lowest g-value !
  // this ensures that all promissing nodes at weak preimage level i
  // are expanded before any nodes at level i+1 when computing a SC 
  // precomponent 
  ghSCQueue searchQueue(u,1.0,1.0); 

  if (verbosity > 0)
    cout << "Inserting goal children on queue: ";

  insertPrunedChildren(T,searchQueue,goal,0,hGoal,coveredStates);

  if (verbosity > 0)
    cout << endl;

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while ( (coveredStates & init) == bddfalse)  
    {

      // initialize data structures for next
      // strong cyclic plan precomponent
      bdd preCompSA      = bddfalse;
      bdd preCompS       = bddfalse;
      bdd preCompCandSA  = bddfalse;
      map<int,bdd> preCompCandSpart;
      preCompCandSpart.clear();      	   
      bdd preCompCandS   = bddfalse;
      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      if (verbosity > 0)
	{
	  preCompNum++;
	  if (verbosity > 0) expandClk.start();
	  cout << "Computing PreCompNo : " << preCompNum << endl;
	}

      // statistic vars
      int nodeNum = 0;


      // find next precomponent
      while (preCompS == bddfalse)
	{
	  // exit with failure if search queue is empty
	  if (searchQueue.empty())
	    return 0;
	  
	  // 1) pop top node (parent node) of the search queue
          //    and prune its SAs wrt. what is already covered
	  
	  // pop top node from search queue
	  double f; 
	  int g; 
	  int h;
	  bdd parentSA = searchQueue.popTop(f,g,h);


	  // prune parent for states already covered by states 
	  // in the plan and SAs already in the candidate
	  parentSA &= !preCompCandSA;

	  if (verbosity > 0)
	    {
	      nodeNum++;
	      cout << "  |Q|=" << searchQueue.size() << " weakNode:" << nodeNum << " size=" 
		   << bdd_nodecount(parentSA) << " (f=" << f << ",g=" << g << ",h=" << h << ")";
	      if (parentSA == bddfalse) cout << endl;
	    }

	  
	  // 2) Add parent to candidate precomponent if not empty
          //    and insert its children on the search queue
	  if (parentSA != bddfalse)
	    {
	      // Add parent to precomponent candidates
	      preCompCandSA |= parentSA;
	      bdd parentS = bdd_exist(parentSA,Ainfo.Avars);
	      preCompCandS |= parentS; 
	      if (preCompCandSpart.count(h) == 1)
		preCompCandSpart[h] |= parentS;
	      else
		preCompCandSpart[h] = parentS;

	      if (verbosity > 0)
		cout << " |CandSA|=" << bdd_nodecount(preCompCandSA) << " Children=";

	      // insert children of parent node in search queue
	      insertPrunedChildren(T,searchQueue,parentS,g,h,coveredStates | preCompCandSA);



	      preCompSA = preCompCandSA;
	      bdd OldpreCompSA = !preCompCandSA; // using negation for bottom element !
	      
	      while ( OldpreCompSA != preCompSA )
		{
		  OldpreCompSA = preCompSA;
		  preCompSA = PruneUnconnected(T,Ainfo,coveredStates,
					       PruneOutgoing(T,Ainfo,coveredStates,preCompSA));
		}
	      
	      preCompS = bdd_exist(preCompSA,Ainfo.Avars);
	    }
	  else
	    {
	      preCompSA = bddfalse;
	      preCompS = bddfalse;
	    }
	}

      //////////////////////////////////
      // non-empty precomponent exists
      // add it to strong cyclic plan
      //////////////////////////////////
      
      if (verbosity > 0)
	cout << "  Found preCompSA size = " << bdd_nodecount(preCompSA) 
	     << " Texp = " << expandClk.stop() << endl;



      // 1) add to unpartitioned covered states and plan
      coveredStates |= preCompS;
      universalPlan |= preCompSA;
      
      // 2) add new states to partitioned covered states structure
      for (map<int,bdd>::iterator ib = preCompCandSpart.begin(); 
	   ib != preCompCandSpart.end(); ++ib)       
	if (coveredStatesPart.count(ib->first) == 1)
	  coveredStatesPart[ib->first] |= (ib->second & preCompS);	      
	else
	  coveredStatesPart[ib->first] = (ib->second & preCompS);	      
     
      if (verbosity > 0)
	cout << "  Inserting SC children:";

      // 3) fill search queue with children of current SC plan
      searchQueue.clear();
      for (map<int,bdd>::iterator ib = coveredStatesPart.begin(); 
	   ib != coveredStatesPart.end(); ++ib)
	{
	  if (verbosity > 0)
	    cout << "  h=" << ib->first << " ";
	  insertPrunedChildren(T,searchQueue,ib->second,0,ib->first,coveredStates);          
	}

      if (verbosity > 0)
	cout << endl;
    }


  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = preCompNum;
    }
  
  return 1;
}
  










///////////////////////////////////////////////////////////
//
// Strong cyclic search ECP-99
//
///////////////////////////////////////////////////////////
// 
// This algorithm has been re-written to closely follow 
// the format in the "strong planning revisited" paper by
// Vardi et al, ECP-99. This version builds the plan 
// incrementally from the goal states (Fig. 5 in paper)
//
// This algorithm is implemented as close (also variable
// and function namewise) to this algorithm as possible 
//  
///////////////////////////////////////////////////////////




// IN
//  T      : disjunctive partitioning where transitions are 
//           associated with action identifiers
//  Ainfo  : action identifier BDD variable info 
//  init   : initial state
//  goal   : goal states
//  SCP    : reference to strong cyclic plan (implicit output)
// OUT
//  1: success, 0: failure
int strongCyclicSearchECP99(TRel& T, ActInfo& Ainfo,bdd init,bdd goal,bdd& SCP) {
  
  // statistic vars
  timer expandClk,searchClk;
  int it = 0;

  
  if (verbosity > 0)
    searchClk.start();  
  

  ///////////////////////////////
  // instantiate plan and search 
  ///////////////////////////////  
  
  bdd I = init & !goal;
  SCP = bddfalse;
  bdd AccSA = bddfalse;
  bdd OldAccSA = !AccSA; // use the negation of what it is old of as the bottom value
  
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while (  ( States(Ainfo,SCP) & I ) != I   &&    AccSA != OldAccSA)    
    {

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  {
	    cout << "Time out!";
	    return 0;
	  }
      

      OldAccSA = AccSA;
      AccSA = OneStepBack(T,Ainfo,goal,AccSA);
      SCP = AccSA;

      bdd OldSCP = !SCP;


      if (verbosity > 0)
	{
	  it++;
	  if (verbosity > 0) expandClk.start();
	  cout << " it= " << it << " |AccSA|=" <<  bdd_nodecount(AccSA);
	}


      /////////////////////////////////
      // Prune SAs
      /////////////////////////////////
      
      while ( OldSCP != SCP )
	{
	  OldSCP = SCP;
	  SCP = PruneUnconnected(T,Ainfo,goal,PruneOutgoing(T,Ainfo,goal,SCP));
	}


      if (verbosity > 0)
	cout << " Texp = " << expandClk.stop() << endl;
    }
      
  
  if ( ( States(Ainfo,SCP) & I ) != I  )
    {
      if (verbosity > 0) cout << "AccSA could not be incremented !\n"; 	    
      return 0;
    }
  else 
    {
      if (verbosity > 0)
	{
	  Tsearch = searchClk.stop();
	  cout << "Total search time: " << Tsearch << " seconds\n";
	  iterationNum = it;
	}
      return 1;
    }
    
}











///////////////////////////////////////////////////////////
//
// Strong cyclic heuristic search
//
// 
// This algorithm is identical to the algorithm in the 
// "strong planning revisited" paper by
// Vardi et al, ECP-99 (Fig. 5 in paper), except that 
// it uses a function ExpandGuided to expand the candidate 
// SAs (AccSA) in a directed fashion (f-level by f-level) 
//  
///////////////////////////////////////////////////////////


// IN
//  T      : disjunctive branching partitioning where transitions are 
//           associated with action identifiers
//  Ainfo  : action identifier BDD variable info 
//  init   : initial state (max one)
//  goal   : goal states (identical h-value)
//  hGoal  : h-value of goal states
//  SCP    : reference to strong cyclic plan (implicit output)
// OUT
//  1: success, 0: failure
int strongCyclicHeuristicSearchECP99(TRel& T, ActInfo& Ainfo,bdd init,bdd goal,int hGoal,bdd& SCP) {
  
  // statistic vars
  timer expandClk,searchClk;
  int it = 0;

  
  if (verbosity > 0)
    searchClk.start();  
  

  ///////////////////////////////
  // instantiate plan and search 
  ///////////////////////////////  

  Smap S(hGoal,goal);
  
  bdd I = init & !goal;
  SCP = bddfalse;
  bdd AccSA = bddfalse;
  bdd OldAccSA = !AccSA; // use the negation of what it is old of as the bottom value
  
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while (  ( States(Ainfo,SCP) & I ) != I   &&    AccSA != OldAccSA)    
    {

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  {
	    cout << "Time out!";
	    return 0;
	  }
      

      if (verbosity > 0)
	{
	  it++;
	  if (verbosity > 0) expandClk.start();
	  cout << " it= " << it << " |AccSA|=" <<  bdd_nodecount(AccSA);
	}


      OldAccSA = AccSA;
      AccSA |= S.expandGuided(T,Ainfo,AccSA,goal);
      SCP = AccSA;

      bdd OldSCP = !SCP;


      /////////////////////////////////
      // Prune SAs
      /////////////////////////////////
      
      while ( OldSCP != SCP )
	{
	  OldSCP = SCP;
	  SCP = PruneUnconnected(T,Ainfo,goal,PruneOutgoing(T,Ainfo,goal,SCP));
	}


      if (verbosity > 0)
	cout << " Texp = " << expandClk.stop() << endl;
    }
      
  
  if ( ( States(Ainfo,SCP) & I ) != I  )
    {
      if (verbosity > 0) cout << "AccSA could not be incremented !\n"; 	    
      return 0;
    }
  else 
    {
      if (verbosity > 0)
	{
	  Tsearch = searchClk.stop();
	  cout << "Total search time: " << Tsearch << " seconds\n";
	  iterationNum = it;
	}
      return 1;
    }
    
}




































///////////////////////////////////////////////////////////
//
// Weak/optimisitc search algortihms
//
///////////////////////////////////////////////////////////




// IN
//  T             : disjunctive partitioning with variables identifying actions
//  init          : initial state (max one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
// OUT
//  1: success, 0: failure
int weakSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan) {
  
  // reached states at level 0, frontier at level 0  
  bdd reached = goal;
  
  // frontier states encoded in current variables
  bdd frontier;          
  bdd frontierStates;          
    
  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize;
  int it;

  if (verbosity > 0)
      searchClk.start();  


  // make universal plan by backward chaining from goal
  it = 0; // iteration count
  
  universalPlan = bddfalse;
  //main loop
  while ((reached & init) != init)
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      if (verbosity > 0)
	{
	  it++;
	  cout <<  " it=" << it << " Backward expanding size=" << bdd_nodecount(reached);
	  fringeSize.push_back(bdd_nodecount(reached));
	  cout.flush();
	  expandClk.start();	  
	}

      frontier = preImage(T,reached); // act*current bdd vars.
      frontier = frontier & !reached;       // prune frontier from states already reached
      frontierStates = bdd_exist(frontier,Ainfo.Avars); // current vars.

      if (verbosity > 0)
	{
	  cout << " Texpand=" << expandClk.stop() << endl;
	}
      

      if (frontier != bddfalse)
	{
	  universalPlan |= frontier;
	  reached |= frontierStates;
	}
      else
	{
	  if (verbosity > 0)
	    {
	      Tsearch = searchClk.stop();
	      cout << "Total search time: " << Tsearch << " seconds\n";
	      iterationNum = it;
	      double aveFringe = 0.0;
	      for (int i = 0; i < fringeSize.size(); i++)
		aveFringe += fringeSize[i];
	      aveFringeSize = aveFringe / double(fringeSize.size()); 
	      cout << "Average number of nodes in fringe: " 
		   << aveFringeSize << endl;
	    }
	  return 0;
	}
    }


  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = it;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }

  if ( (reached & init) == init)  
    return 1;
  else
    {
      cout << "Timed out !\n";
      return 0;
    }
}











// IN
//  T             : disjunctive branching partitioning where transitions are 
//                  associated with action identifiers
//  Ainfo         : action identifier BDD variable info 
//  init          : initial state (max one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
//  hgoal         : heuristic value of the goal state (recall the search is backward)
//  u             : upperLimit of nodes in the search queue
// OUT
//  1: success, 0: failure
int weakHeuristicSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan,
			int hGoal,int u) {

  double f; // dummy 
  int g;    // dummy
  int h;

  ghQueue searchQueue(u,0.0,1.0); 

  // statistics vars
  vector<int> fringeSize;
  timer expandClk,searchClk;
  int maxQsize = 0;
  int it = 0;


  h = hGoal;
  universalPlan = bddfalse;
  bdd coveredStates = goal; // states covered by the plan 


  if (verbosity > 0)
      searchClk.start();  


  // initialize search queue by inserting all goal state children 
  // go through each branching partition and add the new nodes to searchQueue
  if (verbosity > 0)
    cout << "Initializing searchQ with children of goal state:[";
  
  for (int i = 0; i < T.p.size(); i++)
    {
      // compute child i
      // 1: compute partion preImage
      bdd childSA = partitionPreImage(T,goal,i);
      
      // 2:prune child for states already covered by the universal plan  
      childSA = childSA & !coveredStates;
      
      // insert child on searchQueue if nonempty
      if (childSA == bddfalse)
	{
	  // child should not be added, but write a dot to tell that a child was processed
	  if (verbosity > 0)
	    {
	      cout << ".";	   
	      cout.flush();
	    }
	}
      else
	{
	  // write child summary
	  if (verbosity > 0)
	    {
	      if (T.p[i].dh > 0)
		cout << "+";
	      else if (T.p[i].dh < 0)
		cout << "-";
	      else
		cout << "0";		      
	      cout.flush();
	    }
	  
	  // add child to search queue 
	  searchQueue.insert(childSA,false,0,h + T.p[i].dh);
	}
    }
  if (verbosity > 0)
    cout << "]\n";


  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while (!searchQueue.empty() && (coveredStates & init) == bddfalse)  
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;
      
      
      // pop parent node 
      // (obs: g,h,and f value written to g,h,and f as a side effect)
      // (obs: g and f not used for anything)
      bdd parentSA = searchQueue.popTop(f,g,h);
      bdd parentS = bdd_exist(parentSA,Ainfo.Avars);
      
      // update plan with parent node
      universalPlan |= parentSA;
      coveredStates |= parentS;

      
      if (verbosity > 0)
	{
	  it++;
	  cout << "|Q|=" << searchQueue.size() << " it=" << it << " Expanding (h=" 
	       << h << ",size=" << bdd_nodecount(parentS) << ") children=[";
	  if (searchQueue.size() > maxQsize) maxQsize = searchQueue.size();
	  fringeSize.push_back(bdd_nodecount(parentS));
	  cout.flush();
	  expandClk.start();	  
	}
      
      
      // go through each branching partition and add the new nodes to searchQueue
      for (int i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  // 1: compute partion preImage
	  bdd childSA = partitionPreImage(T,parentS,i);
	  
	  // 2:prune child for states already covered by the universal plan  
	  childSA = childSA & !coveredStates;
	  	  
	  // insert child on searchQueue if nonempty
	  if (childSA == bddfalse)
	    {
	      // child should not be added, but write a dot to tell that a child was processed
	      if (verbosity > 0)
		{
		  cout << ".";	   
		  cout.flush();
		}
	    }
	  else
	    {
	      // write child summary
	      if (verbosity > 0)
		  {
		    if (T.p[i].dh > 0)
		      cout << "+";
		    else if (T.p[i].dh < 0)
		      cout << "-";
		    else
		      cout << "0";		      
		    cout.flush();
		  }
	      
	      // add child to search queue 
	      searchQueue.insert(childSA,false,0,h + T.p[i].dh);
	    }
	}
      
      if (verbosity > 0)
	cout << "] Texpand=" << expandClk.stop() << endl;      
    }
  

  
  // main loop ended either with success, failure, or timeout
  
  // however, always compute search statistics
  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      iterationNum = it;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  

 
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      if (Tsearch > timeout)
	{
	  cout << "TIME OUT !\n";
	  return 0;
	}
      else
	cout << "Total search time: " << Tsearch << " seconds\n";
    }


  // case solution found
  if ((coveredStates & init) != bddfalse)
    return 1;
  else return 0;
}































///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////
//
// Adversarial Planning
//
///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////






// IN
//  T             : disjunctive branching partitioning where transitions are 
//                  associated with action identifiers
//  Ainfo         : action identifier BDD variable info 
//  init          : initial state (max one)
//  goal          : goal states
//  universalPlan : reference to universal plan (implicit output)
// OUT
//  1: success, 0: failure
int strongCyclicAdversarialSearch(TRel& T,TRel& Tenv, ActInfo& Ainfo,bdd init,bdd goal,bdd& universalPlan) {
  
  // statistic vars
  timer expandClk,searchClk;
  int preCompNum = 0;

  
  if (verbosity > 0)
    searchClk.start();  

  // A(s,a_e) : applicable environment actions. 
  //            only needs to be computed once
  bdd A = App(Tenv,Ainfo);


  ///////////////////////////////
  // instantiate plan and search 
  ///////////////////////////////  
  
  bdd coveredStates = goal; // states covered by the plan
  universalPlan = bddfalse;
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  while ( (coveredStates & init) != init )  
    {

      // initialize data structures for next
      // strong cyclic plan precomponent
      bdd preCompSA      = bddfalse;
      bdd preCompS       = bddfalse;
      bdd preCompCandSA  = bddfalse;
      bdd preCompCandS   = bddfalse;
      

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;
      
      if (verbosity > 0)
	{
	  preCompNum++;
	  if (verbosity > 0) expandClk.start();
	  cout << "PreCompNo : " << preCompNum << endl;
	}


      // find next precomponent
      while (preCompS == bddfalse)
	{
	  // statistic vars
	  int preimageNum = 0;
	  

	  // 1) compute weak preimage SAs to precomponent candidate SAs
	  bdd weakPreImageSA = preImage(T,coveredStates | preCompCandS);     

	  // prune preimage for states already covered by states 
	  // in the plan 
	  weakPreImageSA &= !coveredStates;

	  
	  // Add weak preimage to precomponent candidates
	  bdd OldpreCompCandSA = preCompCandSA;
	  preCompCandSA |= weakPreImageSA;
	  preCompCandS |= bdd_exist(weakPreImageSA,Ainfo.Avars);

	  
	  // 2) Add preimage to candidate precomponent if not empty
	  if (preCompCandSA == OldpreCompCandSA)
	    {
	      // no SC plan exists !
	      if (verbosity > 0) cout << "  Could not extend preCompCandSA!\n"; 	    
	      return 0;
	    }
	  else
	    {
	      if (verbosity > 0)
		{
		  preimageNum++;
		  cout << "  Weak preimageSA : " << preimageNum << " size=" 
		       << bdd_nodecount(weakPreImageSA);
		}
	      

	      
	      preCompSA = preCompCandSA;
	      bdd OldpreCompSA = !preCompCandSA; // using negation for bottom element !
	      
	      while ( OldpreCompSA != preCompSA )
		{
		  OldpreCompSA = preCompSA;
		  preCompSA = PruneOutgoing(T,Ainfo,coveredStates,preCompSA);
		  preCompSA = PruneUnconnected(T,Ainfo,coveredStates,preCompSA);
		  preCompSA = PruneUnfair(T,Tenv,Ainfo,A,coveredStates,preCompSA);
		}
	      
	      preCompS = bdd_exist(preCompSA,Ainfo.Avars);
	      
	    }
	}
      
      //////////////////////////////////
      // non-empty precomponent exists
      // add it to strong cyclic plan
      //////////////////////////////////
      
      if (verbosity > 0)
	cout << "  Found non-empty preCompSA size=" << bdd_nodecount(preCompSA) 
	     << " Texp=" << expandClk.stop() << endl;

      // add to covered states and plan
      coveredStates |= preCompS;
      universalPlan |= preCompSA;      
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = preCompNum;
    }

  
  return 1;
}
  


























///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////
//
// 1-fault tolerant planning
//
///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////




// IN
//  Teff          : disjunctive partitioning with variables identifying actions
//                  representing non-faulting transitions of actions
//  Terr          : disjunctive partitioning with variables identifying actions
//                  representing faulting transitions of actions
//  init          : initial state (max one)
//  goal          : goal states
//  USA           : reference to universal plan (implicit output)
//  RSA           : reference to recovery plan (implicit output)
// OUT
//  1: success, 0: failure
int faultTolerantSearch(TRel& Teff, TRel& Terr, ActInfo& Ainfo, bdd init, bdd goal, bdd& USA, bdd& RSA) {
  
  // init plans
  USA = bddfalse;
  RSA = bddfalse;
  
  // reached states at level 0
  bdd US = goal;
  bdd RS = goal;
  
  
  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize;
  int itU = 0;
  int itR = 0;
  if (verbosity > 0)
    searchClk.start();  
  
  
  //main loop
  while ((US & init) != init)
    {
      if (verbosity > 0)
	{
	  itU++;
	  cout <<  " itU=" << itU;
	  cout.flush();
	  expandClk.start();	  
	}

      // U Precomponent encoded in act*current bdd vars
      bdd UpreCompSAcand = preImage(Teff,US); 
      
      // prune states in plan
      UpreCompSAcand &= !US;
      
      if (UpreCompSAcand == bddfalse)
	{
	  if (verbosity > 0)
	    cout << " Empty US preCompCand No solution exists! \n";
	  return 0;
	}
      
      // find SAs that with error can lead out of RS
      bdd outgoingSAs = preImage(Terr,!RS);
      // prune U precomponent from outgoing SAs
      bdd UpreCompSA = UpreCompSAcand & !outgoingSAs;
      
      if (verbosity > 0)
	if (UpreCompSA == bddfalse)
	  {
	    cout << " ExtR:";
	    cout.flush();
	  }
      
      while (UpreCompSA == bddfalse)
	{
	  if (verbosity > 0)
	    {
	      itR++;
	      cout << " " << itR;
	      cout.flush();
	    }
	  
	  // extend RS A vars * current state vars
	  bdd RpreCompSA = preImage(Teff,RS);
	  // prune states in RS
	  RpreCompSA &= !RS;
	  
	  if (RpreCompSA == bddfalse)
	    {
	      if (verbosity > 0)
		cout << " Empty RS preComponent No solution exists! \n";
	      return 0;
	    }
	  
	  // add to RSA
	  RSA |= RpreCompSA;
	  // add to RS
	  RS |= bdd_exist(RpreCompSA,Ainfo.Avars);

	  // see if UpreCompSA now is non-empty
	  outgoingSAs = preImage(Terr,!RS);
	  UpreCompSA = UpreCompSAcand & !outgoingSAs;
	}
      
      // succeeded to find a U preComponent
      // update U
      USA |= UpreCompSA;
      US |= bdd_exist(UpreCompSA,Ainfo.Avars);
      
      if (verbosity > 0)
	{
	  cout <<  " size=" << bdd_nodecount(UpreCompSA) 
	       << " Texp=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(UpreCompSA));
	  cout.flush();
	  expandClk.start();	  
	}
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = itU;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  return 1;
}







// IN
//  Teff          : disjunctive partitioning with variables identifying actions
//                  representing non-faulting transitions of actions
//  Terr          : disjunctive partitioning with variables identifying actions
//                  representing faulting transitions of actions
//  init          : initial state (only one)
//  goal          : goal states
//  USA           : reference to universal plan (implicit output)
//  RSA           : reference to recovery plan (implicit output)
// OUT
//  1: success, 0: failure
int faultTolerantSearchOpt(TRel& Teff, TRel& Terr, ActInfo& Ainfo, bdd init, bdd goal, bdd& USA, bdd& RSA) {
  
  
  // overhead
  int overhead = -1;
  
  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize; // ends up being the grand average of all fringes !
  int itU;
  int itR;
  int itUTotal = 0;
  if (verbosity > 0)
    searchClk.start();  
  
  
  //main loop: keep increasing overhead until either success or failure:
  while (1) 
    {
      // next overhead
      overhead++;

      if (verbosity > 0) 
	{
	  cout << "OVERHEAD : " << overhead << "\n";
	  cout << "=============\n\n";
	  cout.flush();
	}

      // init plans
      USA = bddfalse;
      RSA = bddfalse;
      
      // reached states at level 0
      bdd US = goal;
      bdd RS = goal;

      // statistics
      int itU = 0;
      int itR = 0;
  
      // try to find a plan with current overhead 
      while (1)
	{
	  // return plan if init covered
	  if ((US & init) != bddfalse)
	    return 1;

	  itU++;
	  if (verbosity > 0)
	    {
	      itUTotal++;
	      cout <<  " itU=" << itU;
	      cout.flush();
	      expandClk.start();	  
	    }
	  
	  // U Precomponent encoded in act*current bdd vars
	  bdd UpreCompSAcand = preImage(Teff,US); 
	  
	  // prune states in plan
	  UpreCompSAcand &= !US;
	  
	  if (UpreCompSAcand == bddfalse)
	    {
	      if (verbosity > 0)
		cout << " Empty US preCompCand No solution exists! \n";
	      return 0;
	    }
	  
	  // find SAs that with error can lead out of union(US,RS)
	  bdd outgoingSAs = preImage(Terr,!RS);
	  // prune U precomponent from outgoing SAs
	  bdd UpreCompSA = UpreCompSAcand & !outgoingSAs;
	  
	  if (verbosity > 0)
	    if (UpreCompSA == bddfalse)
	      {
		cout << " ExtR:";
		cout.flush();
	      }

	  while ( (UpreCompSA == bddfalse) && (itR - itU < overhead - 1) )
	    {
	      itR++;
	      if (verbosity > 0)
		{
		  cout << " " << itR;
		  cout.flush();
		}
	      
	      // extend RS A vars * current state vars
	      bdd RpreCompSA = preImage(Teff,RS);
	      // prune states in RS
	      RpreCompSA &= !RS;
	      
	      if (RpreCompSA == bddfalse)
		{
		  if (verbosity > 0)
		    cout << " Empty RS preComponent No solution exists! \n";
		  return 0;
		}
	      
	      // add to RSA
	      RSA |= RpreCompSA;
	      // add to RS
	      RS |= bdd_exist(RpreCompSA,Ainfo.Avars);
	      
	      // see if UpreCompSA now is non-empty
	      outgoingSAs = preImage(Terr,!RS);
	      UpreCompSA = UpreCompSAcand & !outgoingSAs;
	    }

	  if ( UpreCompSA == bddfalse)
	    {
	      // no plan with this overhead brak out
	      if (verbosity > 0)
		{
		  cout << " No plan with overhead " << overhead << " !\n";
		  cout.flush();
		}	      
	      break;
	    }
	  else
	    {	  
	      // succeeded to find a U preComponent
	      // update U
	      USA |= UpreCompSA;
	      US |= bdd_exist(UpreCompSA,Ainfo.Avars);
	      
	      if (verbosity > 0)
		{
		  cout <<  " size=" << bdd_nodecount(UpreCompSA) 
		       << " Texp=" << expandClk.stop() << endl;
		  fringeSize.push_back(bdd_nodecount(UpreCompSA));
		  cout.flush();
		  expandClk.start();	  
		}
	    }
	}
      

    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = itUTotal;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  return 1;
}














///////////////////////////////////////////////////////////
//
// Strategy I
//
// Guided 1-fault tolerant planning
// algorithm number
// 
// 1: Uses state-set branching to guide both USA and RSA 
//    towards the initial states
// 2: Uses state-set branching to guide the USA towards 
//    the initial states and Hamming distance to guide 
//    RSA towards the fault states of USA
///////////////////////////////////////////////////////////




// 1
// IN
//  Teff          : disjunctive branching partitioning with variables identifying actions
//                  representing non-faulting transitions of actions
//  Terr          : ordinary disjunctive partitioning with variables identifying actions
//                  representing faulting transitions of actions
//  init          : initial state (max one)
//  goal          : goal states
//  USA           : reference to universal plan (implicit output)
//  RSA           : reference to recovery plan (implicit output)
//  hGoal         : h-value of goal states (we assume they all ar
//  u             : upperLimit of nodes in the recovery search queue
// OUT
//  1: success, 0: failure
int guidedFaultTolerantSearch1(TRel& Teff, TRel& Terr, ActInfo& Ainfo, bdd init, bdd goal, 
			       bdd& USA, bdd& RSA, int hGoal,int u) {


  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize;
  int itU = 0;
  int itR = 0;
  if (verbosity > 0)
    searchClk.start();  

  
  // init USA 
  // ========
  USA = bddfalse;
  bdd US = goal;
  
  
  // init RSA and RS search queue
  // ============================
  // obs: ties in the f-value of the search queue are broken according
  // lowest g-value !
  // this ensures that all promissing nodes at weak preimage level i
  // are expanded before any nodes at level i+1 when computing a SC 
  // precomponent (obs: search queue contains state-action pairs)
  
  RSA = bddfalse;
  bdd RS = goal;
  
  ghSCQueue Q(u,1.0,1.0);
  if (verbosity > 0)
    cout << "Inserting goal children on Recovery search queue: ";

  insertPrunedChildren(Teff,Q,goal,0,hGoal,goal);
  
  if (verbosity > 0)
    cout << endl;
  
  
  
     
  //////////////////////////////////// 
  // find initial (q) and (qPart) 
  // backwards frontier of US
  //////////////////////////////////// 
  if (verbosity > 0)
    cout << "Initializing q and qPart: [";
  
  bdd q = bddfalse;
  map<int,bdd> qPart; 
  
  // go through all partitions of Teff
  for (int i = 0; i < Teff.p.size(); i++)
    {
      // 1) compute partion preImage and prune goal states
	bdd sa = partitionPreImage(Teff,goal,i) & !goal;
	
	// 2) add sa to q and qPart if nonempty
	if (sa == bddfalse)
	  {
	    // sa should not be added, but write a dot to tell that a child was processed
	    if (verbosity > 0)
	      {
		cout << ".";	   
		cout.flush();
	      }
	  }
	else
	  {
	    // write sa summary
	    if (verbosity > 0)
	      {
		if (Teff.p[i].dh > 0)
		  cout << "+";
		else if (Teff.p[i].dh < 0)
		  cout << "-";
		else
		  cout << "0";		      
		cout.flush();
	      }
	    // add sa to q and qPart 
	    // OBS: in ICAPS and IJCAI papers I assume that delta h is given in
	    // forward direction !!!
	    q |= sa;
	    if (qPart.count(hGoal + Teff.p[i].dh) == 1)
	      qPart[hGoal + Teff.p[i].dh] |= sa;	      
	    else
	      qPart[hGoal + Teff.p[i].dh] = sa;	      
	  }
    }    
  if (verbosity > 0)
    cout << "]\n";  

  
    
  //////////////////////////////////// 
  // Main Loop
  //////////////////////////////////// 
  while ((US & init) != init)
    {
      // exit if q is empty
      if (q == bddfalse)
	{
	  if (verbosity > 0)
	    cout << "Empty frontier of 1-fault tolerant plan.\n";
	  return 0;
	}


      if (verbosity > 0)
	{
	  itU++;
	  cout <<  " itU=" << itU;
	  cout.flush();
	  expandClk.start();	  
	}

    
      //////////////////////////////////// 
      // 1) find subset p of q that can be
      //    recovered by extending the 
      //    recovery plan
      //////////////////////////////////// 
      
      
      // initially p equals q pruned for sa's leading out of RS
      bdd p = q & !preImage(Terr,!RS);
      
      if (verbosity > 0)
	if (p == bddfalse)
	  {
	    cout << " ExtR:";
	    cout.flush();
	  }
      
      // extend the recovery plan 
      while (p == bddfalse)
	{
	  if (verbosity > 0)
	    {
	      itR++;
	    }
	  
	  // if recovery search queue is empty, no extension is possible 
	  // return with failure
	  if (Q.empty()) 
	    {
	      if (verbosity > 0)
		cout << " recovery search queue us empty.\n";
	      return 0;
	    }
	  
	  
	  // 1) pop top node (parent node) of the recovery search queue	  
	  double f; int g; int h;
	  bdd parentSA = Q.popTop(f,g,h);

	  // 2) prune parent for states already covered by the recovery plan
	  parentSA &= !RS; 
       
	  // 3) if parent is non-empty, add it to recovery plan
          //    and compute children
	  if (parentSA != bddfalse)
	    {
	      bdd parentS = bdd_exist(parentSA,Ainfo.Avars);
	      RSA |= parentSA;
	      RS |= parentS;      

	      if (verbosity > 0)
		cout << " (#=" << itR << ",size=" << bdd_nodecount(parentSA) << ",f=" << f << ",g=" << g << ",h=" << h << ",";
	  
	      insertPrunedChildren(Teff,Q,parentS,g,h,RS);
	      
	      if (verbosity > 0)
		cout << ")";
	    }
	  else
	    // write a dot to indicate that a node was popped
	    if (verbosity > 0)
	      cout << " (.)";
	    	  
	  // 4) see if the set of recovered states p now is non-empty
	  p = q & !preImage(Terr,!RS);
	}



      //////////////////////////////////// 
      // succeeded to find a non-empty
      // extension of the 1-fault tolerant plan
      // 2) find the frontier component with
      //    least h-value with states overlapping
      //    with p and add these states to
      //    the 1-fault tolerant plan  
      //////////////////////////////////// 

      // 1)
      bdd sa;   // component to add to plan
      int hMin; // its associated h-value        
      // go through qPart (ascending h-values), compute overlap if found and stop
      for (map<int,bdd>::iterator qParti = qPart.begin(); qParti != qPart.end(); ++qParti)
	if ( (sa = qParti->second & p) != bddfalse )
	  {
	    hMin = qParti->first;
	    break;
	  }

      // 2) add sa to USA
      bdd sas = bdd_exist(sa,Ainfo.Avars);
      USA |= sa;
      US |= sas;
      
      // 3) update q and qPart 
      q &= !sas;
      qPart[hMin] &= !sas; // obs. we don;t have to remove states from any other partition of qPart
                           // since the state does not exist with two different h-values


      if (verbosity > 0)
	cout << " [";  

      // go through all partitions of Teff
      for (int i = 0; i < Teff.p.size(); i++)
	{
	  // 1) compute partion preImage and prune for states already in US
	  bdd saC = partitionPreImage(Teff,sas,i) & !US;
	  
	  // 2) add saC to q and qPart if nonempty
	  if (saC == bddfalse)
	    {
	      // saC should not be added, but write a dot to tell that a child was processed
	      if (verbosity > 0)
		{
		  cout << ".";	   
		  cout.flush();
		}
	    }
	  else
	    {
	      // write sa summary
	      if (verbosity > 0)
		{
		  if (Teff.p[i].dh > 0)
		    cout << "+";
		  else if (Teff.p[i].dh < 0)
		    cout << "-";
		  else
		    cout << "0";		      
		  cout.flush();
		}
	      // add saC to q and qPart 
	      // OBS: in ICAPS and IJCAI papers I assume that delta h is given in
	      // forward direction !!!
	      q |= saC;
	      if (qPart.count(hMin + Teff.p[i].dh) == 1)
		qPart[hMin + Teff.p[i].dh] |= saC;	      
	      else
		qPart[hMin + Teff.p[i].dh] = saC;	      
	    }
	}    
      
      if (verbosity > 0)
	{
	  cout <<  "] hMin=" << hMin << " size=" << bdd_nodecount(sa) 
	       << " Texp=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(sa));
	  cout.flush();
	  expandClk.start();	  
	}       
    }
      

  // found a solution or solution trivial zero sized
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = itU;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  return 1;
}




















//
// Strategy II
//
// IN
//  Teff          : disjunctive branching partitioning with variables identifying actions
//                  representing non-faulting transitions of actions
//  Terr          : ordinary disjunctive partitioning with variables identifying actions
//                  representing faulting transitions of actions
//  init          : initial state (max one)
//  goal          : goal states
//  USA           : reference to universal plan (implicit output)
//  RSA           : reference to recovery plan (implicit output)
//  hGoal         : h-value of goal states (we assume they all ar
//  u             : upperLimit of nodes in the recovery search queue
// OUT
//  1: success, 0: failure
int guidedFaultTolerantSearch2(TRel& Teff, TRel& Terr, ActInfo& Ainfo, bdd init, bdd goal, 
			       bdd& USA, bdd& RSA, int hGoal,int u) {


  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize;
  int itU = 0;
  int itR = 0;
  if (verbosity > 0)
    searchClk.start();  

  
  // init USA 
  // ========
  USA = bddfalse;
  bdd US = goal;
  map<int,bdd> USpart; 
  USpart[hGoal] = goal;	      

  
  
  // init RSA 
  // =========  
  RSA = bddfalse;
  bdd RS = goal;
  
  
    
  //////////////////////////////////// 
  // Main Loop
  //////////////////////////////////// 
  while ((US & init) != init)
    {


      if (verbosity > 0)
	{
	  itU++;
	  cout <<  " itU=" << itU;
	  cout.flush();
	  expandClk.start();	  
	}

    
      // 1) Expand the backward frontier of USA (USfrontier)
      
      map<int,bdd> USfrontier;
      computeUSfrontier(Teff,USpart,US,USfrontier);

      // 2) Extend the USA precomponent candidate with SA
      //    with ascending h-values. 
      //    Extend RSA guided toward current error states
      //    (the number of extensions equal to the minimum 
      //     hamming distance to current error states) after
      //    each extension of the candidate


      bdd USAPreCompCandSA  = bddfalse;
      bdd USAPreCompCandS   = bddfalse;
      bdd USAPreCompCandSerr = bddfalse;
      bdd prunedCandidate = bddfalse; // we might have an empty frontier ! 
      map<int,bdd>::iterator maxPart = USfrontier.end();
      for (map<int,bdd>::iterator ibi = USfrontier.begin(); 
	   ibi != USfrontier.end(); ++ibi)
	{
	  if (verbosity > 0)
	    cout << " USCand+=" << ibi->first;

          USAPreCompCandSA |= ibi->second;
	  USAPreCompCandS |= bdd_exist(ibi->second,Ainfo.Avars);
	  USAPreCompCandSerr |= image(Terr,USAPreCompCandS);
	  
	  extendRSA2(Teff,Ainfo,RSA,RS,USAPreCompCandSerr);
	  
	  // prune uncovered SAs
	  prunedCandidate = USAPreCompCandSA & !preImage(Terr,!RS);
	  
	  if (prunedCandidate != bddfalse)
	    {
	      maxPart = ibi;
	      ++maxPart;
	      break; // we don't want to consider further partitions
	    }
	}
      
      // if the pruned candidate is empty continue extending RSA
      // until either a fixed point of RSA (including having covered all error states) 
      // is found or a non-empty pruned candidate
      bdd OldRSA = !RSA;
      while (prunedCandidate == bddfalse && OldRSA != RSA)
	{
	  extendRSA2(Teff,Ainfo,RSA,RS,USAPreCompCandSerr);
	  prunedCandidate = USAPreCompCandSA & !preImage(Terr,!RS);	  
	}
      
      if (prunedCandidate == bddfalse)
	{
	  // no recoverable preimage exists
	  cout << "No pruned US partition exists!\n";
	  return 0;
	}
      else
	{
	  // add fraction of each partition overlaping with the 
          // pruned candidate
	  for (map<int,bdd>::iterator ibi = USfrontier.begin(); 
	       ibi != maxPart; ++ibi)
	    if (USpart.count(ibi->first) == 1)
	      USpart[ibi->first] |= bdd_exist(prunedCandidate & ibi->second,Ainfo.Avars);
	    else
	      USpart[ibi->first] = bdd_exist(prunedCandidate & ibi->second,Ainfo.Avars);
      
	  USA |= prunedCandidate;
	  US |= bdd_exist(prunedCandidate,Ainfo.Avars);
      
	  if (verbosity > 0)
	    {
	      cout << " size=" << bdd_nodecount(prunedCandidate) 
	       << " Texp=" << expandClk.stop() << endl;
	      fringeSize.push_back(bdd_nodecount(prunedCandidate));
	      cout.flush();
	      expandClk.start();
	    }
	}
    }

  
  // found a solution or solution trivial zero sized
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = itU;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  return 1;
}




/*

//
// Strategy III
//
// IN
//  Teff          : disjunctive branching partitioning with variables identifying actions
//                  representing non-faulting transitions of actions
//  Terr          : ordinary disjunctive partitioning with variables identifying actions
//                  representing faulting transitions of actions
//  init          : initial state (max one)
//  goal          : goal states
//  USA           : reference to universal plan (implicit output)
//  RSA           : reference to recovery plan (implicit output)
//  hGoal         : h-value of goal states (we assume they all ar
//  u             : upperLimit of nodes in the recovery search queue
// OUT
//  1: success, 0: failure
int guidedFaultTolerantSearch3(TRel& Teff, TRel& Terr, ActInfo& Ainfo, bdd init, bdd goal, 
			       bdd& USA, bdd& RSA, int hGoal,int u) {


  // statistics
  timer expandClk,searchClk;
  vector<int> fringeSize;
  int itU = 0;
  int itR = 0;
  if (verbosity > 0)
    searchClk.start();  


  // expand decision clock
  timer lastExpandClk;
  double TlastExpand = 0.01;

  
  // init USA 
  // ========
  USA = bddfalse;
  bdd US = goal;
  map<int,bdd> USpart; 
  USpart[hGoal] = goal;	      

  
  
  // init RSA 
  // =========  
  RSA = bddfalse;
  bdd RS = goal;
  
  
    
  //////////////////////////////////// 
  // Main Loop
  //////////////////////////////////// 
  while ((US & init) != init)
    {


      if (verbosity > 0)
	{
	  itU++;
	  cout <<  " itU=" << itU;
	  cout.flush();
	  expandClk.start();	  
	}

      // start expansion clock
      lastExpandClk.start();
    
      // 1) Expand the backward frontier of USA (USfrontier)
      
      map<int,bdd> USfrontier;
      computeUSfrontier(Teff,USpart,US,USfrontier);

      // 2) Extend the USA precomponent candidate with SA
      //    with ascending h-values. 
      //    Extend RSA blind until
      //    a) Candidate has a covered subset 
      //       (prune R to only cover this subset)
      //    b) Texpand > TlastExp in which case the next 
      //       partition is addded to the candidate
      //    If the complete frontier candidate is not covered
      //    R is expanded blindly

      bdd USAPreCompCandSA  = bddfalse;
      bdd USAPreCompCandS   = bddfalse;
      bdd prunedCandidate = bddfalse; // we might have an empty frontier !
                                     
      map<int,bdd>::iterator maxPart = USfrontier.end();
      for (map<int,bdd>::iterator ibi = USfrontier.begin(); 
	   ibi != USfrontier.end(); ++ibi)
	{
	  if (verbosity > 0)
	    cout << " USCand+=" << ibi->first;

          USAPreCompCandSA |= ibi->second;
	  USAPreCompCandS |= bdd_exist(ibi->second,Ainfo.Avars);
	  
	  prunedCandidate = extendRSA3timed(Teff,Terr,Ainfo,RSA,RS,USAPreCompCandSA,TlastExpand);
	  
	  if (prunedCandidate != bddfalse)
	    {
	      maxPart = ibi;
	      ++maxPart;
	      break; // we don't want to consider further partitions
	    }
	}
      
      // if the pruned candidate is empty continue extending RSA
      // until either a fixed point of RSA (including having covered all error states) 
      // is found or a non-empty pruned candidate
      bdd OldRSA = !RSA;
      while (prunedCandidate == bddfalse && OldRSA != RSA)	
	prunedCandidate = extendRSA3toFixPoint(Teff,Terr,Ainfo,RSA,RS,USAPreCompCandSA);
      
      // no solution exists if candidate is still empty at this point
      if (prunedCandidate == bddfalse)
	{
	  cout << "No pruned US partition exists!\n";
	  return 0;
	}
      else
	{
	  // add fraction of each partition overlaping with the 
          // pruned candidate
	  for (map<int,bdd>::iterator ibi = USfrontier.begin(); 
	       ibi != maxPart; ++ibi)
	    if (USpart.count(ibi->first) == 1)
	      USpart[ibi->first] |= bdd_exist(prunedCandidate & ibi->second,Ainfo.Avars);
	    else
	      USpart[ibi->first] = bdd_exist(prunedCandidate & ibi->second,Ainfo.Avars);
      
	  USA |= prunedCandidate;
	  US |= bdd_exist(prunedCandidate,Ainfo.Avars);

	  // make a new last expansion estimate
	  TlastExpand = lastExpandClk.stop();
      
	  if (verbosity > 0)
	    {
	      cout << " size=" << bdd_nodecount(prunedCandidate) 
	       << " Texp=" << expandClk.stop() << endl;
	      fringeSize.push_back(bdd_nodecount(prunedCandidate));
	      cout.flush();
	      expandClk.start();
	    }
	}
    }

  
  // found a solution or solution trivial zero sized
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      iterationNum = itU;
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  return 1;
}

*/
