////////////////////////////////////////////////////////
// File  : main.cc   
// Desc. : Main file for BDDSearch
// Author: Rune M. Jensen, CS, CMU 
// Date  : 04/06/02
////////////////////////////////////////////////////////

#include <fstream>
#include <stream.h>
#include <string.h>
#include <stdio.h> 
#include <stdlib.h>
#include <unistd.h>
#include "bddLayout.hpp"
#include "common.hpp"
#include "main.hpp"
#include "timer.hpp"
#include "extNADL.hpp"
#include "pddl.hpp"
#include "numDomain.hpp"
#include "reachInfo.hpp"
#include "balancedInfo.hpp"
#include "partition.hpp"
#include "detSearchAlg.hpp"
#include "nonDetSearchAlg.hpp"
#include "singleState.actions.hpp"
#include "singleState.search.hpp"
#include "symbolicHeuristic.hpp"
#include "bdda.search.hpp"
#include "SA.hpp"
#include "universal.hpp"
#include "checkPlan.hpp"


////////////////////////
// global variables
////////////////////////


// global control
int verbosity;
int timeout;



// statistics
double Ttotal   = -1.0;
double Tparse   = -1.0;
double Tanalyse = -1.0;
double Tinit    = -1.0;
double Trel     = -1.0;
double Tact     = -1.0;

double Tsearch              = -1.0;
double Textract             = -1.0;
double aveFringeSize        = -1.0;
int    maxSearchQueueSize   = -1;
int    iterationNum         = -1;
int    forwardExpansionNum  = -1;
int    backwardExpansionNum = -1;

int    solutionLength       = -1;        
int    solutionSize         = -1; // no nodes in solution BDD for non-det planning        
int    heuristicBDDsize     = -1;
int    totalFormulaSize     = -1;
int    SizeTact             = -1;
int    SizeT                = -1;
int    SizeTbranch          = -1;
int    spaceSize            = -1; // number of BDD vars in state rep.

int main(int argc,char **argv)  {

  // unistd.h
  extern char *optarg;

  // lex/yacc
  extern FILE *yyin;
  
  extern FILE *xxin;
  extern int xxlineno;
  

  /////////////////////////////////////////////////////////////////
  // BDDSearch Parameters
  /////////////////////////////////////////////////////////////////

  // I/O
  verbosity      = 1;
  timeout        = 500; 
  inputFormatType inputFormat      = it_none;
  char domainName[MAXNAMELENGTH]   = "";
  char problemName[MAXNAMELENGTH]  = "";
  char solutionName[MAXNAMELENGTH] = "";
  char expName[MAXNAMELENGTH]      = "";

  // BuDDy
  dynReorderType dynReorder = dt_none;
  int initBddNodes        = 1000000;
  int initCache           = 100000;
  int bddVarNum = -1; // don't worry it will be instantiated;

  // Partitioning
  int partLimit           = 5000;

  
  // Search 
  algorithmType algorithm = at_none;
  heuristicType heuristic = ht_none;
  bool frontierSetSimpl   = false; // what we mean is frontier set simplification with Courdert's trick.
                                   // of course in blind search we subtract already reached states

  // SetA*, A* and FminA* specific
  double gWeight          = 1.0;
  double hWeight          = 1.0;
  double squizeFactor     = 1.0; // no squize
  int upperBound          = 5000; 


  /////////////////////////////////////////////////////////////////
  // Initialize timers
  /////////////////////////////////////////////////////////////////
  timer totalClk,parseClk,analyseClk,initClk,relClk,searchClk;

  // always measure total time
  resettimers();
  totalClk.start();

  if (verbosity > 0)
    {      
      parseClk.start();
    }




  /////////////////////////////////////////////////////////////////
  // Parse Options
  /////////////////////////////////////////////////////////////////

  if (argc == 1)
    {
      cout << endl;
      printUsage();
      exit(-1);
    }
    

  int c;
  while ( (c = getopt(argc,argv,"hv:i:d:p:o:l:z:e:n:c:r:t:a:g:x:y:u:f")) != -1)
    switch (c) 
      {
      case 'h':
	cout << endl;
	printUsage();
	exit(-1);
	break;
	
      case 'v':
	verbosity = atoi(optarg);	
	break;

      case 'i':
	if (string(optarg) == "PDDL") inputFormat = it_PDDL;
	else if (string(optarg) == "extNADL") inputFormat = it_extNADL;
	else 
	  {
	    cout << optarg << " is an invalid argument of option -i\n";
	    exit(-1);
	  }		     
	break;
	
      case 'd':
	strcpy(domainName,optarg);
	break;
	
      case 'p':
	strcpy(problemName,optarg);
	break;
	
      case 'o':
	strcpy(solutionName,optarg);
	break;

      case 'e':
	strcpy(expName,optarg);
	break;
	
      case 'n':
	initBddNodes = atoi(optarg);	
	break;

      case 'l':
	timeout = atoi(optarg);	
	break;


      case 'c':
	initCache = atoi(optarg);	
	break;

      case 'r':
	if (string(optarg) == "Off") dynReorder = dt_none;
	if (string(optarg) == "Win2ite") dynReorder = dt_win2ite;
	else 
	  {
	    cout <<  optarg << " is an invalid argument of option -r\n";
	    exit(-1);
	  }		     	
	break;
	
      case 't':
	partLimit = atoi(optarg);
	break;

      case 'a':
        // Deterministic alg.
	if      (string(optarg) == "Bidir")               algorithm = at_bidir;
	else if (string(optarg) == "Forward")             algorithm = at_forward;
	else if (string(optarg) == "Backward")            algorithm = at_backward;
	else if (string(optarg) == "SetAstar")            algorithm = at_setAstar;
	else if (string(optarg) == "Astar")               algorithm = at_Astar;
	else if (string(optarg) == "BDDAstar")            algorithm = at_BDDAstar;
	else if (string(optarg) == "PureBDDAstar")        algorithm = at_pureBDDAstar;
	else if (string(optarg) == "PubBDDAstar")         algorithm = at_PubBDDAstar;
	else if (string(optarg) == "PubPureBDDAstar")     algorithm = at_PubPureBDDAstar;
	else if (string(optarg) == "FminAstar")           algorithm = at_FminAstar;
	else if (string(optarg) == "PureFminAstar")       algorithm = at_pureFminAstar;
        // non-Deterministic alg.
	else if (string(optarg) == "Strong")              algorithm = at_strong;
	else if (string(optarg) == "StrongH")             algorithm = at_strongH;
	else if (string(optarg) == "StrongCyclic")        algorithm = at_strongCyclic;
	else if (string(optarg) == "StrongCyclicH")       algorithm = at_strongCyclicH;
	else if (string(optarg) == "Weak")                algorithm = at_weak;
	else if (string(optarg) == "WeakH")               algorithm = at_weakH;
	else if (string(optarg) == "FaultTolerant")       algorithm = at_faultTolerant;         
	else if (string(optarg) == "FaultTolerantOpt")    algorithm = at_faultTolerantOpt;         
	else if (string(optarg) == "GuidedFaultTolerant") algorithm = at_guidedFaultTolerant;
	else if (string(optarg) == "StrongCyclicAdv")     algorithm = at_strongCyclicAdv;         
	else 
	  {
	    cout << optarg << " is an invalid argument of option -a\n";
	    exit(-1);
	  }		     
	break;

      case 'g':
	if (string(optarg)      == "MinHamming") heuristic = ht_minHamming;
	else if (string(optarg) == "Manhattan")  heuristic = ht_manhattan;
	else if (string(optarg) == "HSPr")       heuristic = ht_HSPr;
	else 
	  {
	    cout << optarg << " is an invalid argument of option -g\n";
	    exit(-1);
	  }		     
	break;

      case 'x':
	gWeight = atof(optarg);
	break;

      case 'y':
	hWeight = atof(optarg);
	break;

      case 'z':
	squizeFactor = atof(optarg);	
	break;

      case 'u':
	upperBound = atoi(optarg);
	break;

      case 'f':
	frontierSetSimpl = true;	
	break;

      default:
	cout << "option -h prints usage\n";
	exit(-1);
	break;
      }



 
  /////////////////////////////////////////////////////////////////
  // Make solution file name
  /////////////////////////////////////////////////////////////////
  
  if (string(solutionName) == "")
    {
      int i;
      // make solution filename 
	if (inputFormat == it_PDDL)
	  {
	    for (i=0; problemName[i] != '.' && problemName[i] != 0; i++)
	      solutionName[i] = problemName[i];
	  }
	else
	  {
	    for (i=0; domainName[i] != '.' && domainName[i] != 0; i++)
	      solutionName[i] = domainName[i];
	  }
	solutionName[i] = 0;
	strcat(solutionName,".sol");
    }
  

  

  /////////////////////////////////////////////////////////////////
  // Read input files
  /////////////////////////////////////////////////////////////////


  extern extNADLproblem extNADLprob;
  extern PDDLdomain  PDDLdom; 
  extern PDDLproblem PDDLprob;

  switch (inputFormat) 
    {
    case it_extNADL :
      
      // parse extNADL problem
      yyin = fopen(domainName,"r");
      if (yyin == NULL) 
	{
	  cout << "main.cc main : Cannot open \"" << domainName << "\"\n";
	  exit(1);
	}
      
      if (yyparse()) exit(-1); 
      
         
      if (verbosity > 2)
	extNADLprob.print();
      
      break;
      
    
    
    case it_PDDL :
      
      // Parse domain and problem
      if ( (xxin = fopen(domainName,"r")) == NULL)
	{
	  cout << "Cannot open \"" << domainName << "\"\n"; 
	  exit(-1);
	}
      
      xxlineno = 0;
      if (xxparse()) exit(-1); 
    
      if (verbosity > 1) cout << "Parsing problem...\n";
      if ( (xxin = fopen(problemName,"r")) == NULL)
	{
	  cout << "Cannot open \"" << problemName << "\"\n"; 
	  exit(-1);
	}
    
      xxlineno = 0;
      if (xxparse()) exit(-1); 
      

      if (verbosity > 2) 
	{
	  PDDLdom.print();
	  PDDLprob.print();
	}
      break;
      

    default:
      cout << "main.cc main : switch option not covered\nexiting\n";
      exit(-1);
      break;
    }


  if (verbosity > 0)
    {
      Tparse = parseClk.stop();
      cout << "Tparse =" << Tparse << " sec\n";
    }
  






  // choose between non-det or det search
  switch (algorithm)
    {
      ////////////////////////////////
      // non deterministic universal 
      // planning
      ////////////////////////////////
    case at_strong:
    case at_strongH:
    case at_strongCyclic:
    case at_strongCyclicH:
    case at_weak:
    case at_weakH:
    case at_strongCyclicAdv:
      {
	if (verbosity > 0)
	  {
	    initClk.start(); 
	  }
	
	// make OBDD layout
	NADLNonDetbddLayout layout;
	layout.initialize(extNADLprob);
	
	if (verbosity > 2)
	  layout.print();
	
	// initialize buddy package
	bdd_init(initBddNodes,initCache);
	bdd_setvarnum(layout.bddVarNum);
	bdd_setmaxincrease(5000000);
	switch (dynReorder) 
	  {
	  case dt_none:
	    // do nothing
	    break;
	  case dt_win2ite:
	    layout.mkBlocks();
	    bdd_reorder_verbose(2);
	    bdd_autoreorder(BDD_REORDER_WIN2ITE);
	    break;
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }

	
	if (verbosity > 0)
	  {
	    Tinit = initClk.stop();
	    cout << "Tinit =" << Tinit << " sec\n";
	    relClk.start(); 
	  }
	
	// declare disjunctive partitioned transition relation
	TRel Traw;
	TRel T;

	TRel TrawEnv; // used by strong cyclic adversarial 
	TRel Tenv;    // used by strong cyclic adversarial 

	bdd init;
	bdd goal;
	int hGoal;
	ActInfo Ainfo;
	
	Ainfo.initialize(layout);
	init = formula2bdd(layout,extNADLprob.init);
	goal = formula2bdd(layout,extNADLprob.goal);
	
	// optimize disjunctive partitioning according to 
	// algorithm type
	switch (algorithm)
	  {
	  case at_strong:
	  case at_strongCyclic:
	  case at_weak:
	    // make regular disjunctive partitioning
	    NADLNonDetmkT(layout,Traw);
	    // quantify environment actions if any	   
 	    if (layout.envActive)
	      abstractEnvAct(Traw,Ainfo);
	    optimize(Traw,T,partLimit);	    
	    break;

	  case at_strongCyclicAdv:
	    // make regular disjunctive partitioning
	    NADLNonDetmkT(layout,Traw);
	    // quantify environment actions if any	   
	    abstractEnvAct(Traw,Ainfo);
	    optimize(Traw,T,partLimit);

	    // make disjunctive partitioning with environment
            // actions
	    NADLNonDetmkT(layout,TrawEnv);
	    optimize(TrawEnv,Tenv,partLimit);
	    break;
	   
	  case at_strongH:
	  case at_strongCyclicH:	  
	  case at_weakH:
	    if (!extNADLprob.heuOptGoal)
	      {
		cout << "main.cc : Goal heuristic value needed in extNADL domain\n" 
		     << "exiting\n";
		exit(1);
	      }
	    // make disjunctive branching partitioning
	    hGoal = extNADLprob.heuGoal; 
	    NADLNonDetHeumkT(layout,Traw);	    
	    optimizeBranching(Traw,T,partLimit);
	    break;
	    
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }

	
	if (verbosity > 0)
	  {
	    Trel = relClk.stop();
	    SizeT = T.size();
	  }
		
	if (verbosity > 2)
	  {
	    cout << "\nRaw  partitioning\n";
	    Traw.print();
	    cout << "\n\nOptimized partitioning\n";
	    T.print();

	    if (algorithm == at_strongCyclicAdv)
	      {
		cout << "\nRaw partitioning with environment actions\n";
		TrawEnv.print();
		cout << "\n\nOptimized partitioning with environment actions\n";
		Tenv.print();
	      }
	  }
	
	
	// search
	bdd universalPlan;
	int res;

	switch (algorithm)
	  {
	  case at_strong:
	    res = strongSearch(T,Ainfo,init,goal,universalPlan);
	    break;
	  case at_strongCyclic:
	    res = strongCyclicSearch(T,Ainfo,init,goal,universalPlan);	    
	    break;
	  case at_strongCyclicAdv:
	    res = strongCyclicAdversarialSearch(T,Tenv,Ainfo,init,goal,universalPlan);
	    break;
	  case at_weak:
	    res = weakSearch(T,Ainfo,init,goal,universalPlan);	    
	    break;	    
	  case at_strongH:	    
	    res = strongHeuristicSearch(T,Ainfo,init,goal,universalPlan,hGoal,upperBound);
	    break;
	  case at_strongCyclicH:
	    res = strongCyclicHeuristicSearch(T,Ainfo,init,goal,universalPlan,hGoal,upperBound);
	    break;
	  case at_weakH:
	    res = weakHeuristicSearch(T,Ainfo,init,goal,universalPlan,hGoal,upperBound);
	    break;
	    
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }


	if (res)
	  cout << "Solution found" << endl;
	else
	  cout << "No solution exists" << endl;
	
	if (verbosity > 0)
	  solutionSize = bdd_nodecount(universalPlan);


	// debug	
	//SA SP("SP.txt",layout);
	//SP.print(universalPlan);

	//planIsTotal(T,Ainfo,init,goal,universalPlan);
	//planIsTotal(Teff,Ainfo,init,goal,recoveryPlan);
	//faultCovered(Teff,Terr,Ainfo,init,goal,universalPlan,recoveryPlan);

		   
	
      }
      break;
      




      ////////////////////////////////
      // non deterministic 1-fault
      // tolerant planning
      ////////////////////////////////
    case at_faultTolerant:
    case at_faultTolerantOpt:
      {
	if (verbosity > 0)
	  {
	    initClk.start(); 
	  }
	
	// make OBDD layout
	NADLNonDetbddLayout layout;
	layout.initialize(extNADLprob);
	
	if (verbosity > 2)
	  layout.print();
	
	// initialize buddy package
	bdd_init(initBddNodes,initCache);
	bdd_setvarnum(layout.bddVarNum);
	switch (dynReorder) 
	  {
	  case dt_none:
	    // do nothing
	    break;
	  case dt_win2ite:
	    layout.mkBlocks();
	    bdd_reorder_verbose(2);
	    bdd_autoreorder(BDD_REORDER_WIN2ITE);
	    break;
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }
	
	if (verbosity > 0)
	  {
	    Tinit = initClk.stop();
	    cout << "Tinit =" << Tinit << " sec\n";
	    relClk.start(); 
	  }
	
	// make two disjunctive partitioned transition relations
	// 1) one for successful effects, and
	TRel TrawEff;
	TRel Teff;
	NADLNonDetmkT(layout,TrawEff);
	optimize(TrawEff,Teff,partLimit);

	// 2) one for errors
	TRel TrawErr;
	TRel Terr;
	NADLNonDetmkTErr(layout,TrawErr);
	optimize(TrawErr,Terr,partLimit);

	// make goal and init formulas
	bdd init;
	bdd goal;
	ActInfo Ainfo;
	
	Ainfo.initialize(layout);
	init = formula2bdd(layout,extNADLprob.init);
	goal = formula2bdd(layout,extNADLprob.goal);
	
	if (verbosity > 0)
	  {
	    Trel = relClk.stop();
	    SizeT = Teff.size() + Terr.size();
	  }
		
	if (verbosity > 2)
	  {
	    cout << "\nEff  partitioning\n";
	    Teff.print();
	    cout << "\nErr  partitioning\n";
	    Terr.print();
	  }
	
	
	// search
	bdd universalPlan;
	bdd recoveryPlan;
	int res;
	
	switch (algorithm)
	  {
	  case at_faultTolerant:
	    res = faultTolerantSearch(Teff,Terr,Ainfo,init,goal,universalPlan,recoveryPlan);
	    break;
	  case at_faultTolerantOpt:
	    res = faultTolerantSearchOpt(Teff,Terr,Ainfo,init,goal,universalPlan,recoveryPlan);
	    break;	    
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }


	if (res)
	  cout << "Solution found" << endl;
	else
	  cout << "No solution exists" << endl;
	
	//planIsTotal(Teff,Ainfo,init,goal,universalPlan);
	//planIsTotal(Teff,Ainfo,init,goal,recoveryPlan);
	//faultCovered(Teff,Terr,Ainfo,init,goal,universalPlan,recoveryPlan);

	if (verbosity > 0)
	  solutionSize = bdd_nodecount(universalPlan) + bdd_nodecount(recoveryPlan);	
      
      }
      break;
      















      ////////////////////////////////
      // guided 1-fault tolerant 
      // planning
      ////////////////////////////////
    case at_guidedFaultTolerant:
      {
	if (verbosity > 0)
	  {
	    initClk.start(); 
	  }
	
	// make OBDD layout
	NADLNonDetbddLayout layout;
	layout.initialize(extNADLprob);
	
	if (verbosity > 2)
	  layout.print();
	
	// initialize buddy package
	bdd_init(initBddNodes,initCache);
	bdd_setvarnum(layout.bddVarNum);
	switch (dynReorder) 
	  {
	  case dt_none:
	    // do nothing
	    break;

	  case dt_win2ite:
	    layout.mkBlocks();
	    bdd_reorder_verbose(2);
	    bdd_autoreorder(BDD_REORDER_WIN2ITE);
	    break;

	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }
	
	if (verbosity > 0)
	  {
	    Tinit = initClk.stop();
	    cout << "Tinit =" << Tinit << " sec\n";
	    relClk.start(); 
	  }
	
	
	// 1) make a disjunctive branching partitioning for primary
        //    effects 
	TRel TrawEff;
	TRel Teff;	    
	if (!extNADLprob.heuOptGoal)
	  {
	    cout << "main.cc : Goal heuristic value needed in extNADL domain\n" 
		 << "exiting\n";
	    exit(1);
	  }
	int hGoal = extNADLprob.heuGoal;
	if (verbosity > 0) cout << "Making branching partitioning\n";
	NADLNonDetHeumkT(layout,TrawEff);	    
	optimizeBranching(TrawEff,Teff,partLimit);


	// 2) make a regular disjunctive partitioning for secondary 
        //    effects
	TRel TrawErr;
	TRel Terr;
	if (verbosity > 0) cout << "Making error partitioning\n";
	NADLNonDetmkTErr(layout,TrawErr);
	optimize(TrawErr,Terr,partLimit);

	// make goal and init formulas
	bdd init;
	bdd goal;
	ActInfo Ainfo;
	
	Ainfo.initialize(layout);
	init = formula2bdd(layout,extNADLprob.init);
	goal = formula2bdd(layout,extNADLprob.goal);
	
	if (verbosity > 0)
	  {
	    Trel = relClk.stop();
	    SizeT = Teff.size() + Terr.size();
	  }
		
	if (verbosity > 2)
	  {
	    cout << "\nEff  partitioning\n";
	    Teff.print();
	    cout << "\nErr  partitioning\n";
	    Terr.print();
	  }
	
	
	// search
	bdd universalPlan;
	bdd recoveryPlan;
	int res;
	
	switch (algorithm)
	  {
	  case at_guidedFaultTolerant:
	    res = guidedFaultTolerantSearch2(Teff,Terr,Ainfo,init,goal,
					     universalPlan,recoveryPlan,hGoal,upperBound);
	    break;
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;
	  }


	if (res)
	  cout << "Solution found" << endl;
	else
	  cout << "No solution exists" << endl;

	// debug
	//SA USA("USA.txt",layout);
	//SA RSA("RSA.txt",layout);
	//USA.print(universalPlan);
	//RSA.print(recoveryPlan);

	// debug
	//planIsTotal(Teff,Ainfo,init,goal,universalPlan);
	//planIsTotal(Teff,Ainfo,init,goal,recoveryPlan);
	//faultCovered(Teff,Terr,Ainfo,init,goal,universalPlan,recoveryPlan);

	if (verbosity > 0)
	  solutionSize = bdd_nodecount(universalPlan) + bdd_nodecount(recoveryPlan);	
      
      }
      break;
      



















      /////////////////////////////////////////////////////////////////
      // Deterministic search 
      /////////////////////////////////////////////////////////////////      
    default:
      {
	list<string> solution;
	
	
	switch (algorithm) 
	  {
	  case at_bidir:
	  case at_forward:
	  case at_backward:
	  case at_setAstar:    
	  case at_FminAstar:
	  case at_pureFminAstar:
	    {
	      /////////////////////////////////////////////////////////////////
	      // BDD-based search
	      /////////////////////////////////////////////////////////////////
	      
	      
	      if (verbosity > 0)
		analyseClk.start();
	      
	      //////////////////////////
	      // 1) Analyse 
	      //////////////////////////
	      
	      numDomain numDom;
	      
	      reachInfo rInfo;
	      balancedInfo bInfo;
	      PDDLbddLayout PDDLlayout;
	      NADLbddLayout NADLlayout;
	      
	      switch (inputFormat) {
		
	      case it_extNADL:	
		NADLlayout.initialize(extNADLprob);
		
		if (verbosity > 2)
		  NADLlayout.print();
		break;
		
		
	      case it_PDDL:	
		numDom.initialize(PDDLdom,PDDLprob);
		
		rInfo.analyse(numDom,squizeFactor); 
		bInfo.analyse(rInfo,numDom);
		PDDLlayout.initialize(rInfo,bInfo,numDom);
		
		if (verbosity > 2)
		  {
		    PDDLdom.print();
		    PDDLprob.print();
		    numDom.print();
		    rInfo.print(numDom);
		    bInfo.print(numDom);
		    PDDLlayout.print(numDom);
		  }
		break;
		
	      default:
		cout << "main.cc main : switch option not covered\nexiting\n";
		exit(-1);
		break;		  
	      }
	      
           
	      
	      if (verbosity > 0)
		{
		  Tanalyse = analyseClk.stop();
		  cout << "Tanalyse =" << Tanalyse << " sec\n";
		  initClk.start(); 
		}
	      
	
      
	
	      //////////////////////////
	      // 2) Init BuDDy
	      //////////////////////////

	      switch (inputFormat) 
		{
		case it_extNADL:
		  bddVarNum = NADLlayout.bddVarNum;
		  break;
	    
		case it_PDDL:
		  bddVarNum = PDDLlayout.bddVarNum;
		  break;
		  
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		}	
	      
	      // initialize bdd package
	      bdd_init(initBddNodes,initCache);
	      bdd_setvarnum(bddVarNum);
	      
	      if (verbosity > 0)
		{
		  Tinit = initClk.stop();
		  cout << "Tinit =" << Tinit << " sec\n";
		  relClk.start(); 
		}
	      


	      //////////////////////////
	      // 3)
	      // Transition Relation
	      // +
	      //////////////////////////
	      
	      TRel Tact;
	      TRel Tbranch;
	      TRel T;
	      bdd init;
	      bdd goal;
	      int hInit;
	      
	      switch (inputFormat) 
		{
		case it_extNADL:
		  
		  init = formula2bdd(NADLlayout,extNADLprob.init);
		  goal = formula2bdd(NADLlayout,extNADLprob.goal);
		  
		  switch (algorithm) 
		    {
		    case at_bidir :
		    case at_forward :
		    case at_backward :
		      // make NADL partitions for BLIND search
		      NADLmkTact(NADLlayout,Tact);
		      optimize(Tact,T,partLimit);      
		      
		      if (verbosity > 0)
			{
			  SizeTact = Tact.size();
			  SizeT = T.size();
			}
		      
		      if (verbosity > 2)
			{
			  cout << "\nAction partitioning for solution extraction\n";
			  Tact.print();
			  cout << "\n\nOptimized search partitioning\n";
			  T.print();
			}
		      break;
		      
	  
		    case at_setAstar:
		      
		      switch (heuristic) 
			{
			case ht_minHamming:
			  cout << "main.cc : no support for Buddymod\n";
			  exit(1);
			  //hInit = bdd_lowerboundH(init,goal);
			  // make NADL partitions for HEURISTIC search    
			  NADLmkTact(NADLlayout,Tact);
			  // doing MinHamming for now
			  //splitTactPartitions(Tact,Tbranch,goal);
			  optimizeBranching(Tbranch,T,partLimit);     
		    
			  if (verbosity > 0)
			    {
			      SizeTact = Tact.size();
			      SizeT = T.size();
			    }
			  
			  if (verbosity > 2)
			    {
			      cout << "\nAction partitioning for solution extraction\n";
			      Tact.print();
			      cout << "\n\nSearch branching partitioning\n";
			      Tbranch.print();
			      cout << "\n\nOptimized search branching partitioning\n";
			      T.print();
			    }
			  break;


			case ht_manhattan:
			  hInit = ((extNADLprob.env.front()).actions.front()).trans.front().dg;
			  NADLmkTact(NADLlayout,Tact);
			  // since each actions in the n^2 puzzle has dh encoded 
			  // as action value
			  // make the correction: dh = dg, dg = 1
			  cout << "Warning !!! making 15 puzzle correction: Does this apply to your exp. ?\n";
			  modifyTact15puzzle(Tact);
			  // optimize the branching partition
			  optimizeBranching(Tact,T,partLimit);     
			  
			  if (verbosity > 0)
			    {
			      SizeTact = Tact.size();
			      SizeT = T.size();
			    }
			  
			  if (verbosity > 2)
			    {
			      cout << "\nAction partitioning for solution extraction\n";
			      Tact.print();
			      cout << "\n\nOptimized search branching partitioning\n";
			      T.print();
			    }
			  break;
			  
			default:
			  cout << "main.cc main : switch option not covered\nexiting\n";
			  exit(-1);
			  break;			
			}		
		      break; 





		    case at_FminAstar:
		      switch (heuristic) 
			{
			case ht_minHamming:
			  cout << "main.cc no support for BuDDyMod\n";
			  exit(1);
			  hInit = 1; //bdd_lowerboundH(init,goal);
			  // make NADL partitions for HEURISTIC search    
			  NADLmkTact(NADLlayout,Tact);		    
			  //splitTactPartitions(Tact,Tbranch,goal);
			  NADLFminModify(Tbranch,gWeight,hWeight);
			  optimizeBranching(Tbranch,T,partLimit);     
			  
			  if (verbosity > 0)
			    {
			      SizeTact = Tact.size();
			      SizeT = T.size();
			    }
			  
			  if (verbosity > 2)
			    {
			      cout << "\nAction partitioning for solution extraction\n";
			      Tact.print();
			      cout << "\n\nSearch branching partitioning\n";
			      Tbranch.print();
			      cout << "\n\nOptimized search branching partitioning\n";
			      T.print();
			    }
			  break;


			case ht_manhattan:
			  hInit = ((extNADLprob.env.front()).actions.front()).trans.front().dg;
			  // make NADL partitions for n^2 -1 puzzle search    
			  NADLmkTact(NADLlayout,Tact);
			  // since each action in the puzzle has dh encoded as action value
			  // make the correction: dh = dg, dg = 1
			  modifyTact15puzzle(Tact);
			  NADLFminModify(Tact,gWeight,hWeight);
			  // optimize the branching partition
			  optimizeBranching(Tact,T,partLimit);     
			  
			  if (verbosity > 0)
			    {
			      SizeTact = Tact.size();
			      SizeT = T.size();
			    }
			  
			  if (verbosity > 2)
			    {
			      cout << "\nAction partitioning for solution extraction\n";
			      Tact.print();
			      cout << "\n\nOptimized search branching partitioning\n";
			      T.print();
			    }
			  break;
		    
			default:
			  cout << "main.cc main : switch option not covered\nexiting\n";
			  exit(-1);
			  break;			
			}		
		      break; 

		


		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;			
		    }
		  break;
	
	
	    
		case it_PDDL:
	    
	    
		  switch (algorithm) 
		    {	      
		    case at_bidir :
		    case at_forward :
		    case at_backward :
		      init = init2bdd(PDDLlayout,numDom,numDom.init);
		      goal = goal2bdd(PDDLlayout,numDom,numDom.goal);
		      // make PDDL partitions for BLIND search
		      // Tbranch just a dummy parameter. It is not used for blind search
		      PDDLmkTact(PDDLlayout,rInfo,numDom,Tact,ht_none); 
		      optimize(Tact,T,partLimit);      
		      
		      if (verbosity > 0)
			{
			  SizeTact = Tact.size();
			  SizeT = T.size();
			}
		      
		      if (verbosity > 2)
			{
			  cout << "\nAction partitioning for solution extraction\n";
			  Tact.print();
			  cout << "\n\nOptimized search partitioning\n";
			  T.print();
			}
		      break;
	  	  
		    case at_setAstar :
		      switch (heuristic)
			{
			case ht_HSPr:
			  init = init2bdd(PDDLlayout,numDom,numDom.init);
			  // we use init2bdd also for the goal since it is
			  // assumed to be a single state
			  goal = init2bdd(PDDLlayout,numDom,numDom.goal);
			  
			  // make PDDL partitions for HEURISTIC search    
			  hInit = hHSPr(PDDLlayout,rInfo,numDom.goal); 
			  
			  PDDLmkTact(PDDLlayout,rInfo,numDom,Tact,heuristic);
			  optimizeBranching(Tact,T,partLimit);      
			  
			  if (verbosity > 0)
			    {
			      SizeTact = Tact.size();
			      SizeT = T.size();
			    }
			  
			  if (verbosity > 2)
			    {
			      cout << "\nAction partitioning for solution extraction\n";
			      Tact.print();
			      cout << "\n\nOptimized search branching partitioning\n";
			      T.print();
			    }
			  break;

			case ht_minHamming:
			  cout << "no support for BuDDy MOD\n";
			  exit(1);
			  // make PDDL partitions for HEURISTIC search    
			  init = init2bdd(PDDLlayout,numDom,numDom.init);
			  goal = goal2bdd(PDDLlayout,numDom,numDom.goal);
			  //hInit = bdd_lowerboundH(init,goal);
			  
			  PDDLmkTact(PDDLlayout,rInfo,numDom,Tact,heuristic);
			  //splitTactPartitions(Tact,Tbranch,goal);
			  optimizeBranching(Tbranch,T,partLimit);      

			  if (verbosity > 0)
			    {
			      SizeTact = Tact.size();
			      SizeT = T.size();
			    }
			  
			  if (verbosity > 2)
			    {
			      cout << "\nAction partitioning for solution extraction\n";
			      Tact.print();
			      cout << "\n\nSearch branching partitioning\n";
			      Tbranch.print();
			      cout << "\n\nOptimized search branching partitioning\n";
			      T.print();
			    }
			  break;


		    		    
			default:
			  cout << "main.cc main : switch option not covered\nexiting\n";
			  exit(-1);
			  break;	
			}
		      break;
		

		    case at_FminAstar:
		      init = init2bdd(PDDLlayout,numDom,numDom.init);
		      // we use init2bdd also for the goal since it is
		      // assumed to be a single state
		      goal = init2bdd(PDDLlayout,numDom,numDom.goal);
		      
		      hInit = hHSPr(PDDLlayout,rInfo,numDom.goal); 
		      
		      PDDLmkTact(PDDLlayout,rInfo,numDom,Tact,heuristic);
		      FminModify(Tact,gWeight,hWeight);
		      optimizeBranching(Tact,T,partLimit);      
		      

		      if (verbosity > 0)
			{
			  SizeTact = Tact.size();
			  SizeT = T.size();
			}

		      if (verbosity > 2)
			{
			  cout << "\nAction partitioning for solution extraction\n";
			  Tact.print();
			  cout << "\n\nOptimized search branching partitioning\n";
			  T.print();
			}
		      break;
		      
		    case at_pureFminAstar:
		      init = init2bdd(PDDLlayout,numDom,numDom.init);
		      // we use init2bdd also for the goal since it is
		      // assumed to be a single state
		      goal = init2bdd(PDDLlayout,numDom,numDom.goal);
		      
		      hInit = hHSPr(PDDLlayout,rInfo,numDom.goal); 
		      
		      PDDLmkTact(PDDLlayout,rInfo,numDom,Tact,heuristic);
		      pureFminModify(Tact);
		      optimizeBranching(Tact,T,partLimit);      
		      
		      if (verbosity > 0)
			{
			  SizeTact = Tact.size();
			  SizeT = T.size();
			}
		      
		      if (verbosity > 2)
			{
			  cout << "\nAction partitioning for solution extraction\n";
			  Tact.print();
			  cout << "\n\nOptimized search branching partitioning\n";
			  T.print();
			}
		      break;
		      
		
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }
		  break;
		  
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		}
      
      

	      if (verbosity > 0)
		{
		  Trel = relClk.stop();
		  cout << "Trel =" << Trel << " sec\n";
		}
	      
    

	      //////////////////////////
	      // 4) Search
	      //////////////////////////
      
      
	      switch (algorithm) 
		{
		case at_bidir:
		  if (verbosity > 0)
		    cout << "Bidir blind search\n";
		  solution = bidir(Tact,T,init,goal,frontierSetSimpl);
		  break;
		  
		case at_forward:        
		  if (verbosity > 0)
		    cout << "Forward blind search\n";
		  solution = forward(Tact,T,init,goal,frontierSetSimpl);
		  break;
		  
		case at_backward:
		  if (verbosity > 0)
		    cout << "Backward blind search\n";
		  solution = backward(Tact,T,init,goal,frontierSetSimpl);
		  break;
		  
		case at_setAstar:
		  switch (inputFormat) 
		    {
		    case it_PDDL :
		      switch (heuristic) 
			{
			case ht_HSPr:
			  // HSPr is a heuristic for backward search: use special
			  // SetA* search algorithm 
			  if (verbosity > 0)
			    cout << "SetA* backward heuristic search\n";
			  solution = SetAstarBackward(Tact,T,goal,init,hInit,upperBound,gWeight,hWeight);
			  break;
			  
			case ht_minHamming:
			  // use regular forward SetA* search
			  if (verbosity > 0)
			    cout << "SetA* forward heuristic search\n";
			  solution = SetAstar(Tact,T,init,goal,hInit,upperBound,gWeight,hWeight);
			  break;
			  
			default:
			  cout << "main.cc main : switch option not covered\nexiting\n";
			  exit(-1);
			  break;	
			}
		      break;
		      
		    case it_extNADL:
		      switch (heuristic) 
			{
			case ht_minHamming:
			case ht_manhattan:
			  // use regular SetA* search
			  if (verbosity > 0)
			    cout << "SetA* forward heuristic search\n";
			  solution = SetAstar(Tact,T,init,goal,hInit,upperBound,gWeight,hWeight);
			  break;
			  
			default:
			  cout << "main.cc main : switch option not covered\nexiting\n";
			  exit(-1);
			  break;	
			}
		      break;
		      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }
		  break;
		  

		case at_FminAstar:
		  switch (inputFormat) 
		    {
		    case it_PDDL:
		      // Assumes HSPr is a heuristic and backward search
		      if (verbosity > 0)
			cout << "FminAstar backward HSPr heuristic search\n";
		      solution = FminAstarBackward(Tact,T,goal,init,hInit,upperBound,gWeight,hWeight);
		      break;
		      
		    case it_extNADL:
		      switch (heuristic)
			{
			case ht_minHamming:
			  if (verbosity > 0)
			    cout << "FminAstar forward minHamming heuristic search\n";
			  solution = FminAstar(Tbranch,T,init,goal,hInit,upperBound,gWeight,hWeight);
			  break;
			  
			case ht_manhattan:
			  if (verbosity > 0)
			    cout << "FminAstar forward manhattan heuristic search\n";
			  solution = FminAstar(Tact,T,init,goal,hInit,upperBound,gWeight,hWeight);
			  break;
		  
			default:
			  cout << "main.cc main : switch option not covered\nexiting\n";
			  exit(-1);
			  break;		     
			}
		      break;
	      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;		     
		    }
		  break;
	  
		case at_pureFminAstar:
		  // Assumes HSPr is a heuristic and backward search
		  if (verbosity > 0)
		    cout << "PureFminAstar backward HSPr heuristic search\n";
		  solution = pureFminAstarBackward(Tact,T,goal,init,hInit);
		  break;


		  

		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;		      
		}
	      
	    } // end of Determnistic BDD-based search
	    break;
	    
      
	    /////////////////////////////////////////////////////////////////
	    // Single state deterministic 
	    // search
	    /////////////////////////////////////////////////////////////////
	    
      
	  case at_Astar :
	    {
	      // 1) analyse domain 
	      // Compared to BDD-based search, we can skip the balanced predicate analysis
	      if (verbosity > 0)
		analyseClk.start();
	      
	      numDomain numDom;
	      reachInfo rInfo;
	      numDom.initialize(PDDLdom,PDDLprob);        
	      rInfo.analyse(numDom,squizeFactor);
	      
	      if (verbosity > 2)
		{
		  PDDLdom.print();
		  PDDLprob.print();
		  numDom.print();
		  rInfo.print(numDom);
		}
	      
	      if (verbosity > 0)
		{
		  Tanalyse = analyseClk.stop();
		  cout << "Tanalyse =" << Tanalyse << " sec\n";
		  relClk.start(); 
		}
	      
	      // 2) generate vector of ground, reachable actions
	      vector<singleStateAction> singleAction;
	      mkActions(singleAction,numDom,rInfo,heuristic);
	      
	      if (verbosity > 2)
		{
		  cout << "\nActions of single-state search doamins\n";
		  for (int i = 0; i < singleAction.size(); i++)
		    singleAction[i].print(numDom,rInfo);
		}
	      
	      if (verbosity > 0)
		{
		  Tact = relClk.stop();
		  cout << "Tact =" << Tact << " sec\n";
		}
	      
	      // 3) search
	      switch (heuristic) 
		{
		case ht_HSPr:
		  {
		    // HSPr is a heuristic for backward search: use special
		    // A* search algorithm 
		    set<int> init = goal2intSet(rInfo,numDom.goal);
		    set<int> goal = init2intSet(rInfo);
		    int hInit = hHSPrSingle(rInfo,numDom.goal); 
		    
		    if (verbosity > 0)
		      cout << "A* backward heuristic search\n";
		    solution = AstarBackward(numDom,rInfo,singleAction,init,goal,hInit,gWeight,hWeight);
		    break;
		  }
		  
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		}
	      break;		
	    }
	    break;
	    

	    
	    
	    
	    
	    
      
	    /////////////////////////////////////////////////////////////////
	    // Optimized versions of 
	    // BDDA* and Pure-BDDA*
	    // search
	    /////////////////////////////////////////////////////////////////
      
	  case at_pureBDDAstar:
	  case at_BDDAstar:
	    {
	      if (verbosity > 0)
		analyseClk.start();
	      
	      //////////////////////////
	      // 1) Analyse 
	      //////////////////////////
	      
	      numDomain numDom;
	      reachInfo rInfo;
	      balancedInfo bInfo;
	      PDDLbddBDDAlayout PDDLBDDAlayout;
	      NADLbddBDDAlayout NADLBDDAlayout;
	      
	      
	      switch (inputFormat) {
	  
	      case it_extNADL:	
		NADLBDDAlayout.fhMax = 100;
		NADLBDDAlayout.initialize(extNADLprob);
	  
	  
		if (verbosity > 2)
		  NADLBDDAlayout.print();
		break;
		
    	
	      case it_PDDL:	
		{
		  numDom.initialize(PDDLdom,PDDLprob);        
		  rInfo.analyse(numDom,squizeFactor);  
		  bInfo.analyse(rInfo,numDom);
		  // OBS : manually choosing max value of f and h
		  PDDLBDDAlayout.fhMax = 180;
		  PDDLBDDAlayout.initialize(rInfo,bInfo,numDom);
		  
		  if (verbosity > 2)
		    {
		      PDDLdom.print();
		      PDDLprob.print();
		      numDom.print();
		      rInfo.print(numDom);
		      bInfo.print(numDom);
		      PDDLBDDAlayout.print(numDom);
		    }
		}
		break;
		
	      default:
		cout << "main.cc main : switch option not covered\nexiting\n";
		exit(-1);
		break;		  
	      }
	      
           
      
	      if (verbosity > 0)
		{
		  Tanalyse = analyseClk.stop();
		  cout << "Tanalyse =" << Tanalyse << " sec\n";
		  initClk.start(); 
		}
	      
	      

	      //////////////////////////
	      // 2) Init BuDDy
	      //////////////////////////
	      
	      switch (inputFormat) 
		{
		case it_extNADL:
		  bddVarNum = NADLBDDAlayout.bddVarNum;
		  break;
		  
		case it_PDDL:
		  bddVarNum = PDDLBDDAlayout.bddVarNum;
		  break;
	    
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		}	
	      
	      // initialize bdd package
	      bdd_init(initBddNodes,initCache);
	      bdd_setvarnum(bddVarNum);
	
	      if (verbosity > 0)
		{
		  Tinit = initClk.stop();
		  cout << "Tinit =" << Tinit << " sec\n";
		  relClk.start(); 
		}
	      


	      //////////////////////////
	      // 3) Build
	      // Transition Relation
	      // + h and formula 
	      // structures
	      //////////////////////////
	      
	      TRel Tact;
	      TRel T;
	      bdd heuristicBdd;
	      bdd init;
	      bdd goal;
	      FHinfo fhInfo;
	      
	      switch (inputFormat) 
		{
		case it_PDDL:
		  {
		    // Assume the HSPr heuristic
		    // build and optimize a disjunctive partitioning
		    // similar to the SetAstar case
		    PDDLmkTact(PDDLBDDAlayout,rInfo,numDom,Tact,ht_none);
		    optimize(Tact,T,partLimit);     	    
		    
		    // init fhInfo
		    fhInfo.initialize(PDDLBDDAlayout);
		    
		    // make init, goal and heuristic function encoding
		    init = init2bdd(PDDLBDDAlayout,numDom,numDom.init);
		    // we also use init2bdd for the goal state, since it, as the init state
		    // is assumed to be a single state
		    goal = init2bdd(PDDLBDDAlayout,numDom,numDom.goal); 
		    
		    heuristicBdd = PDDLSymbolicHSPr(PDDLBDDAlayout,rInfo,numDom,fhInfo);
		    
		    if (verbosity > 0)
		      {
			heuristicBDDsize = bdd_nodecount(heuristicBdd);
			cout << "Heuristic BDD size=" << heuristicBDDsize << endl;
		      }
      
		    if (verbosity > 2)
		      {
			cout << "\nAction partitioning for solution extraction\n";
			Tact.print();
			cout << "\n\nOptimized search partitioning\n";
			T.print();
		      }
		  }
		  break;
		
		  
		case it_extNADL:
		  {
		    switch (heuristic) {

		    case ht_minHamming:		
		      // so far assuming the problem is the 15puzzle 
		      // build and optimize a disjunctive partitioning
		      // similar to the SetAstar case
		      NADLmkTact(NADLBDDAlayout,Tact);
		      optimize(Tact,T,partLimit);     	    
		      
		      init = formula2bdd(NADLBDDAlayout,extNADLprob.init);
		      goal = formula2bdd(NADLBDDAlayout,extNADLprob.goal);
		      
		      // init fhInfo
		      fhInfo.initialize(NADLBDDAlayout);
		      
		      heuristicBdd = symbolicMinHamming(NADLBDDAlayout,fhInfo,goal);

		      
		      if (verbosity > 0)
			{
			  heuristicBDDsize = bdd_nodecount(heuristicBdd);
			  cout << "Heuristic BDD size=" << heuristicBDDsize << endl;
			}
		      
		      if (verbosity > 2)
			{
			  cout << "\nAction partitioning for solution extraction\n";
			  Tact.print();
			  cout << "\n\nOptimized search partitioning\n";
			  T.print();
			}	     
		      break;


		    case ht_manhattan:		
		      // build and optimize a disjunctive partitioning
		      // similar to the SetAstar case
		      NADLmkTact(NADLBDDAlayout,Tact);
		      optimize(Tact,T,partLimit);     	    
		      
		      init = formula2bdd(NADLBDDAlayout,extNADLprob.init);
		      goal = formula2bdd(NADLBDDAlayout,extNADLprob.goal);
		  
		      // init fhInfo
		      fhInfo.initialize(NADLBDDAlayout);
		      
		      heuristicBdd = NADLSymbolicManhattan(NADLBDDAlayout,fhInfo);
		      
		      if (verbosity > 0)
			{
			  heuristicBDDsize = bdd_nodecount(heuristicBdd);
			  cout << "Heuristic BDD size=" << heuristicBDDsize << endl;
			}
		      
		      if (verbosity > 2)
			{
			  cout << "\nAction partitioning for solution extraction\n";
			  Tact.print();
			  cout << "\n\nOptimized search partitioning\n";
			  T.print();
			}		
		      break;
		      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }



		  }
		  break;



		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		  
		}
      
      

	      if (verbosity > 0)
		{
		  Trel = relClk.stop();
		  cout << "Trel =" << Trel << " sec\n";
		}
      

	      //////////////////////////
	      //
	      // search
	      // 
	      //////////////////////////

	      switch (algorithm) 
		{
		  
		case at_pureBDDAstar:
		  switch (inputFormat) 
		    {
		    case it_PDDL:
		      
		      if (verbosity > 0)
			{
			  SizeTact = Tact.size();
			  SizeT = T.size();
			}
		      
		      solution = pureBDDAstarBackward(PDDLBDDAlayout,fhInfo,heuristicBdd,Tact,T,goal,init);       
		      break;
	      	      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }        
		  break;
		  
		  
		case at_BDDAstar:
		  switch (inputFormat) 
		    {
		    case it_PDDL:
		      {
			BDDAinfo bddaInfo;
			bddaInfo.initialize(PDDLBDDAlayout);
			
			// expand transition relations such that each 
			// partition has a complete IV expression.
			// This is needed in order to decide the change of h
			IVextend(bddaInfo,T);
			IVextend(bddaInfo,Tact);
			
			if (verbosity > 0)
			  {
			    SizeTact = Tact.size();
			    SizeT = T.size();
			  }
			
			solution = BDDAstarBackward(PDDLBDDAlayout,fhInfo,bddaInfo,
						    heuristicBdd,Tact,T,goal,init);
		      }
		      break;
		      

		    case it_extNADL:
		      {
			BDDAinfo bddaInfo;
			bddaInfo.initialize(NADLBDDAlayout);
			
			// expand transition relations such that each 
			// partition has a complete IV expression.
			// This is needed in order to decide the change of h
			IVextend(bddaInfo,T);
			IVextend(bddaInfo,Tact);
			
			if (verbosity > 0)
			  {
			    SizeTact = Tact.size();
			    SizeT = T.size();
			  }
			
			solution = BDDAstar(NADLBDDAlayout,fhInfo,bddaInfo,heuristicBdd,Tact,T,init,goal);
		      }
		      break;

	      	      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }        	  
		  break;
	  
	  
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;		      
		}
      
      
	    } // at_BDDA* and at_pureBDDA*
	    break;

	    










      
	    /////////////////////////////////////////////////////////////////
	    // The published versions of 
	    // BDDA* and Pure-BDDA*
	    // search
	    /////////////////////////////////////////////////////////////////
	    
	  case at_PubPureBDDAstar:
	  case at_PubBDDAstar:
	    {
	      if (verbosity > 0)
		analyseClk.start();
	      
	      //////////////////////////
	      // 1) Analyse 
	      //////////////////////////
	      
	      numDomain numDom;
	      reachInfo rInfo;
	      balancedInfo bInfo;
	      PDDLbddPubBDDAlayout PubPDDLBDDAlayout;
	      NADLbddPubBDDAlayout PubNADLBDDAlayout;
	      

	
	      switch (inputFormat) {
		

	      case it_extNADL:
		{
		  PubNADLBDDAlayout.fhMax = 100;
		  PubNADLBDDAlayout.initialize(extNADLprob);
		  
		  
		  if (verbosity > 2)
		    PubNADLBDDAlayout.print();
		  
		}
		break;
		
	      	
	      case it_PDDL:	
		{
		  numDom.initialize(PDDLdom,PDDLprob);        
		  rInfo.analyse(numDom,squizeFactor);  
		  bInfo.analyse(rInfo,numDom);
		  // OBS : manually choosing max value of f and h
		  PubPDDLBDDAlayout.fhMax = 180;
		  PubPDDLBDDAlayout.initialize(rInfo,bInfo,numDom);
		  
		  if (verbosity > 2)
		    {
		      PDDLdom.print();
		      PDDLprob.print();
		      numDom.print();
		      rInfo.print(numDom);
		      bInfo.print(numDom);
		      PubPDDLBDDAlayout.print(numDom);
		    }
		}
		break;
		
		
	      default:
		cout << "main.cc main : switch option not covered\nexiting\n";
		exit(-1);
		break;		  
	      }
	      
           
      
	      if (verbosity > 0)
		{
		  Tanalyse = analyseClk.stop();
		  cout << "Tanalyse =" << Tanalyse << " sec\n";
		  initClk.start(); 
		}



	      //////////////////////////
	      // 2) Init BuDDy
	      //////////////////////////
	      
	      switch (inputFormat) 
		{
		case it_extNADL:
		  bddVarNum = PubNADLBDDAlayout.bddVarNum;
		  break;
		  
		case it_PDDL:
		  bddVarNum = PubPDDLBDDAlayout.bddVarNum;
		  break;
		  
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		}	
      
	      // initialize bdd package
	      bdd_init(initBddNodes,initCache);
	      bdd_setvarnum(bddVarNum);
	      
	      if (verbosity > 0)
		{
		  Tinit = initClk.stop();
		  cout << "Tinit =" << Tinit << " sec\n";
		  relClk.start(); 
		}



	      //////////////////////////
	      // 3) Build
	      // Transition Relation
	      // + h and formula 
	      // structures
	      //////////////////////////

	      TRel Tact;
	      TRel T;
	      bdd heuristicBdd;
	      bdd init;
	      bdd goal;
	      PubFHinfo fhInfo;
	      
	      switch (inputFormat) 
		{
		case it_PDDL:
		  {
		    // build and optimize a disjunctive partitioning
		    // similar to the SetAstar case
		    PDDLmkTact(PubPDDLBDDAlayout,rInfo,numDom,Tact,ht_none);
		    // use monolithic transition relation Set partLimit "infinit"
		    optimize(Tact,T,100000000);     	    
		    
		    // init fhInfo
		    fhInfo.initialize(PubPDDLBDDAlayout);
		
		    // make init, goal and heuristic function encoding
		    init = init2bdd(PubPDDLBDDAlayout,numDom,numDom.init);
		    // we also use init2bdd for the goal state, since it, as the init state
		    // is assumed to be a single state
		    goal = init2bdd(PubPDDLBDDAlayout,numDom,numDom.goal); 
		    
		    heuristicBdd = PubPDDLSymbolicHSPr(PubPDDLBDDAlayout,rInfo,numDom,fhInfo);
		    
		    if (verbosity > 0)
		      {
			heuristicBDDsize = bdd_nodecount(heuristicBdd);
			cout << "Heuristic BDD size=" << heuristicBDDsize << endl;
		      }
		    
		    if (verbosity > 2)
		      {
			cout << "\nAction partitioning for solution extraction\n";
			Tact.print();
			cout << "\n\nOptimized search partitioning\n";
			T.print();
		      }
		  }
		  break;
		  

		case it_extNADL:
		  {
		    switch (heuristic) 
		      {
		      case ht_minHamming:		  
			// so far assuming the problem is the 15puzzle 
			// build and optimize a disjunctive partitioning
			// similar to the SetAstar case
			NADLmkTact(PubNADLBDDAlayout,Tact);
			optimize(Tact,T,100000000);     	    
			
			init = formula2bdd(PubNADLBDDAlayout,extNADLprob.init);
			goal = formula2bdd(PubNADLBDDAlayout,extNADLprob.goal);
			
			// init fhInfo
			fhInfo.initialize(PubNADLBDDAlayout);
			
			heuristicBdd = PubSymbolicMinHamming(PubNADLBDDAlayout,fhInfo,goal);		 
			
			if (verbosity > 0)
			  {
			    heuristicBDDsize = bdd_nodecount(heuristicBdd);
			    cout << "Heuristic BDD size=" << heuristicBDDsize << endl;
			  }
			
			if (verbosity > 2)
			  {
			    cout << "\nAction partitioning for solution extraction\n";
			    Tact.print();
			    cout << "\n\nOptimized search partitioning\n";
			    T.print();
			  }		  
			break;
			
		  
		      case ht_manhattan:		  
			// build and optimize a disjunctive partitioning
			// similar to the SetAstar case
			NADLmkTact(PubNADLBDDAlayout,Tact);
			optimize(Tact,T,100000000);     	    
			
			init = formula2bdd(PubNADLBDDAlayout,extNADLprob.init);
			goal = formula2bdd(PubNADLBDDAlayout,extNADLprob.goal);
			
			// init fhInfo
			fhInfo.initialize(PubNADLBDDAlayout);
			
			heuristicBdd = PubNADLSymbolicManhattan(PubNADLBDDAlayout,fhInfo);
			
			if (verbosity > 0)
			  {
			    heuristicBDDsize = bdd_nodecount(heuristicBdd);
			    cout << "Heuristic BDD size=" << heuristicBDDsize << endl;
			  }
		  
			if (verbosity > 2)
			  {
			    cout << "\nAction partitioning for solution extraction\n";
			    Tact.print();
			    cout << "\n\nOptimized search partitioning\n";
			    T.print();
			  }		  
			break;
		      }
		  }
		  break;
	    
	    
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;	
		  
		}
	      
	  
	  
	      if (verbosity > 0)
		{
		  Trel = relClk.stop();
		  cout << "Trel =" << Trel << " sec\n";
		}
	

	      //////////////////////////
	      //
	      // search
	      // 
	      //////////////////////////
	      
	      switch (algorithm) 
		{
	  
		case at_PubPureBDDAstar:
		  switch (inputFormat) 
		    {
		    case it_PDDL:
		      
		      if (verbosity > 0)
			{
			  SizeTact = Tact.size();
			  SizeT = T.size();
			}
		      
		      solution = PubpureBDDAstarBackward(PubPDDLBDDAlayout,fhInfo,heuristicBdd,Tact,T,goal,init);       
		      break;
		      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }        
		  break;
		  
	    
		case at_PubBDDAstar:
		  switch (inputFormat) 
		    {
		    case it_PDDL:
		      {
			PubBDDAinfo bddaInfo;
			bddaInfo.initialize(PubPDDLBDDAlayout);
			
			if (verbosity > 0)
			  {
			    SizeTact = Tact.size();
			    SizeT = T.size();
			  }
			
			solution = PubBDDAstarBackward(PubPDDLBDDAlayout,fhInfo,bddaInfo,
						       heuristicBdd,Tact,T,goal,init);
		      }
		      break;
		      

		    case it_extNADL:
		      {
			PubBDDAinfo bddaInfo;
			bddaInfo.initialize(PubNADLBDDAlayout);
			
			if (verbosity > 0)
			  {
			    SizeTact = Tact.size();
			    SizeT = T.size();
			  }
			
			solution = PubBDDAstar(PubNADLBDDAlayout,fhInfo,bddaInfo,heuristicBdd,Tact,T,init,goal);
		      }
		      break;


		      
		    default:
		      cout << "main.cc main : switch option not covered\nexiting\n";
		      exit(-1);
		      break;	
		    }        	  
		  break;
	    
	    
		default:
		  cout << "main.cc main : switch option not covered\nexiting\n";
		  exit(-1);
		  break;		      
		}
	
       
	    } // at_pubBDDA* and at_PubpureBDDA*
	    break;
      
	    
	  default:
	    cout << "main.cc main : switch option not covered\nexiting\n";
	    exit(-1);
	    break;		      
	  } // main switch on deterministic search algorithm type
      

  
	/////////////////////////////////////////////////////////////////
	// write solution
	/////////////////////////////////////////////////////////////////
	if (solution.size())
	  {
	    ofstream out(solutionName,ios::out);
	    int step = 1;
	    if (verbosity > 1)
	      cout << "Solution:\n";
	    for (list<string>::iterator lsi = solution.begin(); lsi != solution.end(); ++ lsi)
	      {
		out << step << " : " << (*lsi) << endl;
		if (verbosity > 1)
		  cout << step << " : " << (*lsi) << endl;
		step++;
	      }
	    
	    if (verbosity > 0)
	      cout << solution.size() << " step solution found and written to " << solutionName << "\n";
	  }
	else
	  if (verbosity > 0)
	    cout << "No solution exists (or solution trivial of zero length)\n";    

	solutionLength = solution.size();
	
      } // deterministic algorithms
      break;
    } // outermost algorithm switch
  


  
  /////////////////////////////////////////////////////////////////
  // write statistics
  /////////////////////////////////////////////////////////////////

  Ttotal = totalClk.stop();      
    
      
  ofstream expF;
  bool Newfile;
  
  if (string(expName) != "" )
    {
      expF.open(expName,ios_base::app);
      if (int(expF.tellp()) == 0) // make header of experimental file
	{
	  if (Ttotal >= 0)
	    {
	      expF.width(8);
	      expF << "Ttotal ";
	    }
	  if (Tparse >= 0)		
	    {
	      expF.width(8);
	      expF << "Tparse ";
	    }
	  if (Tanalyse >= 0)		
	    {
	      expF.width(8);
	      expF << "Tanal ";
	    }
	  if (Tinit >= 0)
	    {
	      expF.width(8);
	      expF << "Tinit ";
	    }
	  if (Trel >= 0)
	    {
	      expF.width(8);				  
	      expF << "Trel ";
	    }
	  if (Tact >= 0)
	    {
	      expF.width(8);		  
	      expF << "Tact ";
	    }
	  if (Tsearch >= 0)
	    {
	      expF.width(8);
	      expF << "Tsearch ";
	    }
	  if (Textract >= 0)
	    {
	      expF.width(8);
	      expF << "Text ";
	    }
	  if (spaceSize >= 0)
	    {
	      expF.width(8);		
	      expF << "|space|";
	    }
	  if (solutionLength >= 0)
	    {
	      expF.width(8);
	      expF << "|sol| ";
	    }
	  if (solutionSize >= 0)
	    {
	      expF.width(8);
	      expF << "|solBDD| ";
	    }
	  if (aveFringeSize >= 0)
	    {
	      expF.width(8);
	      expF << "|Fringe| ";
	    }
	  if (maxSearchQueueSize >= 0)
	    {
	      expF.width(8);
	      expF << "|maxQ| ";
	    }
	  if (iterationNum >= 0)
	    {
	      expF.width(8);
	      expF << "#it ";
	    }
	  if (forwardExpansionNum >= 0)
	    {
	      expF.width(8);
	      expF << "#itForw ";
	    }
	  if (backwardExpansionNum >= 0)
	    {
	      expF.width(8);
	      expF << "#itBackw ";
	    }
	  if (SizeTact >= 0)
	    {
	      expF.width(8);
	      expF << "|Tact| ";
	    }
	  if (SizeT >= 0)
	    {
	      expF.width(8);
	      expF << "|T| ";
	    }
	  if (SizeTbranch >= 0)
	    {
	      expF.width(8);
	      expF << "|Tbranch| ";
	    }
	  if (heuristicBDDsize >= 0)
	    {
	      expF.width(8);
	      expF << "|h| ";
	    }
	  if (totalFormulaSize >= 0)
	    {
	      expF.width(8);
	      expF << "|formulas| ";
	    }
	  expF.width(8);
	  expF << "Problem ";
	  expF.width(8);
	  expF << "n ";
	  expF.width(8);
	  expF << "c ";
	  expF << endl;
	}
	}
  
  cout << "\nStatistics:\n";
  cout << "===========\n";
  
  
  if (Ttotal >= 0)
	{
	  cout << "Ttotal=              " << Ttotal << " (sec)\n";
	  if (string(expName) != "" ) expF.width(7);
	  if (string(expName) != "" ) expF << Ttotal << " ";
	}
  if (Tparse >= 0)
    {
      cout << "Tparse=              " << Tparse << " (sec)\n";               
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Tparse << " ";
    }
  if (Tanalyse >= 0)
    {
      cout << "Tanalyse=            " << Tanalyse << " (sec)\n";            
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Tanalyse << " ";
    }
  if (Tinit >= 0)
    {
      cout << "Tinit=               " << Tinit << " (sec)\n";                
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Tinit << " ";
    }
  if (Trel >= 0)
    {
      cout << "Trel=                " << Trel << " (sec)\n";                
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Trel << " ";
    }
  if (Tact >= 0)
    {
      cout << "Tact=                " << Tact << " (sec)\n";                
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Tact << " ";
    }
  if (Tsearch >= 0)
    {
      cout << "Tsearch=             " << Tsearch << " (sec)\n";             
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Tsearch << " ";
    }
  if (Textract >= 0)
    {
      cout << "Textract=            " << Textract << " (sec)\n";            
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << Textract << " ";
    }
  if (spaceSize >= 0)
    {
      cout << "number of BDD vars in state encoding=  " << spaceSize << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << spaceSize << " ";
    }
  
  if (solutionLength >= 0)
    {
      cout << "solution length=     " << solutionLength << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << solutionLength << " ";
    }
  
  if (solutionSize >= 0)
    {
      cout << "solution BDD size=     " << solutionSize << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << solutionSize << " ";
    }
  
  if (aveFringeSize >= 0)
    {
      cout << "aveFringeSize=       " << aveFringeSize << " nodes\n";
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << aveFringeSize << " ";
    }
  if (maxSearchQueueSize >= 0)
    {
      cout << "maxSearchQueueSize=  " << maxSearchQueueSize << endl;  
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << maxSearchQueueSize << " ";
    }
  if (iterationNum >= 0)
    {
      cout << "iteration number=    " << iterationNum << endl; 
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << iterationNum << " ";
    }
  if (forwardExpansionNum >= 0)
    {
      cout << "forwardExpansionNum= " << forwardExpansionNum << endl; 
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << forwardExpansionNum << " ";
    }
  if (backwardExpansionNum >= 0)
    {
      cout << "backwardExpansionNum=" << backwardExpansionNum << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << backwardExpansionNum << " ";
    }
  if (SizeTact >= 0)
    {
      cout << "SizeTact=            " << SizeTact << endl;
	  if (string(expName) != "" ) expF.width(7);
	  if (string(expName) != "" ) expF << SizeTact << " ";
    }
  if (SizeT >= 0)
    {
      cout << "SizeT=               " << SizeT << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << SizeT << " ";
    }
  if (SizeTbranch >= 0)
    {
      cout << "SizeTbranch=         " << SizeTbranch << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << SizeTbranch << " ";
    }
  if (heuristicBDDsize >= 0)
    {
      cout << "heuristicBDDsize=    " << heuristicBDDsize << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << heuristicBDDsize << " ";
    }
  if (totalFormulaSize >= 0)
    {
      cout << "totalFormulaSize=    " << totalFormulaSize << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << totalFormulaSize << " ";
    }
  
  if (inputFormat == it_PDDL)
    {
      cout << "ProblemName=         " << problemName << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << problemName << " ";
    }
  else
    {
      cout << "ProblemName=         " << domainName << endl;
      if (string(expName) != "" ) expF.width(7);
      if (string(expName) != "" ) expF << domainName << " ";
    }
  
  cout << "InitBddNodes=        " << initBddNodes << endl;
  if (string(expName) != "" ) expF.width(7);
  if (string(expName) != "" ) expF << initBddNodes << " ";
  cout << "InitCache=           " << initCache << endl;
  if (string(expName) != "" ) expF.width(7);
  if (string(expName) != "" ) expF << initCache << " ";
  
  if (string(expName) != "" ) expF << endl;
  
  
  
  return 0; 
}




void  printUsage() {
printf("USAGE:  bdds -d domainFile [-iavponcrthwuf]                                                  \n");
printf("											     \n");
printf("OPTIONS:										     \n");
printf("											     \n");
printf("       -h                                                                                    \n");
printf("	      Print usage and exit          	    		                             \n");	  
printf("											     \n");
printf("       -l NUM                              					             \n");
printf("	      Time out limit in seconds. (Default 500)                                       \n");
printf("											     \n");
printf("       -v NUM                            						     \n");
printf("	      Debug verbosity=NUM (default 1)			      			     \n");
printf("											     \n");
printf("       -i TYPE                            					             \n");
printf("	      Type of input format. The input format can either be PDDL or extNADL. For      \n");
printf("	      TYPE=PDDL both a domain and problem file must be given (using the -d           \n");
printf("	      and -p option). If TYPE=extNADL only a domain file has to be given (using      \n");
printf("              the -d option)                                                                 \n");
printf("											     \n");
printf("       -d FILE                                                                               \n");
printf("              Name of domain file read by BDDsearch                                          \n");
printf("											     \n");
printf("       -p FILE    									     \n");
printf("              Name of problem file read by BDDsearch (only needs to be given for PDDL        \n");
printf("       	      problems)									     \n");
printf("       											     \n");
printf("       -o FILE										     \n");
printf("              Name of output file created and written by BDDsearch. For the extNADL          \n");
printf("              input format, the default is the domain file changed to .sol. For		     \n");
printf("              the PDDL input format, it is the problem file changed to .sol		     \n");
printf("       											     \n");
printf("       -e FILE										     \n");
printf("              Name of experimental file collecting statistical data.                         \n");
printf("              Data are appended to this file only if a name is given		             \n");
printf("											     \n");
printf("       -n NUM                              					             \n");
printf("	      Initial number of allocated BDD nodes by BuDDy (default 1000000)               \n");
printf("											     \n");
printf("       -c NUM                                                                                \n");
printf("	      Initial cache size used by BuDDy (default  100000)                             \n");
printf("											     \n");
printf("       -r TYPE                             					             \n");
printf("	      Type dynamic variable reordering of BuDDy. Possible values for TYPE are:       \n");
printf("              										     \n");
printf("              Off        : No dynamic reordering (default)	 	                     \n");
printf("              Win2ite    : Win2Ite sliding window reorder	 	                     \n");
printf("											     \n");
printf("       -t NUM										     \n");
printf("              Disjunctive partitioning threshold = NUM (default 5000)                        \n");
printf("											     \n");
printf("       -a TYPE                             					             \n");
printf("	      Type of search algorithm. Possible values for TYPE are:    		     \n");
printf("                  									     \n");
printf("              Deterministic Search							     \n");
printf("              --------------------							     \n");
printf("              Bidir             : BDD-based breadth-first bidirectional search               \n");
printf("	      Forward           : BDD-based breadth-first forward search 		     \n");
printf("	      Backward          : BDD-based breadth-first backward search		     \n");
printf("	      SetAstar          : BDD-based weighted A* search (f = wg*g + wh*h)	     \n");
printf("              Astar             : Single-state weighted A* search 			     \n");
printf("              BDDAstar          : BDDA* search     (improved BDDA*, AIJ 2002)   	     \n");
printf("              PureBDDAstar      : pureBDDA* search (improved pureBDDA*, AIJ 2002) 	     \n");
printf("              PubBDDAstar       : BDDA* search     (as defined in Edelkamp et al. 99,00,01)  \n");
printf("              PubPureBDDAstar   : pureBDDA* search (as defined in Edelkamp et al. 00,01)     \n");
printf("              FminAstar         : weighted BDDA* search using the SetA* approach             \n");
printf("              PureFminAstar     : pureBDDA* search using the SetA* approach                  \n");
printf("                  									     \n");
printf("              Non-Deterministic Search							     \n");
printf("              ------------------------							     \n");
printf("              Strong             : Strong universal planning                                 \n");
printf("              StrongH            : Heuristic strong universal planning (ICAPS-03)            \n");
printf("              StrongCyclic       : Strong cyclic universal planning                          \n");
printf("              StrongCyclicH      : Heuristic strong cyclic universal planning (ICAPS-03)     \n");
printf("              Weak               : Weak universal planning                                   \n");
printf("              WeakH              : Weak Heuristic universal planning (ICAPS-03)              \n");
printf("              FaultTolerant      : 1-fault tolerant planning                                 \n");
printf("              FaultTolerantOpt   : Optimal 1-fault tolerant planning                         \n");
printf("              GuidedFaultTolerant: Guided 1-fault tolerant planning (ICAPS-03)               \n");
printf("              StrongCyclicAdv    : Strong Cyclic Adversarial Planning  (ECP-01)              \n");
printf("											     \n");
printf("       -g TYPE									             \n");
printf("	      Type of heuristic function used by SetA* or A*. Possible values    	     \n");
printf("              for TYPE are:								     \n");
printf("											     \n");
printf("              MinHamming : Minimum Hamming distance (default)	                             \n");
printf("              Manhattan  : Sum of manhattan distances 	                                     \n");
printf("	      HSPr       : The HSPr heuristic as in Edelkamp & Helmert AIPS-00               \n");
printf("			   Only applies to PDDL problems				     \n");
printf("											     \n");
printf("       -x NUM                                                                                \n");
printf("	      G-weight used by SetA*, A*, and FminA* (f = wg * g + wh * h).                  \n");
printf("	      NUM is in the range [0;1] (default 1.0)					     \n");
printf("											     \n");
printf("       -y NUM                                                                                \n");
printf("	      H-weight used by SetA*, A*, and FminA* (f = wg * g + wh * h).                  \n");
printf("	      NUM is in the range [0;1] (default 1.0)					     \n");
printf("											     \n");
printf("       -z NUM							               		     \n");
printf("              squize factor of fact depth levels. ]0.0,1.0]. 			             \n");
printf("              The lower number the less information in HSPr.(default 1.0 [no squize])        \n");
printf("											     \n");
printf("       -u NUM                                                                                \n");
printf("	      Upper bound of the size of BDDs in the search queue of SetA*                   \n");
printf(" 	      (default 5000)								     \n");
printf("											     \n");
printf("       -f                                                                                    \n");
printf("	      Minimization of frontier sets (Coudert et al. 89). This option only has        \n");
printf("              an effect for Bidir, Forward and Backward BDD-based search. (default: off)     \n");
}






