///////////////////////////////////////////////////////
// File  : extNADL.hpp   
// Desc. : Domain theory representation for extNADL-domains
//         extended such that each action has a value
//         or cost associated
// Author: Rune M. Jensen
// Date  : 10/12/02
///////////////////////////////////////////////////////

#ifndef NADLHPP
#define NADLHPP

#include <set>
#include <list>
#include <string>
#include "common.hpp"

///////////////////////////////////////////////////////
//                Number expressions                
///////////////////////////////////////////////////////

enum relOp {ro_lt,ro_gt,ro_eq,ro_ne};
enum expType {nt_number, nt_plus, nt_minus, nt_varCur,nt_varNext};
 

struct numberExp {
  expType    type;
  int        number;
  numberExp* left;
  numberExp* right;
  string     var;
  numberExp(expType t,int n,numberExp* l,numberExp* r,string v)  
  {type = t; number = n;left = l;right = r;var = v;}
  numberExp() {}
  void minus2plus();
  void print();
};

struct numberProp {
  relOp      op;
  numberExp* left;
  numberExp* right;
  numberProp(relOp o,numberExp* l,numberExp* r) 
    {op = o; left = l; right = r;}
  numberProp() {}
  void minus2plus();
  void print();
};


///////////////////////////////////////////////////////
//                       atoms                      
///////////////////////////////////////////////////////

enum atomType {at_numProp, at_boolVarCur, at_boolVarNext};

struct atom {
  atomType    type;
  numberProp* prop;
  string      var;
  atom(atomType t,numberProp* p,string v) 
  {type = t;prop = p;var = v;}
  atom() {};
  void print();
};


///////////////////////////////////////////////////////
//               Fomula representation              
///////////////////////////////////////////////////////


enum formulaType {ft_atom,ft_neg,ft_and,ft_or,ft_impl,
              ft_biImpl,ft_ite, ft_true, ft_false, ft_paren};

struct formula {
  formulaType type;
  atom*       atomic;
  formula*    f1;
  formula*    f2;
  formula*    f3;
  formula(formulaType t, atom* a, formula* f1c, formula* f2c, formula* f3c)
    {type = t; atomic = a; f1 = f1c; f2 = f2c; f3 = f3c;}
  formula() {}
  void print();
};


///////////////////////////////////////////////////////
//               Transition decleration representation              
///////////////////////////////////////////////////////

struct extNADLtran {
  bool        dgOpt;   // true of the transition decl has a value associated
  bool        dhOpt;   // true of the transition decl has a heuristic value associated
  bool        errOpt;   // true of the transition decl has a error formula associated
  int         dg;
  int         dh;
  set<string> mod;
  formula*    pre;
  formula*    eff;
  formula*    err;

  extNADLtran(int* dgi,int* dhi, set<string>* m, formula* p, formula* e,formula* er);
  extNADLtran() {}
  void print();
}; 




///////////////////////////////////////////////////////
//               Action representation              
///////////////////////////////////////////////////////

struct extNADLaction {
  string            name;
  list<extNADLtran> trans;
  extNADLaction(string n,list<extNADLtran>* trs) 
  { name = n; trans = *trs; }
  extNADLaction() {}
  void print();
}; 


///////////////////////////////////////////////////////
//            agent  representation                 
///////////////////////////////////////////////////////

struct agent {
  string              name;
  list<extNADLaction> actions;
  agent(string n, list<extNADLaction>* a) 
  {name = n; actions = *a;}
  agent() {}
  void print();
};


///////////////////////////////////////////////////////
//       varlst, vartype representation             
///////////////////////////////////////////////////////

enum vType {vt_bool, vt_nat};

struct varType {
  vType type;
  int   range;
  varType(vType t, int r)
    {type = t; range = r;}
  varType() {}
  void print();
};



struct varDecl {
  varType type;
  set<string> ids;
  varDecl(varType* t, set<string>* i) 
  {type = *t; ids = *i;}
  varDecl() {}
  void print();
};


///////////////////////////////////////////////////////
//               Problem representation             
///////////////////////////////////////////////////////

struct extNADLproblem {
  list<varDecl> vars;
  list<agent>   sys;
  bool          envOpt;
  list<agent>   env;
  formula*      init;
  bool          heuOptInit;
  int           heuInit;
  formula*      goal;
  bool          heuOptGoal;
  int           heuGoal;  
  void initialize(list<varDecl>* v, list<agent>* s, list<agent>* e, formula* i,int* hoi, formula* g,int* hog);
  extNADLproblem() {}
  void print();
}; 


///////////////////////////////////////////////////////
//  function prototypes 
///////////////////////////////////////////////////////

int yyparse();


#endif
