////////////////////////////////////////////////////////
// File  : detSearchAlg.cc  
// Desc. : search algorithm implementations 
// Author: Rune M. Jensen CS, CMU
// Date  : 1/13/02 (non-det 8/28/02)
////////////////////////////////////////////////////////

#include <bdd.h>
#include <math.h>
#include <queue>
#include <list>
#include "main.hpp"
#include "numDomain.hpp"
#include "bddLayout.hpp"
#include "partition.hpp"
#include "universal.hpp"
#include "searchQueue.hpp"
#include "searchAlgAuxFunc.hpp"
#include "detSearchAlg.hpp"
#include "timer.hpp"



////////////////////////////////////////////////////////
//  
// Deterministic search algorithms
// 1/13/02 (for AIJ paper)
//   
////////////////////////////////////////////////////////


////////////////////////////////////////////////////////
//   Bidiresctional Search
// 
//   1) Bidirectional image/preimage calculation
// 
//   Preconditions:
//   1) The initial state is assumed to be totally defined 
//	(otherwise the forward search is inconsistent)
////////////////////////////////////////////////////////


//IN
// Tact     : disjunctive partitioning where each partition corresponds to
//               a single action (for action extraction)
// T        : disjunctive backward/forward partitioning  (search)
// init        : initial state
// goal        : goal states
//OUT
// solution (empty if no result)
list<string> bidir(TRel& Tact,TRel& T,bdd init, bdd goal, bool frontierSimpl) {

  int i;
  stateSet preImageSet(MAXSOLLENGTH);
  stateSet imageSet(MAXSOLLENGTH);  
  int stepf;
  int stepb;
  bdd initPoint;
  bdd vf,vb;  // forward and backward visited states
  bdd vfOld,vbOld;
  timer imageTimer, preImageTimer,searchClk,extractClk;
  list<string> solution;
  bdd frontier;
  vector<int> fringeSize;
  bdd simpl;


  //  bidirectional search
  stepf=0;
  stepb=0;
  vf = init;
  vb = goal;
  vfOld = bddfalse;
  vbOld = bddfalse;
  preImageSet.state[0] = goal;
  imageSet.state[0] = init;
  double lastImageCPUtime = -10.0;
  double lastPreImageCPUtime = -10.0;

  if (verbosity > 0)
    searchClk.start();


  while ((imageSet.state[stepf] & preImageSet.state[stepb]) == bddfalse) 
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      
      //  Heuristc: expand the one that took least time at last expansion
      if (lastImageCPUtime < lastPreImageCPUtime)
	{
	  // make image
	  imageTimer.start();
	  stepf++;

	  // frontier set simplification
	  if (frontierSimpl) {
	    if (stepf == 1)
	      frontier = vf;
	    else
	      {
		simpl = bdd_simplify(imageSet.state[stepf-1],!vfOld);
		if (bdd_nodecount(simpl) < bdd_nodecount(imageSet.state[stepf-1]))		  
		    frontier = simpl;
		else
		  frontier = imageSet.state[stepf-1];
	      }
	  }
	  else
	    frontier = imageSet.state[stepf-1];

	  imageSet.state[stepf] = image(T,frontier);
	  imageSet.state[stepf] = pruneStates(imageSet.state[stepf],vf);
	  if (imageSet.state[stepf] != bddfalse)
	    {
	      vfOld = vf;
	      vf |= imageSet.state[stepf];
	      lastImageCPUtime = imageTimer.stop();
	      if (verbosity > 0) 
		{
		  cout << "Step " << stepf + stepb << "  Image " << stepf << "  Time "
		       << lastImageCPUtime << "  Fringe " << bdd_nodecount(imageSet.state[stepf]) 
		       << "/" << bdd_nodecount(vf) << "\n";
		  fringeSize.push_back(bdd_nodecount(imageSet.state[stepf]));
		}
	    }
	  else
	    {
	      cout << "Empty Image: No solution exists!\n";
	      return solution; //empty
	    }
	}
      else
	{
	  // Make preImage
	  preImageTimer.start();
	  stepb++;	  

	  // frontier set simplification
	  if (frontierSimpl) {
	    if (stepb == 1)
	      frontier = vb;
	    else
	      {
		simpl = bdd_simplify(preImageSet.state[stepb-1],!vbOld);
		if (bdd_nodecount(simpl) < bdd_nodecount(preImageSet.state[stepb-1]))
		  frontier = simpl;
		else
		  frontier = preImageSet.state[stepb-1];
	      }
	  }
	  else
	    frontier = preImageSet.state[stepb-1];



	  preImageSet.state[stepb] = preImage(T,frontier);
	    
	  preImageSet.state[stepb] = pruneStates(preImageSet.state[stepb],vb);
	  if (preImageSet.state[stepb] != bddfalse)
	    {
	      vbOld = vb;
	      vb |= preImageSet.state[stepb];
	      lastPreImageCPUtime = preImageTimer.stop();
	      if (verbosity > 0) {
		cout << "Step " << stepf + stepb << "  PreIm " << stepb << "  Time " 
		     << lastPreImageCPUtime  << "  Fringe " 
		     << bdd_nodecount(preImageSet.state[stepb]) << "/" << bdd_nodecount(vb) << "\n";
		fringeSize.push_back(bdd_nodecount(preImageSet.state[stepb]));
	      }
	    }
	  else
	    {
	      cout << "Empty PreImage: No solution exists!\n";
	      return solution; // empty
	    } 
	}
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      forwardExpansionNum = stepf;
      backwardExpansionNum = stepb;
      iterationNum = stepf + stepb;
    }

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  // post: the last preImage and image overlaps 
  //       collapse it to one state in the intersection

  //select one state in overlap
  bdd ints;
  ints = imageSet.state[stepf] & preImageSet.state[stepb];
  ints = bdd_satone(ints);
  imageSet.state[stepf] = ints;
  preImageSet.state[stepb] = ints;

  
  bdd nextState;
  // Extract solution from forward traversal of preImages
  for (i= stepb-1; i >=0; i--)
    {
      int j=-1;
      nextState = bddfalse;
      while (nextState == bddfalse)
	{
	  j++;      
	  nextState = successorOfAct(Tact,preImageSet.state[i+1],preImageSet.state[i],j);
	}
      if (verbosity > 1) cout << "Extracted forward: " << Tact.p[j].actName << endl;
      cout.flush();
      preImageSet.state[i] = nextState;
      solution.push_back(Tact.p[j].actName);      
    }

  bdd previousState;
  // Extract solution from backward traversal of images
  for (i = stepf-1; i >=0; i--)
    {
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = predecessorOfAct(Tact,imageSet.state[i],imageSet.state[i+1],j);
	}
      if (verbosity > 1) cout << "Extracted backward: " << Tact.p[j].actName << endl;
      cout.flush();
      imageSet.state[i] = previousState;
      solution.push_front(Tact.p[j].actName);      
    }

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in forward fringe: " 
	   << aveFringeSize << endl;
    }


  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " << Textract << " seconds\n";
    }
  
  return solution;
}




////////////////////////////////////////////////////////
// FORWARD SEARCH
//
// 1) image calculation
//    using a forward dis partitioning and a regular
//    action backward partitioning
//
// Preconditions:
// 1) The initial state is assumed to be totally defined 
//   (otherwise the forward search is inconsistent)
////////////////////////////////////////////////////////


//IN
// Tact     : disjunctive partitioning where each partition corresponds to
//               a single action (for action extraction)
// T        : disjunctive backward/forward partitioning  (search)
// init        : initial state
// goal        : goal states
// OUT
// solution (empty if no result)
list<string> forward(TRel& Tact,TRel& T,bdd init, bdd goal, bool frontierSimpl) {

  int i;
  stateSet preImageSet(MAXSOLLENGTH);
  stateSet imageSet(MAXSOLLENGTH);  
  int stepf;
  int stepb;
  bdd initpoint;
  bdd vf,vb;  // forward and backward visited states
  bdd vfOld,vbOld;
  timer imageTimer, preImageTimer,searchClk,extractClk;
  list<string> solution;
  bdd frontier;
  vector<int> fringeSize;


  //  bidirectional search
  stepf=0;
  stepb=0;
  vf = init;
  vb = goal;
  vfOld = bddfalse;
  vbOld = bddfalse;
  preImageSet.state[0] = goal;
  imageSet.state[0] = init;
  double lastImageCPUtime = -10.0;
  double lastPreImageCPUtime = -10.0;

  if (verbosity > 0)
    searchClk.start();

  while ((imageSet.state[stepf] & preImageSet.state[stepb]) == bddfalse) 
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      // make image
      imageTimer.start();
      stepf++;
      
      // frontier set simplification
      if (frontierSimpl) {
	if (stepf == 1)
	  frontier = vf;
	else
	  frontier = bdd_simplify(imageSet.state[stepf-1],!vfOld);
      }
      else
	frontier = imageSet.state[stepf-1];
      
      imageSet.state[stepf] = image(T,frontier);
      imageSet.state[stepf] = pruneStates(imageSet.state[stepf],vf);
      if (imageSet.state[stepf] != bddfalse)
	{
	  vfOld = vf;
	  vf |= imageSet.state[stepf];
	  lastImageCPUtime = imageTimer.stop();

	  if (verbosity > 0)
	    {
	      cout << "Step " << stepf + stepb << "  Image " << stepf << "  Time " 
		   << lastImageCPUtime << "  Fringe " << bdd_nodecount(imageSet.state[stepf]) 
		   << "/" << bdd_nodecount(vf)<< "\n";
	      fringeSize.push_back(bdd_nodecount(imageSet.state[stepf]));
	    }
	}
      else
	{
	  cout << "Empty Image: No solution exists!\n";
	  return solution; //empty
	}
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = stepf + stepb;
    }


  // post: the last image and goal overlaps 
  //       collapse it to one state in the intersection

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  //select one state in overlap
  bdd ints;
  ints = imageSet.state[stepf] & preImageSet.state[stepb];
  ints = bdd_satone(ints);
  imageSet.state[stepf] = ints;
  preImageSet.state[stepb] = ints;
  

  bdd previousState;
  // Extract solution from backward traversal of images
  for (i = stepf-1; i >=0; i--)
    {
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = predecessorOfAct(Tact,imageSet.state[i],imageSet.state[i+1],j);
	}
      if (verbosity > 1) cout << "Extracted backward: " << Tact.p[j].actName << endl;
      cout.flush();
      imageSet.state[i] = previousState;
      solution.push_front(Tact.p[j].actName);      
    }
   

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in backward/forward fringe: " 
	   << aveFringeSize << endl;
    }

  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " << Textract << " seconds\n";
    }
  
  return solution;
}





////////////////////////////////////////////////////////
//  DISJUNCTIVE BACKWARDSOLUTION
// 
//  1) Backward preImage calculation
//     using a Dis partitioning 
// 
//  Preconditions:
//  1) The initial state is assumed to be totally defined 
//     (otherwise the forward search is inconsistent)
////////////////////////////////////////////////////////


//IN
// Tact     : disjunctive partitioning where each partition corresponds to
//               a single action (for action extraction)
// T        : disjunctive backward/forward partitioning  (search)
// init        : initial state
// goal        : goal states
// OUT
// solution (empty if no result)
list<string> backward(TRel& Tact,TRel& T,bdd init, bdd goal, bool frontierSimpl) {

  int i;
  stateSet preImageSet(MAXSOLLENGTH);
  stateSet imageSet(MAXSOLLENGTH);  
  int stepf;
  int stepb;
  bdd initPoint;
  bdd vf,vb;  // forward and backward visited states
  bdd vfOld,vbOld;
  timer imageTimer, preImageTimer,searchClk,extractClk;
  list<string> solution;
  bdd frontier;
  vector<int> fringeSize;


  //  bidirectional search
  stepf=0;
  stepb=0;
  vf = init;
  vb = goal;
  vfOld = bddfalse;
  vbOld = bddfalse;
  preImageSet.state[0] = goal;
  imageSet.state[0] = init;
  double lastImageCPUtime = -10.0;
  double lastPreImageCPUtime = -10.0;

  if (verbosity > 0)
    searchClk.start();

  while ((imageSet.state[stepf] & preImageSet.state[stepb]) == bddfalse) 
    {

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      
      // Make preImage
      preImageTimer.start();
      stepb++;	  
      
      // frontier set simplification
      if (frontierSimpl) {
	if (stepb == 1)
	  frontier = vb;
	else
	    frontier = bdd_simplify(preImageSet.state[stepb-1],!vbOld);
      }	    
      else
	frontier = preImageSet.state[stepb-1];
	 
       
      preImageSet.state[stepb] = preImage(T,frontier);
      
      preImageSet.state[stepb] = pruneStates(preImageSet.state[stepb],vb);
      if (preImageSet.state[stepb] != bddfalse)
	{
	  vbOld = vb;
	  vb |= preImageSet.state[stepb];
	  lastPreImageCPUtime = preImageTimer.stop();
	  if (verbosity > 0)
	    {
	      cout << " Step " << stepf + stepb << "  PreIm " << stepb 
		   << "  Time " << lastPreImageCPUtime  << "  Fringe " 
		   << bdd_nodecount(preImageSet.state[stepb]) << "/" 
		   << bdd_nodecount(vb) << "\n";
	      fringeSize.push_back(bdd_nodecount(preImageSet.state[stepb]));
	    }
	}
      else
	{
	  cout << "Empty PreImage: No solution exists!\n";
	  return solution; // empty
	} 
    }

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = stepf + stepb;
    }


  // post: the last preImage and image overlaps 
  //       collapse it to one state in the intersection

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;



  //select one state in overlap
  bdd ints;
  ints = imageSet.state[stepf] & preImageSet.state[stepb];
  ints = bdd_satone(ints);
  imageSet.state[stepf] = ints;
  preImageSet.state[stepb] = ints;
  

  bdd nextState;
  // Extract solution from forward traversal of preImages
  for (i= stepb-1; i >=0; i--)
    {
      int j=-1;
      nextState = bddfalse;
      while (nextState == bddfalse)
	{
	  j++;      
	  nextState = successorOfAct(Tact,preImageSet.state[i+1],preImageSet.state[i],j);
	}
      if (verbosity > 1) cout << "Extracted forward: " << Tact.p[j].actName << endl;
      cout.flush();
      preImageSet.state[i] = nextState;
      solution.push_back(Tact.p[j].actName);      
    }

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in backward fringe: " 
	   << aveFringeSize << endl;
    }

  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " << Textract << " seconds\n";
    }
    
  return solution;
}




























////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
// 
//	     ---- HEURISTIC SEARCH ----
//   
////////////////////////////////////////////////////////
////////////////////////////////////////////////////////



////////////////////////////////////////////////////////
//  SetA*
//  				
//  Weighted A* search algorithm generalized to sets 
//  of states 
// 
//
////////////////////////////////////////////////////////



//IN
// Tact     : disjunctive partitioning where each partition corresponds
//               to an action (for action extraction)
// T        : disjunctive split partitioning where each partition corresponds to a 
//               splitted action fraction 
// init        : initial states (= PDDL goal states)
// goal        : goal states (= PDDl init state)
// l           : lower bound on BDDs in queue structure (currently not used for anything)
// u           : upper bound on BDDs in queue structure
// wg           : 
// wh           : f = wg * g + wh*h
//OUT
// solution (empty if no result)
list<string> SetAstar(TRel& Tact,TRel& T,bdd init, bdd goal,int hInit,
		      int upperBound, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;

  ghQueue ghQ(upperBound,wg,wh);
  reachedStates RS(MAXDEPTH);  
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = hInit;
  g = 0;
  goalOverLap = (init & goal) != bddfalse;
  ghQ.insert(init,goalOverLap,g,h);
  RS.update(init,g);


  if (verbosity > 0)
      searchClk.start();  
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    { 

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

     
      currentSet = ghQ.popTop(f,g,h);
      it++;
      if (verbosity > 0)
	{
	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	  expandClk.start();	  
	}

	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionImage(T,currentSet,i);
	  
	  // prune child according to reached states
	  child = RS.prune(child,g + T.p[i].dg);
	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		  {
		    if (T.p[i].dh < 0)
		      cout << "+";
		    else if (T.p[i].dh > 0)
		      cout << "-";
		    else
		      cout << "0";		      
		    cout.flush();
		  }
	      
	      goalOverLap = (child & goal) != bddfalse;
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,h + T.p[i].dh);
	      RS.update(child,g + T.p[i].dg);
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      
      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      iterationNum = it;
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  
  
  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
      solution =  RS.extractSolution(Tact,goal,ghQ.gOfTop());      

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  return solution;
}







////////////////////////////////////////////////////////
//    SetAstarBackward
//  				
//  Weighted A* search algorithm generalized to sets 
//  of states. This algorithm simulates a backward search.
//  Thus init = PDDL goal states and goal = PDDL init states 
// 
//  Comments :
//         Used for the HSPr heuristic
//
//  Assumptions :
// 
//  1) init is a single state
//  2) goal is a single state (otherwise it's the heuristic 
//     value for backward search is ill defined)
//
////////////////////////////////////////////////////////



//IN
// Tact         : disjunctive partitioning where each partition corresponds
//                to an action (for action extraction)
// T            : disjunctive split partitioning where each partition corresponds to a 
//                splitted action fraction 
// init         : initial states paired with their H-value (= PDDL goal state)
// goal         : goal states (= PDDL init state)
// upperBound   : upper bound on BDDs in queue structure
// wg           : 
// wh           : f = wg * g + wh*h
//OUT
// solution (empty if no result)
list<string> SetAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,
			      int upperBound, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;

  ghQueue ghQ(upperBound,wg,wh);
  reachedStates RS(MAXDEPTH);  
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = hInit;
  g = 0;
  goalOverLap = (init & goal) != bddfalse;
  ghQ.insert(init,goalOverLap,g,h);
  RS.update(init,g);


  if (verbosity > 0)
      searchClk.start();  
  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      currentSet = ghQ.popTop(f,g,h);
      it++;
      if (verbosity > 0)
	{
	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionPreImage(T,currentSet,i);
	  
	  // prune child according to reached states
	  child = RS.prune(child,g + T.p[i].dg);
	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dh > 0)
		    cout << "+";
		  else if (T.p[i].dh < 0)
		    cout << "-";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = (child & goal) != bddfalse;
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,h - T.p[i].dh);
	      RS.update(child,g + T.p[i].dg);
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      
      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
      solution =  RS.extractSolutionBackward(Tact,goal,ghQ.gOfTop());      

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  return solution;
}













////////////////////////////////////////////////////////
//  
//  FminAstar
//  				
//  A*  search. Expands the 
//  whole fmin set in each iteration (thus, its purpose
//  is to simulate BDDA* with the SetA*
//  approach). 
//
//  The branching parition has been changed such that dg 
//  actually corresponds to df (actually RESOLUTION times df 
//  Since dg must be an integer)
//
//  df = RESOLUTION*wg*dg + RESOLUTION*wh*dh     <=>
//
//  Obs notice that this changes Finit from
//  Finit = wh*Hinit to
//  Finit = RESOLTION*wh*Hinit 
//
//  ==============================================
//  Simulates BDDA* when wg = 1  and wh = 1 and 
//  the upperbound is infinit
//  ==============================================
//
//
//  Assumptions :
// 
//  1) init is a single state
//
////////////////////////////////////////////////////////



//IN
// Tact       : disjunctive partitioning where each partition corresponds
//          	to an action (for action extraction). In this partition dg
//          	corresponds to df for weighted Astar scaled, to allow integer
//              representation (dg = RESOLUTION*wg*dg + RESOLUTION*wh*dh) 
// T          : disjunctive split partitioning search partition
// init       : initial state
// goal       : goal states 
// hInit      : h value in init        
// u          : upper bound (should be inifinit to simulate BDDA*)
// wg         :
// wh         : Currently NOT ACTIVE !!
 //OUT
// solution (empty if no result)
// OBS OBS OBS
//  f = RESOLTUION*wg*g + RESOLTUION*wh*h
//  however this information is assumed to be written into g thus wg = 1, wh = 0
list<string> FminAstar(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,int u,double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;
  vector< pair<int,bdd> > fringe;

  //  f = RESOLTUION*wg*g + RESOLTUION*wh*h
  //  however this infor is assumed to be written into g thus wg = 1, wh = 0
  ghQueue ghQ(u,1.0,0.0); 
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ by inserting init
  h = 0;
  g = int(WEIGHTRESOLUTION*wh*hInit);
  goalOverLap = ( (init & goal) != bddfalse );
  // wg = 1.0, wh = 0.0 such that f = g
  ghQ.insert(init,goalOverLap,g,h); 

  if (verbosity > 0)
      searchClk.start();  

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      currentSet = ghQ.popTop(f,g,h);

      pair<int,bdd> elem(g,currentSet);
      fringe.push_back(elem);

	   	
      it++;
      if (verbosity > 0)
	{
	  if ( (currentSet & goal) != bddfalse ) cout << "T";

	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionImage(T,currentSet,i);
	  	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dg > 0)
		    cout << "-";
		  else if (T.p[i].dg < 0)
		    cout << "+";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = ( (child & goal) != bddfalse );
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,0);	      
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      

      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
    {
      // extract the solution in a similar way to BDDA*

      bdd currentState = ghQ.popTop(f,g,h);
      currentState = bdd_satone(goal & currentState);
      int currentF = g; 

      // find the fringe of the currentState
      int currentFringe = fringe.size();

      int nextFringe;
      bool nextFringeFound;

      // Extract solution from backward traversal of fringe
      while ( (currentState & init) == bddfalse )
	{
	  nextFringeFound = false;
	  nextFringe = currentFringe;
	  while ( !nextFringeFound )
	    {
	      nextFringe--;
	      if (nextFringe < 0)
		{
		  cout << "deterministic.search.cc : FminAstar : No more" 
		       << " fringes left\nexiting\n";
		  exit(1);
		}
	      
	      // go through each action in order to see if it could lead 
	      // to the fringe (break if solution found
	      for (int j = 0; j < Tact.p.size(); j++)
		{
		  bdd nextState = predecessorOfAct(Tact,fringe[nextFringe].second,currentState,j); 

		  if (nextState != bddfalse && fringe[nextFringe].first == (currentF - Tact.p[j].dg) )
		    {
		      // nextFringe found, update currentState
		      nextFringeFound = true;
		      currentF = fringe[nextFringe].first;
		      currentState = bdd_satone(nextState);
		      currentFringe = nextFringe;
		      if (verbosity > 1) 
			cout << "Extracted backward: " << Tact.p[j].actName << endl;
		      solution.push_front(Tact.p[j].actName);      
		      break;
		    }
		}
	    }
	}
    }


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }


  return solution;
}






////////////////////////////////////////////////////////
//  
//  FminAstarBackward
//  				
//  A*  search. Expands the 
//  whole fmin set in each iteration (thus, its purpose
//  is to simulate BDDA* with the SetA*
//  approach). This algorithm simulates a backward search.
//  Thus init = PDDL goal states and goal = PDDL init states 
//
//  The branching parition has been changed such that dg 
//  actually corresponds to df (actually RESOLUTION times df 
//  Since dg must be an integer)
//
//  df = RESOLUTION*wg*dg + RESOLUTION*wh*dh     <=>
//
//  Obs notice that this changes Finit from
//  Finit = wh*Hinit to
//  Finit = RESOLTION*wh*Hinit 
//
//  ============================================
//  Simulates BDDA* when wg = 1.0, wh = 1.0 and  
//  the upperbound is infinit
//  ============================================
//
//  Comments :
//         Used for the HSPr heuristic 
//
//  Assumptions :
// 
//  1) init is a single state
//  2) goal is a single state (otherwise it's the heuristic 
//     value for backward search is ill defined)
//
////////////////////////////////////////////////////////



//IN
// Tact       : disjunctive partitioning where each partition corresponds
//          	to an action (for action extraction). In this partition dg
//          	corresponds to df for weighted Astar scaled, to allow integer
//              representation (dg = RESOLUTION*wg*dg + RESOLUTION*wh*dh) 
// T          : disjunctive split partitioning search partition
// init       : initial states paired with their H-value (= PDDL goal state)
// goal       : goal states (= PDDL init state)
// hInit      : h value in init        
// u            : upper bound (should be inifinit to simulate BDDA*)
// wg           : OBS NOT ACTIVE !!!!!!
// wh           : OBS NOT ACTIVE !!!!!!
//                 f = RESOLTUION*wg*g + RESOLTUION*wh*h
//OUT
// solution (empty if no result)
list<string> FminAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,int u, double wg, double wh) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;
  vector< pair<int,bdd> > fringe;

  // wg = 1.0, wh = 0.0 such that f = g
  ghQueue ghQ(u,1.0,0.0); 
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = 0;
  g = int(WEIGHTRESOLUTION*wh*hInit);
  goalOverLap = ( (init & goal) != bddfalse );
  ghQ.insert(init,goalOverLap,g,h); 


  if (verbosity > 0)
      searchClk.start();  

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      currentSet = ghQ.popTop(f,g,h);

      pair<int,bdd> elem(g,currentSet);
      fringe.push_back(elem);

	   	
      it++;
      if (verbosity > 0)
	{
	  if ( (currentSet & goal) != bddfalse ) cout << "T";

	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionPreImage(T,currentSet,i);
	  	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dg > 0)
		    cout << "-";
		  else if (T.p[i].dg < 0)
		    cout << "+";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = ( (child & goal) != bddfalse );
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,0);	      
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      

      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
    {
      // extract the solution in a similar way to BDDA*

      bdd currentState = ghQ.popTop(f,g,h);
      currentState = bdd_satone(goal & currentState);
      int currentF = g; 

      // find the fringe of the currentState
      int currentFringe = fringe.size();

      int nextFringe;
      bool nextFringeFound;

      // Extract solution from forward traversal of fringe
      while ( (currentState & init) == bddfalse )
	{
	  nextFringeFound = false;
	  nextFringe = currentFringe;
	  while ( !nextFringeFound )
	    {
	      nextFringe--;
	      if (nextFringe < 0)
		{
		  cout << "deterministic.search.cc : FminAstar : No more" 
		       << " fringes left\nexiting\n";
		  exit(1);
		}
	      
	      // go through each action in order to see if it could lead 
	      // to the fringe (break if solution found
	      for (int j = 0; j < Tact.p.size(); j++)
		{
		  bdd nextState = successorOfAct(Tact,currentState,fringe[nextFringe].second,j);
		  if (nextState != bddfalse && fringe[nextFringe].first == (currentF - Tact.p[j].dg) )
		    {
		      // nextFringe found, update currentState
		      nextFringeFound = true;
		      currentF = fringe[nextFringe].first;
		      currentState = bdd_satone(nextState);
		      currentFringe = nextFringe;
		      if (verbosity > 1) 
			cout << "Extracted forward: " << Tact.p[j].actName << endl;
		      solution.push_back(Tact.p[j].actName);      
		      break;
		    }
		}
	    }
	}
    }


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }


  return solution;
}






////////////////////////////////////////////////////////
//  
//  PureFminAstarBackward
//  				
//  best-first  search. Expands the 
//  whole fmin set in each iteration (thus, its purpose
//  is to simulate pure-BDDA* the SetA*
//  approach). This algorithm simulates a backward search.
//  Thus init = PDDL goal states and goal = PDDL init states 
// 
//  Comments :
//         Used for the HSPr heuristic 
//
//  Assumptions :
// 
//  1) init is a single state
//  2) goal is a single state (otherwise it's the heuristic 
//     value for backward search is ill defined)
//
////////////////////////////////////////////////////////



//IN
// Tact       : disjunctive partitioning where each partition corresponds
//          	to an action (for action extraction). In this partition dg
//          	corresponds to df (e.g. dg = dg + dh for Astar and dg = dh for
//          	pure Astar)
// T          : disjunctive split partitioning search partition
// init       : initial states paired with their H-value (= PDDL goal state)
// goal       : goal states (= PDDL init state)
//OUT
// solution (empty if no result)
list<string> pureFminAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit) {

  double f;
  int g,h,i;
  bool goalOverLap;
  bdd currentSet;
  vector<int> fringeSize;
  vector< pair<int,bdd> > fringe;
  bdd reach = bddfalse; // reach set

  // wg = 1.0 and wh = 0.0 such that f = g (we don't seperate the information)
  ghQueue ghQ(100000000,1.0,0.0); // upper bound is infinit
  timer expandClk,searchClk,extractClk;

  int maxQsize = 0;
  int it = 0;

  
  // initialize gsQ and RS by inserting init
  h = 0;
  g = hInit;
  goalOverLap = (init & goal) != bddfalse;
  ghQ.insert(init,goalOverLap,g,h); 
  reach |= init;


  if (verbosity > 0)
      searchClk.start();  

  
  /////////////////////////////////
  // Main loop
  /////////////////////////////////
  bool stop;
  if (ghQ.empty()) stop = true;
  else stop = ghQ.topHasGoalOverlap();

  while ( !stop )
    {      

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      currentSet = ghQ.popTop(f,g,h);
      it++;

      // insert fmin set on fringe
      pair<int,bdd> elem(g,currentSet);
      fringe.push_back(elem);


      if (verbosity > 0)
	{
	  cout << "|Q|=" << ghQ.size() << " it=" << it << 
	    " Expanding (f=" << f << ",g=" << g << ",h=" << 
	    h <<",size=" << bdd_nodecount(currentSet) << ") children=[";
	  if (ghQ.size() > maxQsize) maxQsize = ghQ.size();
	  fringeSize.push_back(bdd_nodecount(currentSet));
	  cout.flush();
	}


	  
      if (verbosity > 0) 
	expandClk.start();	  
      
      // go through each branch and add the new nodes to ghQ
      for (i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  bdd child;
	  child = partitionPreImage(T,currentSet,i);
	  // prune reached children
	  child = child & !reach;
	  	  
	  // add child to ghQ if nonempty
	  if (child != bddfalse)
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dh > 0)
		    cout << "+";
		  else if (T.p[i].dh < 0)
		    cout << "-";
		  else
		    cout << "0";
		  cout.flush();
		}

	      goalOverLap = (child & goal) != bddfalse;
	      ghQ.insert(child,goalOverLap,g + T.p[i].dg,0);
	      reach |= child;
	    }
	}
      
      if (verbosity > 0)
	cout << "] Tsplit=" << expandClk.stop() << endl;
      
      if (ghQ.empty()) stop = true;
      else stop = ghQ.topHasGoalOverlap();
    }
  
  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
  

  // extract solution
  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (ghQ.empty())
    {
      // no solution to extract
    }
  else
    {
      // extract the solution in a similar way to BDDA*

      bdd currentState = ghQ.popTop(f,g,h);
      currentState = bdd_satone(goal & currentState);
      int currentF = g; 
				    
      // find the fringe of the currentState
      int currentFringe = fringe.size();
      int nextFringe;
      bool nextFringeFound;

      // Extract solution from forward traversal of fringe
      while ( (currentState & init) == bddfalse )
	{
	  nextFringeFound = false;
	  nextFringe = currentFringe;
	  while ( !nextFringeFound )
	    {
	      nextFringe--;
	      if (nextFringe < 0)
		{
		  cout << "deterministic.search.cc : FminAstar : No more" 
		       << " fringes left\nexiting\n";
		  exit(1);
		}
	      
	      // go through each action in order to see if it could lead 
	      // to the fringe (break if solution found
	      for (int j = 0; j < Tact.p.size(); j++)
		{
		  bdd nextState = successorOfAct(Tact,currentState,fringe[nextFringe].second,j);
		  if (nextState != bddfalse && fringe[nextFringe].first == currentF - Tact.p[j].dg)
		    {
		      // nextFringe found, update currentState
		      nextFringeFound = true;
		      currentF = fringe[nextFringe].first;
		      currentFringe = nextFringe;
		      currentState = bdd_satone(nextState);
		      if (verbosity > 1) 
			cout << "Extracted forward: " << Tact.p[j].actName << endl;
		      solution.push_back(Tact.p[j].actName);      
		      break;
		    }
		}
	    }
	}
    }


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  return solution;
}


