///////////////////////////////////////////////////////
// File  : detAndNonDet.search.hpp  
// Desc. : Header for detAndNonDet.search.cc
// Author: Rune M. Jensen CS, CMU
// Date  : 1/13/02 (non-det 8/28/02)
///////////////////////////////////////////////////////

#ifndef SEARCHHPP
#define SEARCHHPP

#include <list>
#include <utility>
#include <queue>

#include "bddLayout.hpp"
#include "partition.hpp"
#include "universal.hpp"


///////////////////////////////////////////////////////
// Statistic externs
///////////////////////////////////////////////////////

extern double Tsearch;
extern double Textract;
extern double aveFringeSize;
extern int    maxSearchQueueSize;
extern int    forwardExpansionNum;
extern int    backwardExpansionNum;
extern int    iterationNum;
extern int    timeout;



struct stateSet {
  bdd *state;
  int size;
  stateSet(bdd *st,int s) {state = st; size=s;}
  stateSet(int s) {state = new bdd[s]; size=s;}
};


///////////////////////////////////////////////////////
// Structures for Priority Queue datastructure
///////////////////////////////////////////////////////


struct bddNode {
  int size;
  bdd b;
  bddNode(int si, bdd bi) {size = si; b = bi;}
};


struct bddNodeCmp {
  bool operator() (bddNode n1,bddNode n2) 
  {return (n1.size > n2.size);}
};


struct Mvalue {
  priority_queue<bddNode,vector<bddNode>,bddNodeCmp> q;      
                      // BDD priority queue
  bool goalOverlap;   // true if some state in the node overlap with the goal states
  Mvalue(priority_queue<bddNode,vector<bddNode>,bddNodeCmp> qi,bool goalOverlapi)
  {q = qi; goalOverlap = goalOverlapi;}
  Mvalue() {};        // default value needed for map
    // debug { goalOverlap = false; }
};


struct Qvalue {
  double f;
  int g;
  int h;
  Qvalue(double fi,int gi,int hi) 
  {f = fi; g = gi; h = hi;} 
};


// lowest f value gets highest priority
// ties are broken in favor of lowest h value
struct QvalueCmp {
  bool operator() (Qvalue v1,Qvalue v2) 
  {if (v1.f == v2.f) return (v1.h > v2.h);
  else return (v1.f > v2.f);}
};


// ghqueue abstract datatype
struct ghQueue {
  int upperBound;         // coalescing upper bound
  double wg;
  double wh;
  int bddNum;             // number of BDDs in the queue 
  map< pair<int,int> , Mvalue > M;                   // mapping from (g,h) -> bdd node priority queue
  priority_queue<Qvalue,vector<Qvalue>,QvalueCmp> Q; // pqueue keeping track of (g,h) entry 
                                                     // with highest priority 
  ghQueue(int u, double wgi, double whi) 
  {upperBound = u; wg = wgi; wh = whi; bddNum = 0;}
  bdd popTop(double& f,int& g,int& h);
  void insert(bdd set,bool goalOverlap,int g,int h);
  bool empty() { Q.empty(); };
  void clear();
  bool topHasGoalOverlap();
  int  gOfTop();
  int size() { return bddNum; }
  void print();
  
};




///////////////////////////////////////////////////////
// Structure for loop detection
///////////////////////////////////////////////////////


// within reachable sets for pruning A* search paths
struct  reachedStates {
  vector<bdd> withinSet;      // keeps the set of states seen within the given g-level        
  int seenDepth;
  void update(bdd states,int g); 
  bdd prune(bdd states,int g);
  bdd within(int g); // returns the within set of g
  list<string> extractSolution(TRel& TAct, bdd goal, int gLevel);
  list<string> extractSolutionBackward(TRel& TAct, bdd goal, int gLevel);
  void print();
  reachedStates(int maxDepth)  // init maxDepth to above the max search depth
    {seenDepth=0; withinSet.resize(maxDepth+2,bddfalse);}
};




  

///////////////////////////////////////////////////////
// function prototypes
///////////////////////////////////////////////////////



// aux functions
bdd preImage(TRel& T, bdd v);
bdd image(TRel& T, bdd v);
bdd predecessorOfAct(TRel& Tact,bdd set1,bdd set2,int act);
bdd successorOfAct(TRel& Tact,bdd set1,bdd set2,int act);

////////////////////////////////
// deterministic Search
////////////////////////////////

// blind search
list<string> bidir(TRel& TAct,TRel& T,bdd init, bdd goal, bool frontierSimpl);
list<string> forward(TRel& TAct,TRel& T,bdd init, bdd goal, bool frontierSimpl);
list<string> backward(TRel& TAct,TRel& T,bdd init, bdd goal, bool frontierSimpl);

// heuristic search
list<string> SetAstar(TRel& TAct,TRel& T,bdd init, bdd goal, 
		      int hinit,int upperBound, double wg, double wh);
list<string> SetAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,
		      int hinit,int upperBound, double wg, double wh);
list<string> FminAstar(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,int u, double wg, double wh);
list<string> FminAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit,int u, double wg, double wh);
list<string> pureFminAstarBackward(TRel& Tact,TRel& T, bdd init, bdd goal,int hInit);

////////////////////////////////
// non-deterministic Search
////////////////////////////////

// aux. functions
bdd strong(TRel& T, bdd r, bdd s);


// strong search
int strongSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan);
int strongHeuristicSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan,
			  int hGoal,int u);
int strongCyclicSearch(TRel& T, ActInfo& Ainfo,bdd init,bdd goal,bdd& universalPlan);
int strongCyclicHeuristicSearch(TRel& T, ActInfo& Ainfo, bdd init,bdd goal, 
				bdd& universalPlan,int hGoal,int u);
int weakSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan);
int weakHeuristicSearch(TRel& T, ActInfo& Ainfo, bdd init, bdd goal, bdd& universalPlan,
			int hGoal,int u);




#endif  
