//////////////////////////////////////////////////////////
// File  : bdda.search.cc
// Desc. : pure-BDDA* and BDDA* algorithm implementations
// Author: Rune M. Jensen
// Date  : 5/8/02
//////////////////////////////////////////////////////////

#include <string>
#include <list>
#include <stream.h>
#include <vector>

#include "bdda.search.hpp"
#include "bddLayout.hpp"
#include "symbolicHeuristic.hpp"
#include "partition.hpp"
#include "timer.hpp"
#include "searchAlgAuxFunc.hpp" 



//////////////////////////////////////////////////////////
//
// pure-BDDA* for searching backwards using e.g. 
// the HSPr heuristic
// (pure-BDDA* = weight A* with w = 1.0 =
//  best-first search)
//
// 1) This algorithm assumes unit edge cost
// 2) the algorithm only uses the h-encoding
//    since f = h 
//////////////////////////////////////////////////////////

//IN
// bInfo     : data structure with h and f substitution and 
//             quantification information
// heuristic : bdd encoding of f-value and state pairs
//             (represented in current state variables)
// TAct      : transition relation where each state equals
//             an action (disjunctive partitioning)
// T         : an optimized transition relation  
//OUT
// solution
list<string> pureBDDAstarBackward(PDDLbddBDDAlayout& layout,FHinfo& fhInfo,
				  bdd heuristic,TRel &Tact,TRel &T,bdd init, bdd goal) {
  
  bdd open;   // search queue of A*
  bdd min;    // part of searc queue with minimum hvalue
  bdd reach;  // states reached so far
  vector<bdd> fringe; // vector of reached states
  vector<int> fringeSize;
  timer expandClk,searchClk,extractClk;
  bdd next;
  
  // statistics
  int it = 0;

  if (verbosity > 0)
    searchClk.start();

    
  // initialize open, reach and fringe
  open = heuristic & init;
  reach = init;
  fringe.push_back(init);
  
  
  // main loop
  while (open != bddfalse)
    {      

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      // statistics
      if (verbosity > 0)	
	expandClk.start();

      // find hmin set
      int hmin =  hMin(layout,min,open);
      
      // break out if goal reached
      if ( (min & goal) != bddfalse )
	break;

      it++;

      

      // subtract min from open
      open = open & !min;
      
      // abstract h values from min
      min = bdd_exist(min,fhInfo.Hcurrent);
      
      // find next states from min (backward, so we take the preimage)
      next = preImage(T,min);

      // prune states seen so far
      next = next & !reach;

      // update reach and fringe
      reach |= next;
      fringe.push_back(next);

      // update open
      open = open | (next & heuristic);       


      if (verbosity > 0)
	{
	  cout << "Step=" << it << " Hmin=" << hmin << " minSize=" << bdd_nodecount(min) << " Timg=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(min));
	} 
    }

  // while loop ended : either because open is empty or because a 
  // solution was found

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = it;
    }



  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  bdd ints = bdd_exist(min & goal,fhInfo.Hcurrent); 
  

  if (ints != bddfalse)
    {
      bdd currentState = bdd_satone(ints);          
      bdd nextState;
      bdd nextStates; 
      int fringeNo = fringe.size() - 1;

  
      //select one state in intersection
      // Extract solution from forward traversal of preImages
      while ( (currentState & init) == bddfalse )
	{
	  // 1) find the first fringe overlapping with 
	  //    the next states of the current state
	  nextStates = image(T,currentState);
	  while (  ( fringe[fringeNo] & nextStates ) ==  bddfalse )
	    fringeNo--;

	  // 2) find a particular action leading to one of the states in the overlap
	  int j = -1;
	  nextState = bddfalse;
	  while (nextState == bddfalse)
	    {
	      j++;      
	      nextState = successorOfAct(Tact,currentState,fringe[fringeNo],j);
	    }
	  if (verbosity > 1) cout << "Extracted forward: " << Tact.p[j].actName << endl;
	  cout.flush();
	  currentState = nextState;
	  solution.push_back(Tact.p[j].actName);      
	  fringeNo--; // look at next fringe
	}


    }



  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }



  if (verbosity > 0)
    {
      Textract =  extractClk.stop();
      cout << "Total extraction time: " << Textract << " seconds\n";
    }
  

  return solution;
}






//////////////////////////////////////////////////////////
// Published version of pure-BDDA*
//
// pure-BDDA* for searching backwards using e.g. 
// the HSPr heuristic
// (pure-BDDA* = weight A* with w = 1.0 =
//  best-first search)
//
// 1) This algorithm assumes unit edge cost
// 2) the algorithm only uses the h-encoding
//    since f = h 
//////////////////////////////////////////////////////////

//IN
// bInfo     : data structure with h and f substitution and 
//             quantification information
// heuristic : bdd encoding of f-value and state pairs
//             (represented in current state variables)
// TAct      : transition relation where each state equals
//             an action (disjunctive partitioning)
// T         : a monolithic transition relation  
//OUT
// solution
list<string> PubpureBDDAstarBackward(PDDLbddPubBDDAlayout& layout,PubFHinfo& fhInfo,
				  bdd heuristic,TRel &Tact,TRel &T,bdd init, bdd goal) {
  
  bdd open;   // search queue of A*
  bdd min;    // part of searc queue with minimum hvalue
  bdd reach;  // states reached so far
  vector<int> fringeSize;
  timer expandClk,searchClk,extractClk;
  bdd next;
  
  // statistics
  int it = 0;

  if (verbosity > 0)
    searchClk.start();

    
  // initialize open, reach and fringe
  open = heuristic & init;
  reach = init;
  
  
  // main loop
  while (open != bddfalse)
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      // statistics
      if (verbosity > 0)	
	expandClk.start();

      // find hmin set
      int hmin =  PubhMin(layout,min,open);
      
      // break out if goal reached
      if ( (min & goal) != bddfalse )
	break;

      it++;

      

      // subtract min from open
      open = open & !min;
      
      // abstract h values from min
      min = bdd_exist(min,fhInfo.Hcurrent);
      
      // find next states from min (backward, so we take the preimage)
      next = preImage(T,min);

      // prune states seen so far
      next = next & !reach;

      // update reach and fringe
      reach |= next;

      // update open
      open = open | (next & heuristic);       


      if (verbosity > 0)
	{
	  cout << "Step=" << it << " Hmin=" << hmin << " minSize=" << bdd_nodecount(min) << " Timg=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(min));
	} 
    }

  // while loop ended : either because open is empty or because a 
  // solution was found

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = it;
    }



  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;


  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }



  if (verbosity > 0)
    {
      Textract =  extractClk.stop();
      cout << "Total extraction time: " << Textract << " seconds\n";
    }
  

  return solution;
}







//////////////////////////////////////////////////////////
//
// BDDA* for regular forward search
//
// 1) This algorithm assumes unit edge costs
//////////////////////////////////////////////////////////

//IN
// fhInfo     : data structure with h and f substitution and 
//              quantification information
// bddaInfo   : datastructure holding 
//              quantification info and a vector
//              of expansion formulas used by BDDA* 
// heuristic  : bdd encoding of f-value and state pairs
//              (represented in current state variables)
// TAct       : transition relation where each state equals
//              an action (disjunctive partitioning)
// T          : an optimized transition relation
// init       : initial state
// goal       : goal state
//OUT
// solution
list<string> BDDAstar(NADLbddBDDAlayout& layout,FHinfo& fhInfo,BDDAinfo& bddaInfo,
		      bdd heuristic,TRel &Tact,TRel &T,bdd init, bdd goal) {
  
  bdd open;       // search queue of BDDA*
  bdd newOpen;    // child states f pairs
  bdd min;        // part of search queue with minimum f-value
  bdd heuristicm; // heuristic in next H variables
      
  vector<bdd> fringe; // vector of reached states
  vector<int> fringeSize;
  timer expandClk,searchClk,extractClk;


  
  // statistics
  int it = 0;
  if (verbosity > 0)
    searchClk.start();

  ///////////////////
  // initialize
  ///////////////////

  // initialize open, reach, fringe and heursiticm
  open = bdd_replace(heuristic,fhInfo.Hcurrent2Fcurrent) & init;
  fringe.push_back(open);

  // construct h' 
  heuristicm = bdd_replace(heuristic,fhInfo.Hcurrent2next);
  heuristicm = bdd_replace(heuristicm,bddaInfo.current2next);
  
  
  // main loop
  while (open != bddfalse)
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      // statistics
      if (verbosity > 0)	
	expandClk.start();

      // find hmin set
      int fmin = fMin(layout,min,open);
      
      // check if goal reached
      if ( (min & goal) != bddfalse )
	break; // break out of while loop

      it++;

      // update fringe
      fringe.push_back(min);      
            
      // subtract min from open
      open &= !min;
      
      // abstract f values from min (we know it equals fmin for all states)
      min = bdd_exist(min,fhInfo.Fcurrent);
      
      
      // find next states from min (forward, so we take the image)
      newOpen = imageOpen(bddaInfo,fhInfo,T,heuristic,heuristicm,min,fmin);


      // update open
      open |= newOpen;


      if (verbosity > 0)
	{
	  cout << "Step=" << it << " Fmin=" << fmin << " minSize=" 
	       << bdd_nodecount(min) << " Timg=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(min));
	} 
    }

  // while loop ended : either because open is empty or because a 
  // solution was found

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = it;
    }



  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;

  
  bdd ints = min & goal;
  


  if (ints != bddfalse)
    {
      bdd currentState = bdd_satone(ints);          
      bdd nextState;
      bdd nextStates; 
      int fringeNo = fringe.size() - 1;
      
      
      //select one state in intersection
      // Extract solution from backward traversal of images
      while ( (currentState & init) == bddfalse )
	{
	  // 1) find the first fringe overlapping with 
	  //    the next states of the current state
	  int fVal = bvecEcoding2int(layout.fVars.length,layout.fVars.current,currentState);
	  currentState = bdd_exist(currentState,fhInfo.Fcurrent);
	  nextStates = preImageOpen(bddaInfo,fhInfo,T,heuristic,heuristicm,currentState,fVal);
	  while (  ( fringe[fringeNo] & nextStates ) ==  bddfalse )
	    fringeNo--;
	  
	  // 2) find a particular action leading to one of the states in the overlap
	  int j = -1;
	  nextState = bddfalse;
	  while (nextState == bddfalse)
	    {
	      j++;      
	      nextState = predecessorOfActOpen(bddaInfo,fhInfo,Tact,heuristic,heuristicm,currentState,fVal,j);
	      nextState &= fringe[fringeNo];
	      
	    }
	  if (verbosity > 1) cout << "Extracted backward: " << Tact.p[j].actName << endl;
	  cout.flush();
	  currentState = nextState;
	  solution.push_front(Tact.p[j].actName);
	  fringeNo--; // look at next fringe
 	}
    }




  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }


  if (verbosity > 0)
    {
      Textract =  extractClk.stop();
      cout << "Total extraction time: " << Textract << " seconds\n";
    }
  




  return solution;
}













//////////////////////////////////////////////////////////
//
// BDDA* for searching backwards using e.g. 
// the HSPr heuristic
//
// 1) This algorithm assumes unit edge costs
//////////////////////////////////////////////////////////

//IN
// bInfo     : data structure with h and f substitution and 
//             quantification information
//bddaInfo   : datastructure holding 
//             quantification info and a vector
//             of expansion formulas used by BDDA* 
// heuristic : bdd encoding of f-value and state pairs
//             (represented in current state variables)
// TAct      : transition relation where each state equals
//             an action (disjunctive partitioning)
// T         : an optimized transition relation  
//OUT
// solution
list<string> BDDAstarBackward(PDDLbddBDDAlayout& layout,FHinfo& fhInfo,BDDAinfo& bddaInfo,
			      bdd heuristic,TRel &Tact,TRel &T,bdd init, bdd goal) {
  
  bdd open;       // search queue of BDDA*
  bdd newOpen;    // child states f pairs
  bdd min;        // part of searc queue with minimum f-value
  bdd heuristicm; // heuristic in next H variables
      
  vector<bdd> fringe; // vector of reached states
  vector<int> fringeSize;
  timer expandClk,searchClk,extractClk;


  
  // statistics
  int it = 0;
  if (verbosity > 0)
    searchClk.start();


  // initialize open, reach, fringe and heursiticm
  open = bdd_replace(heuristic,fhInfo.Hcurrent2Fcurrent) & init;

  fringe.push_back(open);
  heuristicm = bdd_replace(heuristic,fhInfo.Hcurrent2next);
  heuristicm = bdd_replace(heuristicm,bddaInfo.current2next);
  
  
  // main loop
  while (open != bddfalse)
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      // statistics
      if (verbosity > 0)	
	expandClk.start();

      // find hmin set
      int fmin =  fMin(layout,min,open);
      
      // check if goal reached
      if ( (min & goal) != bddfalse )
	break; // break out of while loop

      it++;

      // update fringe
      fringe.push_back(min);
      
            
      // subtract min from open
      open &= !min;
      
      // abstract f values from min (we know it equals fmin for all states)
      min = bdd_exist(min,fhInfo.Fcurrent);
      
      
      // find next states from min (backward, so we take the preimage)
      newOpen = preImageOpen(bddaInfo,fhInfo,T,heuristic,heuristicm,min,fmin);


      // update open
      open |= newOpen;


      if (verbosity > 0)
	{
	  cout << "Step=" << it << " Fmin=" << fmin << " minSize=" 
	       << bdd_nodecount(min) << " Timg=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(min));
	} 
    }

  // while loop ended : either because open is empty or because a 
  // solution was found

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = it;
    }



  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;

  
  bdd ints = min & goal;
  

  if (ints != bddfalse)
    {
      bdd currentState = bdd_satone(ints);          
      bdd nextState;
      bdd nextStates; 
      int fringeNo = fringe.size() - 1;

  
      //select one state in intersection
      // Extract solution from forward traversal of preImages
      while ( (currentState & init) == bddfalse )
	{
	  // 1) find the first fringe overlapping with 
	  //    the next states of the current state
	  int fVal = bvecEcoding2int(layout.fVars.length,layout.fVars.current,currentState);
	  currentState = bdd_exist(currentState,fhInfo.Fcurrent);
	  nextStates = imageOpen(bddaInfo,fhInfo,T,heuristic,heuristicm,currentState,fVal);
	  while (  ( fringe[fringeNo] & nextStates ) ==  bddfalse )
	    fringeNo--;

	  // 2) find a particular action leading to one of the states in the overlap
	  int j = -1;
	  nextState = bddfalse;
	  while (nextState == bddfalse)
	    {
	      j++;      
	      nextState = successorOfActOpen(bddaInfo,fhInfo,Tact,heuristic,heuristicm,currentState,fVal,j);
	      nextState &= fringe[fringeNo];
	      
	    }
	  if (verbosity > 1) cout << "Extracted forward: " << Tact.p[j].actName << endl;
	  cout.flush();
	  currentState = nextState;
	  solution.push_back(Tact.p[j].actName);
	  fringeNo--; // look at next fringe
 	}
    }



  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }


  if (verbosity > 0)
    {
      Textract =  extractClk.stop();
      cout << "Total extraction time: " << Textract << " seconds\n";
    }
  




  return solution;
}












//////////////////////////////////////////////////////////
// Published version of BDDA* 
// regular BDDA* searching forwards
//
// 1) This algorithm assumes unit edge costs
//////////////////////////////////////////////////////////

//IN
// fhInfo     : data structure with h and f substitution and 
//              quantification information
// bddaInfo   : datastructure holding 
//              quantification info and a vector
//              of expansion formulas used by BDDA* 
// heuristic  : bdd encoding of f-value and state pairs
//              (represented in current state variables)
// TAct       : transition relation where each state equals
//              an action (disjunctive partitioning)
// T          : an optimized transition relation
// init       : initial state
// goal       : goal state
//OUT
// solution
list<string> PubBDDAstar(NADLbddPubBDDAlayout& layout,PubFHinfo& fhInfo,PubBDDAinfo& bddaInfo,
		      bdd heuristic,TRel &Tact,TRel &T,bdd init, bdd goal) {
  
  bdd open;       // search queue of BDDA*
  bdd newOpen;    // child states f pairs
  bdd min;        // part of search queue with minimum f-value
  bdd heuristicm; // heuristic in next H variables
      
  vector<int> fringeSize;
  timer expandClk,searchClk,extractClk;


  
  // statistics
  int it = 0;
  if (verbosity > 0)
    searchClk.start();

  ///////////////////
  // initialize
  ///////////////////

  // initialize open, reach, fringe and heursiticm
  open = bdd_replace(heuristic,fhInfo.Hcurrent2Fcurrent) & init;

  // construct h' 
  heuristicm = bdd_replace(heuristic,fhInfo.Hcurrent2next);
  heuristicm = bdd_replace(heuristicm,bddaInfo.current2next);
  
  
  // main loop
  while (open != bddfalse)
    {      
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      // statistics
      if (verbosity > 0)	
	expandClk.start();

      // find hmin set
      int fmin = PubfMin(layout,min,open);
      
      // check if goal reached
      if ( (min & goal) != bddfalse )
	break; // break out of while loop

      it++;
            
      // subtract min from open
      open &= !min;
      
            
      // find next states from min (forward, so we take the image)
      newOpen = PubImageOpen(bddaInfo,fhInfo,T,heuristic,heuristicm,min);


      // update open
      open |= newOpen;


      if (verbosity > 0)
	{
	  cout << "Step=" << it << " Fmin=" << fmin << " minSize=" 
	       << bdd_nodecount(min) << " Timg=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(min));
	} 
    }

  // while loop ended : either because open is empty or because a 
  // solution was found

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = it;
    }



  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;

  

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }


  if (verbosity > 0)
    {
      Textract =  extractClk.stop();
      cout << "Total extraction time: " << Textract << " seconds\n";
    }
  

  return solution;
}







//////////////////////////////////////////////////////////
// Published version of BDDA* 
// BDDA* for searching backwards using e.g. 
// the HSPr heuristic
//
// 1) This algorithm assumes unit edge costs
//////////////////////////////////////////////////////////

//IN
// bInfo     : data structure with h and f substitution and 
//             quantification information
//bddaInfo   : datastructure holding 
//             quantification info and a vector
//             of expansion formulas used by BDDA* 
// heuristic : bdd encoding of f-value and state pairs
//             (represented in current state variables)
// TAct      : transition relation where each state equals
//             an action (disjunctive partitioning)
// T         : an optimized transition relation  
//OUT
// solution
list<string> PubBDDAstarBackward(PDDLbddPubBDDAlayout& layout,PubFHinfo& fhInfo,PubBDDAinfo& bddaInfo,
			      bdd heuristic,TRel &Tact,TRel &T,bdd init, bdd goal) {
  
  bdd open;       // search queue of BDDA*
  bdd newOpen;    // child states f pairs
  bdd min;        // part of searc queue with minimum f-value
  bdd heuristicm; // heuristic in next H variables
      
  vector<int> fringeSize;
  timer expandClk,searchClk,extractClk;


  
  // statistics
  int it = 0;
  if (verbosity > 0)
    searchClk.start();


  // initialize open, reach, fringe and heursiticm
  open = bdd_replace(heuristic,fhInfo.Hcurrent2Fcurrent) & init;

  heuristicm = bdd_replace(heuristic,fhInfo.Hcurrent2next);
  heuristicm = bdd_replace(heuristicm,bddaInfo.current2next);
  
  
  // main loop
  while (open != bddfalse)
    {      

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;


      // statistics
      if (verbosity > 0)	
	expandClk.start();

      // find fmin set
      int fmin =  PubfMin(layout,min,open);
      
      // check if goal reached
      if ( (min & goal) != bddfalse )
	break; // break out of while loop

      it++;
      
            
      // subtract min from open
      open &= !min;
      
            
      // find next states from min (backward, so we take the preimage)
      newOpen = PubPreImageOpen(bddaInfo,fhInfo,T,heuristic,heuristicm,min);


      // update open
      open |= newOpen;


      if (verbosity > 0)
	{
	  cout << "Step=" << it << " Fmin=" << fmin << " minSize=" 
	       << bdd_nodecount(min) << " Timg=" << expandClk.stop() << endl;
	  fringeSize.push_back(bdd_nodecount(min));
	} 
    }

  // while loop ended : either because open is empty or because a 
  // solution was found

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      extractClk.start();
      iterationNum = it;
    }



  list<string> solution;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return solution;
  

  if (verbosity > 0) 
    {
      double aveFringe = 0.0;
      for (int i = 0; i < fringeSize.size(); i++)
	aveFringe += fringeSize[i];
      aveFringeSize = aveFringe / double(fringeSize.size()); 
      cout << "Average number of nodes in fringe: " 
	   << aveFringeSize << endl;
    }


  if (verbosity > 0)
    {
      Textract =  extractClk.stop();
      cout << "Total extraction time: " << Textract << " seconds\n";
    }
  




  return solution;
}


