////////////////////////////////////////////////////////
// File  : bddLayout.hpp   
// Desc. : BDD definition of DOP
// Author: Rune M. Jensen
// Date  : 1/12/02
////////////////////////////////////////////////////////

#ifndef BDDLAYOUTHPP
#define BDDLAYOUTHPP

#include <string>
#include <vector>
#include <set>
#include <map>
#include <bdd.h>
#include "common.hpp"
#include "pddl.hpp"
#include "extNADL.hpp"
#include "numDomain.hpp"
#include "reachInfo.hpp"
#include "balancedInfo.hpp"


// externs
extern int spaceSize;


////////////////////////////////////////////////////////
//
//  Base Classes
//
////////////////////////////////////////////////////////

struct  varDef {
  int        length;        // num of variables
  int        *current;      // mapping to current state bddvars 
  int        *next;         // mapping to next state bddvars
  varDef() {}
  varDef(int l,int *c, int *n) 
  { length = l; current = c; next = n; }
  void mkBlock();
  void print();
};



////////////////////////////////////////////////////////
//
//  PDDL  Structures
//
////////////////////////////////////////////////////////


struct  PDDLvarDef : varDef {
  int seenFacts; // number of facts of theis rep. seen (only used during instantiation) 
  PDDLvarDef()   // default value needed for map (seenfacts given its initial value)
  { seenFacts = 0; } 
  void print();
};


//  A group defines the layout of a subset
//  of fluents (e.g. a set of balanced fluents)

enum fluentGroupType {fgt_single, fgt_balanced};

struct fluentGroup {

  fluentGroupType type;

  // if balanced  
  set< vector<int> > properties; // properties of group
  map<int,int> factValue;        // mapping from fact ID to binary value of fact
  map<int,PDDLvarDef> vars;      // mapping from single valued object to vardef   

  // if single
  // vars above //               // mapping from fact to vardef of fact
  
  void print(numDomain &numDom,int objMask,int predMask,
	     int predShift,int objShift,vector<int> predArity);
};

  

// BDD variable layout of PDDL domain
//  BDD: 
//    regular x,x' interleaving
//  Fluents: 
//    Are represented in groups
//    of balanced or individually represented 
//    fluents
//
//  fluents are ordered closely to the order
//  they appear in in the PDDL domain definition
//  such that their ordering can be changed by
//  changing the ordering in the PDDL domain
struct PDDLbddLayout {

  // bit vector representation vars
  int predShift;
  int objShift;                 
  int predMask;
  int objMask;
  int maxArity;
  vector<int> predArity;

  set<int> fluents;                   // set of fluents in domain (1,2,3,...)
  map<int,int>  fluent2group;         // mapping from fluent Id to the Id of the group that represents it
  vector<fluentGroup> group;          // groups in layout
  set< pair<int,int> > stateVarPairs; // currentVar nextVar pairs                                   
  set<int> currentVars;               // set of all current BDDs variables in layout
  set<int> nextVars;                  // set of all next BDDs variables in layout

  void initialize(reachInfo &rInfo,balancedInfo &bInfo, numDomain &numDom); 
                                // constructs groups and layout bdd vars

  int bddVarNum;                // number of bdd variables of a single state
  PDDLvarDef &bddVars(int fact);// returns the PDDLvardef of fact
  bdd expr(numDomain &numDom,int factid, int prime); 
                                // returns bdd expr of fact                 
  void print(numDomain &numDom);
};






// BDD variable layout of PDDL domain
// specialized to BDDA*
//
//  BDD: 
//    regular x,x' interleaving prefixed
//    with a binary f encoding
struct PDDLbddBDDAlayout : PDDLbddLayout {

  int fhMax;          // maxvalue of f and h
  varDef fVars;       // f-value bit vector representation
  varDef hVars;       // h-value bit vector representation

  void initialize(reachInfo &rInfo,balancedInfo &bInfo, numDomain &numDom); 
  void print(numDomain &numDom);
};






// BDD variable layout of PDDL domain
// specialized to BDDA*
//
//  BDD: 
//    regular x,x' interleaving prefixed
//    with a binary f encoding
struct PDDLbddPubBDDAlayout : PDDLbddLayout {

  int fhMax;          // maxvalue of f and h
  varDef fVars;       // f-value bit vector representation
  varDef hVars;       // h-value bit vector representation

  void initialize(reachInfo &rInfo,balancedInfo &bInfo, numDomain &numDom); 
  void print(numDomain &numDom);
};













////////////////////////////////////////////////////////
//
//  NADL  Structures
//
////////////////////////////////////////////////////////



////////////////////////////////////////////////////////
//  NADL  Structures (Deterministic)
////////////////////////////////////////////////////////


struct actDef {
  string name;
  vector<extNADLtran> tran;
  void print();
};


struct agtDef {
  string name;
  vector<actDef>  act;
  agtDef() {}
  agtDef(string n, vector<actDef> ac)
    { name = n; act = ac; }
  void print();
};

// NADL vardef

enum bType {bt_action, bt_bit, bt_bool, bt_nat};


struct  NADLvarDef : varDef {
  string     name;
  bType      type;     
  NADLvarDef(string na, bType bt, int l, int *c, int *n) : varDef(l,c,n)
  { name = na; type = bt; } 	
  NADLvarDef() : varDef(0,NULL,NULL)  // default value needed for map
     { name = ""; type = bt_bool;}
  void print();
};



////////////////////////////////////////////////////////
//  NADLbddLayout
//
//  total bdd variable layout description 
//    
// Interleaved Ordering of S and S'  variables: 
//
// S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
//
////////////////////////////////////////////////////////
struct NADLbddLayout {
  agtDef             agt;        // system agent
  vector<NADLvarDef>     var;    // NADL variable definitions
  map<string,NADLvarDef> varMap; // Map from variable names to variable definition
  vector<string>     varNames;   // vector of variable names
  int                bddVarNum;  // total number of BDD variables in the layout
  void initialize(extNADLproblem &prob); 
  void print();
};






////////////////////////////////////////////////////////
//  NADLbddBDDAlayout
//
//  description of BDD variables in NADL domain 
//  specialized for BDDA*
//
//  total bdd variable layout description 
//    
// Interleaved Ordering of S and S'  variables: 
//
// S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
//
// with binary encoding of f-value and h, h' value
// in the middle
////////////////////////////////////////////////////////
struct NADLbddBDDAlayout : NADLbddLayout {
  int fhMax;          // maxvalue of f and h
  varDef fVars;       // f-value bit vector representation
  varDef hVars;       // h-value bit vector representation
  void initialize(extNADLproblem &prob);
  void print();
};





////////////////////////////////////////////////////////
//  NADLbddPubBDDAlayout
//
//  description of BDD variables in NADL domain 
//  specialized for BDDA*
//
//  total bdd variable layout description 
//    
// Interleaved Ordering of S and S'  variables: 
//
// S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
//
// prefixed with binary encoding of f-value
//
////////////////////////////////////////////////////////
struct NADLbddPubBDDAlayout : NADLbddLayout {
  int fhMax;          // maxvalue of f and h
  varDef fVars;       // f-value bit vector representation
  varDef hVars;       // h-value bit vector representation
  void initialize(extNADLproblem &prob);
  void print();
};




////////////////////////////////////////////////////////
//  NADL  Structures (Non-Deterministic)
////////////////////////////////////////////////////////




////////////////////////////////////////////////////////
//  NADLNonDetbddLayout
//
//  total bdd variable layout description 
//    
// Interleaved Ordering of S and S'  variables: 
//
// Ae1,...,AeP,As1...,Ask,S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
//
////////////////////////////////////////////////////////
struct NADLNonDetbddLayout : NADLbddLayout {
  NADLvarDef actID;     // OBDD vars used for system action IDs     (A1,..,Ak)
  agtDef     agtEnv;    // environment agent (if present)                       
  NADLvarDef actIDenv;  // vars used for for environment action IDs (Ae1,..,AeP)
  bool envActive;       // true if environment agent is present
  void initialize(extNADLproblem &prob); 
  void mkBlocks();
  void print();
};










// structure  for representing binary number expresions
struct binary {
  int  digitNum;
  bdd *digit;
  binary(int dn, bdd *d)
    { digitNum = dn; digit = d; }
};




////////////////////////////////////////////////////////
//             function prototypes                     
////////////////////////////////////////////////////////

// Shared
bdd int2bdd(int *var,int length,int n);
bdd IV(set< pair<int,int> > vars);

// PDDL
int hHSPr(PDDLbddLayout &layout,reachInfo &rInfo,vector< vector<int> > s);

// NADL
bdd formula2bdd(NADLbddLayout &layout,formula *f) ;
bdd A(NADLbddLayout &layout,int i);
bdd trsGroup(NADLbddLayout &layout,int i,int j);
bdd trsGroupEnv(NADLNonDetbddLayout &layout,int i,int j);
bdd trsGroupErr(NADLbddLayout &layout,int i,int j);
bdd atom2bdd(NADLbddLayout &layout,atom *a);
bdd numProp2bdd(NADLbddLayout &layout,numberProp *prop);
binary *numExp2binary(NADLbddLayout &layout,numberExp *num);




#endif  
