////////////////////////////////////////////////////////
// File  : bddLayout.cc   
// Desc. : BDD layouts for PDDL and NADL 
// Author: Rune M. Jensen
// Date  : 4/7/02
////////////////////////////////////////////////////////

#include<vector>
#include<set>
#include<map>
#include<bdd.h>
#include<math.h>
#include "main.hpp"
#include "set.hpp"
#include "pddl.hpp"
#include "extNADL.hpp"
#include "numDomain.hpp"
#include "reachInfo.hpp"
#include "balancedInfo.hpp"
#include "bddLayout.hpp"

////////////////////////////////////////////////////////
// 
//  varDef member functions
//  
////////////////////////////////////////////////////////

void varDef::mkBlock() {
  int first = 1000000; // infty approx.
  int last = 0; 
  
  if (length > 0)
    {
      for (int i = 0; i < length; i++)
	{
	  if (current[i] < first)
	    first = current[i];
	  if (next[i] < first)
	    first = next[i];

	  if (current[i] > last)
	    last = current[i];
	  if (next[i] > last)
	    last = next[i];
	}
      
      bdd_intaddvarblock(first,last,1);	
    }
}



////////////////////////////////////////////////////////
// 
//  Shared Functions
//  
////////////////////////////////////////////////////////



// IN
//   var    : bdd var mapping
//   length : number of bdd vars
//   n      : integer to encode.
//   OUT
//   bdd : BDD with the binary encoding of n
//         starting with the most significant bit. 
bdd int2bdd(int *var,int length,int n)  {
  bdd res;
  int j;  

  res = bddtrue;
  for (j=length-1; j >= 0; j--)
    {
      if (n % 2) 
	res &= bdd_ithvar(var[j]); 
      else
	res &= bdd_nithvar(var[j]);
      n /= 2;
    }
return res;
}





// IV(curvars,layout)
// IN
//  curVars  : set of BDD variables on current state form which  must be held constant
//  layout   : BDD layout definition
//
// OUT
//  bdd expression for the invariable expression
//  constraining variables in facts
bdd IV(set< pair<int,int> > vars) {
  int i,j;
  bdd res;

  res = bddtrue;
  
  for (set< pair<int,int> >::iterator var = vars.begin(); var  != vars.end(); ++var)
    { 
      res &= bdd_ithvar(var->first) >> bdd_ithvar(var->second);
      res &= bdd_ithvar(var->first) << bdd_ithvar(var->second);      
    }
  return res;
}



////////////////////////////////////////////////////////
// 
//  PDDL member Functions
//  
////////////////////////////////////////////////////////


// 1. sorts out the fluents in groups
// 2. traverse reached facts in order to find value domain of balanced fluents
// 3. lay out the BDD var map of fluents
// Obs: layout does not initialize the BDD package. This must be done before
//      using the expression function
void PDDLbddLayout::initialize(reachInfo &rInfo,balancedInfo &bInfo,numDomain &numDom) {
  
  // misc aux variables for bit vector rep.
  predShift = rInfo.predShift; 
  objShift  = rInfo.objShift;               
  predMask  = rInfo.predMask;
  objMask   = rInfo.objMask;
  predArity = rInfo.predArity;


  // make groups such that predicates in the groups are ordered
  // as closely as possible to how they appear in the PDDL domain definition
  // in this way alternative orderings can be considered by changing the orderings
  // of these predicates

  int groupNo = 0;

  for (int i=0; i < numDom.predNum; i++)
    if (!setMember(rInfo.staticPreds,i+1)) fluents.insert(i+1);
	
  set<int> needGroup = fluents;


  // *** Step 1 ***
  // init groups, by traversing the domain predicates
  for (int i=0; i < numDom.predNum; i++) 
    if (setMember(needGroup,i+1))
      {
	// see if fluent is balanced
	int balancedSet = -1;
	
	for (int j=0; j < bInfo.cv.size(); j++)
	  {
	    if (balancedSet > -1) break;
	    for (set< vector<int> >::iterator sv = bInfo.cv[j].begin(); sv != bInfo.cv[j].end(); ++sv)
	      if (i+1 == (*sv)[0]) 
		{
		  balancedSet = j;
		  break;
		}
	  }

	if (balancedSet > -1)
	  { ////////////////////////////////////////////
	    // init group for balanced set of properties
	    ////////////////////////////////////////////

	    set<int> fluentsInGroup;

	    // remove fluents in group from fluents needing a rep.
	    for (set< vector<int> >::iterator sv = bInfo.cv[balancedSet].begin(); 
		 sv != bInfo.cv[balancedSet].end(); ++sv)
	      {
		needGroup.erase((*sv)[0]);
		fluentsInGroup.insert((*sv)[0]);
	      }

	    // update fluent2group map
	    for (set<int>::iterator si = fluentsInGroup.begin(); si != fluentsInGroup.end(); ++si)
	      fluent2group[*si] = groupNo;

	    // init group
	    fluentGroup g;
	    g.type = fgt_balanced;
	    g.properties = bInfo.cv[balancedSet];
	    group.push_back(g);

	    groupNo++;
	  }
	else
	  { ////////////////////////////////////////////
	    // init group for single fluent
	    ////////////////////////////////////////////

	    // remove fluent from fluents needing a rep.
	    needGroup.erase(i+1);

	    // update fluent2group map
	    fluent2group[i+1] = groupNo;
	    
	    // init group
	    fluentGroup g;
	    g.type = fgt_single;
	    group.push_back(g);

	    groupNo++;
	  }	      
      }
  
  
  
  //     Step 2    
  // go through all reached facts in order to update group info
  for (set<int>::iterator si = rInfo.closedFacts.begin(); si != rInfo.closedFacts.end(); ++si) 
    {
      // get the fact
      vector<int> fact;
      int id = *si;
      id2fact(id,fact,objMask,predMask,
	      predShift,objShift,
	      predArity);
      
      if (setMember(fluents,fact[0]))
	{
	  // group of fluent
	  int fluentGroup = fluent2group[fact[0]];
	  
	  if (group[fluentGroup].type == fgt_balanced)
	    { 
	      // find PDDLvarDef of single valued object
	      int obj;
	      for (set< vector<int> >::iterator sv = group[fluentGroup].properties.begin(); 
		   sv != group[fluentGroup].properties.end(); ++sv)
		if ( (*sv)[0] == fact[0] ) obj = fact[ (*sv)[1] ];
	      
	      PDDLvarDef fluentVars = group[fluentGroup].vars[obj];
	      group[fluentGroup].factValue[id] = fluentVars.seenFacts;
	      fluentVars.seenFacts++;
	      // update PDDLvarDef of single valued object
	      group[fluentGroup].vars[obj] = fluentVars;	     
	    }
	  else
	    { // group[fluentGroup].type == fgt_single
	      
	      // find PDDLvarDef of fact	      
	      PDDLvarDef fluentVars = group[fluentGroup].vars[id];
	      group[fluentGroup].factValue[id] = fluentVars.seenFacts;
	      fluentVars.seenFacts++;
	      // update PDDLvarDef of single valued object
	      group[fluentGroup].vars[id] = fluentVars;	     
	    }	  
	}	  
    }
  
  //    Step 3 
  // assign bdd variables to groups in the order they appear in
  // the groups

  bddVarNum = 0;
  spaceSize = 0;

  for (int i=0; i < group.size(); i++)
    if (group[i].type == fgt_balanced)
      {
	// go through PDDLvarDefs
	for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
	      miv != group[i].vars.end(); ++miv)
	  {
	    PDDLvarDef vd = miv->second;
	    
	    vd.length = int(ceil(log(vd.seenFacts)/log(2)));
	    vd.current = new int[vd.length];
	    vd.next = new int[vd.length];
	    
	    for (int j=0; j<vd.length; j++)
	      {
		int cur;
		cur = bddVarNum;
		currentVars.insert(bddVarNum);
		vd.current[j] = bddVarNum++;
		spaceSize++;
		pair<int,int> elem(cur,bddVarNum);
		stateVarPairs.insert(elem);
		nextVars.insert(bddVarNum);	   
		vd.next[j] = bddVarNum++;
	      }
	    
	    group[i].vars[miv->first] = vd;
	  }
      }
    else
      { // group[i].type == fgt_single
	// go through PDDLvarDefs
	for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
	      miv != group[i].vars.end(); ++miv)
	  {
	    PDDLvarDef vd = miv->second;
	    
	    vd.length = 1;
	    vd.current = new int[1];
	    vd.next = new int[1];

	    int cur;
	    cur = bddVarNum;
	    currentVars.insert(bddVarNum);
	    vd.current[0] = bddVarNum++;
	    spaceSize++;
	    pair<int,int> elem(cur,bddVarNum);
	    stateVarPairs.insert(elem);
	    nextVars.insert(bddVarNum);
	    vd.next[0] = bddVarNum++;
	  	    
	    group[i].vars[miv->first] = vd;
	  }
      }
}
    









// THE BDDA* equivalent !
// 1. sorts out the fluents in groups
// 2. traverse reached facts in order to find value domain of balanced fluents
// 3. lay out the BDD var map of fluents
// Obs: layout does not initialize the BDD package. This must be done before
//      using the expression function
void PDDLbddBDDAlayout::initialize(reachInfo &rInfo,balancedInfo &bInfo,numDomain &numDom) {
  
  // misc aux variables for bit vector rep.
  predShift = rInfo.predShift; 
  objShift  = rInfo.objShift;               
  predMask  = rInfo.predMask;
  objMask   = rInfo.objMask;
  predArity = rInfo.predArity;


  // make groups such that predicates in the groups are ordered
  // as closely as possible to how they appear in the PDDL domain definition
  // in this way alternative orderings can be considered by changing the orderings
  // of these predicates

  int groupNo = 0;

  for (int i=0; i < numDom.predNum; i++)
    if (!setMember(rInfo.staticPreds,i+1)) fluents.insert(i+1);
	   
  set<int> needGroup = fluents;


  // *** Step 1 ***
  // init groups, by traversing the domain predicates
  for (int i=0; i < numDom.predNum; i++) 
    if (setMember(needGroup,i+1))
	 {
	   // see if fluent is balanced
	   int balancedSet = -1;
	   
	   for (int j=0; j < bInfo.cv.size(); j++)
	     {
	       if (balancedSet > -1) break;
	       for (set< vector<int> >::iterator sv = bInfo.cv[j].begin(); sv != bInfo.cv[j].end(); ++sv)
		 if (i+1 == (*sv)[0]) 
		   {
		     balancedSet = j;
		     break;
		   }
	     }

	   if (balancedSet > -1)
	     { ////////////////////////////////////////////
	       // init group for balanced set of properties
	       ////////////////////////////////////////////

	       set<int> fluentsInGroup;

	       // remove fluents in group from fluents needing a rep.
	       for (set< vector<int> >::iterator sv = bInfo.cv[balancedSet].begin(); 
		    sv != bInfo.cv[balancedSet].end(); ++sv)
		 {
		   needGroup.erase((*sv)[0]);
		   fluentsInGroup.insert((*sv)[0]);
		 }

	       // update fluent2group map
	       for (set<int>::iterator si = fluentsInGroup.begin(); si != fluentsInGroup.end(); ++si)
		 fluent2group[*si] = groupNo;

	       // init group
	       fluentGroup g;
	       g.type = fgt_balanced;
	       g.properties = bInfo.cv[balancedSet];
	       group.push_back(g);

	       groupNo++;
	     }
	   else
	     { ////////////////////////////////////////////
	       // init group for single fluent
	       ////////////////////////////////////////////

	       // remove fluent from fluents needing a rep.
	       needGroup.erase(i+1);

	       // update fluent2group map
	       fluent2group[i+1] = groupNo;
	       
	       // init group
	       fluentGroup g;
	       g.type = fgt_single;
	       group.push_back(g);

	       groupNo++;
	     }	      
	 }
  
  
  
  //     Step 2    
  // go through all reached facts in order to update group info
  for (set<int>::iterator si = rInfo.closedFacts.begin(); si != rInfo.closedFacts.end(); ++si) 
    {
	 // get the fact
	 vector<int> fact;
	 int id = *si;
	 id2fact(id,fact,objMask,predMask,
		 predShift,objShift,
		 predArity);
	 
	 if (setMember(fluents,fact[0]))
	   {
	     // group of fluent
	     int fluentGroup = fluent2group[fact[0]];
	     
	     if (group[fluentGroup].type == fgt_balanced)
	       { 
		 // find PDDLvarDef of single valued object
		 int obj;
		 for (set< vector<int> >::iterator sv = group[fluentGroup].properties.begin(); 
		      sv != group[fluentGroup].properties.end(); ++sv)
		   if ( (*sv)[0] == fact[0] ) obj = fact[ (*sv)[1] ];
		 
		 PDDLvarDef fluentVars = group[fluentGroup].vars[obj];
		 group[fluentGroup].factValue[id] = fluentVars.seenFacts;
		 fluentVars.seenFacts++;
		 // update PDDLvarDef of single valued object
		 group[fluentGroup].vars[obj] = fluentVars;	     
	       }
	     else
	       { // group[fluentGroup].type == fgt_single
		 
		 // find PDDLvarDef of fact	      
		 PDDLvarDef fluentVars = group[fluentGroup].vars[id];
		 group[fluentGroup].factValue[id] = fluentVars.seenFacts;
		 fluentVars.seenFacts++;
		 // update PDDLvarDef of single valued object
		 group[fluentGroup].vars[id] = fluentVars;	     
	       }	  
	   }	  
    }
  
  // step 3: assign bdd variables to groups in the order they appear in
  //    the groups

  
  bddVarNum = 0;
  spaceSize = 0;
  
  bool done = false;
  int stateLength = 0;

  // count the number of necessary BDD variables in a 
  // state encoding in order to put the f and h 
  // encoding in the middle
  for (int i = 0; i < group.size(); i++)
    if (group[i].type == fgt_balanced)
      {
	for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
	      miv != group[i].vars.end(); ++miv)
	  {
	    PDDLvarDef vd = miv->second;	    
	    stateLength += int(ceil(log(vd.seenFacts)/log(2)));
	  }
      }
    else
      {   
	for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
	      miv != group[i].vars.end(); ++miv)
	  stateLength++;
      }



  // 2) assign the state BDD varaibles   
  for (int i = 0; i < group.size(); i++)
    {
      if (group[i].type == fgt_balanced)
	{
	  // go through PDDLvarDefs
	  for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
		miv != group[i].vars.end(); ++miv)
	    {
	      // insert f and h if time is up (gripper problem 20)
	      if (bddVarNum > stateLength && !done)
		{
		  // 1) Assign BDD-variables to the f and h encoding in binary 
		
		  fVars.length = int(ceil(log(fhMax)/log(2)));
		  fVars.current = new int[fVars.length];
		  fVars.next = new int[fVars.length];
		  hVars.length = int(ceil(log(fhMax)/log(2)));
		  hVars.current = new int[hVars.length];
		  hVars.next = new int[hVars.length];
		  
		  for (int z = 0; z < fVars.length; z++)
		    {
		      fVars.current[z] = bddVarNum++;
		      hVars.current[z] = bddVarNum++;
		      hVars.next[z] = bddVarNum++;
		    }	
		  done = true;
		}
	        
	      PDDLvarDef vd = miv->second;
	      
	      vd.length = int(ceil(log(vd.seenFacts)/log(2)));
	      vd.current = new int[vd.length];
	      vd.next = new int[vd.length];
	      
	      for (int j=0; j<vd.length; j++)
		{
		  int cur;
		  cur = bddVarNum;
		  currentVars.insert(bddVarNum);
		  vd.current[j] = bddVarNum++;
		  spaceSize++;
		  pair<int,int> elem(cur,bddVarNum);
		  stateVarPairs.insert(elem);
		  nextVars.insert(bddVarNum);	   
		  vd.next[j] = bddVarNum++;
		}
	      
	      group[i].vars[miv->first] = vd;
	    }
	}
      else
	{   
	  // group[i].type == fgt_single
	  // go through PDDLvarDefs
	  for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
		miv != group[i].vars.end(); ++miv)
	    {

	      // insert f and h if time is up (gripper problem 20)
	      if (bddVarNum > stateLength && !done)
		{
		  // 1) Assign BDD-variables to the f and h encoding in binary 
		  
		  fVars.length = int(ceil(log(fhMax)/log(2)));
		  fVars.current = new int[fVars.length];
		  fVars.next = new int[fVars.length];
		  hVars.length = int(ceil(log(fhMax)/log(2)));
		  hVars.current = new int[hVars.length];
		  hVars.next = new int[hVars.length];
		  
		  for (int z = 0; z < fVars.length; z++)
		    {
		      fVars.current[z] = bddVarNum++;
		      hVars.current[z] = bddVarNum++;
		      hVars.next[z] = bddVarNum++;
		    }	
		  done = true;
		}


	      PDDLvarDef vd = miv->second;
	      
	      vd.length = 1;
	      vd.current = new int[1];
	      vd.next = new int[1];
	      
	      int cur;
	      cur = bddVarNum;
	      currentVars.insert(bddVarNum);
	      vd.current[0] = bddVarNum++;
	      spaceSize++;
	      pair<int,int> elem(cur,bddVarNum);
	      stateVarPairs.insert(elem);	    
	      nextVars.insert(bddVarNum);
	      vd.next[0] = bddVarNum++;
	      
	      group[i].vars[miv->first] = vd;
	    }
	}
    }
}








// Published BDD layout for BDDA*
// f varibles comes first in the ordering
// not interleaved with f' , h or h' variables
// (Edelkamp & Helmert 2000, Edelkamp & Reffel 1999)  
// 1. sorts out the fluents in groups
// 2. traverse reached facts in order to find value domain of balanced fluents
// 3. lay out the BDD var map of fluents
// Obs: layout does not initialize the BDD package. This must be done before
//      using the expression function
void PDDLbddPubBDDAlayout::initialize(reachInfo &rInfo,balancedInfo &bInfo,numDomain &numDom) {
  
  // misc aux variables for bit vector rep.
  predShift = rInfo.predShift; 
  objShift  = rInfo.objShift;               
  predMask  = rInfo.predMask;
  objMask   = rInfo.objMask;
  predArity = rInfo.predArity;


  // make groups such that predicates in the groups are ordered
  // as closely as possible to how they appear in the PDDL domain definition
  // in this way alternative orderings can be considered by changing the orderings
  // of these predicates

  int groupNo = 0;

  for (int i=0; i < numDom.predNum; i++)
    if (!setMember(rInfo.staticPreds,i+1)) fluents.insert(i+1);
	   
  set<int> needGroup = fluents;


  // *** Step 1 ***
  // init groups, by traversing the domain predicates
  for (int i=0; i < numDom.predNum; i++) 
    if (setMember(needGroup,i+1))
	 {
	   // see if fluent is balanced
	   int balancedSet = -1;
	   
	   for (int j=0; j < bInfo.cv.size(); j++)
	     {
	       if (balancedSet > -1) break;
	       for (set< vector<int> >::iterator sv = bInfo.cv[j].begin(); sv != bInfo.cv[j].end(); ++sv)
		 if (i+1 == (*sv)[0]) 
		   {
		     balancedSet = j;
		     break;
		   }
	     }

	   if (balancedSet > -1)
	     { ////////////////////////////////////////////
	       // init group for balanced set of properties
	       ////////////////////////////////////////////

	       set<int> fluentsInGroup;

	       // remove fluents in group from fluents needing a rep.
	       for (set< vector<int> >::iterator sv = bInfo.cv[balancedSet].begin(); 
		    sv != bInfo.cv[balancedSet].end(); ++sv)
		 {
		   needGroup.erase((*sv)[0]);
		   fluentsInGroup.insert((*sv)[0]);
		 }

	       // update fluent2group map
	       for (set<int>::iterator si = fluentsInGroup.begin(); si != fluentsInGroup.end(); ++si)
		 fluent2group[*si] = groupNo;

	       // init group
	       fluentGroup g;
	       g.type = fgt_balanced;
	       g.properties = bInfo.cv[balancedSet];
	       group.push_back(g);

	       groupNo++;
	     }
	   else
	     { ////////////////////////////////////////////
	       // init group for single fluent
	       ////////////////////////////////////////////

	       // remove fluent from fluents needing a rep.
	       needGroup.erase(i+1);

	       // update fluent2group map
	       fluent2group[i+1] = groupNo;
	       
	       // init group
	       fluentGroup g;
	       g.type = fgt_single;
	       group.push_back(g);

	       groupNo++;
	     }	      
	 }
  
  
  
  //     Step 2    
  // go through all reached facts in order to update group info
  for (set<int>::iterator si = rInfo.closedFacts.begin(); si != rInfo.closedFacts.end(); ++si) 
    {
	 // get the fact
	 vector<int> fact;
	 int id = *si;
	 id2fact(id,fact,objMask,predMask,
		 predShift,objShift,
		 predArity);
	 
	 if (setMember(fluents,fact[0]))
	   {
	     // group of fluent
	     int fluentGroup = fluent2group[fact[0]];
	     
	     if (group[fluentGroup].type == fgt_balanced)
	       { 
		 // find PDDLvarDef of single valued object
		 int obj;
		 for (set< vector<int> >::iterator sv = group[fluentGroup].properties.begin(); 
		      sv != group[fluentGroup].properties.end(); ++sv)
		   if ( (*sv)[0] == fact[0] ) obj = fact[ (*sv)[1] ];
		 
		 PDDLvarDef fluentVars = group[fluentGroup].vars[obj];
		 group[fluentGroup].factValue[id] = fluentVars.seenFacts;
		 fluentVars.seenFacts++;
		 // update PDDLvarDef of single valued object
		 group[fluentGroup].vars[obj] = fluentVars;	     
	       }
	     else
	       { // group[fluentGroup].type == fgt_single
		 
		 // find PDDLvarDef of fact	      
		 PDDLvarDef fluentVars = group[fluentGroup].vars[id];
		 group[fluentGroup].factValue[id] = fluentVars.seenFacts;
		 fluentVars.seenFacts++;
		 // update PDDLvarDef of single valued object
		 group[fluentGroup].vars[id] = fluentVars;	     
	       }	  
	   }	  
    }
  
  // Step 3
  //
  // 1) Assign BDD-variables to the f and h encoding in binary 
  // 2) assign bdd variables to groups in the order they appear in
  //    the groups

  
  bddVarNum = 0;
  spaceSize = 0;
  
  // 1) assign the first BDD variables to the f-encoding and h-encoding
  fVars.length = int(ceil(log(fhMax)/log(2)));
  fVars.current = new int[fVars.length];
  fVars.next = new int[fVars.length];
  hVars.length = int(ceil(log(fhMax)/log(2)));
  hVars.current = new int[hVars.length];
  hVars.next = new int[hVars.length];

  for (int i = 0; i < fVars.length; i++)
    fVars.current[i] = bddVarNum++;
  
  for (int i = 0; i < fVars.length; i++)
    fVars.next[i] = bddVarNum++;
  
  for (int i = 0; i < hVars.length; i++)
    hVars.current[i] = bddVarNum++;
  
  for (int i = 0; i < hVars.length; i++)
    hVars.next[i] = bddVarNum++;
  
  
  // 2) assign the state BDD varaibles 
  for (int i=0; i < group.size(); i++)
    if (group[i].type == fgt_balanced)
	 {
	   // go through PDDLvarDefs
	   for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
		 miv != group[i].vars.end(); ++miv)
	     {
	       PDDLvarDef vd = miv->second;
	       
	       vd.length = int(ceil(log(vd.seenFacts)/log(2)));
	       vd.current = new int[vd.length];
	       vd.next = new int[vd.length];
	       
	       for (int j=0; j<vd.length; j++)
		 {
		   int cur;
		   cur = bddVarNum;
		   currentVars.insert(bddVarNum);
		   vd.current[j] = bddVarNum++;
		   spaceSize++;
		   pair<int,int> elem(cur,bddVarNum);
		   stateVarPairs.insert(elem);
		   nextVars.insert(bddVarNum);	   
		   vd.next[j] = bddVarNum++;
		 }
	       
	       group[i].vars[miv->first] = vd;
	     }
	 }
    else
	 { // group[i].type == fgt_single
	   // go through PDDLvarDefs
	   for ( map<int,PDDLvarDef>::iterator miv = group[i].vars.begin();
		 miv != group[i].vars.end(); ++miv)
	     {
	       PDDLvarDef vd = miv->second;
	       
	       vd.length = 1;
	       vd.current = new int[1];
	       vd.next = new int[1];

	       int cur;
	       cur = bddVarNum;
	       currentVars.insert(bddVarNum);
	       vd.current[0] = bddVarNum++;
	       spaceSize++;
	       pair<int,int> elem(cur,bddVarNum);
	       stateVarPairs.insert(elem);	    
	       nextVars.insert(bddVarNum);
	       vd.next[0] = bddVarNum++;
		       
	       group[i].vars[miv->first] = vd;
	     }
	 }
}
    

















////////////////////////////////////////////
// PDDL User functions
////////////////////////////////////////////
	

//IN
// layout    : PDDLbddlayout
// rInfo     : data from reachability analysis
// s         : vector of true facts (obs s can only represent a single state)
//OUT
// HSPr-value of the state
int hHSPr(PDDLbddLayout &layout,reachInfo &rInfo,vector< vector<int> > s) {

  int sum = 0;

  //sum depths of facts in s
  for (int i=0; i < s.size(); i++)
    if (setMember(layout.fluents,(s[i])[0]))
      {
	int id = fact2id(s[i],layout.predShift,layout.objShift);
	sum += rInfo.depth[id];
      }
  return sum;  
}







//IN
// factId  : fact id (bit vector rep)
//OUT
// PDDLvarDef of fact
PDDLvarDef &PDDLbddLayout::bddVars(int factId) {

      // expand the fact
      vector<int> fact;
      id2fact(factId,fact,objMask,predMask,
	      predShift,objShift,
	      predArity);
  
      if (!setMember(fluents,fact[0]))
	{
	  cout << "PDDLbddLayout.cc : PDDLbddLayout::bddvars : Predicate no. " << fact[0];
	  cout << "is not represented in state\nexiting\n";
	  exit(1);
	}

      int groupNo = fluent2group[ fact[0] ];
      
      if (group[groupNo].type == fgt_balanced)
	{
	  // find PDDLvarDef of single valued object
	  int obj;
	  for (set< vector<int> >::iterator sv = group[groupNo].properties.begin(); 
	       sv != group[groupNo].properties.end(); ++sv)
	    if ( (*sv)[0] == fact[0] ) obj = fact[ (*sv)[1] ];
	  
	  return group[groupNo].vars[obj];
	}
      else
	{
	  // group[groupNo].type == fgt_single

	  // find PDDLvarDef of fact	      
	  return group[groupNo].vars[factId];
	}  
}









//IN
// fact    : fact id (bit vector rep)
// prime   : 0: current state rep., 1: next state rep.
//OUT
// bdd expression of fact
bdd PDDLbddLayout::expr(numDomain &numDom,int factId,int prime) {                              

  // expand the fact
  vector<int> fact;
  id2fact(factId,fact,objMask,predMask,
	  predShift,objShift,
	  predArity);
  
  if (!setMember(fluents,fact[0]))
    {
      cout << "bddlayouot.cc : PDDLbddLayout::bddvars : Predicate " << numDom.pred[fact[0]-1];
      cout << " is not represented in state\nexiting\n";
      exit(1);
    }
  
  int groupNo = fluent2group[ fact[0] ];
  
  if (group[groupNo].type == fgt_balanced)
    {
      // find PDDLvarDef of single valued object
      int obj;
      for (set< vector<int> >::iterator sv = group[groupNo].properties.begin(); 
	   sv != group[groupNo].properties.end(); ++sv)
	if ( (*sv)[0] == fact[0] ) obj = fact[ (*sv)[1] ];      
      PDDLvarDef fluentVars = group[groupNo].vars[obj];

      if (fluentVars.seenFacts == 0)
	{
	  cout << "bddlayouot.cc : PDDLbddLayout::Expr (balanced) : ";
	  printFact(fact,numDom);
	  cout << " has no encoding\nexiting\n";
	  exit(1);
	}

      if (prime)
	return int2bdd(fluentVars.next,fluentVars.length,group[groupNo].factValue[factId]);
      else
	return int2bdd(fluentVars.current,fluentVars.length,group[groupNo].factValue[factId]);
    }
  else
    {
      // group[groupNo].type == fgt_single
      
      // find PDDLvarDef of fact	      
      PDDLvarDef fluentVars = group[groupNo].vars[factId];

      if (fluentVars.seenFacts == 0)
	{
	  cout << "bddlayouot.cc : PDDLbddLayout::bddExpr (single) : ";
	  printFact(fact,numDom);
	  cout << " has no encoding\nexiting\n";
	  exit(1);
	}
      
      if (prime)      
	return bdd_ithvar(fluentVars.next[0]);
      else
	return bdd_ithvar(fluentVars.current[0]);
    }  
}












////////////////////////////////////////////
// PDDL Print functions
////////////////////////////////////////////


// base classes

void varDef::print() {

  cout << "    length : " << length << endl;
  cout << "    current : ";
  for (int i=0; i < length; i++)
    cout << current[i] << " ";
  cout << endl;
  cout << "    next : ";
  for (int i=0; i < length; i++)
    cout << next[i] << " ";
  cout << endl;
}

// PDDL Structures

void PDDLvarDef::print() {

  cout << "    seenFacts : " << seenFacts << endl;
  cout << "    varDef\n";
  varDef::print();
}


void fluentGroup::print(numDomain &numDom,int objMask,int predMask,
			int predShift,int objShift,vector<int> predArity) {

  if (type == fgt_balanced)
    {
      cout << "   Balanced\n";
      cout << "   Properties:\n";
      for (set< vector<int> >::iterator sv = properties.begin(); sv != properties.end(); ++sv)
	cout << "    Fluent : " << numDom.pred[ (*sv)[0] - 1 ] << " Arg: " << (*sv)[1] << endl;
      cout << "   FactValues:\n";
      for (map<int,int>::iterator m = factValue.begin(); m != factValue.end(); ++m)
	{
	  cout << "    Fact : ";
	  vector<int> fact;
	  int id = m->first;
	  id2fact(id,fact,objMask,predMask,predShift,objShift,predArity);
	  printFact(fact,numDom);
	  cout << " value : " << m->second << endl;
	}
      cout << "   VarDefs:\n";
      for (map<int,PDDLvarDef>::iterator mv = vars.begin(); mv != vars.end(); ++mv)
	{
	  cout << "    Object : " << numDom.obj[mv->first - 1] << endl;
	  mv->second.print();
	}
    }
  else
    { // type == ftg_single
      cout << "   Single\n";
      cout << "   PDDLvarDefs:\n";
      for (map<int,PDDLvarDef>::iterator mv = vars.begin(); mv != vars.end(); ++mv)
	{
	  cout << "    Fact : ";
	  vector<int> fact;
	  int id = mv->first;
	  id2fact(id,fact,objMask,predMask,predShift,objShift,predArity);
	  printFact(fact,numDom);
	  cout << endl;
	  mv->second.print();
	}           
    }
}



void PDDLbddLayout::print(numDomain &numDom) {

  cout << "\n\nPDDL BDD Layout\n";
  cout << " FluentMap\n";
  for (map<int,int>::iterator m = fluent2group.begin(); m != fluent2group.end(); ++m)  
    cout << "   " << numDom.pred[m->first - 1] << " : " << m->second << endl;
  cout << " Groups\n";
  for (int i=0; i < group.size(); i++)
    {
      cout << "  Group " << i << endl;
      group[i].print(numDom,objMask, predMask,predShift,objShift,predArity);
    }
  cout << " Current BDD variables: ";
  setPrint(currentVars);
  cout << endl;
  cout << " Next BDD variables: ";
  setPrint(nextVars);
  cout << endl;
}




void PDDLbddBDDAlayout::print(numDomain &numDom) {
  cout << "\nPDDLbddBDDAlayout\n";
  cout << " f encoding\n";
  fVars.print();
  cout << endl;
  cout << " h encoding\n";
  hVars.print();
  PDDLbddLayout::print(numDom);
}  




void PDDLbddPubBDDAlayout::print(numDomain &numDom) {
  cout << "\nPubPDDLbddBDDAlayout\n";
  cout << " f encoding\n";
  fVars.print();
  cout << endl;
  cout << " h encoding\n";
  hVars.print();
  PDDLbddLayout::print(numDom);
}  








































////////////////////////////////////////////////////////
// 
//  NADL Structures  
//  
////////////////////////////////////////////////////////



//IN
// prob: NADL descripton internal rep.
// only one system agent assumed
// no environment agents
//OUT
// a bdd layout and various domain info
void NADLbddLayout::initialize(extNADLproblem &prob) {
  
  // we assume 1 system agent and no environment agents
  if (prob.sys.size() != 1 || prob.env.size() != 0)
    {
      cout << "bddLayout.cc :NADLbddLayout::initialize :"
	   << " 1 system and at most 1 environment agent expected\nexiting\n";
      exit(1);
    }


  // add the system agent
  vector<actDef> actv;
  for (list<extNADLaction>::iterator acti = prob.sys.front().actions.begin(); 
       acti != prob.sys.front().actions.end(); ++acti)
    {
      actDef act;
      act.name = acti->name;
      act.tran.push_back(acti->trans.front());	  
      actv.push_back(act);
    }
  agt = agtDef(prob.sys.front().name,actv);



  ////////////////////////////////////////////////////////
  // build the varDef                   
  // Interleaved Ordering of S and S'  variables: 
  //
  // S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
  //
  ////////////////////////////////////////////////////////

  bddVarNum = 0;
  spaceSize = 0;
  
  for (list<varDecl>::iterator vardi = prob.vars.begin(); 
       vardi != prob.vars.end(); ++vardi)
    {
      int     length;
      bType   type;
      

      // find the bdd variable length of variable type 
      if (vardi->type.type == vt_bool) 
	{
	  type = bt_bool;
	  length = 1;
	}
      else 
	{
	  type = bt_nat;
	  length = vardi->type.range;
	}
      
      // assign bdd variable space for each variable
      for (set<string>::iterator si = vardi->ids.begin(); si != vardi->ids.end(); ++si)      
	{	
	  // make mapping to bddvars
	  int *current = new int[length];
	  int *next    = new int[length];
	  for (int i = 0; i<length; i++)
	    {
	      current[i] = bddVarNum;
	      bddVarNum++;
	      spaceSize++;
	      next[i] = bddVarNum;
	      bddVarNum++;
	    }
	  
	  // save var description in vector and map
	  var.push_back(NADLvarDef(*si,type,length,current,next));
	  varMap[*si] = NADLvarDef(*si,type,length,current,next);
	  
	  // add variable to variable list
          varNames.push_back(*si);
	}      
    }
}








// BDDA* equivalent
//IN
// prob: NADL descripton internal rep.
//OUT
// a bdd layout and various domain info
void NADLbddBDDAlayout::initialize(extNADLproblem &prob) {


  // we assume 1 system agent and no environment agents
  if (prob.sys.size() != 1 || prob.env.size() != 0)
    {
      cout << "bddLayout.cc :NADLbddBDDAlayout::initialize :"
	   << " 1 system and at most 1 environment agent expected\nexiting\n";
      exit(1);
    }

  
  // add the system agent
  vector<actDef> actv;
  for (list<extNADLaction>::iterator acti = prob.sys.front().actions.begin(); 
       acti != prob.sys.front().actions.end(); ++acti)
    {
      actDef act;
      act.name = acti->name;
      act.tran.push_back(acti->trans.front());	  
      actv.push_back(act);
    }
  agt = agtDef(prob.sys.front().name,actv);
  

  ////////////////////////////////////////////////////////
  // Build the varDef
  // Interleaved Ordering of S and S'  with f,h,h'
  // in the middle
  // 
  // S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...f1,h1,h1' ..., ...
  //
  ////////////////////////////////////////////////////////

  // 1) measure state length
  int stateLength = 0;
  for (list<varDecl>::iterator vardi = prob.vars.begin(); 
       vardi != prob.vars.end(); ++vardi)
    if (vardi->type.type == vt_bool) 
      for (set<string>::iterator si = vardi->ids.begin(); si != vardi->ids.end(); ++si)      
	stateLength++;
    else 
      for (set<string>::iterator si = vardi->ids.begin(); si != vardi->ids.end(); ++si)      	
	stateLength += vardi->type.range;  

  fVars.length = int(ceil(log(fhMax)/log(2)));
  fVars.current = new int[fVars.length];
  fVars.next = new int[fVars.length];
  hVars.length = int(ceil(log(fhMax)/log(2)));
  hVars.current = new int[hVars.length];
  hVars.next = new int[hVars.length];


  bddVarNum = 0;
  bool done = false;

  // 2) allocate the state BDD variables (allocate the f and h 
  //    vars half way through the state vars
  for (list<varDecl>::iterator vardi = prob.vars.begin(); 
       vardi != prob.vars.end(); ++vardi)
    {

      int     length;
      bType   type;
      

      // find the bdd variable length of variable type 
      if (vardi->type.type == vt_bool) 
	{
	  type = bt_bool;
	  length = 1;
	}
      else 
	{
	  type = bt_nat;
	  length = vardi->type.range;
	}
      
      // assign bdd variable space for each variable
      for (set<string>::iterator si = vardi->ids.begin(); si != vardi->ids.end(); ++si)      
	{

	  // put f,h and h' vars into the ordering when the middle is reached
	  if ( (bddVarNum > stateLength)  && !done)
	    {	  
	      // allocate f and h low level interleaved
	      for (int i = 0; i < fVars.length; i++)
		{
		  fVars.current[i] = bddVarNum++;
		  hVars.current[i] = bddVarNum++;
		  hVars.next[i] = bddVarNum++;
		}	 
	      done = true;
	    }


	  // make mapping to bddvars
	  int *current = new int[length];
	  int *next    = new int[length];
	  for (int i = 0; i<length; i++)
	    {
	      current[i] = bddVarNum;
	      bddVarNum++;
	      spaceSize++;
	      next[i] = bddVarNum;
	      bddVarNum++;
	    }
	  
	  // save var description in vector and map
	  var.push_back(NADLvarDef(*si,type,length,current,next));
	  varMap[*si] = NADLvarDef(*si,type,length,current,next);
	  
	  // add variable to variable list
          varNames.push_back(*si);
	}      
    }
}











// PubBDDA* equivalent
//IN
// prob: NADL descripton internal rep.
//OUT
// a bdd layout and various domain info
void NADLbddPubBDDAlayout::initialize(extNADLproblem &prob) {
  

  // we assume 1 system agent and no environment agents
  if (prob.sys.size() != 1 || prob.env.size() != 0)
    {
      cout << "bddLayout.cc :NADLbddPubBDDAlayout::initialize :"
	   << " 1 system and at most 1 environment agent expected\nexiting\n";
      exit(1);
    }


  // add the system agent
  vector<actDef> actv;
  for (list<extNADLaction>::iterator acti = prob.sys.front().actions.begin(); 
       acti != prob.sys.front().actions.end(); ++acti)
    {
      actDef act;
      act.name = acti->name;
      act.tran.push_back(acti->trans.front());	  
      actv.push_back(act);
    }
  agt = agtDef(prob.sys.front().name,actv);
  
  

  ////////////////////////////////////////////////////////
  // Build the varDef
  // 1) allocte variables for the f-encofding                   
  // 2) Interleaved Ordering of S and S'  variables: 
  // 
  // S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
  //
  ////////////////////////////////////////////////////////

  bddVarNum = 0;

  
  // 1) assign the first BDD variables to the f-encoding and h-encoding
  fVars.length = int(ceil(log(fhMax)/log(2)));
  fVars.current = new int[fVars.length];
  fVars.next = new int[fVars.length];
  for (int i = 0; i < fVars.length; i++)
    fVars.current[i] = bddVarNum++;
  for (int i = 0; i < fVars.length; i++)
    fVars.next[i] = bddVarNum++;
    
  
  hVars.length = int(ceil(log(fhMax)/log(2)));
  hVars.current = new int[hVars.length];
  hVars.next = new int[hVars.length];
  for (int i = 0; i < hVars.length; i++)
    hVars.current[i] = bddVarNum++;
  for (int i = 0; i < hVars.length; i++)
    hVars.next[i] = bddVarNum++;


  // 2) allocate the state BDD variables
  for (list<varDecl>::iterator vardi = prob.vars.begin(); 
       vardi != prob.vars.end(); ++vardi)
    {
      int     length;
      bType   type;
      

      // find the bdd variable length of variable type 
      if (vardi->type.type == vt_bool) 
	{
	  type = bt_bool;
	  length = 1;
	}
      else 
	{
	  type = bt_nat;
	  length = vardi->type.range;
	}
      
      // assign bdd variable space for each variable
      for (set<string>::iterator si = vardi->ids.begin(); si != vardi->ids.end(); ++si)      
	{	
	  // make mapping to bddvars
	  int *current = new int[length];
	  int *next    = new int[length];
	  for (int i = 0; i<length; i++)
	    {
	      current[i] = bddVarNum;
	      bddVarNum++;
	      spaceSize++;
	      next[i] = bddVarNum;
	      bddVarNum++;
	    }
	  
	  // save var description in vector and map
	  var.push_back(NADLvarDef(*si,type,length,current,next));
	  varMap[*si] = NADLvarDef(*si,type,length,current,next);
	  
	  // add variable to variable list
          varNames.push_back(*si);
	}      
    }
}
































/////////////////////////////////////////////////////
/////////////////////////////////////////////////////
//
// Non-deterministic Planning
//
/////////////////////////////////////////////////////
/////////////////////////////////////////////////////




//IN
// prob: Extended NADL descripton internal rep.
//OUT
// extended NADL layout 
// one system agent
// at most one environement agent
void NADLNonDetbddLayout::initialize(extNADLproblem &prob) {

  // we assume 1 system agent and at most 1 environment agents
  if (prob.sys.size() != 1 || prob.env.size() > 1)
    {
      cout << "bddLayout.cc :NADLNonDetbddLayout::initialize :"
	   << " 1 system and at most 1 environment agent expected\nexiting\n";
      exit(1);
    }
  
  envActive = ( prob.env.size() > 0 );
  
  // add the system agent
  vector<actDef> actv;
  for (list<extNADLaction>::iterator acti = prob.sys.front().actions.begin(); 
       acti != prob.sys.front().actions.end(); ++acti)
    {
      actDef act;
      for (list<extNADLtran>::iterator trani = acti->trans.begin(); 
	   trani != acti->trans.end(); ++trani)
	act.tran.push_back(*trani);	  
      act.name = acti->name;
      actv.push_back(act);
    }
  agt = agtDef(prob.sys.front().name,actv);
    

  if ( envActive )
    {
      // add environemt agent (if present) 
      vector<actDef> actv;
      for (list<extNADLaction>::iterator acti = prob.env.front().actions.begin(); 
	   acti != prob.env.front().actions.end(); ++acti)
	{
	  actDef act;
	  for (list<extNADLtran>::iterator trani = acti->trans.begin(); 
	       trani != acti->trans.end(); ++trani)
	    act.tran.push_back(*trani);	  	  
	  act.name = acti->name;
	  act.tran.push_back(acti->trans.front());	  
	  actv.push_back(act);
	}
      agtEnv = agtDef(prob.env.front().name,actv);
    }

  ////////////////////////////////////////////////////////
  // Build the varDef
  // 1) allocte variables for environment and
  //    system action ID encofding                   
  // 3) Interleaved Ordering of S and S'  variables: 
  // 
  // Ae1,...,Aek,As1,...,Ask,S11,S11',...,S1nS'1n, S21,S21',... ,S2m,S2m' ...
  //
  ////////////////////////////////////////////////////////
  
  bddVarNum = 0;
  
  // 1) allocte variables for environment and system action ID encoding                   
  if ( envActive )
    { 
      if (agtEnv.act.size() == 1)
	actIDenv.length = 1; // we need a bit even if only a single action is present
      else
	actIDenv.length = int(ceil(log(agtEnv.act.size())/log(2)));
      actIDenv.current = new int[actIDenv.length];
      for (int i = 0; i < actIDenv.length; i++)
	actIDenv.current[i] = bddVarNum++;
      // dummy assignment to avoid seg fault when printing
      actIDenv.next = new int[actIDenv.length];
      for (int i = 0; i < actIDenv.length; i++)
	actIDenv.next[i] = 0;
    }

  if (agt.act.size() == 1)
    actID.length = 1;
  else
    actID.length = int(ceil(log(agt.act.size())/log(2)));
  actID.current = new int[actID.length];
  for (int i = 0; i < actID.length; i++)
    actID.current[i] = bddVarNum++;
  // dummy assignment to avoid seg fault when printing
  actID.next = new int[actID.length];
  for (int i = 0; i < actID.length; i++)
    actID.next[i] = 0;
  
     
    
  
  // 2) allocate the state BDD variables
  for (list<varDecl>::iterator vardi = prob.vars.begin(); 
       vardi != prob.vars.end(); ++vardi)
    {
      int     length;
      bType   type;
      

      // find the bdd variable length of variable type 
      if (vardi->type.type == vt_bool) 
	{
	  type = bt_bool;
	  length = 1;
	}
      else 
	{
	  type = bt_nat;
	  length = vardi->type.range;
	}
      
      // assign bdd variable space for each variable
      for (set<string>::iterator si = vardi->ids.begin(); si != vardi->ids.end(); ++si)      
	{	
	  // make mapping to bddvars
	  int *current = new int[length];
	  int *next    = new int[length];
	  for (int i = 0; i<length; i++)
	    {
	      current[i] = bddVarNum;
	      bddVarNum++;
	      spaceSize++;
	      next[i] = bddVarNum;
	      bddVarNum++;
	    }
	  
	  // save var description in vector and map
	  var.push_back(NADLvarDef(*si,type,length,current,next));
	  varMap[*si] = NADLvarDef(*si,type,length,current,next);
	  
	  // add variable to variable list
          varNames.push_back(*si);
	}      
    }
}






//IN
// 
//OUT
// makes blocks around bdd variables for dynamic variable 
// reordering
void NADLNonDetbddLayout::mkBlocks() { 
  for (int i = 0; i < var.size(); i++)
    var[i].mkBlock();
  
  actID.mkBlock();
  
  if ( envActive )
    actIDenv.mkBlock();
}
  
























////////////////////////////////////////////
// NADL Functions for generating BDDs 
// from formulas
////////////////////////////////////////////

bdd formula2bdd(NADLbddLayout &layout,formula *f) {

bdd b,l,r,e;

  switch (f->type) {
  case ft_atom:
    b = atom2bdd(layout,f->atomic);
    break;

  case ft_neg:
    b = formula2bdd(layout,f->f1);
    b = !b;
    break;
    
  case ft_and:
    l = formula2bdd(layout,f->f1);
    r = formula2bdd(layout,f->f2);
    b = l & r;
    break;
    
  case ft_or:
    l = formula2bdd(layout,f->f1);
    r = formula2bdd(layout,f->f2);
    b = l | r;
    break;
    
  case ft_impl:
    l = formula2bdd(layout,f->f1);
    r = formula2bdd(layout,f->f2);
    b = l >> r;
    break;
    
  case ft_biImpl:
    l = formula2bdd(layout,f->f1);
    r = formula2bdd(layout,f->f2);
    b = (l >> r) & (l << r);
    break;
    
  case ft_ite:
    e = formula2bdd(layout,f->f1);
    l = formula2bdd(layout,f->f2);
    r = formula2bdd(layout,f->f3);
    b = bdd_ite(e,l,r); 
    break;

  case ft_true:
    b = bddtrue;
    break;

  case ft_false:
    b = bddfalse;
    break;

  case ft_paren:
    b = formula2bdd(layout,f->f1);
    break;
  }
return b;
}



bdd atom2bdd(NADLbddLayout &layout,atom *a) {
  
  bdd b;
  
  switch (a->type) {
    case at_numProp:
      a->prop->minus2plus();
      b = numProp2bdd(layout,a->prop);
      break;
    case at_boolVarCur:
      {
	NADLvarDef vd(layout.varMap[a->var]);
	if (vd.current != NULL) 
	  b = bdd_ithvar(vd.current[0]);
	else
	  {
	    cout << "\nbddLayout, atom2bdd: variable " << a->var;
	    cout << " is not defined\nexiting\n";
	    exit(1);
	  }
      }
      break;

    case at_boolVarNext:
      {
	NADLvarDef vd(layout.varMap[a->var]);
	if (vd.next != NULL) 
	  b = bdd_ithvar(vd.next[0]);
	else
	  {
	    cout << "\nbddLayout, atom2bdd: variable " << a->var;
	    cout << "' is not defined\nexiting\n";
	    exit(1);
	  }
      }
      break;

  }
  return b;
}
 

//  Functions for generating BDD's from numProps

bdd numProp2bdd(NADLbddLayout &layout,numberProp *prop) {
  bdd b,e,ld,rd;
  binary *l, *r;
  numberExp *c;
  numberProp *propc;
  int i,j,digitNum;

  switch (prop->op) {
  case ro_lt:
    c = prop->left;
    prop->left = prop->right;
    prop->right = c;
    prop->op = ro_gt;
    b = numProp2bdd(layout,prop); 
    break;
  case ro_ne:
    propc = new numberProp;
    propc->op = ro_eq;
    propc->left  = prop->left;
    propc->right = prop->right;
    b = !numProp2bdd(layout,propc);
    break;
  case ro_eq:
    l = numExp2binary(layout,prop->left);
    r = numExp2binary(layout,prop->right);
    
    b = bddtrue;
    
    // test if l = r
    i = 0;
    while (i < l->digitNum || i < r->digitNum)
      {
	if (i < l->digitNum && i < r->digitNum)
	  {
	    b &= l->digit[i] >> r->digit[i];
	    b &= l->digit[i] << r->digit[i];
	  }
	else
	  if ( i < l->digitNum )
	    {
	      b &= !l->digit[i];
	    }
	  else
	    {
	      b &= !r->digit[i];
	    }
	i++;
      }
    break;

  case ro_gt:
    l = numExp2binary(layout,prop->left);
    r = numExp2binary(layout,prop->right);
    
    digitNum = l->digitNum > r->digitNum ? l->digitNum : r->digitNum;
    
    b = bddfalse;
    for (i=0; i<digitNum; i++)
      {
	// all digits before digit_i must be equal
	e = bddtrue;
	for (j = i+1; j<digitNum; j++)
	  {
	    ld = j >= l->digitNum ? bddfalse : l->digit[j];
	    rd = j >= r->digitNum ? bddfalse : r->digit[j];
	    e &= ld >> rd;
	    e &= ld << rd;
	  }

	ld = i >= l->digitNum ? bddfalse : l->digit[i];
	rd = i >= r->digitNum ? bddfalse : r->digit[i];

	// if e is true then if ld > rd: l > r
	b |= e & ld & !rd; 
      }
    break;
  }
  return b;
}


binary *numExp2binary(NADLbddLayout &layout,numberExp *num) {
  binary *b,*l,*r;
  bdd    *digit,c;
  int     digitNum,i,number;

  switch (num->type) {
  case nt_minus:
    cout << "\n\nfsm.cc numExp2binary: error type of numExp cannot be nt_minus\nexiting\n";
    exit(1);
    break;

  case nt_number:
    number = num->number;
    // the digitNum of 0 is 0, of 1 is 1, of 2,3 is 2, of 4,5,6,7 is 3 etc.
    digitNum = int(ceil(log(number+1)/log(2)));
    
    if (digitNum > 0)
      digit = new bdd[digitNum];
    else
      digit = NULL;
    
    // initialize digit array
    for (i=0; i<digitNum; i++)
      {
	if (number % 2)
	  digit[i] = bddtrue;
	else
	  digit[i] = bddfalse;
	number /= 2;
      }
    
    b = new binary(digitNum,digit);
    break;

  case nt_varCur:
    {
      NADLvarDef vard(layout.varMap[num->var]);
      if (vard.current == NULL) 
	{
	  cout << "\nbddLayout.cc numExp2binary: variable " << num->var;
	  cout << " is not defined\nexiting\n";
	  exit(1);
	}      
      else
	{
	  digitNum = vard.length;
	  
	digit = new bdd[digitNum];
	
	//intialize the digit array
	for (i=0; i < digitNum; i++)
	  digit[i] = bdd_ithvar(vard.current[digitNum - 1 - i]);
	
	b = new binary(digitNum,digit);
	}
    }
    break;


  case nt_varNext:
    {
      NADLvarDef vard(layout.varMap[num->var]);
      if (vard.next == NULL) 
	{
	  cout << "\nbddLayout.hpp numExp2binary: variable " << num->var;
	  cout << "' is not defined\nexiting\n";
	  exit(1);
	}      
      else
	{
	  digitNum = vard.length;
	  
	digit = new bdd[digitNum];
	
	//intialize the digit array
	for (i=0; i < digitNum; i++)
	  digit[i] = bdd_ithvar(vard.next[digitNum - 1 - i]);
	
	b = new binary(digitNum,digit);
	}
    }
    break;


  case nt_plus:
    l = numExp2binary(layout,num->left);
    r = numExp2binary(layout,num->right);

    // define result binary
    digitNum = l->digitNum > r->digitNum ? l->digitNum + 1 : r->digitNum + 1;
    digit = new bdd[digitNum];

    // set initial carry
    c = bddfalse;
    i = 0;

    // add the binaries
    while ( i < l->digitNum || i < r->digitNum )
      {
	if ( i < l->digitNum && i < r->digitNum )
	  {
	    digit[i] = l->digit[i] ^ r->digit[i] ^ c;
	    c = l->digit[i] & r->digit[i] |
	        l->digit[i] & c |
	        r->digit[i] & c;
	  }
	else
	  if ( i < l->digitNum )
	    {
	      digit[i] = l->digit[i] ^ c;
	      c = l->digit[i] & c;
	    }
	  else
	    {
	      digit[i] = r->digit[i] ^ c;
	      c = r->digit[i] & c;
	    }
	i++;
      }
    
    // post: i = last digit in result (corresponds to carry)
    digit[i] = c;

    b = new binary(digitNum,digit);
    break;
  }
  return b;
}




////////////////////////////////////////////////////////
// Functions for Generating an action expression
////////////////////////////////////////////////////////

  
// IN
//  layout : domain definition. 
//  i      : Action i of  system
// OUT
//  bdd expression for the precondition
//  of action(i,j).
bdd P(NADLbddLayout &layout,int i) {
  return formula2bdd(layout,layout.agt.act[i].tran[0].pre);
}

// IN
//  layout : domain definition. 
//  i      : Action i of system agent
// OUT
//  bdd expression for the effect
bdd E(NADLbddLayout &layout,int i) {
  return formula2bdd(layout,layout.agt.act[i].tran[0].eff);
}




// A(layout,i)
// IN 
//  i : system action number
//  layout : domain definition
// OUT
//  bdd representation of the action
bdd A(NADLbddLayout &layout,int i) {
  return P(layout,i) & E(layout,i);
}


////////////////////////////////////////////////////////
// Functions for Generating a transition group expression
////////////////////////////////////////////////////////


//IN
// layout : BDD layout
//      i : action no. of system agent 
//      j : transition group no of action i
//OUT
// BDD of the transition group
bdd trsGroup(NADLbddLayout &layout,int i,int j) {

  bdd res;
  res  = formula2bdd(layout,layout.agt.act[i].tran[j].pre);
  res &= formula2bdd(layout,layout.agt.act[i].tran[j].eff);
  
  return res;
}
 



//IN
// layout : BDD layout
//      i : action no. of environment agent 
//      j : transition group no of action i
//OUT
// BDD of the transition group
bdd trsGroupEnv(NADLNonDetbddLayout &layout,int i,int j) {

  bdd res;
  res  = formula2bdd(layout,layout.agtEnv.act[i].tran[j].pre);
  res &= formula2bdd(layout,layout.agtEnv.act[i].tran[j].eff);
  
  return res;
}



//IN
// layout : BDD layout
//      i : action no. of system agent 
//      j : transition group no of action i
//OUT
// BDD of the transition group
bdd trsGroupErr(NADLbddLayout &layout,int i,int j) {

  bdd res;
  res  = formula2bdd(layout,layout.agt.act[i].tran[j].pre);
  res &= formula2bdd(layout,layout.agt.act[i].tran[j].err);
  
  return res;
}


////////////////////////////////////////////
// NADL Print functions
////////////////////////////////////////////




void actDef::print() {
  cout << "Action " << name << endl;
  cout << "  transition groups :\n";
  for (int i = 0; i < tran.size(); i++)
    {
	tran[i].print();
	cout << endl;
    }
}


void agtDef::print() {
  cout << "Actions " << name << "\n\n";
  for (int i = 0; i < act.size(); i++)
    {
	act[i].print();
	cout << endl;
    }
}


void NADLvarDef::print() {
  cout << "name:" << name << "\n";
  cout << "type: ";
  switch (type) {
  case bt_bool: 
    cout << "bool\n";
    break;
  case bt_nat: 
    cout << "nat\n";
    break;
  }

  cout << "VarDef:\n";
  varDef::print();
}


void NADLbddLayout::print() {
  cout << "System Agent : " << agt.name << endl;
  cout << "NADLVarDef:\n\n";
  for (int i = 0; i < var.size(); i++)
    {
      var[i].print();
      cout << "\n\n";
    }

  cout << "\nvarNames : ";
  for (int i = 0; i < varNames.size(); i++)
    cout << varNames[i] << " ";
  cout << "\n\n";
}


void NADLbddBDDAlayout::print() {
  cout << "\nNADLbddBDDAlayout\n";
  cout << " f encoding\n";
  fVars.print();
  cout << endl;
  cout << " h encoding\n";
  hVars.print();
  NADLbddLayout::print();
}  

void NADLbddPubBDDAlayout::print() {
  cout << "\nNADLbddPubBDDAlayout\n";
  cout << " f encoding\n";
  fVars.print();
  cout << endl;
  cout << " h encoding\n";
  hVars.print();
  NADLbddLayout::print();
}  




void NADLNonDetbddLayout::print() {
  cout << "\nNADLNonDetBDDlayout\n\n";
  if (envActive)
    {
      cout << "actID env:\n\n";
      actIDenv.print();
      cout << "\n";      
    }
  cout << "actID sys:\n\n";
  actID.print();
  cout << "\n\n";      
  NADLbddLayout::print();
}

