//////////////////////////////////////////////////////// 
// File  : balancedfluents.cc   
// Desc. : Functions for deriving sets of balanced fluents
//	   of a domain
// Author: Rune M. Jensen
// Date  : 1/10/02
//////////////////////////////////////////////////////// 

#include<set>
#include<map>
#include<vector>
#include "set.hpp"
#include "reachInfo.hpp"
#include "balancedInfo.hpp"

// Aux. functions


// function for translating a property to a bit vector 
// form: [argno-1,predno-1]
// argno: 1,2,...
// predno: 1,2,...
inline int property2id(vector<int> &property,int &predShift) {

  int id = 0;
  id += property[1]-1;
  id = id << predShift;
  id += property[0]-1;
  return id;
}



// convert a bitvector rep. of a property to 
// regular vector rep. of a property
inline void id2property(int &id, vector<int> &property, 
			int &predMask,
			int &predShift) {
  property.clear();
  int id0 = id;
  int pred = id0 & predMask;
  property.push_back(pred+1);
  id0 = id0 >> predShift;
  property.push_back( id0 + 1 );
}






////////////////////////////////////////////////////////// 
// Step 1: make the candidate set of 
//         balanced fluents, s       
// comment: s is a set of set of     
//          properties               
////////////////////////////////////////////////////////// 

// Initially each candidate is a singleton
// set for each property
//
// To get the final candidate set we 
// consider every pair of properties
// p1 and p2 from distinct predicates
// and union the sets they belong to iff
// (dom(p1) <= dom(p2) or dom(p1) >= dom(p2)
// and all properties in set(p1) and set(p2) 
// are distinct 
void balancedInfo::makeCandidates() {

  // initiate s
  set<int> e;
  for (map< int, set<int> >::iterator mi = propertyDom.begin(); mi != propertyDom.end(); ++mi)
    {
      e.clear();
      e.insert(mi->first);
      s.insert(e);
    }

  // go through all property pairs of distinct predicates 
  for (map< int, set<int> >::iterator mi1 = propertyDom.begin(); mi1 != propertyDom.end(); ++mi1)
    for (map< int, set<int> >::iterator mi2 = propertyDom.begin(); mi2 != propertyDom.end(); ++mi2)
      {
	vector<int> property1;
	int id1 = mi1->first;
	id2property(id1,property1,predMask,predShift);

	vector<int> property2;
	int id2 = mi2->first;
	id2property(id2,property2,predMask,predShift);
	
	// check if the predicate of the two properties are distinct 
	if (property1[0] != property2[0]) 
	  // union their sets if their domains are subsets of each other
	  if (setDifference(mi1->second,mi2->second).empty() ||
	      setDifference(mi2->second,mi1->second).empty() )
	    {
	      // find set of property 1 and 2
	      set<int> s1;
	      set<int> s2;	      
	      for (set< set<int> >::iterator si = s.begin(); si != s.end(); ++si)
		{
		  set<int> s = *si;
		  if (setMember(s,id1)) s1 = *si;
		  if (setMember(s,id2)) s2 = *si;
		}

	      set<int> newe = setUnion(s1,s2);
	      // check if all predicates of properties in union are of distinct predicates
	      bool checkOk = true;
	      
	      for (set<int>::iterator sia = newe.begin(); sia != newe.end(); ++sia)
		{
		  vector<int> propertya;
		  int ida = *sia;
		  id2property(ida,propertya,predMask,predShift);
		  set<int>::iterator sibStart = sia;
		  ++sibStart;

		  for (set<int>::iterator sib = sibStart; sib != newe.end(); ++sib)
		    {
		      vector<int> propertyb;
		      int idb = *sib;
		      id2property(idb,propertyb,predMask,predShift);
		      if (propertya[0] == propertyb[0])
			{
			  checkOk = false;
			  break;
			}
		    }
		  
		  if (!checkOk) break;
		}
	      
		
	      if (checkOk)
		{
		  s.erase(s1);
		  s.erase(s2);
		  s.insert(newe);
		}
	    }
      }	     
}	      





////////////////////////////////////////////////////////// 
// Step 2: Go through candidate set,s  
//         and produce the true set of 
//         balanced properties         
//                                     
//   a) prove that the claim holds     
//      in the initial state           
//   b) prove claim holds for each     
//      action                         
//                                     
// Invariant:                          
//   at any time a property belongs to 
//   most one balanced set candidate.     
////////////////////////////////////////////////////////// 

// IN
// bproperties : the set of balanced properties candidate
// OUT
// true if candidate is balanced in init
bool balancedInfo::holdsInInit(set<int> bProperties,numDomain &numDom) {
  
  // find the union of the property domains
  set<int> dom;
  for (set<int>::iterator si = bProperties.begin(); si != bProperties.end(); ++si)
    dom = setUnion(dom,propertyDom[*si]);
  
  // find the set of predicates and their associated argument
  // in the set of properties 
  set<int> predicatesInSet;  // set of predicates in property set
  map<int,int> predicateArg; // mapping from predicate to the property argument of the predicate
  for (set<int>::iterator si = bProperties.begin(); si != bProperties.end(); ++si)
    {
      vector<int> property;
      int id = *si;
      id2property(id,property,predMask,predShift);
      predicatesInSet.insert(property[0]);
      predicateArg[property[0]] = property[1]; 
    }
  
  
  // fil out a mapping from each property object in dom to the number of true properties in init
  // (should be exactly 1 for each object)
  map<int,int> propertyNum;
  for (int i = 0; i < numDom.init.size(); i++)
    if (setMember(predicatesInSet,(numDom.init[i])[0]))
      propertyNum[(numDom.init[i])[ predicateArg[ (numDom.init[i])[0] ] ] ]++; 
  
  // check if all objects are balanced
  for (set<int>::iterator si = dom.begin(); si != dom.end(); ++si)
    if (propertyNum[*si] != 1) return false;
  
  return true;
}


  

// IN
// bproiepties : the set of balanced properties candidate
// OUT
// true if candidate is balanced in init
bool balancedInfo::holdsForActions(set<int> bProperties,numDomain &numDom) {

  // find the set of predicates in the balanced set candidate 
  // and their associated argument 
  set<int> predicatesInSet;  // set of predicates in property set
  map<int,int> predicateArg; // mapping from predicate to the property argument of the predicate
  for (set<int>::iterator si = bProperties.begin(); si != bProperties.end(); ++si)
    {
      vector<int> property;
      int id = *si;
      id2property(id,property,predMask,predShift);
      predicatesInSet.insert(property[0]);
      predicateArg[property[0]] = property[1]; 
    }


  // go through each action
  for (int i=0; i<numDom.act.size(); i++)
    {
      // preds in add
      set<int> predicatesInAdd;
      for (int j=0; j < numDom.act[i].add.size(); j++)
	predicatesInAdd.insert( (numDom.act[i].add[j])[0]);

      // preds in del
      set<int> predicatesInDel;
      for (int j=0; j < numDom.act[i].del.size(); j++)
	predicatesInDel.insert( (numDom.act[i].del[j])[0]);
      
      
      // case 1: neither del or add refers to a predicate in candidate
      if (setIntersection(predicatesInSet,predicatesInAdd).empty() &&
	  setIntersection(predicatesInSet,predicatesInDel).empty())
	{} // ok do nothing
	else
	  // check if swap
	  {
	    set<int> addSet = setIntersection(predicatesInSet,predicatesInAdd);
	    set<int> delSet = setIntersection(predicatesInSet,predicatesInDel);

	    // we require only one property predicate touched in add and del
            // this might be a too strong requirement, but it works on the 
            // examples used for AAAI-02
	    if (addSet.size() == 1 && delSet.size() == 1)
	      { 
		int add = *(addSet.begin());
		int del = *(delSet.begin());

		// find the add and del predicates in add and del list of action
		vector<int> addPredicate;
		for (int j=0; j < numDom.act[i].add.size(); j++)
		  if ( (numDom.act[i].add[j])[0] == add ) addPredicate = numDom.act[i].add[j];

		vector<int> delPredicate;
		for (int j=0; j < numDom.act[i].del.size(); j++)
		  if ( (numDom.act[i].del[j])[0] == del ) delPredicate = numDom.act[i].del[j];
		
		// check that the add and del predicates exchange on the property argument
                // such that balance is maintained
		if ( delPredicate[ predicateArg[del] ] == addPredicate[ predicateArg[add] ] )
		  {} // do nothing
		  else
		    return false;
	      }
	    else
	      return false;
	  }
    }  
  return true;
}


// Note that this function is not completely checking all subsets of
// a candidate property set. However the cases considered suffices for
// the examples seen so far
void balancedInfo::findBalanced(numDomain &numDom) {
  
  // init set of all tried blanced property sets 
  t = s;

  // go through s

  for (set< set<int> >::iterator ssi = s.begin(); ssi != s.end(); ++ssi)
    if ( holdsInInit(*ssi,numDom) && holdsForActions(*ssi,numDom)   ) b.insert(*ssi); else

      // we only consider a subset of all subsets
      switch (ssi->size()) {
      case 1 :
	// do nothing, no subset exists
	break;
      case 2 :
	{
	  // see if the individual elements are balanced
	  set<int> e;
	  for (set<int>::iterator si = ssi->begin(); si != ssi->end(); ++si)
	    {
	      e.clear();
	      e.insert(*si);
	      t.insert(e);
	      if (holdsInInit(e,numDom) && holdsForActions(e,numDom)) b.insert(e);	      
	    }
	}
	break;

      case 3: 
	// erase one element and see if the remaining elements are balanced
	// OBS: incomplete
	for (set<int>::iterator si = ssi->begin(); si != ssi->end(); ++si)
	  {
	    set<int> out;
	    out.clear();
	    out.insert(*si);
	    set<int> in = *ssi;
	    in.erase(*si);
	    t.insert(in);
	    if (holdsInInit(in,numDom) && holdsForActions(in,numDom) )
	      {
		b.insert(in);
		if (holdsInInit(out,numDom) && holdsForActions(out,numDom) ) b.insert(out);
		break;
	      }
	  }
	break;

      case 4: 
	// erase one element and see if the remaining elements are balanced
	// OBS: incomplete
	for (set<int>::iterator si = ssi->begin(); si != ssi->end(); ++si)
	  {
	    set<int> out;
	    out.clear();
	    out.insert(*si);
	    set<int> in = *ssi;
	    in.erase(*si);
	    t.insert(in);
	    if (holdsInInit(in,numDom) && holdsForActions(in,numDom) )
	      {
		b.insert(in);
		if (holdsInInit(out,numDom) && holdsForActions(out,numDom) ) b.insert(out);
		break;
	      }
	  }
	break;


      }
}


////////////////////////////////////////////////////////// 
// Step 3: Find the the best coverage     
//         of the predicates             
//                                       
// Algorithm:                            
//   a) Rank each balanced set according 
//      to how many objects the set      
//      involves                         
//                                       
//   b) Take sets with higest rank and   
//      exclude subsequent sets          
//      conflicting with taken sets      
//                                          
////////////////////////////////////////////////////////// 
void balancedInfo::coverPredicates() {
      
  // rank each set
  // rank equals sum of dom(properties in balanced set)
  map<set<int>, int > rank;
  for (set< set<int> >::iterator vi = b.begin(); vi != b.end(); ++vi)
    {
      // find the union of the property domains
      set<int> dom;
      dom.clear();
      for (set<int>::iterator si = vi->begin(); si != vi->end(); ++si)
	dom = setUnion(dom,propertyDom[*si]);
      rank[*vi] = dom.size();
    }
    
  
  // define a working set
  set< set<int> > w = b;

  set<int> exclude;
  while (!w.empty())
    {
      // find higest rank
      set<int> maxRank;
      int max = 0;
      for (set< set<int> >::iterator vi = w.begin(); vi != w.end(); ++vi)
	if (rank[*vi] > max)
	  {
	    max = rank[*vi];
	    maxRank = *vi;
	  }

      // predicates in max rank
      set<int> predicatesInMax;
      predicatesInMax.clear();
      for (set<int>::iterator si = maxRank.begin(); si != maxRank.end(); ++si)
	{
	  vector<int> property;
	  int id = *si;
	  id2property(id,property,predMask,predShift);
	  predicatesInMax.insert(property[0]);
	}

      // if max don't overlaps with already covered predicates then insert
      if (setIntersection(predicatesInMax,exclude).empty())
	{
	  c.insert(maxRank);
	  exclude = setUnion(exclude,predicatesInMax);
	}
      
      // erase maxrank from working set
      w.erase(maxRank);
    }

  // make expanded vectorform of c (called cv)
  for (set< set<int> >::iterator vi = c.begin(); vi != c.end(); ++vi)
    {
      set< vector<int> > curBalancedSet;
      
      for (set<int>::iterator si = vi->begin(); si != vi->end(); ++si)
	{
	  vector<int> property;
	  int id = *si;
	  id2property(id,property,predMask,predShift);
	  curBalancedSet.insert(property);
	}
      
      cv.push_back(curBalancedSet);
    }
       

}
  




////////////////////////////////////////////////////////// 
//    misc print functions 
////////////////////////////////////////////////////////// 

 void printProperty(vector<int> &property,numDomain &numDom) {
  cout << numDom.pred[property[0]-1] << "[" << property[1] << "]";
}


void balancedInfo::print(numDomain &numDom) {

  cout << "Property domain\n";   
  for (map< int, set<int> >::iterator mi = propertyDom.begin(); mi != propertyDom.end(); ++mi)
    {
      vector<int> property;
      int id = mi->first;
      id2property(id,property,predMask,predShift);
      cout << "  ";
      printProperty(property,numDom);
      cout << " : ";
      for (set<int>::iterator si = mi->second.begin(); si != mi->second.end(); ++si)
	cout << numDom.obj[*si-1] << " ";
      cout << "\n";
    }
  cout << "\n";
  

  cout << "Balanced fluents initial candidates\n";
  for (set< set<int> >::iterator ssi = s.begin(); ssi != s.end(); ++ssi)
    {
      for (set<int>::iterator si = ssi->begin(); si != ssi->end(); ++si)
	{
	  vector<int> property;
	  int id = *si;
	  id2property(id,property,predMask,predShift);
	  cout << "  ";
	  printProperty(property,numDom);
	}
      cout << "\n   Balance holds in init: " << holdsInInit(*ssi,numDom);
      cout << "\n   Balance holds in acts: " << holdsForActions(*ssi,numDom) << endl;
      
    }
  cout << endl;

  cout << "Balanced fluents tried candidates\n";
  for (set< set<int> >::iterator ssi = t.begin(); ssi != t.end(); ++ssi)
    {
      for (set<int>::iterator si = ssi->begin(); si != ssi->end(); ++si)
	{
	  vector<int> property;
	  int id = *si;
	  id2property(id,property,predMask,predShift);
	  cout << "  ";
	  printProperty(property,numDom);
	}
      cout << "\n   Balance holds in init: " << holdsInInit(*ssi,numDom);
      cout << "\n   Balance holds in acts: " << holdsForActions(*ssi,numDom) << endl;
      
    }
  cout << endl;
  
  cout << "Proved Balanced fluents\n";
  for (set< set<int> >::iterator ssi = b.begin(); ssi != b.end(); ++ssi)
    {
      for (set<int>::iterator si = ssi->begin(); si != ssi->end(); ++si)
	{
	  vector<int> property;
	  int id = *si;
	  id2property(id,property,predMask,predShift);
	  cout << "  ";
	  printProperty(property,numDom);
	}
      cout << endl;
    }

  cout << "Chosen non-conflicting balanced fluents\n";
  for (int i=0; i<cv.size();i++)
    for (set< vector<int> >::iterator si = cv[i].begin(); si != cv[i].end(); ++si)
	{
	  cout << "  ";
	  vector<int> property = *si; 
	  printProperty(property,numDom);
	}  
      cout << endl;
}




//     Balancedinfo Constructor
//     init. misc vars
//     + addad all the analysis
void balancedInfo::analyse(reachInfo &reached,numDomain &numDom) {

  predShift = reached.predShift;
  predMask =  reached.predMask;
  
  // make the property domain
 
  // init. the domain of each property.
  // we only consider properties of fluents since
  // static predicates don't change
  for (set<int>::iterator si=reached.closedFacts.begin(); si != reached.closedFacts.end(); ++si)
    {
      // get the fact
      vector<int> fact;
      int id = *si;
      id2fact(id,fact,reached.objMask,reached.predMask,
	      reached.predShift,reached.objShift,
	      reached.predArity);
      
      // if predicate of fact is a fluent then update property domain
      if (!setMember(reached.staticPreds,fact[0]))
	// go through args
	for (int i=0; i<fact.size()-1; i++)
	  {
	    // compute property id
	    vector<int> property;
	    property.push_back(fact[0]);
	    property.push_back(i+1);
	    int propid = property2id(property,predShift);
	    propertyDom[propid].insert(fact[i+1]);
	  }
    }	      

  // analyse data
  makeCandidates();
  findBalanced(numDom);
  coverPredicates();
}

