#include <stdlib.h>
#include <stdio.h>
#include <values.h>
#include <time.h>
#include <math.h>

#define RANDOM(x) (random()%(x))
#define SEED(x) srandom((unsigned int)(x));
#define RANDOM_MAX RAND_MAX

#define MAX_TARGETS     40
#define MAX_INSTRUMENTS 15
#define MAX_QUALITIES   10
#define MAX_GOALS       10
#define CIRCLE_LENGTH   60
#define SLOW_TURN_SPEED 2.0
#define SLOW_TURN_ACC   2.0
#define HARD_TURN_SPEED 2.0
#define HARD_TURN_FUEL  1.0

#define MAX(x,y) (((x) < (y)) ? (y) : (x))
#define MIN(x,y) (((x) < (y)) ? (x) : (y))
#define ANGLE_DISTANCE(x,y) (x < y ? MIN(y - x, (CIRCLE_LENGTH - y) + x) : MIN(x - y, (CIRCLE_LENGTH - x) + y))

unsigned int n_targets;
unsigned int n_ref_targets;
unsigned int n_instruments;
unsigned int min_cal_targets;
unsigned int max_cal_targets;
unsigned int n_qualities;
float redundancy_factor;
unsigned int n_goals;
float easy_turn_fuel_cons;

int hard_turn;
int downlink;
unsigned int domain_type;

unsigned int power_min, power_max, power_div;
unsigned int store_min, store_max, store_div;
float fuel_avail, power_avail, store_avail, bandwidth_avail;

int n_instances;

unsigned int target_orientation[MAX_TARGETS];
unsigned int goal_target[MAX_GOALS];
unsigned int goal_quality[MAX_GOALS];
unsigned char instrument_capability[MAX_INSTRUMENTS][MAX_QUALITIES];
unsigned char calibration_target[MAX_INSTRUMENTS][MAX_TARGETS];

char fname[255];

void generate_instance() {
  unsigned int k, i, n, occ;
  unsigned int min_support, max_support;
  unsigned int n_support[MAX_QUALITIES];

  for (k = 0; k < n_targets+n_ref_targets; k++) {
    target_orientation[k] = RANDOM(CIRCLE_LENGTH - 1) + 1;
    do {
      occ = 0;
      for (i = 0; (i < k) && !occ; i++)
	if (target_orientation[i] == target_orientation[k]) occ = 1;
      if (occ) target_orientation[k] += 1;
    } while (occ);
  }

  for (k = 0; k < n_goals; k++) {
    goal_target[k] = RANDOM(n_targets)+1;
    goal_quality[k] = RANDOM(n_qualities)+1;
  }

  min_support = (unsigned int)(((redundancy_factor*n_qualities)/
				(float)n_instruments)+0.99);
  max_support = (unsigned int)(((n_instruments*min_support)/
				(float)n_qualities)+0.99)+1;
  for (k = 0; k < n_qualities; k++) n_support[k] = 0;

  /* printf("min_support = %u\n", min_support); */
  /* printf("max_support = %u\n", max_support); */

  for (k = 0; k < n_instruments; k++) {
    /* printf("cam%d:", k+1); */
    for (i = 0; i < n_qualities; i++)
      instrument_capability[k][i] = 0;
    for (n = 0; n < min_support; n++) {
      i = RANDOM(n_qualities);
      while (n_support[i] == max_support) i = (i+1)%n_qualities;
      instrument_capability[k][i] = 1;
      /* printf(" q%d", i+1); */
      n_support[i] += 1;
    }
    /* printf("\n"); */

    for (i = 0; i < n_targets+n_ref_targets; i++)
      calibration_target[k][i] = 0;
    for (n = RANDOM(max_cal_targets - min_cal_targets) + min_cal_targets;
	 n > 0; n--) {
      i = RANDOM(n_ref_targets);
      calibration_target[k][i] = 1;
    }
  }

  for (k = 0; k < n_qualities; k++)
    if (n_support[k] == 0) {
      i = RANDOM(n_instruments);
      instrument_capability[i][k] = 1;
    }
}

float slow_turn_time(unsigned int x, unsigned int y) {
  return (ANGLE_DISTANCE(x,y)/SLOW_TURN_SPEED)+SLOW_TURN_ACC;
}

float hard_turn_time(unsigned int x, unsigned int y) {
  return sqrt(2.0*(float)ANGLE_DISTANCE(x,y))/HARD_TURN_SPEED;
}

float hard_turn_fuel(unsigned int x, unsigned int y) {
  return (ANGLE_DISTANCE(x,y)/HARD_TURN_FUEL);
}

void write_instance(int n, char* name) {
  FILE *f;
  unsigned int k, i;

  f = fopen(name, "w");
  if (f == NULL) {
    printf("exception: can't open \"%s\"\n", name);
    exit(0);
  }
  fprintf(f, "(define (problem spacecraft%u_p%d)\n", domain_type, n);
  fprintf(f, " (:domain spacecraft%u)\n", domain_type);
  fprintf(f, " (:requirements :strips :typing)\n");
  fprintf(f, " (:objects");
  for (k = 0; k < n_ref_targets; k++) fprintf(f, " ref%d", k+1);
  for (k = 0; k < n_targets; k++) fprintf(f, " a%d", k+1);
  fprintf(f, " - direction");
  for (k = 0; k < n_instruments; k++) fprintf(f, " cam%d", k+1);
  fprintf(f, " - instrument");
  for (k = 0; k < n_qualities; k++) fprintf(f, " q%d", k+1);
  fprintf(f, " - quality");
  fprintf(f, ")\n");

  fprintf(f, " (:init\n");
  for (k = 0; k < n_ref_targets; k++) {
    fprintf(f, "  (time_to_turn Earth ref%d %.1f)",
	    k+1, slow_turn_time(0, target_orientation[k]));
    fprintf(f, " (time_to_turn ref%d Earth %.1f)\n",
	    k+1, slow_turn_time(0, target_orientation[k]));
    if (hard_turn) {
      fprintf(f, "  (hard_turn_time Earth ref%d %.1f)",
	      k+1, hard_turn_time(0, target_orientation[k]));
      fprintf(f, " (hard_turn_time ref%d Earth %.1f)\n",
	      k+1, hard_turn_time(0, target_orientation[k]));
      fprintf(f, "  (hard_turn_fuel Earth ref%d %.1f)",
	      k+1, hard_turn_fuel(0, target_orientation[k]));
      fprintf(f, " (hard_turn_fuel ref%d Earth %.1f)\n",
	      k+1, hard_turn_fuel(0, target_orientation[k]));
    }
    for (i = 0; i < n_ref_targets; i++) if (i != k) {
      fprintf(f, "  (time_to_turn ref%d ref%d %.1f)",
	      k+1, i+1, slow_turn_time(target_orientation[i],
				       target_orientation[k]));
      fprintf(f, " (time_to_turn ref%d ref%d %.1f)\n",
	      i+1, k+1, slow_turn_time(target_orientation[i],
				       target_orientation[k]));
      if (hard_turn) {
	fprintf(f, "  (hard_turn_time ref%d ref%d %.1f)",
		k+1, i+1, hard_turn_time(target_orientation[i],
					 target_orientation[k]));
	fprintf(f, " (hard_turn_time ref%d ref%d %.1f)\n",
		i+1, k+1, hard_turn_time(target_orientation[i],
					 target_orientation[k]));
	fprintf(f, "  (hard_turn_fuel ref%d ref%d %.1f)",
		k+1, i+1, hard_turn_fuel(target_orientation[i],
					 target_orientation[k]));
	fprintf(f, " (hard_turn_fuel ref%d ref%d %.1f)\n",
		i+1, k+1, hard_turn_fuel(target_orientation[i],
					 target_orientation[k]));
      }
    }
    for (i = 0; i < n_targets; i++) {
      fprintf(f, "  (time_to_turn ref%d a%d %.1f)",
	      k+1, i+1, slow_turn_time(target_orientation[i+n_ref_targets],
				       target_orientation[k]));
      fprintf(f, " (time_to_turn a%d ref%d %.1f)\n",
	      i+1, k+1, slow_turn_time(target_orientation[i+n_ref_targets],
				       target_orientation[k]));
      if (hard_turn) {
	fprintf(f, "  (hard_turn_time ref%d a%d %.1f)",
		k+1, i+1, hard_turn_time(target_orientation[i+n_ref_targets],
					 target_orientation[k]));
	fprintf(f, " (hard_turn_time a%d ref%d %.1f)\n",
		i+1, k+1, hard_turn_time(target_orientation[i+n_ref_targets],
					 target_orientation[k]));
	fprintf(f, "  (hard_turn_fuel ref%d a%d %.1f)",
		k+1, i+1, hard_turn_fuel(target_orientation[i+n_ref_targets],
					 target_orientation[k]));
	fprintf(f, " (hard_turn_fuel a%d ref%d %.1f)\n",
		i+1, k+1, hard_turn_fuel(target_orientation[i+n_ref_targets],
					 target_orientation[k]));
      }
    }
  }
  for (k = 0; k < n_targets; k++) {
    fprintf(f, "  (time_to_turn Earth a%d %.1f)",
	    k+1, slow_turn_time(0, target_orientation[k+n_ref_targets]));
    fprintf(f, " (time_to_turn a%d Earth %.1f)\n",
	    k+1, slow_turn_time(0, target_orientation[k+n_ref_targets]));
    if (hard_turn) {
      fprintf(f, "  (hard_turn_time Earth a%d %.1f)",
	      k+1, hard_turn_time(0, target_orientation[k+n_ref_targets]));
      fprintf(f, " (hard_turn_time a%d Earth %.1f)\n",
	      k+1, hard_turn_time(0, target_orientation[k+n_ref_targets]));
      fprintf(f, "  (hard_turn_fuel Earth a%d %.1f)",
	      k+1, hard_turn_fuel(0, target_orientation[k+n_ref_targets]));
      fprintf(f, " (hard_turn_fuel a%d Earth %.1f)\n",
	      k+1, hard_turn_fuel(0, target_orientation[k+n_ref_targets]));
    }
    for (i = 0; i < n_targets; i++) if (i != k) {
      fprintf(f, "  (time_to_turn a%d a%d %.1f)",
	      k+1, i+1, slow_turn_time(target_orientation[i+n_ref_targets],
				       target_orientation[k+n_ref_targets]));
      fprintf(f, " (time_to_turn a%d a%d %.1f)\n",
	      i+1, k+1, slow_turn_time(target_orientation[i+n_ref_targets],
				       target_orientation[k+n_ref_targets]));
      if (hard_turn) {
	fprintf(f, "  (hard_turn_time a%d a%d %.1f)",
		k+1, i+1, hard_turn_time(target_orientation[i+n_ref_targets],
					 target_orientation[k+n_ref_targets]));
	fprintf(f, " (hard_turn_time a%d a%d %.1f)\n",
		i+1, k+1, hard_turn_time(target_orientation[i+n_ref_targets],
					 target_orientation[k+n_ref_targets]));
	fprintf(f, "  (hard_turn_fuel a%d a%d %.1f)",
		k+1, i+1, hard_turn_fuel(target_orientation[i+n_ref_targets],
					 target_orientation[k+n_ref_targets]));
	fprintf(f, " (hard_turn_fuel a%d a%d %.1f)\n",
		i+1, k+1, hard_turn_fuel(target_orientation[i+n_ref_targets],
					 target_orientation[k+n_ref_targets]));
      }
    }
  }
  for (k = 0; k < n_instruments; k++) {
    for (i = 0; i < n_qualities; i++)
      if (instrument_capability[k][i])
	fprintf(f, "  (capability cam%d q%d)\n", k+1, i+1);
    for (i = 0; i < n_ref_targets; i++)
      if (calibration_target[k][i])
	fprintf(f, "  (calibration_target cam%d ref%d)\n", k+1, i+1);
    fprintf(f, "  (power_required cam%d %.1f)\n", k+1,
	    (RANDOM((power_max-power_min)+1)+power_min)/(float)power_div);
  }
  if (downlink) for (k = 0; k < n_qualities; k++) {
    fprintf(f, "  (image_size q%d %.1f)\n", k+1,
	    (RANDOM((store_max-store_min)+1)+store_min)/(float)store_div);
  }
  fprintf(f, "  (pointing Earth))\n");

  if (downlink)
    fprintf(f, " (:resources (fuel %.1f) (power %.1f) (storage %.1f))\n",
	    fuel_avail, power_avail, store_avail);
  else
    fprintf(f, " (:resources (fuel %.1f) (power %.1f))\n",
	    fuel_avail, power_avail);

  fprintf(f, " (:goal (and");
  for (k = 0; k < n_goals; k++) {
    if (downlink)
      fprintf(f, " (image_sent a%d q%d)", goal_target[k], goal_quality[k]);
    else
      fprintf(f, " (have_image a%d q%d)", goal_target[k], goal_quality[k]);
  }
  fprintf(f, "))\n)\n");
  fclose(f);
}

void write_domain() {
  FILE *f;

  sprintf(fname, "sc%u.pddl", domain_type);
  f = fopen(fname, "w");
  if (f == NULL) {
    printf("exception: can't open \"%s\"\n", fname);
    exit(0);
  }

  fprintf(f, "(define (domain spacecraft%u)\n", domain_type);
  fprintf(f, " (:requirements :strips :typing)\n");
  fprintf(f, " (:types direction instrument quality)\n");
  fprintf(f, " (:constants Earth - direction)\n");

  fprintf(f, " (:predicates\n");
  fprintf(f, "   (pointing ?d - direction)\n");
  fprintf(f, "   (power_on ?i - instrument)\n");
  fprintf(f, "   (calibrated ?i - instrument)\n");
  fprintf(f, "   (have_image ?d - direction ?q - quality)\n");
  if (downlink)
    fprintf(f, "   (image_sent ?d - direction ?q - quality)\n");
  fprintf(f, "   (time_to_turn ?d1 ?d2 - direction ?t)\n");
  if (hard_turn) {
    fprintf(f, "   (hard_turn_time ?d1 ?d2 - direction ?t)\n");
    fprintf(f, "   (hard_turn_fuel ?d1 ?d2 - direction ?t)\n");
  }
  fprintf(f, "   (capability ?i - instrument ?q - quality)\n");
  fprintf(f, "   (calibration_target ?i - instrument ?d - direction)\n");
  fprintf(f, "   (power_required ?i - instrument ?p)\n");

  if (downlink)
    fprintf(f, "   (image_size ?q - quality ?s)\n");
  fprintf(f, "  )\n");
  if (downlink)
    fprintf(f, " (:resources power storage use_of_link)\n");
  else
    fprintf(f, " (:resources power)\n");
  fprintf(f, " (:consumable fuel)\n\n");

  /* standard turning and pointing operators */
  fprintf(f, " (:action turn_to\n   :parameters (?d_new - direction)\n   :vars (?d_prev - direction)\n   :precondition (and (pointing ?d_prev) (not (= ?d_prev ?d_new)))\n   :effect (and (not (pointing ?d_prev)) (pointing ?d_new))\n   :resources ((fuel %.1f))\n   :delay ?t : (time_to_turn ?d_prev ?d_new ?t))\n\n (:action NOOP\n   :parameters (?d - direction)\n   :effect (pointing ?d)\n   :resources ((fuel 0.5)))\n\n", easy_turn_fuel_cons);

  /* hard turn operator */
  if (hard_turn)
    fprintf(f, " (:action turn_hard\n   :parameters (?d_new - direction)\n   :vars (?d_prev - direction)\n   :precondition (and (pointing ?d_prev) (not (= ?d_prev ?d_new)))\n   :effect (and (not (pointing ?d_prev)) (pointing ?d_new))\n   :resources ((fuel ?a : (hard_turn_fuel ?d_prev ?d_new ?a)))\n   :delay ?t : (hard_turn_time ?d_prev ?d_new ?t))\n\n");

  /* instrument management */
  fprintf(f, " (:action switch_on\n  :parameters (?i - instrument)\n  :effect (and (power_on ?i) (not (calibrated ?i)))\n  :resources ((power ?p : (power_required ?i ?p)))\n  :delay 2.0)\n\n (:action NOOP\n  :parameters (?i - instrument)\n  :effect (power_on ?i)\n  :resources ((power ?p : (power_required ?i ?p))))\n\n (:action calibrate\n  :parameters (?i - instrument ?d - direction)\n  :precondition (and (power_on ?i) (pointing ?d) (calibration_target ?i ?d))\n  :effect (calibrated ?i)\n  :delay 1.0)\n\n");

  /* taking and sending images */
  if (downlink)
    fprintf(f, " (:action take_image\n  :parameters (?i - instrument ?q - quality ?d - direction)\n  :precondition (and (power_on ?i) (calibrated ?i) (capability ?i ?q)\n	      (pointing ?d))\n  :effect (have_image ?d ?q)\n  :resources ((storage ?s : (image_size ?q ?s)))\n  :delay 1.0)\n\n (:action NOOP\n  :parameters (?d - direction ?q - quality)\n  :effect (have_image ?d ?q)\n  :resources ((storage ?s : (image_size ?q ?s))))\n\n (:action send_image\n  :parameters (?d - direction ?q - quality)\n  :precondition (and (have_image ?d ?q) (pointing Earth))\n  :effect (and (not (have_image ?d ?q)) (image_sent ?d ?q))\n  :resources ((power 2.0))\n  :delay 2.0)\n");
  else
    fprintf(f, " (:action take_image\n  :parameters (?i - instrument ?q - quality ?d - direction)\n  :precondition (and (power_on ?i) (calibrated ?i) (capability ?i ?q) (pointing ?d))\n  :effect (have_image ?d ?q)\n  :delay 1.0)\n");

  fprintf(f, " )\n");
  fclose(f);
}

int main(int argc, char* argv[]) {
  time_t t_zero;
  unsigned long r_seed;
  int write_d, print_help, k;

  n_targets = 10;
  n_ref_targets = 5;
  n_instruments = 4;
  min_cal_targets = 1;
  max_cal_targets = 3;
  n_qualities = 3;
  redundancy_factor = 1.5;
  n_goals = 4;
  hard_turn = 0;
  downlink = 0;

  power_min = 2;
  power_max = 10;
  power_div = 2;
  store_min = 2;
  store_max = 10;
  store_div = 2;
  fuel_avail = 100.0;
  power_avail = 5.0;
  store_avail = 10.0;
  bandwidth_avail = 1.0;
  easy_turn_fuel_cons = 3.0;

  write_d = 1;
  n_instances = 1;

  print_help = 0;
  r_seed = 0;

  for (k = 0; k < argc; k++) {
    if ((strcmp(argv[k],"-t") == 0) && (k < argc - 1))
      n_targets = atoi(argv[++k]);
    if ((strcmp(argv[k],"-rt") == 0) && (k < argc - 1))
      n_ref_targets = atoi(argv[++k]);
    if ((strcmp(argv[k],"-i") == 0) && (k < argc - 1))
      n_instruments = atoi(argv[++k]);
    if ((strcmp(argv[k],"-q") == 0) && (k < argc - 1))
      n_qualities = atoi(argv[++k]);
    if ((strcmp(argv[k],"-rf") == 0) && (k < argc - 1))
      redundancy_factor = atof(argv[++k]);
    if ((strcmp(argv[k],"-c") == 0) && (k < argc - 2)) {
      min_cal_targets = atoi(argv[k+1]);
      max_cal_targets = atoi(argv[k+2]);
      k += 2;
    }
    if ((strcmp(argv[k],"-p") == 0) && (k < argc - 3)) {
      power_min = atoi(argv[k+1]);
      power_max = atoi(argv[k+2]);
      power_div = atoi(argv[k+3]);
      k += 3;
    }
    if ((strcmp(argv[k],"-s") == 0) && (k < argc - 3)) {
      store_min = atoi(argv[k+1]);
      store_max = atoi(argv[k+2]);
      store_div = atoi(argv[k+3]);
      k += 3;
    }
    if (((strcmp(argv[k],"-o") == 0) || (strcmp(argv[k],"-g") == 0)) &&
	(k < argc - 1))
      n_goals = atoi(argv[++k]);
    if ((strcmp(argv[k],"-fuel") == 0) && (k < argc - 1))
      fuel_avail = atof(argv[++k]);
    if ((strcmp(argv[k],"-power") == 0) && (k < argc - 1))
      power_avail = atof(argv[++k]);
    if ((strcmp(argv[k],"-store") == 0) && (k < argc - 1))
      store_avail = atof(argv[++k]);
    if ((strcmp(argv[k],"-etfc") == 0) && (k < argc - 1))
      easy_turn_fuel_cons = atof(argv[++k]);

    if (strcmp(argv[k],"-hardturn") == 0) hard_turn = 1;
    if (strcmp(argv[k],"-downlink") == 0) downlink = 1;

    if ((strcmp(argv[k],"-domain") == 0) || (strcmp(argv[k],"-d") == 0))
      write_d = !write_d;
    if ((strcmp(argv[k],"-n") == 0) && (k < argc - 1))
      n_instances = atoi(argv[++k]);
    if ((strcmp(argv[k],"-help") == 0) || (strcmp(argv[k],"-h") == 0) ||
	(strcmp(argv[k],"-?") == 0)) print_help = 1;
    if ((strcmp(argv[k],"-r") == 0) && (k < argc - 1)) {
      r_seed = atol(argv[++k]);
    }
  }

  if (print_help) {
    printf("%s [options] [-n <instances>]\n", argv[0]);
    printf(" -t <n>          : set #targets = <n>\n");
    printf(" -rt <n>         : set #reference targets = <n>\n");
    printf(" -i <n>          : set #instruments = <n>\n");
    printf(" -q <n>          : set #image qualities = <n>\n");
    printf(" -rf <f>         : set redundancy factor = <f>\n");
    printf(" -c <min> <max>  : set #calibration targets = [<min>,<max>]\n");
    printf(" -o <n>, -g <n>  : set #observation goals = <n>\n");
    printf(" -p <a> <b> <c>  : set instrument power rq. = [<a>,<b>]/<c>\n");
    printf(" -s <a> <b> <c>  : set image size = [<a>,<b>]/<c>\n");
    printf(" -fuel <f>       : set fuel available = <f>\n");
    printf(" -power <f>      : set power available = <f>\n");
    printf(" -store <f>      : set data storage available = <f>\n");
    printf(" -domain, -d     : write domain/operator file\n");
    printf(" -hardturn       : domain option: fast turns\n");
    printf(" -downlink       : domain option: downlink\n");
    printf(" -r <n>          : set random seed = <n>\n");
    exit(0);
  }

  domain_type = (hard_turn + 2*downlink);
  if (write_d) write_domain();

  if (r_seed != 0) {
    SEED(r_seed);
  }
  else {
    time(&t_zero);
    SEED(t_zero);
  }

  for (k = 0; k < n_instances; k++) {
    generate_instance();
    sprintf(fname, "sc%u_p%d.pddl", domain_type, k+1);
    write_instance(k+1, fname);
  }
}
