#include <stdlib.h>
#include <stdio.h>
#include <values.h>
#include <time.h>
#include <signal.h>

#define RANDOM(x) (random()%(x))
#define SEED(x) srandom((unsigned int)(x));
#define RANDOM_MAX RAND_MAX

#define MAX_VAR 20

const char* graphplan_domain = "(operator LOAD-TRUCK\n (params (<packet> PACKET) (<truck> TRUCK) (<loc> LOCATION))\n (preconds (at <truck> <loc>) (at <packet> <loc>))\n (effects (in <packet> <truck>) (del at <packet> <loc>)))\n\n(operator LOAD-AIRPLANE\n (params (<packet> PACKET) (<airplane> AIRPLANE) (<loc> LOCATION))\n (preconds (at <packet> <loc>) (at <airplane> <loc>))\n (effects (in <packet> <airplane>) (del at <packet> <loc>)))\n\n(operator UNLOAD-TRUCK\n (params (<packet> PACKET) (<truck> TRUCK) (<loc> LOCATION))\n (preconds (at <truck> <loc>) (in <packet> <truck>))\n (effects (at <packet> <loc>) (del in <packet> <truck>)))\n\n(operator UNLOAD-AIRPLANE\n (params (<packet> PACKET) (<airplane> AIRPLANE) (<loc> LOCATION))\n (preconds (in <packet> <airplane>) (at <airplane> <loc>))\n (effects (at <packet> <loc>) (del in <packet> <airplane>)))\n\n(operator DRIVE\n (params (<truck> TRUCK) (<from> LOCATION) (<to> LOCATION) (<city> CITY))\n (preconds (at <truck> <from>) (loc-at <from> <city>) (loc-at <to> <city>))\n (effects  (at <truck> <to>) (del at <truck> <from>)))\n\n(operator FLY\n (params (<airplane> AIRPLANE) (<from> AIRPORT) (<to> AIRPORT))\n (preconds (at <airplane> <from>))\n (effects  (at <airplane> <to>) (del at <airplane> <from>)))\n";
const char* pddl_typed_domain = "(define (domain logistics)\n  (:requirements :strips :typing :equality)\n  (:types airport - location\n	  truck airplane - vehicle\n	  vehicle packet - thing\n	  thing location city)\n  (:predicates (loc-at ?x - location ?y - city)\n	       (at ?x - thing ?y - location)\n	       (in ?x - packet ?y - vehicle))\n\n  (:action load\n    :parameters (?x - packet ?y - vehicle)\n    :vars (?z - location)\n    :precondition (and (at ?x ?z) (at ?y ?z))\n    :effect (and (not (at ?x ?z)) (in ?x ?y)))\n\n  (:action unload\n    :parameters (?x - packet ?y - vehicle)\n    :vars (?z - location)\n    :precondition (and (at ?y ?z) (in ?x ?y))\n    :effect (and (at ?x ?z) (not (in ?x ?y))))\n\n  (:action drive\n    :parameters (?x - truck ?y - location)\n    :vars (?z - location ?c - city)\n    :precondition (and (loc-at ?z ?c) (loc-at ?y ?c) (not (= ?z ?y))\n		       (at ?x ?z))\n    :effect (and (not (at ?x ?z)) (at ?x ?y)))\n\n  (:action fly\n    :parameters (?x - airplane ?y - airport)\n    :vars (?z - airport)\n    :precondition (and (not (= ?z ?y)) (at ?x ?z))\n    :effect (and (not (at ?x ?z)) (at ?x ?y)))\n  )\n";
const char* pddl_untyped_domain = "(define (domain logistics)\n (:requirements :strips :equality)\n (:predicates (location ?x)\n       (airport ?x)\n       (vehicle ?x)\n       (truck ?x)\n       (airplane ?x)\n       (packet ?x)\n       (city ?x)\n       (loc-at ?x ?y)\n       (at ?x ?y)\n       (in ?x ?y))\n  (:action load\n   :parameters (?x ?y ?z)\n   :precondition (and (packet ?x) (vehicle ?y) (location ?z) (at ?x ?z) (at ?y ?z))\n   :effect (and (not (at ?x ?z)) (in ?x ?y)))\n  (:action unload\n   :parameters (?x ?y ?z)\n   :precondition (and (packet ?x) (vehicle ?y) (location ?z) (at ?y ?z) (in ?x ?y))\n   :effect (and (at ?x ?z) (not (in ?x ?y))))\n  (:action drive\n   :parameters (?x ?y ?z ?c)\n   :precondition (and (truck ?x) (location ?y) (location ?z) (city ?c)\n	       (loc-at ?z ?c) (loc-at ?y ?c) (not (= ?z ?y))\n	       (at ?x ?z))\n   :effect (and (not (at ?x ?z)) (at ?x ?y)))\n  (:action fly\n   :parameters (?x ?y ?z)\n   :precondition (and (airplane ?x) (airport ?y) (airport ?z) (not (= ?z ?y))\n	       (at ?x ?z))\n   :effect (and (not (at ?x ?z)) (at ?x ?y)))\n )\n";
const char* pddl_timed_domain = "(define (domain logistics)\n  (:requirements :strips :typing :equality)\n  (:types airport - location\n	  truck airplane - vehicle\n	  vehicle packet - thing\n	  thing location city)\n  (:predicates (loc-at ?x - location ?y - city)\n               (time-to-drive ?x - location ?y - location ?t)\n               (time-to-fly ?x - location ?y - location ?t)\n               (size ?x - packet ?t)\n	       (at ?x - thing ?y - location)\n	       (in ?x - packet ?y - vehicle))\n\n  (:action load\n    :parameters (?x - packet ?y - vehicle)\n    :vars (?z - location)\n    :precondition (and (at ?x ?z) (at ?y ?z))\n    :effect (and (not (at ?x ?z)) (in ?x ?y))\n    :delay ?t : (size ?x ?t))\n\n  (:action unload\n    :parameters (?x - packet ?y - vehicle)\n    :vars (?z - location)\n    :precondition (and (at ?y ?z) (in ?x ?y))\n    :effect (and (at ?x ?z) (not (in ?x ?y)))\n    :delay ?t : (size ?x ?t))\n\n  (:action drive\n    :parameters (?x - truck ?y - location)\n    :vars (?z - location ?c - city)\n    :precondition (and (loc-at ?z ?c) (loc-at ?y ?c) (not (= ?z ?y))\n		       (at ?x ?z))\n    :effect (and (not (at ?x ?z)) (at ?x ?y))\n    :delay ?t : (time-to-drive ?z ?y ?t))\n\n  (:action fly\n    :parameters (?x - airplane ?y - airport)\n    :vars (?z - airport)\n    :precondition (and (not (= ?z ?y)) (at ?x ?z))\n    :effect (and (not (at ?x ?z)) (at ?x ?y))\n    :delay ?t : (time-to-fly ?z ?y ?t))\n  )\n";

int n_cities;
int n_packets;
int n_airplanes;
int inter_city;
int write_graphplan;
int write_pddl;
int pddl_type;
int pddl_timed;
int write_domain;
int write_problem_req;
int n_instances;
int n_variations;

int delay_min[MAX_VAR];
int delay_max[MAX_VAR];

int *packet_location;
int *packet_destination;
int *truck_location;
int *airplane_location;
int *packet_size;
int *time_to_drive;
int *time_to_fly;
char fname[255];

void generate_instance() {
  int k, l;

  for (k = 0; k < n_packets; k++) {
    packet_location[k] = RANDOM(2*n_cities) + 1;
  }
  for (k = 0; k < n_packets; k++) {
    packet_destination[k] = RANDOM(2*n_cities) + 1;
    if (inter_city) {
      while ((packet_destination[k] == packet_location[k]) ||
	     (packet_destination[k] == packet_location[k]+n_cities) ||
	     (packet_destination[k]+n_cities == packet_location[k])) {
	packet_destination[k] = RANDOM(2*n_cities) + 1;
      }
    }
    else {
      while (packet_destination[k] == packet_location[k]) {
	packet_destination[k] = RANDOM(2*n_cities) + 1;
      }
    }
  }
  for (k = 0; k < n_airplanes; k++) {
    airplane_location[k] = RANDOM(n_cities) + 1;
  }
  for (k = 0; k < n_airplanes; k++) {
    truck_location[k] = RANDOM(2);
  }
}

void generate_variation(int var) {
  int k, l;
  for (k = 0; k < n_packets; k++)
    packet_size[k] = RANDOM((delay_max[var] - delay_min[var]) + 1) + delay_min[var];
  for (k = 0; k < n_cities; k++) {
    for (l = k+1; l < n_cities; l++) {
      time_to_fly[k*n_cities + l] = 
	RANDOM((delay_max[var] - delay_min[var]) + 1) + delay_min[var];
      time_to_fly[l*n_cities + k] = time_to_fly[k*n_cities + l];
    }
    time_to_drive[k] = RANDOM((delay_max[var] - delay_min[var]) + 1) + delay_min[var];
  }
}

void write_instance_pddl(int n, int v, char* name) {
  FILE *f;
  int k, l;

  f = fopen(name, "w");
  if (f == NULL) {
    printf("exception: can't open \"%s\"\n", name);
    exit(0);
  }
  if (v > 0) fprintf(f, "(define (problem log_%d_v%d)\n", n, v);
  else fprintf(f, "(define (problem log_%d)\n", n);
  fprintf(f, " (:domain logistics)\n");
  if (write_problem_req) {
    if (pddl_type) fprintf(f, " (:requirements :strips :typing)\n");
    else fprintf(f, " (:requirements :strips)\n");
  }
  fprintf(f, " (:objects");
  for (k = 0; k < n_cities; k++) fprintf(f, " city%d", k+1);
  if (pddl_type) fprintf(f, " - city");
  for (k = 0; k < n_cities; k++) fprintf(f, " office%d", k+1);
  if (pddl_type) fprintf(f, " - location");
  for (k = 0; k < n_cities; k++) fprintf(f, " airport%d", k+1);
  if (pddl_type) fprintf(f, " - airport");
  for (k = 0; k < n_cities; k++) fprintf(f, " truck%d", k+1);
  if (pddl_type) fprintf(f, " - truck");
  for (k = 0; k < n_airplanes; k++) fprintf(f, " airplane%d", k+1);
  if (pddl_type) fprintf(f, " - airplane");
  for (k = 0; k < n_packets; k++) fprintf(f, " packet%d", k+1);
  if (pddl_type) fprintf(f, " - packet");
  fprintf(f, ")\n");
  fprintf(f, " (:init");
  if (!pddl_type) {
    for (k = 0; k < n_cities; k++) fprintf(f, " (city city%d)", k+1);
    for (k = 0; k < n_cities; k++) fprintf(f, " (location office%d)", k+1);
    for (k = 0; k < n_cities; k++) {
      fprintf(f, " (location airport%d)", k+1);
      fprintf(f, " (airport airport%d)", k+1);
    }
    for (k = 0; k < n_cities; k++) {
      fprintf(f, " (vehicle truck%d)", k+1);
      fprintf(f, " (truck truck%d)", k+1);
    }
    for (k = 0; k < n_airplanes; k++) {
      fprintf(f, " (vehicle airplane%d)", k+1);
      fprintf(f, " (airplane airplane%d)", k+1);
    }
    for (k = 0; k < n_packets; k++) fprintf(f, " (packet packet%d)", k+1);
  }
  for (k = 0; k < n_cities; k++) {
    fprintf(f, " (loc-at office%d city%d)", k+1, k+1);
    fprintf(f, " (loc-at airport%d city%d)", k+1, k+1);
  }
  for (k = 0; k < n_cities; k++) {
    if (truck_location[k]) fprintf(f, " (at truck%d airport%d)", k+1, k+1);
    else fprintf(f, " (at truck%d office%d)", k+1, k+1);
  }
  for (k = 0; k < n_airplanes; k++) {
    fprintf(f, " (at airplane%d airport%d)", k+1, airplane_location[k]);
  }
  for (k = 0; k < n_packets; k++) {
    if (packet_location[k] <= n_cities)
      fprintf(f, " (at packet%d office%d)", k+1, packet_location[k]);
    else
      fprintf(f, " (at packet%d airport%d)", k+1, packet_location[k] - n_cities);
  }
  if (pddl_timed) {
    for (k = 0; k < n_cities; k++) {
      for (l = k+1; l < n_cities; l++) {
	fprintf(f, " (time-to-fly airport%d airport%d %u)",
		k+1, l+1, time_to_fly[k*n_cities + l]);
	fprintf(f, " (time-to-fly airport%d airport%d %u)",
		l+1, k+1, time_to_fly[l*n_cities + k]);
      }
      fprintf(f, " (time-to-drive office%d airport%d %u)",
	      k+1, k+1, time_to_drive[k]);
      fprintf(f, " (time-to-drive airport%d office%d %u)",
	      k+1, k+1, time_to_drive[k]);
    }
    for (k = 0; k < n_packets; k++)
      fprintf(f, " (size packet%d %u)", k+1, packet_size[k]);
  }
  fprintf(f, ")\n");
  fprintf(f, " (:goal (and");
  for (k = 0; k < n_packets; k++) {
    if (packet_destination[k] <= n_cities)
      fprintf(f, " (at packet%d office%d)", k+1, packet_destination[k]);
    else
      fprintf(f, " (at packet%d airport%d)", k+1, packet_destination[k] - n_cities);
  }
  fprintf(f, "))\n)\n");
  fclose(f);
}

void write_instance_graphplan(int n, char* name) {
  FILE *f;
  int k, l;

  f = fopen(name, "w");
  if (f == NULL) {
    printf("exception: can't open \"%s\"\n", name);
    exit(0);
  }

  for (k = 0; k < n_cities; k++) {
    fprintf(f, "(city%d CITY)\n", k+1);
  }
  for (k = 0; k < n_cities; k++) {
    fprintf(f, "(office%d LOCATION)\n", k+1);
    fprintf(f, "(airport%d LOCATION)\n", k+1);
  }
  for (k = 0; k < n_cities; k++) {
    fprintf(f, "(airport%d AIRPORT)\n", k+1);
  }
  for (k = 0; k < n_cities; k++) {
    fprintf(f, "(truck%d TRUCK)\n", k+1);
  }
  for (k = 0; k < n_airplanes; k++) {
    fprintf(f, "(airplane%d AIRPLANE)\n", k+1);
  }
  for (k = 0; k < n_packets; k++) {
    fprintf(f, "(packet%d PACKET)\n", k+1);
  }
  fprintf(f, "\n");
  fprintf(f, "(preconds\n");
  for (k = 0; k < n_cities; k++) {
    fprintf(f, " (loc-at office%d city%d)\n", k+1, k+1);
    fprintf(f, " (loc-at airport%d city%d)\n", k+1, k+1);
  }
  for (k = 0; k < n_cities; k++) {
    if (truck_location[k]) fprintf(f, " (at truck%d airport%d)\n", k+1, k+1);
    else fprintf(f, " (at truck%d office%d)\n", k+1, k+1);
  }
  for (k = 0; k < n_airplanes; k++) {
    fprintf(f, " (at airplane%d airport%d)\n", k+1, airplane_location[k]);
  }
  for (k = 0; k < n_packets; k++) {
    if (packet_location[k] <= n_cities)
      fprintf(f, " (at packet%d office%d)\n", k+1, packet_location[k]);
    else
      fprintf(f, " (at packet%d airport%d)\n", k+1, packet_location[k] - n_cities);
  }
  fprintf(f, ")\n");
  fprintf(f, "(effects\n");
  for (k = 0; k < n_packets; k++) {
    if (packet_destination[k] <= n_cities)
      fprintf(f, " (at packet%d office%d)\n", k+1, packet_destination[k]);
    else
      fprintf(f, " (at packet%d airport%d)\n", k+1, packet_destination[k] - n_cities);
  }
  fprintf(f, ")\n");
  fclose(f);
}

void write_domain_files() {
  FILE *f;

  if (write_graphplan) {
    f = fopen("log.ops", "w");
    if (f == NULL) {
      printf("exception: can't open \"log.ops\"\n");
      exit(0);
    }
    fprintf(f, "%s", graphplan_domain);
    fclose(f);
  }
  if (write_pddl) {
    f = fopen("log.pddl", "w");
    if (f == NULL) {
      printf("exception: can't open \"log.pddl\"\n");
      exit(0);
    }
    if (pddl_timed) fprintf(f, "%s", pddl_timed_domain);
    else if (pddl_type) fprintf(f, "%s", pddl_typed_domain);
    else fprintf(f, "%s", pddl_untyped_domain);
    fclose(f);
  }
}

int main(int argc, char* argv[]) {
  time_t t_zero;
  unsigned long r_seed;
  int print_help, k, v;

  n_cities = 3;
  n_packets = 3;
  n_airplanes = 1;
  inter_city = 0;
  write_domain = 0;
  write_problem_req = 0;
  write_graphplan = 0;
  write_pddl = 0;
  pddl_type = 0;
  n_instances = 1;
  print_help = 0;
  r_seed = 0;

  delay_min[0] = 1;
  delay_max[0] = 2;
  n_variations = 0;

  for (k = 0; k < argc; k++) {
    if (((strcmp(argv[k],"-city") == 0) || (strcmp(argv[k],"-c") == 0)) &&
	(k < argc - 1))
      n_cities = atoi(argv[++k]);
    if (((strcmp(argv[k],"-packet") == 0) || (strcmp(argv[k],"-p") == 0)) &&
	(k < argc - 1))
      n_packets = atoi(argv[++k]);
    if (((strcmp(argv[k],"-airplane") == 0) || (strcmp(argv[k],"-a") == 0)) &&
	(k < argc - 1))
      n_airplanes = atoi(argv[++k]);
    if ((strcmp(argv[k],"-intercity") == 0) || (strcmp(argv[k],"-i") == 0))
      inter_city = 1;
    if ((strcmp(argv[k],"-graphplan") == 0) || (strcmp(argv[k],"-g") == 0))
      write_graphplan = 1;
    if ((strcmp(argv[k],"-pddl") == 0) || (strcmp(argv[k],"-l") == 0))
      write_pddl = 1;
    if ((strcmp(argv[k],"-type") == 0) || (strcmp(argv[k],"-y") == 0))
      pddl_type = 1;
    if ((strcmp(argv[k],"-time") == 0) || (strcmp(argv[k],"-t") == 0))
      pddl_timed = 1;
    if ((strcmp(argv[k],"-v") == 0) && (k < argc - 2)) {
      if (n_variations < MAX_VAR) {
	delay_min[n_variations] = atoi(argv[++k]);
	delay_max[n_variations] = atoi(argv[++k]);
	n_variations += 1;
      }
    }
    if ((strcmp(argv[k],"-domain") == 0) || (strcmp(argv[k],"-d") == 0) ||
	(strcmp(argv[k],"-ops") == 0) || (strcmp(argv[k],"-o") == 0))
      write_domain = 1;
    if ((strcmp(argv[k],"-n") == 0) && (k < argc - 1))
      n_instances = atoi(argv[++k]);
    if ((strcmp(argv[k],"-help") == 0) || (strcmp(argv[k],"-h") == 0) ||
	(strcmp(argv[k],"-?") == 0)) print_help = 1;
    if ((strcmp(argv[k],"-r") == 0) && (k < argc - 1)) {
      r_seed = atol(argv[++k]);
    }
  }

  if (n_variations == 0) n_variations = 1;

  if (print_help) {
    printf("%s [-c <cities>][-p <packets>][-a <airplanes>] [options] [-n <instances>]\n",
	   argv[0]);
    printf(" -pddl,-l      : write in PDDL format\n");
    printf(" -type,-y      : use types (PDDL only)\n");
    printf(" -time,-t      : write timed (PDDL only)\n");
    printf(" -graphplan,-g : write in Graphplan format\n");
    printf(" -domain,-d    : write domain/operator file\n");
    printf(" -ops,-o       : write domain/operator file\n");
    exit(0);
  }

  /* alloc structures */
  packet_location = malloc(n_packets*sizeof(int));
  packet_destination = malloc(n_packets*sizeof(int));
  if ((packet_location == NULL) || (packet_destination == NULL)) {
    printf("exception: can't alloc 2*%d bytes\n", n_packets*sizeof(int));
    exit(0);
  }
  airplane_location = malloc(n_airplanes*sizeof(int));
  if (airplane_location == NULL) {
    printf("exception: can't alloc %d bytes\n", n_airplanes*sizeof(int));
    exit(0);
  }
  truck_location = malloc(n_cities*sizeof(int));
  if (truck_location == NULL) {
    printf("exception: can't alloc %d bytes\n", n_cities*sizeof(int));
    exit(0);
  }

  if (pddl_timed) {
    packet_size = malloc(n_packets*sizeof(int));
    time_to_drive = malloc(n_cities*sizeof(int));
    time_to_fly = malloc(n_cities*n_cities*sizeof(int));
    if ((packet_size == NULL) || (time_to_drive == NULL) || (time_to_fly == NULL)) {
      printf("exception: can't alloc X bytes\n");
      exit(0);
    }
  }

  if (write_domain) write_domain_files();

  if (r_seed != 0) {
    SEED(r_seed);
  }
  else {
    time(&t_zero);
    SEED(t_zero);
  }

  for (k = 0; k < n_instances; k++) {
    generate_instance();
    if (write_pddl) {
      if (pddl_timed) {
	for (v = 0; v < n_variations; v++) {
	  generate_variation(v);
	  sprintf(fname, "log_g%d_v%d.pddl", k+1, v+1);
	  write_instance_pddl(k+1, v+1, fname);
	}
      }
      else {
	sprintf(fname, "log_g%d.pddl", k+1);
	write_instance_pddl(k+1, 0, fname);
      }
    }
    if (write_graphplan) {
      sprintf(fname, "log_g%d.fac", k+1);
      write_instance_graphplan(k+1, fname);
    }
  }
}
