/****************************************************** 
 * File  : main.cc   
 * Desc. : generator of IkS artificial domains (Rune M. J.) 
 *         in NADL and PDDL format 
 *
 * Usage : D0S1 
 *           -P : pddl format
 *           -N : NADL format
 *           -n <size> : domain size
 *           -k <k value> : h guided part of domain
 *           -h : print usage 
 *
 *         Produces a file called
 *         IkS_<n>.nadl or IkS_<n>.pddl 
 *
 * Author: Rune M. Jensen
 * Date  : 10/26/01
 ******************************************************/

#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>



void  printusage() {
  printf("USAGE:\n");
  printf("       -P PDDL format          \n");
  printf("       -N NADL format     (default)\n");
  printf("       -n <domain size>   (default 8)\n");
  printf("       -k <1-(n-1)>       (default 4)\n");
  printf("       -h print usage     \n");
  printf("\n");
}



int main(int argc, char **argv) {
  
  int i,j;
  int ind;

  int NADL = 1;
  int n = 8;
  int k = 4;
  char outfilename[512];


  if (argc < 2) {
    printusage();
    exit (1);
  }

  /*** Scan command line ***/
  for (ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'P': NADL = 0;
	break;
      case 'N': 
	break;
      case 'n': n = atoi(argv[++ind]);
	break;
      case 'k': k = atoi(argv[++ind]);
	break;
      case 'h': printusage();
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }


  if (NADL)
    {
      sprintf(outfilename,"IkS_n%dk%d.nadl",n,k);      
      ofstream  out(outfilename,ios::out);
      
      out << "\% File:   IkS_" << n << ".nadl\n";
      out << "\% Desc:   IkS artificial domain (RMJ) of size " << n << endl;
      out << "\%         h-guided part is " << k << endl;
      out << "\% Date:   01\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      out << "  bool Istar";
      for (i = 1; i <= n; i++) 
	out << ",I" << i << ",G" << i;
      
      out << "\n\nSYSTEM\n\n";
      
      out << " agt: sys\n";

      // A1i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n  A1" << i << endl;
	  out << "   con: G" << i << endl;	  
	  out << "   pre: Istar";
	  if (i != 1)
	    out << " /\\ G" << i-1;
	  out << endl;
	  out << "   eff: G" << i << "'\n";
	}


      // A2i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n  A2" << i << endl;
	  out << "   con: I" << i << ",Istar\n";
	  out << "   pre: true\n";
	  out << "   eff: I" << i << "' /\\ ~Istar'\n";
	}



      out << "\n\nENVIRONMENT\n";
      out << "\n\nINITIALLY\n";
      out << "  Istar";
      for (i = 1; i <= n; i++) 
	out << " /\\ ~I" << i;
      out << endl;
      for (i = 1; i <= n; i++) 
	out << " /\\ ~G" << i;
      

      out << "\n\nGOAL\n  ";
      for (i = n-k; i <= n; i++) 
	{
	  out << "G" << i;
	  if (i != n) out << " /\\ ";
	}


      
    }
  else // PDDL format
    {
      // make domain file
      sprintf(outfilename,"IkSdomain_n%dk%d.pddl",n,k);      
      ofstream  out(outfilename,ios::out);

      out << "(define (domain IkS-" << n << ")\n";
      out << "   (:requirements :strips)\n";
      
      out << "   (:predicates (Istar) ";
      
      for (i = 1; i <= n; i++) 
	{
	  out << "(G" << i << ") (I" << i << ")";
	  if (i != n) out << " ";
	}
      
      out << ")";

      // A1i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n\n   (:action A1-" << i << endl;
	  if (i == 1) 
	    out << "     :precondition (Istar)\n";
	  else
	    out << "     :precondition (and (Istar) (G" << i-1 << "))\n";
	  out << "     :effect (G" << i << "))\n ";
	}


      // A2i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n\n   (:action A2-" << i << endl;
	  out << "     :effect (and (I" << i << ") (not (Istar))))\n ";
	}

      
      out << ")";
      
     
       
      // make problem file
      sprintf(outfilename,"IkSproblem_n%dk%d.pddl",n,k);      
      ofstream  out2(outfilename,ios::out);
      
      out2 << "(define (problem IkSp-" << n << ")\n";
      out2 << "  (:domain IkS-" << n << ")\n";
      out2 << "  (:objects)\n";
      out2 << "  (:init (Istar))\n";
      out2 << "  (:goal (and ";
      for (i = n-k; i <= n; i++) 
 	{
	  out2 << "(G" << i << ")";
	  if (i != n) out2 << " ";
 	}
      out2 << ")))\n";            
    }
}
  

