/****************************************************** 
 * File  : main.cc   
 * Desc. : generator of DmS2 artificial domains (Korf) 
 *         in NADL and PDDL format 
 *
 * Usage : D0S1 
 *           -P : pddl format
 *           -N : NADL format
 *           -n <size> : domain size
 *           -h : print usage 
 *
 *         Produces a file called
 *         DmS2_<n>.nadl or DmS2_<n>.pddl 
 *
 * Author: Rune M. Jensen
 * Date  : 10/26/01
 ******************************************************/

#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>



void  printusage() {
  printf("USAGE:\n");
  printf("       -P PDDL format     \n");
  printf("       -N NADL format     (default)\n");
  printf("       -n <domain size>   (default 8)\n");
  printf("       -h print usage     \n");
  printf("\n");
}



int main(int argc, char **argv) {
  
  int i,j;
  int ind;

  int NADL = 1;
  int n = 8;
  char outfilename[512];


  if (argc < 2) {
    printusage();
    exit (1);
  }

  /*** Scan command line ***/
  for (ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'P': NADL = 0;
	break;
      case 'N': 
	break;
      case 'n': n = atoi(argv[++ind]);
	break;
      case 'h': printusage();
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }


  if (NADL)
    {
      sprintf(outfilename,"DmS2_%d.nadl",n);      
      ofstream  out(outfilename,ios::out);
      
      out << "\% File:   DmS2_" << n << ".nadl\n";
      out << "\% Desc:   DmS2 artificial domain (Korf) of size " << n << endl;
      out << "\% Date:   01\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      out << "  bool ";
      for (i = 1; i <= n; i++) 
	{
	  out << "I" << i << ",G" << i << ",P" << i;
	  if (i != n) out << ",";
	}

       
      out << "\n\nSYSTEM\n\n";
      
      out << " agt: sys\n";

      // A1i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n  A1" << i << endl;
	  out << "   con: P" << i; 
	  for (j = 1; j < i; j++) 
	    out << ",I" << j;
	  out << endl;
	  out << "   pre: I" << i << endl;
	  out << "   eff: P" << i << "'";
	  if (i != 1) out << " /\\ ";
	  for (j = 1; j < i; j++) 
	    {
	      out << "~I" << j << "'";
	      if (j != i-1) out << " /\\ ";
	    }
	  out << endl;
	  
	}


      // A2i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n  A2" << i << endl;
	  out << "   con: G" << i; 
	  for (j = 1; j < i; j++) 
	    out << ",P" << j;
	  for (j = 1; j <= n; j++) 
	    out << ",I" << j;
	  out << endl;
	  out << "   pre: P" << i << endl;
	  out << "   eff: G" << i << "'";
	  for (j = 1; j < i; j++)       
	    out << " /\\ ~P" << j << "'";
	  for (j = 1; j <= n; j++)       
	    out << " /\\ ~I" << j << "'";
	  out << endl;
	}



      out << "\n\nENVIRONMENT\n";
      out << "\n\nINITIALLY\n  ";

      for (i = 1; i <= n; i++)
	{
	  out << "I" << i;
	  if (i != n) out << " /\\ ";
	}
      out << endl;
      for (i = 1; i <= n; i++) 
	out << " /\\ ~G" << i;
      out << endl;
      for (i = 1; i <= n; i++) 
	out << " /\\ ~P" << i;
      out << endl;
      
      out << "\n\nGOAL\n  ";

      for (i = 1; i <= n; i++) 
	{
	  out << "G" << i;
	  if (i != n) out << " /\\ ";
	}      
    }
  else // PDDL format
    {
      // make domain file
      sprintf(outfilename,"DmS2domain_%d.pddl",n);      
      ofstream  out(outfilename,ios::out);

      out << "(define (domain DmS2-" << n << ")\n";
      out << "   (:requirements :strips)\n";
      
      out << "   (:predicates ";
      
      for (i = 1; i <= n; i++) 
	{
	  out << "(G" << i << ") (I" << i << ") (P" << i << ")";
	  if (i != n) out << " ";
	}
      
      out << ")";


      // A1i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n\n   (:action A1-" << i << endl;
	  out << "     :precondition (I" << i << ")\n";
	  if (i == 1)
	    out << "     :effect (P" << i << ")";
	  else
	    {
	      out << "     :effect (and (P" << i << ") ";
	      for (j = 1; j < i; j++) 
		{
		  out << "(not (I" << j << "))";
		  if (j != i-1) out << " ";
		}
	      out << ")";
	    }
	  out << ")";
	}


      // A2i
      for (i = 1; i <= n; i++) 
	{
	  out << "\n\n   (:action A2-" << i << endl;
	  out << "     :precondition (P" << i << ")\n";
	  out << "     :effect (and (G" << i << ")";
 	  for (j = 1; j < i; j++) 
	    out << " (not (P" << j << "))";
 	  for (j = 1; j <= n; j++) 
	    out << " (not (I" << j << "))";
	  out << "))";
	}
      
      out << ")";

      // make problem file
      sprintf(outfilename,"DmS2problem_%d.pddl",n);      
      ofstream  out2(outfilename,ios::out);

      out2 << "(define (problem DmS2p-" << n << ")\n";
      out2 << "  (:domain DmS2-" << n << ")\n";
      out2 << "  (:objects)\n";
      out2 << "  (:init ";
      for (i = 1; i <= n; i++) 
 	{
	  out2 << "(I" << i << ")";
	  if (i != n) out2 << " ";
 	}
      out2 << ")\n";
      out2 << "  (:goal (and ";
      for (i = 1; i <= n; i++) 
 	{
	  out2 << "(G" << i << ")";
	  if (i != n) out2 << " ";
 	}
      out2 << ")))\n";            
    }
  
}
