/****************************************************** 
 * File  : main.cc   
 * Desc. : generator of DVM artificial domains (Rune M. J.) 
 *         in NADL and PDDL format 
 *
 * Author: Rune M. Jensen
 * Date  : 10/26/01
 ******************************************************/

#include <vector>
#include <string>
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>



void  printusage() {
  printf("USAGE:\n");
  printf("       -P PDDL format               \n");
  printf("       -N NADL format                                            (default)\n");
  printf("       -x <dependency level [0,z]>                               (default 6)\n");
  printf("       -y <number of boolean position vars for each  moveable>   (default 3)\n");
  printf("       -z <number of moveables [1,2^z-1]>                        (default 6)\n");
  printf("       -h print usage     \n");
  printf("       \n");
  printf("       DVM makes an 8puzzle kind of domain except that moveables\n");
  printf("       have an action to change each of it position vars instead of moving\n");
  printf("       up and down, left and right. \n");
  printf("       \n");
  printf("       In the most dependent version a moveable can only change position if\n");
  printf("       no other moveable is at that position. Otherwise if x=i, i of the moveables move\n");
  printf("       in a single plane, while the rest move in independent planes.\n");
}






// IN
// out : out stream
// k   : moveable (which position must be different fromthe final
//                 position of movable i) 
// i   : current moveable 
// j   : varaible that is being changed by action
// y   : total number of variables
// OUT
//  string rep of equivalence expression
void posEq(ofstream &out,int k,int i,int j, int y) {
  
  out << "~(";
  // gothrough variables
  for (int a = 0; a < y; a++)
    {
      // write equivalence expression
      out << "(X" << k << "_" << a << " <=> ";
      if (a == j) out << "~";
      out << "X" << i << "_" << a << ")";
      if (a != y - 1) out << " /\\ ";
    } 
  out << ")";
}



int main(int argc, char **argv) {
  
  int i,j,k,l;
  int ind;

  int NADL = 1;

  int x = 6;
  int y = 3;
  int z = 6;

  char outfilename[512];


  /*** Scan command line ***/
  for (ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'P': NADL = 0;
	break;
      case 'N': 
	break;
      case 'x': x = atoi(argv[++ind]);
	break;
      case 'y': y = atoi(argv[++ind]);
	break;
      case 'z': z = atoi(argv[++ind]);
	break;
      case 'h': printusage();
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }


  if (NADL)
    {
      sprintf(outfilename,"D%dV%dM%d.nadl",x,y,z);      
      ofstream  out(outfilename,ios::out);
      
      out << "\% File:   D" << x << "V" << y << "M" << z <<  ".nadl\n";
      out << "\% Desc:   DVM artificial domain (RMJ) x=" << x << " y=" << y << " z=" << z << endl;
      out << "\% Date:   01\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      for (j = 0; j < y; j++) 
	{	  
	  out << "  bool ";
	  for (i = 0; i < z; i++) 
	    {
	      out << "X" << i << "_" << j;
	      if (i != z - 1) out << ", ";
	    }
	  out << endl;
	}
    
      
      out << "\n\nSYSTEM\n\n";
      
      out << " agt: sys\n\n";

      for (i = 0; i < z; i++) // i = moveable
	{	  
	  for (j = 0; j < y; j++) // one action for each action to change
		{
		  out << "  ChgX" << i << "_" << j << endl;
		  out << "    val: 1\n"; 
		  out << "    mod: X" << i << "_" << j << endl;
		  out << "    pre: true \n"; 
		  if (i < x)
		    // moveable i is dependent: no other depedent moveables at destination  
		    for (k = 0; k < x; k++)
		      if (k != i) 
			{
			  out << "    /\\ ";
			  posEq(out,k,i,j,y);
			  out << endl;
			}
		  out << "    eff: X" << i << "_" << j << "' <=> ~X" << i << "_" << j << "\n\n";
		}
	}
    
		  
		  

      out << "\n\nENVIRONMENT\n";
      out << "\n\nINITIALLY\n";
      // give moveables a start position corresponding to their number
      for (i = 0; i < z; i++)
	{
	  int v = i;
	  for (j = 0; j < y; j++)
	    {
	      if (v % 2)
		out << " X" << i << "_" << j;
	    else
	      out << "~X" << i << "_" << j;
	      if (j != y - 1) out << " /\\ ";
	      v /= 2;
	    }
	  if (i != z - 1) out << " /\\ \n";
	}
      

      out << "\n\nGOAL\n";
      // give moveables a goal position in descending order
      for (i = 0; i < z; i++)
	{
	  int v = z - i - 1;
	  for (j = 0; j < y; j++)
	    {
	      if (v % 2)
		out << " X" << i << "_" << j;
	      else
		out << "~X" << i << "_" << j;
	      if (j != y - 1) out << " /\\ ";
	      v /= 2;
	    }
	  if (i != z - 1) out << " /\\ \n";
	}
    }
  else // PDDL format
    {


      // make domain file
      sprintf(outfilename,"D%dV%dM%dDom.pddl",x,y,z);      
      ofstream  out(outfilename,ios::out);

      out << "(define (domain D" << x << "V" << y << "M" << z << ")\n";
      out << "   (:requirements :strips)\n";
      
      out << "   (:predicates (bool ?x) (moveable ?x) (imoveable ?x) (at ?i ";
      
      for (j = 0; j < y; j++) 
	{
	  out << "?x" << j;
	  if (j != y - 1) out << " ";
	}

      out << ") (free ";

      for (j = 0; j < y; j++) 
	{
	  out << "?x" << j;
	  if (j != y - 1) out << " ";
	}

      out << "))\n";


      // make dependent change actions
      // True
      // go through position variables
      for (j = 0; j < y; j++)
	{
	  out << "\n\n   (:action mktrue" << j << endl;
	  out << "     :parameters (?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) out << " ?x" << k;
	  out << ")\n";
	  
	  
	  out << "     :precondition (and (moveable ?i)";  
	  for (k = 0; k < y; k++) 
	    if (k != j) out << "\n                        (bool ?x" << k << ")";
	  out << "\n                        (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << ")\n                        (free";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << "))\n";
	  
	  
	  out << "     :effect (and (at ?i";
	  // add list
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << ")\n                  (free";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
		out << " ?x" << k;
	    else
	      out << " False";
	  out << ")";
	  
	  // del list
	  out << "\n                  (not (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << "))\n                  (not (free";
	    for (k = 0; k < y; k++) 
	      if (k != j) 
		out << " ?x" << k;
	      else
		out << " True";
	    out << "))))";	    
	}



      // False
      // go through position variables
      for (j = 0; j < y; j++)
	{
	  out << "\n\n   (:action mkfalse" << j << endl;
	  out << "     :parameters (?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) out << " ?x" << k;
	  out << ")\n";
	  
	  
	  out << "     :precondition (and (moveable ?i)";  
	  for (k = 0; k < y; k++) 
	    if (k != j) out << "\n                        (bool ?x" << k << ")";
	  out << "\n                        (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << ")\n                        (free";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << "))\n";
	  
	  
	  out << "     :effect (and (at ?i";
	  // add list
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << ")\n                  (free";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
		out << " ?x" << k;
	    else
	      out << " True";
	  out << ")";
	  
	  // del list
	  out << "\n                  (not (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << "))\n                  (not (free";
	    for (k = 0; k < y; k++) 
	      if (k != j) 
		out << " ?x" << k;
	      else
		out << " False";
	    out << "))))";	    
	}










      
      // make independent change actions
      
      // True
      // go through position variables
      for (j = 0; j < y; j++)
	{
	  out << "\n\n   (:action imktrue" << j << endl;
	  out << "     :parameters (?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) out << " ?x" << k;
	  out << ")\n";
	  
	  
	  out << "     :precondition (and (imoveable ?i)";  
	  for (k = 0; k < y; k++) 
	    if (k != j) out << "\n                        (bool ?x" << k << ")";
	  out << "\n                        (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << "))\n";
	  
	  
	  out << "     :effect (and (at ?i";
	  // add list
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << ")";
	  
	  // del list
	  out << "\n                  (not (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << "))))";	    
	}


      // False
      // go through position variables
      for (j = 0; j < y; j++)
	{
	  out << "\n\n   (:action imkfalse" << j << endl;
	  out << "     :parameters (?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) out << " ?x" << k;
	  out << ")\n";
	  
	  
	  out << "     :precondition (and (imoveable ?i)";  
	  for (k = 0; k < y; k++) 
	    if (k != j) out << "\n                        (bool ?x" << k << ")";
	  out << "\n                        (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << "))\n";
	  
	  
	  out << "     :effect (and (at ?i";
	  // add list
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " False";
	  out << ")";
	  
	  // del list
	  out << "\n                  (not (at ?i";
	  for (k = 0; k < y; k++) 
	    if (k != j) 
	      out << " ?x" << k;
	    else
	      out << " True";
	  out << "))))";	    
	}


      out << ")";
      
  
          
       
      // make problem file
      sprintf(outfilename,"D%dV%dM%dProb.pddl",x,y,z);           
      ofstream  out2(outfilename,ios::out);
      
      out2 << "(define (problem D" << x << "V" << y << "M" << z << "p)\n";
      out2 << "  (:domain D" << x << "V" << y << "M" << z << ")\n";
      out2 << "  (:objects \n    False\n    True";
      for (i = 0; i < z; i++)
	out2 << "\n    M" << i;
      out2 << ")\n";
      out2 << "  (:init";
      out2 << "\n    (bool False)"; 
      out2 << "\n    (bool True)"; 
      // make typing
      for (i = 0; i < x; i++)
	out2 << "\n    (moveable M" << i << ")";
      for (i = x; i < z; i++)
	out2 << "\n    (imoveable M" << i << ")\n";

      // give moveables a start position corresponding to their number
      for (i = 0; i < z; i++)
	{
	  int v = i;
	  out2 << "\n    (at M" << i;
	  for (j = 0; j < y; j++)
	    {
	      if (v % 2)
		out2 << " True";
	      else 
		out2 << " False";
	      v /= 2;
	    }
	  out2 << ")";
 	}
      
      for (i = x; i < int(pow(2,y)); i++)
	{
	  int v = i;
	  out2 << "\n    (free ";
	  for (j = 0; j < y; j++)
	    {
	      if (v % 2)
		out2 << " True";
	      else 
		out2 << " False";
	      v /= 2;
	    }
	  out2 << ")";
 	}
      out2 << ")\n";	 
      out2 << "  (:goal (and ";

      // give moveables a start position corresponding to their number
      for (i = 0; i < z; i++)
	{
	  int v = z - i - 1;
	  out2 << "\n    (at M" << i;
	  for (j = 0; j < y; j++)
	    {
	      if (v % 2)
		out2 << " True";
	      else 
		out2 << " False";
	       v /= 2;
	      }
	  out2 << ")";
 	}
      out2 << ")))\n";	 
    }
}


