/****************************************************** 
 * File  : main.cc   
 * Desc. : generator of DmS1 artificial domains (Korf) 
 *         in NADL and PDDL format 
 *
 * Usage : D0S1 
 *           -P : pddl format
 *           -N : NADL format
 *           -n <size> : domain size
 *           -h : print usage 
 *
 *         Produces a file called
 *         DmS1_<n>.nadl or DmS1_<n>.pddl 
 *
 * Author: Rune M. Jensen
 * Date  : 10/26/01
 ******************************************************/

#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>



void  printusage() {
  printf("USAGE:\n");
  printf("       -P PDDL format          \n");
  printf("       -N NADL format     (default)\n");
  printf("       -n <domain size>   (default 8)\n");
  printf("       -h print usage     \n");
  printf("\n");
}



int main(int argc, char **argv) {
  
  int i,j;
  int ind;

  int NADL = 1;
  int n = 8;
  char outfilename[512];


  if (argc < 2) {
    printusage();
    exit (1);
  }

  /*** Scan command line ***/
  for (ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'P': NADL = 0;
	break;
      case 'N': 
	break;
      case 'n': n = atoi(argv[++ind]);
	break;
      case 'h': printusage();
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }


  if (NADL)
    {
      sprintf(outfilename,"D0S1_%d.nadl",n);      
      ofstream  out(outfilename,ios::out);
      
      out << "\% File:   D0S1_" << n << ".nadl\n";
      out << "\% Desc:   D0S1 artificial domain (Korf) of size " << n << endl;
      out << "\% Date:   01\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      out << "  bool ";
      for (i = 1; i <= n; i++) 
	{
	  out << "I" << i << ",G" << i;
	  if (i != n) out << ",";
	}
      
      out << "\n\nSYSTEM\n\n";
      
      out << " agt: sys\n";

      for (i = 1; i <= n; i++) 
	{
	  out << "\n  A" << i << endl;
	  out << "   con: G" << i << endl;
	  out << "   pre: I" << i << endl;
	  out << "   eff: G" << i << "'";
	}

      out << "\n\nENVIRONMENT\n";
      out << "\n\nINITIALLY\n  ";

      for (i = 1; i <= n; i++) 
	out << "I" << i << " /\\ ";
      out << endl;
      for (i = 1; i <= n; i++) 
	{
	  out << "~G" << i;
	  if (i != n) out << " /\\ ";
	}



      out << "\n\nGOAL\n  ";

      for (i = 1; i <= n; i++) 
	{
	  out << "G" << i;
	  if (i != n) out << " /\\ ";
	}


      
    }
  else // PDDL format
    {
      // make domain file
      sprintf(outfilename,"D0S1domain_%d.pddl",n);      
      ofstream  out(outfilename,ios::out);

      out << "(define (domain D0S1-" << n << ")\n";
      out << "   (:requirements :strips)\n";
      
      out << "   (:predicates ";
      
      for (i = 1; i <= n; i++) 
	{
	  out << "(G" << i << ") (I" << i << ")";
	  if (i != n) out << " ";
	}
      
      out << ")";

      for (i = 1; i <= n; i++) 
	{
	  out << "\n\n   (:action A" << i << endl;
	  out << "     :precondition (I" << i << ")\n";
	  out << "     :effect (G" << i << "))\n ";
	}
      
      out << ")";
      
     
       
      // make problem file
      sprintf(outfilename,"D0S1problem_%d.pddl",n);      
      ofstream  out2(outfilename,ios::out);
      
      out2 << "(define (problem D0S1p-" << n << ")\n";
      out2 << "  (:domain D0S1-" << n << ")\n";
      out2 << "  (:objects)\n";
      out2 << "  (:init ";
      for (i = 1; i <= n; i++) 
 	{
	  out2 << "(I" << i << ")";
	  if (i != n) out2 << " ";
 	}
      out2 << ")\n";
      out2 << "  (:goal (and ";
      for (i = 1; i <= n; i++) 
 	{
	  out2 << "(G" << i << ")";
	  if (i != n) out2 << " ";
 	}
      out2 << ")))\n";            
    }
}
  

