/****************************************************** 
 * File  : main.cc   
 * Desc. : PDDL NADL translator for AIPS-2000 
 *         Logistics domain (Track1, untyped
 *         base set (does not handle additional set))
 * Author: Rune M. Jensen
 * Date  : 11/10/00
 ******************************************************/
#include <stream.h>
#include <fstream.h>
#include <string.h>
#include <stdio.h> 
#include <stdlib.h>
#include <math.h>
#include <map>
#include <set>
#include <string>
#include "set.h"
#include "common.h"
#include "pddlRcpp.h"
#include "main.h"

using namespace std;


// globals used by LEX/YACC parsing
domain *dom;
problem *prob;


void  printusage() {
  printf("USAGE:\n");
  printf("       -p <problem name>               (default prob)\n");  
  printf("       -h print usage\n");
  printf("      ");
  printf("\n");
}



int numberof(char *str) {
  char  buf[512];
  char *p;
  strcpy(buf,str);
  p = &buf[3];
  
  return atoi(p);
}


char *nameof(char *str) {
  char *buf;
  char *p;
  buf = new char[512];
  strcpy(buf,str);
  buf[3] = '\0';
  
  return buf;
}
  



int main(int argc,char **argv)  {
  extern FILE *yyin;
  extern int yylineno;


  // input parameters
  char pddlname[MAXNAMELENGTH];
  char nadlname[MAXNAMELENGTH];
  char name[MAXNAMELENGTH]  = "prob";

  int ind,i,j;


  if (argc < 2) {
    printusage();
    exit (-1);
  }


  /*** Scan command line ***/
  for (ind = 1; ind < argc; ind++) {

    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'h':   printusage(); exit(0);
	break;
      case 'p': strcpy(name,argv[++ind]);
	break;
      default : printf("Unknown switch '%c'\n", argv[ind][1]);
	break;
      }
    }
  }


  sprintf(pddlname,"%s.pddl",name);
  sprintf(nadlname,"%s.nadl",name);
  

  /*** allocate domain and problem structures **/
  dom =  new domain;
  prob = new problem;

  /*** parse PDDL problem ***/
  if ( (yyin = fopen(pddlname,"r")) == NULL)
    {
      cout << "Cannot open \"" << pddlname << "\"\n"; 
      exit(-1);
    }

  cout << "parsing : " << pddlname << "\n";
  yylineno = 0;
  if (yyparse()) exit(-1); 

  // debug
  // cout << "PDDL PROBLEM :\n";
  // printproblem(prob);
   

  // Make numeric id of trucks, packages, cities, airplanes
  // in this domain there are two subcitites of each city
  // one of them is an airport.. 

  // 
  map<string,int> packagemap,truckmap,airplanemap;
  set<string> packages, trucks, airplanes;


  int packagenum = 0;
  int trucknum = 0;
  int airplanenum = 0;
  
  // go through init and instantiate maps;
  for (predlst* pl = prob->init; pl != NULL; pl = pl->next)
    if (!strcmp(pl->name,"package")) {
      packagemap[pl->args->name] = packagenum++;
      packages.insert(pl->args->name);
    }
    else if (!strcmp(pl->name,"truck")) {
      truckmap[pl->args->name] = trucknum++;
      trucks.insert(pl->args->name);
    }
    else if (!strcmp(pl->name,"airplane")) {
      airplanemap[pl->args->name] = airplanenum++;
      airplanes.insert(pl->args->name);
    }


  // debug
  // cout << packagenum << " " << trucknum << " " << airplanenum << "\n";
  // exit(1);


  ofstream out(nadlname,ios::out);
  out << "\% File : " << nadlname << endl;
  out << "\% Desc : Translation of AIPS-00 Track 1, untyped, logistics problem\n";
  out << "\%        " << pddlname << endl;
  out << "\% Date : 05/02\n";
  out << "\% Auth : Rune M. Jensen CS, CMU\n\n\n";

  out << "VARIABLES\n";

  out << "   nat(1) ";
  for (i = 0; i < trucknum; i++) {
    out << "postruck" << i+1;
    if (i < trucknum-1) out << ", ";
  }

  out << "\n   nat(" << int(ceil(log(trucknum)/log(2))) << ") ";
  for (i = 0; i < airplanenum; i++) {
    out << "posairplane" << i+1;
    if (i < airplanenum-1) out << ", ";
  }

  out << "\n   nat(" << int(ceil( log(3*trucknum + airplanenum) /log(2))) << ") ";
  for (i = 0; i < packagenum; i++) {
    out << "pospackage" << i+1;
    if (i < packagenum-1) out << ", ";
  }

  // package positions are ordered 
  //
  // inairplane
  // ...
  // intruck1
  // ...
  // city 1 subcity1  
  //        airport1  
 

  out << "\n\nSYSTEM\n\n";
  out << "  agt: sys\n\n";
    

 for (i=0; i<packagenum; i++)   // package number
   for (j=0; j<trucknum; j++)    // trucknumber
     {
       out << "  load_Package" << i+1 << "intoTruck" << j+1 << endl;
       out << "    val: 1\n";
       out << "    mod: " << "pospackage" << i+1 << endl;
       out << "    pre: " << "pospackage" << i + 1 << " = " << j*2 << " + postruck" << j+1 << endl;
       out << "    eff: " << "pospackage" << i + 1 << "' = " << 2*trucknum + j << "\n\n";
     }


 for (i=0; i<packagenum; i++)   // package number
   for (j=0; j<airplanenum; j++)// airplane number
     {
       out << "  load_Package" << i+1 << "intoAirplane" << j+1 << endl;
       out << "    val: 1\n";
       out << "    mod: " << "pospackage" << i+1 << endl;
       out << "    pre: " << "pospackage" << i+1 << " = ";
       out << "posairplane" << j+1 << " + posairplane" << j+1 << endl; 
       out << "    eff: " << "pospackage" << i + 1 << "' = " << 3*trucknum + j << "\n\n";
     }


 for (i=0; i<packagenum; i++)   // package number
   for (j=0; j<trucknum; j++)    // truck number
     {
       out << "  unload_Package" << i + 1 << "fromTruck" << j + 1 << endl;
       out << "    val: 1\n";
       out << "    mod: " << "pospackage" << i + 1 << endl;
       out << "    pre: " << "pospackage" << i + 1 << " = " << j + 2*trucknum << endl;
       out << "    eff: " << "pospackage" << i + 1 << "' = " << "postruck" << j+1;
       out << " + " << 2*j << "\n\n";
     }
 
 for (i=0; i<packagenum; i++)   // package number
   for (j=0; j<airplanenum; j++)// airplane number
     {
       out << "  unload_Package" << i + 1 << "fromAirplane" << j + 1 << "\n";
       out << "    val: 1\n";
       out << "    mod: " << "pospackage" << i + 1 << endl;
       out << "    pre: " << "pospackage" << i + 1 << " = " << 3*trucknum + j << "\n";
       out << "    eff: " << "pospackage" << i + 1 << "' = ";
       out << "posairplane" << j+1 << " + posairplane" << j+1 << " \n\n"; 
     }


 for (i=0; i<trucknum; i++)   // truck number
   for (j=0; j<2; j++) // pos (0 - 1)
     {
       out << "  drive_Truck" << i + 1 << "toPos" << j << "\n";
       out << "    val: 1\n";
       out << "    mod: " << "postruck" << i + 1 << "\n";
       out << "    pre: true\n";
       out << "    eff: " << "postruck" << i + 1 << "' = " << j << "\n\n";
     }

 for (i=0; i<airplanenum; i++)   // plane number
   for (j=0; j<trucknum; j++) // pos 
     {
       out << "  fly_Airplane" << i + 1 << "_toPos" << j << "\n";
       out << "    val: 1\n";
       out << "    mod: " << "posairplane" << i + 1 << "\n";
       out << "    pre: true\n";
       out << "    eff: " << "posairplane" << i + 1 << "' = " << j << "\n\n";
     }


 out << "ENVIRONMENT\n \nINITIALLY\n";

 int flag;
 flag  = 0;
 
  for (predlst* pl = prob->init; pl != NULL; pl = pl->next)
    {
      if (!strcmp(pl->name,"at")) {
	if (flag) out << " /\\ "; 
	flag = 1;
	if (strmember(packages,pl->args->name)) {
	  if (!strcmp(nameof(pl->args->next->name),"pos"))
	    out << "pospackage" << packagemap[pl->args->name]+1 << " = " 
	       << 2*(numberof(pl->args->next->name)-1) + 1;
	  else 
	    out << "pospackage" << packagemap[pl->args->name]+1 << " = " 
		 << 2*(numberof(pl->args->next->name)-1);
	}
	else if (strmember(airplanes,pl->args->name)) {
	    out << "posairplane" << airplanemap[pl->args->name]+1 << " = " 
		 << numberof(pl->args->next->name)-1; 
	}
	else if (strmember(trucks,pl->args->name)) {
	  if (!strcmp(nameof(pl->args->next->name),"pos"))
	    out << "postruck" << truckmap[pl->args->name]+1 << " = 1"; 
	  else 
	    out << "postruck" << truckmap[pl->args->name]+1 << " = 0";
	}
      }
    }

  out << "\n\nGOAL\n";

 flag  = 0;
  
  for (predlst* pl = prob->goal; pl != NULL; pl = pl->next)
    {
      if (!strcmp(pl->name,"at")) {
	if (flag) out << " /\\ "; 
	flag = 1;
	if (strmember(packages,pl->args->name)) {
	  if (!strcmp(nameof(pl->args->next->name),"pos"))
	    out << "pospackage" << packagemap[pl->args->name]+1 << " = " 
	       << 2*(numberof(pl->args->next->name)-1) + 1;
	  else 
	    out << "pospackage" << packagemap[pl->args->name]+1 << " = " 
		 << 2*(numberof(pl->args->next->name)-1);
	}
	else if (strmember(airplanes,pl->args->name)) {
	    out << "posairplane" << airplanemap[pl->args->name]+1 << " = " 
		 << numberof(pl->args->next->name)-1; 
	}
	else if (strmember(trucks,pl->args->name)) {
	  if (!strcmp(nameof(pl->args->next->name),"pos"))
	    out << "postruck" << truckmap[pl->args->name]+1 << " = 1"; 
	  else 
	    out << "postruck" << truckmap[pl->args->name]+1 << " = 0";
	}
      }
    }
      
       
  return 0;
}



    
