/////////////////////////////////////////////////////////////////////
// File  : main.cc   
// Desc. : extNADL Generator of an untimed version of 
//         case study 5 of the VHS Esprit project.
//         It models a steel plant of SIDMAR in ghent Belgium.
//
//          Best description of this domain is given in
//          "Scheduling a Steel Plant with Timed Automata", 
//          Ansgar Fehnker.   
//          See model in research journal feb 2002 ->
//
// Author: Rune M. Jensen
// Date  : 10/14/02
/////////////////////////////////////////////////////////////////////


#include <vector>
#include <set>
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


using namespace std; 


void  printusage() {
  printf("USAGE: SIDMAR [-habfo]                                  \n");
  printf("       -h print usage                                   \n");
  printf("       -a NUM (default 2)                               \n");
  printf("          number of ladles of recipe A                  \n");
  printf("       -b NUM (default 2)                               \n");
  printf("          number of ladles of recipe B                  \n");
  printf("       -f <formatNum> (default 0)                       \n");
  printf("          0 : multi agent + error on M1,2,4,5 and       \n");
  printf("              converter 1 and 2                         \n");
  printf("          1 : single agent                              \n");
  printf("          2 : single agent + error on M1,2,4,5 and      \n");
  printf("              converter 1 and 2                         \n");
  printf("          number of ladles of recipe B                  \n");
  printf("       -o FILE (default SIDMAR)                         \n");
  printf("          it will be prefixed with a<a>b<b>             \n");
} 




///////////////////////////////////////////////////
//
// Main function
//
///////////////////////////////////////////////////
int main(int argc, char **argv) {
  
  // heuristic type
  int a = 2;
  int b = 2;
  int format = 0;

  char outName[512] = "SIDMAR";
  char finalOutName[512];


  /*** Scan command line ***/
  for (int ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'a': 
	a = atoi(argv[++ind]);
	break;
      case 'b': 
	b = atoi(argv[++ind]);
	break;
      case 'o': 
	strcpy(outName,argv[++ind]);
	break;
      case 'f': 
	format = atoi(argv[++ind]);
	break;
      case 'h': 
	printusage();
	exit(1);
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }



  
  sprintf(finalOutName,"a%db%d%s.nadl",a,b,outName);      
  ofstream  out(finalOutName,ios::out);

  if (format == 0)
    {
      //////////////////////////////////////////
      //
      // Format 0 Multi-agent, no error
      // 
      //////////////////////////////////////////




      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   SIDMAR problem in format " << format << " with " << a << endl;
      out << "\%         recipe A ladles and " << b << " recipe B ladles\n"; 
      out << "\% Date:   02\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      // starting with crane and occupation variables
      out << "\% starting with crane and occupation variables\n";
      out << "  bool   castToken\n";
      out << "  nat(3) posCrane1, posCrane2\n";
      out << "  bool   freeCP0,freeCP1,freeCP2,freeCP3,freeCP4\n";
      out << "  bool   freeLP1,freeLP2,freeLP3,freeLP4,freeLP5,freeLP6,freeLP6\n";
      out << "  bool   freeLP7,freeLP8,freeLP9,freeLP10,freeLP11,freeLP12,freeLP15,freeLP16\n";
      out << "  nat(3) loadLP13\n";
      out << "  bool   okM1,okM2,okM4,okM5,okConv1,okConv2\n";
      
      // then ladles of recipe A
      out << "\% then ladles of recipe A\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  nat(5) posLadleA" << i << endl;
	  out << "  nat(3) stateLadleA" << i << endl;
	}
      
      // finally ladles of recipe B
      out << "\% finally ladles of recipe B\n";
      for (int i = 1; i <= b; i++)
	{
	  out << "  nat(5) posLadleB" << i << endl;
	  out << "  nat(3) stateLadleB" << i << endl;
	}
      
      
      
      out << "\n\nSYSTEM\n\n";
      
      ////////////////////////////////////
      // converter 1 process
      ////////////////////////////////////
      out << " process:  converter1\n\n";  
  
      for (int i = 1; i <= a; i++)
	{
	  out << "  convertLadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP1" << ",okConv1\n";
	  out << "    pre: posLadleA" << i << " = 0 /\\ stateLadleA" << i << " = 0 /\\ freeLP1 /\\ okConv1\n";
	  out << "    eff: posLadleA" << i << "' = 1 /\\ stateLadleA" << i << "' = 1 /\\ ~freeLP1' /\\ ~okConv1'\n";
	  out << "    err: posLadleA" << i << "' = 0 /\\ stateLadleA" << i << "' = 0 /\\ freeLP1' /\\ ~okConv1'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  convertLadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP1" << ",okConv1\n";
	  out << "    pre: posLadleB" << i << " = 0 /\\ stateLadleB" << i << " = 0 /\\ freeLP1 /\\ okConv1\n";
	  out << "    eff: posLadleB" << i << "' = 1 /\\ stateLadleB" << i << "' = 1 /\\ ~freeLP1' /\\ okConv1'\n";
	  out << "    err: posLadleB" << i << "' = 0 /\\ stateLadleB" << i << "' = 0 /\\ freeLP1' /\\ ~okConv1'\n\n";
	}



      ////////////////////////////////////
      // converter 2 process
      ////////////////////////////////////
      out << " process:  converter2\n\n";  
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  convertLadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP7" << ",okConv2\n";
	  out << "    pre: posLadleA" << i << " = 0 /\\ stateLadleA" << i << " = 0 /\\ freeLP7 /\\ okConv2\n";
	  out << "    eff: posLadleA" << i << "' = 7 /\\ stateLadleA" << i << "' = 1 /\\ ~freeLP7' /\\ okConv2'\n";
	  out << "    err: posLadleA" << i << "' = 0 /\\ stateLadleA" << i << "' = 0 /\\ freeLP7'  /\\ ~okConv2'\n\n";
	}
    
      for (int i = 1; i <= b; i++)
	{
	  out << "  convertLadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP7" << ",okConv2\n";
	  out << "    pre: posLadleB" << i << " = 0 /\\ stateLadleB" << i << " = 0 /\\ freeLP7 /\\ okConv2\n";
	  out << "    eff: posLadleB" << i << "' = 7 /\\ stateLadleB" << i << "' = 1 /\\ ~freeLP7' /\\ okConv2'\n";
	  out << "    err: posLadleB" << i << "' = 0 /\\ stateLadleB" << i << "' = 0 /\\ freeLP7' /\\ ~okConv2'\n\n";
	}


      ////////////////////////////////////
      // machine 1 process
      ////////////////////////////////////
      out << " process:  machine1\n\n";  
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat1LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM1\n";
	  out << "    pre: posLadleA" << i << " = 2 /\\ stateLadleA" << i << " = 1 /\\ okM1\n";
	  out << "    eff: posLadleA" << i << "' = 2 /\\ stateLadleA" << i << "' = 2 /\\ okM1'\n";
	  out << "    err: posLadleA" << i << "' = 2 /\\ stateLadleA" << i << "' = 1 /\\ ~okM1'\n\n";
	}

      for (int i = 1; i <= b; i++)
	{
	  out << "  treat1LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM1\n";
	  out << "    pre: posLadleB" << i << " = 2 /\\ stateLadleB" << i << " = 1 /\\ okM1\n";
	  out << "    eff: posLadleB" << i << "' = 2 /\\ stateLadleB" << i << "' = 2 /\\ okM1'\n";
	  out << "    err: posLadleB" << i << "' = 2 /\\ stateLadleB" << i << "' = 1 /\\ ~okM1'\n\n";
	}
      

      ////////////////////////////////////
      // machine 4 process
      ////////////////////////////////////
      out << " process:  machine4\n\n";  
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat1LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM4\n";
	  out << "    pre: posLadleA" << i << " = 8 /\\ stateLadleA" << i << " = 1 /\\ okM4\n";
	  out << "    eff: posLadleA" << i << "' = 8 /\\ stateLadleA" << i << "' = 2 /\\ okM4'\n";
	  out << "    err: posLadleA" << i << "' = 8 /\\ stateLadleA" << i << "' = 1 /\\ ~okM4'\n\n";
	}

      for (int i = 1; i <= b; i++)
	{
	  out << "  treat1LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM4\n";
	  out << "    pre: posLadleB" << i << " = 8 /\\ stateLadleB" << i << " = 1 /\\ okM4\n";
	  out << "    eff: posLadleB" << i << "' = 8 /\\ stateLadleB" << i << "' = 2 /\\ okM4'\n";
	  out << "    err: posLadleB" << i << "' = 8 /\\ stateLadleB" << i << "' = 1 /\\ ~okM4'\n\n";
	}
      
  


      ////////////////////////////////////
      // machine 2 process
      ////////////////////////////////////
      out << " process:  machine2\n\n";  
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat2LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM2\n";
	  out << "    pre: posLadleA" << i << " = 4 /\\ stateLadleA" << i << " = 2 /\\ okM2\n";
	  out << "    eff: posLadleA" << i << "' = 4 /\\ stateLadleA" << i << "' = 3 /\\ okM2'\n";
	  out << "    err: posLadleA" << i << "' = 4 /\\ stateLadleA" << i << "' = 2 /\\ ~okM2'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat2LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM2\n";
	  out << "    pre: posLadleB" << i << " = 4 /\\ ( stateLadleB" << i << " = 2 \\/ stateLadleB" << i << " = 4) /\\ okM2\n";
	  out << "    eff: posLadleB" << i << "' = 4 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1 /\\ okM2'\n";
	  out << "    err: posLadleB" << i << "' = 4 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1 /\\ ~okM2'\n\n";
	}
      
      
      
      ////////////////////////////////////
      // machine 5 process
      ////////////////////////////////////
      out << " process:  machine5\n\n";  
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat2LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM5\n";
	  out << "    pre: posLadleA" << i << " = 10 /\\ stateLadleA" << i << " = 2 /\\ okM5\n";
	  out << "    eff: posLadleA" << i << "' = 10 /\\ stateLadleA" << i << "' = 3 /\\ okM5'\n";
	  out << "    err: posLadleA" << i << "' = 10 /\\ stateLadleA" << i << "' = 2 /\\ ~okM5'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat2LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM5\n";
	  out << "    pre: posLadleB" << i << " = 10 /\\ ( stateLadleB" << i << " = 2 \\/ stateLadleB" << i << " = 4) /\\ okM5\n";
	  out << "    eff: posLadleB" << i << "' = 10 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1 /\\ okM5'\n";
	  out << "    err: posLadleB" << i << "' = 10 /\\ stateLadleB" << i << "' = stateLadleB" << i << " /\\ ~okM5'\n\n";
	}


      
      ////////////////////////////////////
      // machine 3 process
      ////////////////////////////////////
      out << " process:  machine3\n\n";  
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat3LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 6 /\\ stateLadleB" << i << " = 3\n";
	  out << "    eff: posLadleB" << i << "' = 6 /\\ stateLadleB" << i << "' = 4\n\n";
	}
      



      ////////////////////////////////////
      // caster process
      ////////////////////////////////////
      
      out << " process:  caster\n\n";  
  
      out << "\% begin continuous casting (can only be done once in a batch)\n\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  startLadleA" << i << endl; 
	  out << "    mod: posLadleA" << i << ",castToken,freeLP15,freeLP16\n";
	  out << "    pre: posLadleA" << i << " = 15 /\\ stateLadleA" << i << " = 3 /\\ castToken\n";
	  out << "    eff: posLadleA" << i << "' = 16 /\\ freeLP15' /\\ ~freeLP16' /\\  ~castToken'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  startLadleB" << i << endl; 
	  out << "    mod: posLadleB" << i << ",castToken,freeLP15,freeLP16\n";
	  out << "    pre: posLadleB" << i << " = 15 /\\ stateLadleB" << i << " = 5 /\\ castToken\n";
	  out << "    eff: posLadleB" << i << "' = 16 /\\ freeLP15' /\\ ~freeLP16' /\\  ~castToken'\n\n";
	}
      

      out << "\% swap ladles in continuous casting\n\n";
      for (int i = 1; i <= a; i++)
	for (int j = 1; j <= a; j++)
	  if (i != j)
	    {
	      out << "  swapLadleA" << i << "A" << j << endl; 
	      out << "    mod: posLadleA" << i << ",posLadleA" << j << ",stateLadleA" << j <<endl;
	      out << "    pre: stateLadleA" << i << " = 3 /\\ posLadleA" << i << " = 15 /\\  posLadleA" << j << " = 16\n";
	      out << "    eff: posLadleA" << i << "' = 16 /\\  posLadleA" << j << "' = 15 /\\ stateLadleA" << j << "' = 4\n\n";
	    }
      
      for (int i = 1; i <= a; i++)
	for (int j = 1; j <= b; j++)
	  {
	    out << "  swapLadleA" << i << "B" << j << endl; 
	    out << "    mod: posLadleA" << i << ",posLadleB" << j << ",stateLadleB" << j <<endl;
	    out << "    pre: stateLadleA" << i << " = 3 /\\ posLadleA" << i << " = 15 /\\  posLadleB" << j << " = 16\n";
	    out << "    eff: posLadleA" << i << "' = 16 /\\  posLadleB" << j << "' = 15 /\\ stateLadleB" << j << "' = 6\n\n";
	  }
      


      for (int i = 1; i <= b; i++)
	for (int j = 1; j <= b; j++)
	  if (i != j)
	    {
	      out << "  swapLadleB" << i << "B" << j << endl; 
	      out << "    mod: posLadleB" << i << ",posLadleB" << j << ",stateLadleB" << j <<endl;
	      out << "    pre: stateLadleB" << i << " = 5 /\\ posLadleB" << i << " = 15 /\\  posLadleB" << j << " = 16\n";
	      out << "    eff: posLadleB" << i << "' = 16 /\\  posLadleB" << j << "' = 15 /\\ stateLadleB" << j << "' = 6\n\n";
	    }
      

      for (int i = 1; i <= b; i++)
	for (int j = 1; j <= a; j++)
	  {
	    out << "  swapLadleB" << i << "A" << j << endl; 
	    out << "    mod: posLadleB" << i << ",posLadleA" << j << ",stateLadleA" << j <<endl;
	    out << "    pre: stateLadleB" << i << " = 5 /\\ posLadleB" << i << " = 15 /\\  posLadleA" << j << " = 16\n";
	    out << "    eff: posLadleB" << i << "' = 16 /\\  posLadleA" << j << "' = 15 /\\ stateLadleA" << j << "' = 4\n\n";
	  }
      


      out << "\% end continuous casting \n\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  stopLadleA" << i << endl; 
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP15,freeLP16\n";
	  out << "    pre: posLadleA" << i << " = 16  /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ stateLadleA" << i << "' = 4 /\\ freeLP16' /\\ ~freeLP15'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  stopLadleB" << i << endl; 
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP15,freeLP16\n";
	  out << "    pre: posLadleB" << i << " = 16  /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ stateLadleB" << i << "' = 6 /\\ freeLP16' /\\ ~freeLP15'\n\n";
	}
      
      

      ////////////////////////////////////
      // track1 move processes
      ////////////////////////////////////

      for (int i = 1; i <= a; i++)
	{
	  out << " process:  track1LadleA" << i << "\n\n";  
	  
	  for (int pos = 1; pos <= 5; pos++)
	    {
	      out << "  rightAt" << pos << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos + 1 << " /\\ freeLP" 
	      << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
      
	  for (int pos = 2; pos <= 6; pos++)
	    {
	      out << "  leftAt" << pos << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}

      for (int i = 1; i <= b; i++)
	{
	  out << " process:  track1LadleB" << i << "\n\n";  
	  
	  for (int pos = 1; pos <= 5; pos++)
	    {
	      out << "  rightAt" << pos << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 2; pos <= 6; pos++)
	    {
	      out << "  leftAt" << pos << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}
      

      
      ////////////////////////////////////
      // track2 move processes
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << " process:  track2LadleA" << i << "\n\n";  
	  
	  for (int pos = 7; pos <= 9; pos++)
	    {
	      out << "  rightAt" << pos << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 8; pos <= 10; pos++)
	    {
	      out << "  leftAt" << pos << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << " process:  track2LadleB" << i << "\n\n";  
	  
	  for (int pos = 7; pos <= 9; pos++)
	    {
	      out << "  rightAt" << pos << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 8; pos <= 10; pos++)
	    {
	      out << "  leftAt" << pos << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}





      ////////////////////////////////////
      // crane1 
      ////////////////////////////////////
      
      out << " process:  crane1\n\n";  
      
      for (int pos = 4; pos >= 1; pos--)
	{
	  out << "  upAt" << pos << endl;
	  out << "    mod: posCrane1, freeCP" << pos << ", freeCP" << pos - 1 << endl;
	  out << "    pre: posCrane1 = " << pos << " /\\ freeCP" << pos - 1 << endl; 
	  out << "    eff: posCrane1' = " << pos - 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos - 1 << "'\n\n";
	}
      
      for (int pos = 3; pos >= 0; pos--)
	{
	  out << "  downAt" << pos << endl;
	  out << "    mod: posCrane1, freeCP" << pos << ", freeCP" << pos + 1 << endl;
	  out << "    pre: posCrane1 = " << pos << " /\\ freeCP" << pos + 1 << endl; 
	  out << "    eff: posCrane1' = " << pos + 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos + 1 << "'\n\n";
	}
      
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  loadLadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleA" << i << " = 3 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP3' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadLadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleA" << i << " = 9 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP9' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadLadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleA" << i << " = 13 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP11'\n\n";
	  
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadLadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleA" << i << " = 15 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP15' /\\ ~freeLP11'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  loadLadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleB" << i << " = 3 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP3' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadLadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleB" << i << " = 9 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP9' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadLadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleB" << i << " = 13 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP11'\n\n";
      
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadLadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleB" << i << " = 15 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP15' /\\ ~freeLP11'\n\n";
	}
      


      for (int i = 1; i <= a; i++)
	{
	  out << "  unloadLadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleA" << i << " = 11 /\\ freeLP3\n";
	  out << "    eff: posLadleA" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP11'\n\n";
	  
	  out << "  unloadLadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleA" << i << " = 11 /\\ freeLP9\n";
	  out << "    eff: posLadleA" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP11'\n\n";

	  out << "  unloadLadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleA" << i << " = 11 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleA" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP11'\n\n";

	  out << "  unloadLadleA" << i << "at3\n";
	  out << "    mod: posLadleA" << i << ",freeLP11\n";
	  out << "    pre: posCrane1 = 3 /\\ posLadleA" << i << " = 11\n";
	  out << "    eff: posLadleA" << i << "' = 14 /\\ freeLP11'\n\n";
      
	  out << "  unloadLadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleA" << i << " = 11 /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP11'\n\n";
	}



      for (int i = 1; i <= b; i++)
	{
	  out << "  unloadLadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleB" << i << " = 11 /\\ freeLP3\n";
	  out << "    eff: posLadleB" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP11'\n\n";

	  out << "  unloadLadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleB" << i << " = 11 /\\ freeLP9\n";
	  out << "    eff: posLadleB" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP11'\n\n";


	  out << "  unloadLadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleB" << i << " = 11 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleB" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP11'\n\n";

	  out << "  unloadLadleB" << i << "at3\n";
	  out << "    mod: posLadleB" << i << ",freeLP11\n";
	  out << "    pre: posCrane1 = 3 /\\ posLadleB" << i << " = 11\n";
	  out << "    eff: posLadleB" << i << "' = 14 /\\ freeLP11'\n\n";
      
	  out << "  unloadLadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleB" << i << " = 11 /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP11'\n\n";
	}



      ////////////////////////////////////
      // crane2 
      ////////////////////////////////////
  
      out << " process:  crane2\n\n";  

      for (int pos = 4; pos >= 1; pos--)
	{
	  out << "  upAt" << pos << endl;
	  out << "    mod: posCrane2, freeCP" << pos << ", freeCP" << pos - 1 << endl;
	  out << "    pre: posCrane2 = " << pos << " /\\ freeCP" << pos - 1 << endl; 
	  out << "    eff: posCrane2' = " << pos - 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos - 1 << "'\n\n";
	}
      
      for (int pos = 3; pos >= 0; pos--)
	{
	  out << "  downAt" << pos << endl;
	  out << "    mod: posCrane2, freeCP" << pos << ", freeCP" << pos + 1 << endl;
	  out << "    pre: posCrane2 = " << pos << " /\\ freeCP" << pos + 1 << endl; 
	  out << "    eff: posCrane2' = " << pos + 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos + 1 << "'\n\n";
	}


      for (int i = 1; i <= a; i++)
	{
	  out << "  loadLadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleA" << i << " = 3 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP3' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadLadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleA" << i << " = 9 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP9' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadLadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleA" << i << " = 13 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP12'\n\n";
	  
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadLadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleA" << i << " = 15 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP15' /\\ ~freeLP12'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  loadLadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleB" << i << " = 3 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP3' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadLadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleB" << i << " = 9 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP9' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadLadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleB" << i << " = 13 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP12'\n\n";
      
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadLadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleB" << i << " = 15 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP15' /\\ ~freeLP12'\n\n";
	}



      for (int i = 1; i <= a; i++)
	{
	  out << "  unloadLadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleA" << i << " = 12 /\\ freeLP3\n";
	  out << "    eff: posLadleA" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP12'\n\n";
	  
	  out << "  unloadLadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleA" << i << " = 12 /\\ freeLP9\n";
	  out << "    eff: posLadleA" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP12'\n\n";
	  
	  out << "  unloadLadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleA" << i << " = 12 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleA" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP12'\n\n";
	  
	  out << "  unloadLadleA" << i << "at3\n";
	  out << "    mod: posLadleA" << i << ",freeLP12\n";
	  out << "    pre: posCrane2 = 3 /\\ posLadleA" << i << " = 12\n";
	  out << "    eff: posLadleA" << i << "' = 14 /\\ freeLP12'\n\n";
	  
	  out << "  unloadLadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleA" << i << " = 12 /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP12'\n\n";
	}
      


      for (int i = 1; i <= b; i++)
	{
	  out << "  unloadLadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleB" << i << " = 12 /\\ freeLP3\n";
	  out << "    eff: posLadleB" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP12'\n\n";
	  
	  out << "  unloadLadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleB" << i << " = 12 /\\ freeLP9\n";
	  out << "    eff: posLadleB" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP12'\n\n";


	  out << "  unloadLadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleB" << i << " = 12 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleB" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP12'\n\n";

	  out << "  unloadLadleB" << i << "at3\n";
	  out << "    mod: posLadleB" << i << ",freeLP12\n";
	  out << "    pre: posCrane2 = 3 /\\ posLadleB" << i << " = 12\n";
	  out << "    eff: posLadleB" << i << "' = 14 /\\ freeLP12'\n\n";
	  
	  out << "  unloadLadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleB" << i << " = 12 /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP12'\n\n";
	}

  
      out << "\n\nINITIALLY\n";
      out << "  okM1 /\\ okM2 /\\ okM4 /\\ okM5 /\\ okConv1 /\\ okConv2 /\\\n";
      out << "  ~freeCP0 /\\ freeCP1 /\\ ~freeCP2 /\\ freeCP3 /\\ freeCP4 /\\\n";
      out << "  freeLP1 /\\ freeLP2 /\\ freeLP3 /\\ freeLP4 /\\ freeLP5 /\\ freeLP6 /\\\n";
      out << "  freeLP6 /\\ freeLP7 /\\ freeLP8 /\\ freeLP9 /\\ freeLP10 /\\ freeLP11 /\\\n";
      out << "  freeLP12 /\\ freeLP15 /\\ freeLP16 /\\\n";  
      out << "  loadLP13 = 0 /\\\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  posLadleA" << i <<  " = 0 /\\\n";
	  out << "  stateLadleA" << i << " = 0 /\\\n";
	}	
      for (int i = 1; i <= b; i++)
	{
	  out << "  posLadleB" << i <<  " = 0 /\\\n";
	  out << "  stateLadleB" << i << " = 0 /\\\n";
	}
      out << "  castToken /\\ posCrane1 = 0 /\\ posCrane2 =  2\n";
      
      
      out << "\n\nGOAL\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  posLadleA" << i <<  " = 14 /\\\n";
	  out << "  stateLadleA" << i << " = 4 /\\\n";
	}	
      for (int i = 1; i <= b-1; i++)
	{
	  out << "  posLadleB" << i <<  " = 14 /\\\n";
	  out << "  stateLadleB" << i << " = 6 /\\\n";
	}
      out << "  posLadleB" << b <<  " = 14 /\\\n";
      out << "  stateLadleB" << b << " = 6\n";
    }
  else if (format == 1)
    {












































      ///////////////////////////////////////////////////////////
      //
      // Format 1: single-agent 
      //
      ///////////////////////////////////////////////////////////



      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   SIDMAR problem in format " << format << " with " << a << endl;
      out << "\%         recipe A ladles and " << b << " recipe B ladles\n"; 
      out << "\% Date:   02\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      // starting with crane and occupation variables
      out << "\% starting with crane and occupation variables\n";
      out << "  bool   castToken\n";
      out << "  nat(3) posCrane1, posCrane2\n";
      out << "  bool   freeCP0,freeCP1,freeCP2,freeCP3,freeCP4\n";
      out << "  bool   freeLP1,freeLP2,freeLP3,freeLP4,freeLP5,freeLP6,freeLP6\n";
      out << "  bool   freeLP7,freeLP8,freeLP9,freeLP10,freeLP11,freeLP12,freeLP15,freeLP16\n";
      out << "  nat(3) loadLP13\n";
      
      // then ladles of recipe A
      out << "\% then ladles of recipe A\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  nat(5) posLadleA" << i << endl;
	  out << "  nat(3) stateLadleA" << i << endl;
	}
      
      // finally ladles of recipe B
      out << "\% finally ladles of recipe B\n";
      for (int i = 1; i <= b; i++)
	{
	  out << "  nat(5) posLadleB" << i << endl;
	  out << "  nat(3) stateLadleB" << i << endl;
	}
      
      
      
      out << "\n\nSYSTEM\n\n";
      
      out << " process:  sys\n\n";  

      ////////////////////////////////////
      // converter 1 process
      ////////////////////////////////////
  
      for (int i = 1; i <= a; i++)
	{
	  out << "  convert1LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP1\n";
	  out << "    pre: posLadleA" << i << " = 0 /\\ stateLadleA" << i << " = 0 /\\ freeLP1\n";
	  out << "    eff: posLadleA" << i << "' = 1 /\\ stateLadleA" << i << "' = 1 /\\ ~freeLP1'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  convert1LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP1\n";
	  out << "    pre: posLadleB" << i << " = 0 /\\ stateLadleB" << i << " = 0 /\\ freeLP1\n";
	  out << "    eff: posLadleB" << i << "' = 1 /\\ stateLadleB" << i << "' = 1 /\\ ~freeLP1'\n\n";
	}



      ////////////////////////////////////
      // converter 2 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  convert2LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP7\n";
	  out << "    pre: posLadleA" << i << " = 0 /\\ stateLadleA" << i << " = 0 /\\ freeLP7\n";
	  out << "    eff: posLadleA" << i << "' = 7 /\\ stateLadleA" << i << "' = 1 /\\ ~freeLP7'\n\n";
	}
    
      for (int i = 1; i <= b; i++)
	{
	  out << "  convert2LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP7\n";
	  out << "    pre: posLadleB" << i << " = 0 /\\ stateLadleB" << i << " = 0 /\\ freeLP7\n";
	  out << "    eff: posLadleB" << i << "' = 7 /\\ stateLadleB" << i << "' = 1 /\\ ~freeLP7'\n\n";
	}


      ////////////////////////////////////
      // machine 1 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat1M1LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << endl;
	  out << "    pre: posLadleA" << i << " = 2 /\\ stateLadleA" << i << " = 1\n";
	  out << "    eff: posLadleA" << i << "' = 2 /\\ stateLadleA" << i << "' = 2\n\n";
	}

      for (int i = 1; i <= b; i++)
	{
	  out << "  treat1M1LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 2 /\\ stateLadleB" << i << " = 1\n";
	  out << "    eff: posLadleB" << i << "' = 2 /\\ stateLadleB" << i << "' = 2\n\n";
	}
      

      ////////////////////////////////////
      // machine 4 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat1M4LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << endl;
	  out << "    pre: posLadleA" << i << " = 8 /\\ stateLadleA" << i << " = 1\n";
	  out << "    eff: posLadleA" << i << "' = 8 /\\ stateLadleA" << i << "' = 2\n\n";
	}

      for (int i = 1; i <= b; i++)
	{
	  out << "  treat1M4LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 8 /\\ stateLadleB" << i << " = 1\n";
	  out << "    eff: posLadleB" << i << "' = 8 /\\ stateLadleB" << i << "' = 2\n\n";
	}
      
  


      ////////////////////////////////////
      // machine 2 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat2M2LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << endl;
	  out << "    pre: posLadleA" << i << " = 4 /\\ stateLadleA" << i << " = 2\n";
	  out << "    eff: posLadleA" << i << "' = 4 /\\ stateLadleA" << i << "' = 3\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat2M2LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 4 /\\ ( stateLadleB" << i << " = 2 \\/ stateLadleB" << i << " = 4)\n";
	  out << "    eff: posLadleB" << i << "' = 4 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1\n\n";
	}
      
      
      
      ////////////////////////////////////
      // machine 5 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat2M5LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << endl;
	  out << "    pre: posLadleA" << i << " = 10 /\\ stateLadleA" << i << " = 2\n";
	  out << "    eff: posLadleA" << i << "' = 10 /\\ stateLadleA" << i << "' = 3\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat2M5LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 10 /\\ ( stateLadleB" << i << " = 2 \\/ stateLadleB" << i << " = 4) \n";
	  out << "    eff: posLadleB" << i << "' = 10 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1\n\n";
	}


      
      ////////////////////////////////////
      // machine 3 process
      ////////////////////////////////////
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat3M3LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 6 /\\ stateLadleB" << i << " = 3\n";
	  out << "    eff: posLadleB" << i << "' = 6 /\\ stateLadleB" << i << "' = 4\n\n";
	}
      



      ////////////////////////////////////
      // caster process
      ////////////////////////////////////
      
  
      out << "\% begin continuous casting (can only be done once in a batch)\n\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  startLadleA" << i << endl; 
	  out << "    mod: posLadleA" << i << ",castToken,freeLP15,freeLP16\n";
	  out << "    pre: posLadleA" << i << " = 15 /\\ stateLadleA" << i << " = 3 /\\ castToken\n";
	  out << "    eff: posLadleA" << i << "' = 16 /\\ freeLP15' /\\ ~freeLP16' /\\  ~castToken'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  startLadleB" << i << endl; 
	  out << "    mod: posLadleB" << i << ",castToken,freeLP15,freeLP16\n";
	  out << "    pre: posLadleB" << i << " = 15 /\\ stateLadleB" << i << " = 5 /\\ castToken\n";
	  out << "    eff: posLadleB" << i << "' = 16 /\\ freeLP15' /\\ ~freeLP16' /\\  ~castToken'\n\n";
	}
      

      out << "\% swap ladles in continuous casting\n\n";
      for (int i = 1; i <= a; i++)
	for (int j = 1; j <= a; j++)
	  if (i != j)
	    {
	      out << "  swapLadleA" << i << "A" << j << endl; 
	      out << "    mod: posLadleA" << i << ",posLadleA" << j << ",stateLadleA" << j <<endl;
	      out << "    pre: stateLadleA" << i << " = 3 /\\ posLadleA" << i << " = 15 /\\  posLadleA" << j << " = 16\n";
	      out << "    eff: posLadleA" << i << "' = 16 /\\  posLadleA" << j << "' = 15 /\\ stateLadleA" << j << "' = 4\n\n";
	    }
      
      for (int i = 1; i <= a; i++)
	for (int j = 1; j <= b; j++)
	  {
	    out << "  swapLadleA" << i << "B" << j << endl; 
	    out << "    mod: posLadleA" << i << ",posLadleB" << j << ",stateLadleB" << j <<endl;
	    out << "    pre: stateLadleA" << i << " = 3 /\\ posLadleA" << i << " = 15 /\\  posLadleB" << j << " = 16\n";
	    out << "    eff: posLadleA" << i << "' = 16 /\\  posLadleB" << j << "' = 15 /\\ stateLadleB" << j << "' = 6\n\n";
	  }
      


      for (int i = 1; i <= b; i++)
	for (int j = 1; j <= b; j++)
	  if (i != j)
	    {
	      out << "  swapLadleB" << i << "B" << j << endl; 
	      out << "    mod: posLadleB" << i << ",posLadleB" << j << ",stateLadleB" << j <<endl;
	      out << "    pre: stateLadleB" << i << " = 5 /\\ posLadleB" << i << " = 15 /\\  posLadleB" << j << " = 16\n";
	      out << "    eff: posLadleB" << i << "' = 16 /\\  posLadleB" << j << "' = 15 /\\ stateLadleB" << j << "' = 6\n\n";
	    }
      

      for (int i = 1; i <= b; i++)
	for (int j = 1; j <= a; j++)
	  {
	    out << "  swapLadleB" << i << "A" << j << endl; 
	    out << "    mod: posLadleB" << i << ",posLadleA" << j << ",stateLadleA" << j <<endl;
	    out << "    pre: stateLadleB" << i << " = 5 /\\ posLadleB" << i << " = 15 /\\  posLadleA" << j << " = 16\n";
	    out << "    eff: posLadleB" << i << "' = 16 /\\  posLadleA" << j << "' = 15 /\\ stateLadleA" << j << "' = 4\n\n";
	  }
      


      out << "\% end continuous casting \n\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  stopLadleA" << i << endl; 
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP15,freeLP16\n";
	  out << "    pre: posLadleA" << i << " = 16  /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ stateLadleA" << i << "' = 4 /\\ freeLP16' /\\ ~freeLP15'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  stopLadleB" << i << endl; 
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP15,freeLP16\n";
	  out << "    pre: posLadleB" << i << " = 16  /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ stateLadleB" << i << "' = 6 /\\ freeLP16' /\\ ~freeLP15'\n\n";
	}
      
      

      ////////////////////////////////////
      // track1 move processes
      ////////////////////////////////////

      for (int i = 1; i <= a; i++)
	{
	  
	  for (int pos = 1; pos <= 5; pos++)
	    {
	      out << "  rightTrk1At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos + 1 << " /\\ freeLP" 
	      << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
      
	  for (int pos = 2; pos <= 6; pos++)
	    {
	      out << "  leftTrk1At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}

      for (int i = 1; i <= b; i++)
	{	  
	  for (int pos = 1; pos <= 5; pos++)
	    {
	      out << "  rightTrk1At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 2; pos <= 6; pos++)
	    {
	      out << "  leftTrk1At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}
      

      
      ////////////////////////////////////
      // track2 move processes
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  
	  for (int pos = 7; pos <= 9; pos++)
	    {
	      out << "  rightTrk2At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 8; pos <= 10; pos++)
	    {
	      out << "  leftTrk2At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}
      
      for (int i = 1; i <= b; i++)
	{
	  
	  for (int pos = 7; pos <= 9; pos++)
	    {
	      out << "  rightTrk2At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 8; pos <= 10; pos++)
	    {
	      out << "  leftTrk2At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}





      ////////////////////////////////////
      // crane1 
      ////////////////////////////////////
      
      
      for (int pos = 4; pos >= 1; pos--)
	{
	  out << "  upC1At" << pos << endl;
	  out << "    mod: posCrane1, freeCP" << pos << ", freeCP" << pos - 1 << endl;
	  out << "    pre: posCrane1 = " << pos << " /\\ freeCP" << pos - 1 << endl; 
	  out << "    eff: posCrane1' = " << pos - 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos - 1 << "'\n\n";
	}
      
      for (int pos = 3; pos >= 0; pos--)
	{
	  out << "  downC1At" << pos << endl;
	  out << "    mod: posCrane1, freeCP" << pos << ", freeCP" << pos + 1 << endl;
	  out << "    pre: posCrane1 = " << pos << " /\\ freeCP" << pos + 1 << endl; 
	  out << "    eff: posCrane1' = " << pos + 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos + 1 << "'\n\n";
	}
      
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  loadC1LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleA" << i << " = 3 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP3' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleA" << i << " = 9 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP9' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleA" << i << " = 13 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP11'\n\n";
	  
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC1LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleA" << i << " = 15 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP15' /\\ ~freeLP11'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  loadC1LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleB" << i << " = 3 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP3' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleB" << i << " = 9 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP9' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleB" << i << " = 13 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP11'\n\n";
      
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC1LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleB" << i << " = 15 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP15' /\\ ~freeLP11'\n\n";
	}
      


      for (int i = 1; i <= a; i++)
	{
	  out << "  unloadC1LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleA" << i << " = 11 /\\ freeLP3\n";
	  out << "    eff: posLadleA" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP11'\n\n";
	  
	  out << "  unloadC1LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleA" << i << " = 11 /\\ freeLP9\n";
	  out << "    eff: posLadleA" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleA" << i << " = 11 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleA" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleA" << i << "at3\n";
	  out << "    mod: posLadleA" << i << ",freeLP11\n";
	  out << "    pre: posCrane1 = 3 /\\ posLadleA" << i << " = 11\n";
	  out << "    eff: posLadleA" << i << "' = 14 /\\ freeLP11'\n\n";
      
	  out << "  unloadC1LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleA" << i << " = 11 /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP11'\n\n";
	}



      for (int i = 1; i <= b; i++)
	{
	  out << "  unloadC1LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleB" << i << " = 11 /\\ freeLP3\n";
	  out << "    eff: posLadleB" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleB" << i << " = 11 /\\ freeLP9\n";
	  out << "    eff: posLadleB" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP11'\n\n";


	  out << "  unloadC1LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleB" << i << " = 11 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleB" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleB" << i << "at3\n";
	  out << "    mod: posLadleB" << i << ",freeLP11\n";
	  out << "    pre: posCrane1 = 3 /\\ posLadleB" << i << " = 11\n";
	  out << "    eff: posLadleB" << i << "' = 14 /\\ freeLP11'\n\n";
      
	  out << "  unloadC1LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleB" << i << " = 11 /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP11'\n\n";
	}



      ////////////////////////////////////
      // crane2 
      ////////////////////////////////////
  

      for (int pos = 4; pos >= 1; pos--)
	{
	  out << "  upC2At" << pos << endl;
	  out << "    mod: posCrane2, freeCP" << pos << ", freeCP" << pos - 1 << endl;
	  out << "    pre: posCrane2 = " << pos << " /\\ freeCP" << pos - 1 << endl; 
	  out << "    eff: posCrane2' = " << pos - 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos - 1 << "'\n\n";
	}
      
      for (int pos = 3; pos >= 0; pos--)
	{
	  out << "  downC2At" << pos << endl;
	  out << "    mod: posCrane2, freeCP" << pos << ", freeCP" << pos + 1 << endl;
	  out << "    pre: posCrane2 = " << pos << " /\\ freeCP" << pos + 1 << endl; 
	  out << "    eff: posCrane2' = " << pos + 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos + 1 << "'\n\n";
	}


      for (int i = 1; i <= a; i++)
	{
	  out << "  loadC2LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleA" << i << " = 3 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP3' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleA" << i << " = 9 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP9' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleA" << i << " = 13 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP12'\n\n";
	  
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC2LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleA" << i << " = 15 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP15' /\\ ~freeLP12'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  loadC2LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleB" << i << " = 3 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP3' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleB" << i << " = 9 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP9' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleB" << i << " = 13 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP12'\n\n";
      
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC2LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleB" << i << " = 15 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP15' /\\ ~freeLP12'\n\n";
	}



      for (int i = 1; i <= a; i++)
	{
	  out << "  unloadC2LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleA" << i << " = 12 /\\ freeLP3\n";
	  out << "    eff: posLadleA" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleA" << i << " = 12 /\\ freeLP9\n";
	  out << "    eff: posLadleA" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleA" << i << " = 12 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleA" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at3\n";
	  out << "    mod: posLadleA" << i << ",freeLP12\n";
	  out << "    pre: posCrane2 = 3 /\\ posLadleA" << i << " = 12\n";
	  out << "    eff: posLadleA" << i << "' = 14 /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleA" << i << " = 12 /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP12'\n\n";
	}
      


      for (int i = 1; i <= b; i++)
	{
	  out << "  unloadC2LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleB" << i << " = 12 /\\ freeLP3\n";
	  out << "    eff: posLadleB" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleB" << i << " = 12 /\\ freeLP9\n";
	  out << "    eff: posLadleB" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP12'\n\n";


	  out << "  unloadC2LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleB" << i << " = 12 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleB" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP12'\n\n";

	  out << "  unloadC2LadleB" << i << "at3\n";
	  out << "    mod: posLadleB" << i << ",freeLP12\n";
	  out << "    pre: posCrane2 = 3 /\\ posLadleB" << i << " = 12\n";
	  out << "    eff: posLadleB" << i << "' = 14 /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleB" << i << " = 12 /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP12'\n\n";
	}

  
      out << "\n\nINITIALLY\n";
      out << "  ~freeCP0 /\\ freeCP1 /\\ ~freeCP2 /\\ freeCP3 /\\ freeCP4 /\\\n";
      out << "  freeLP1 /\\ freeLP2 /\\ freeLP3 /\\ freeLP4 /\\ freeLP5 /\\ freeLP6 /\\\n";
      out << "  freeLP6 /\\ freeLP7 /\\ freeLP8 /\\ freeLP9 /\\ freeLP10 /\\ freeLP11 /\\\n";
      out << "  freeLP12 /\\ freeLP15 /\\ freeLP16 /\\\n";  
      out << "  loadLP13 = 0 /\\\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  posLadleA" << i <<  " = 0 /\\\n";
	  out << "  stateLadleA" << i << " = 0 /\\\n";
	}	
      for (int i = 1; i <= b; i++)
	{
	  out << "  posLadleB" << i <<  " = 0 /\\\n";
	  out << "  stateLadleB" << i << " = 0 /\\\n";
	}
      out << "  castToken /\\ posCrane1 = 0 /\\ posCrane2 =  2\n";
      
      
      out << "\n\nGOAL\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  posLadleA" << i <<  " = 14 /\\\n";
	  out << "  stateLadleA" << i << " = 4 /\\\n";
	}	
      for (int i = 1; i <= b-1; i++)
	{
	  out << "  posLadleB" << i <<  " = 14 /\\\n";
	  out << "  stateLadleB" << i << " = 6 /\\\n";
	}
      out << "  posLadleB" << b <<  " = 14 /\\\n";
      out << "  stateLadleB" << b << " = 6\n";
    }
  else if (format == 2)
    {












































      ///////////////////////////////////////////////////////////
      //
      // Format 2: single-agent + error effects on machine 
      // 1,2,3,4, but not 3
      //
      ///////////////////////////////////////////////////////////



      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   SIDMAR problem in format " << format << " with " << a << endl;
      out << "\%         recipe A ladles and " << b << " recipe B ladles\n"; 
      out << "\% Date:   02\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      // starting with crane and occupation variables
      out << "\% starting with crane and occupation variables\n";
      out << "  bool   castToken\n";
      out << "  nat(3) posCrane1, posCrane2\n";
      out << "  bool   freeCP0,freeCP1,freeCP2,freeCP3,freeCP4\n";
      out << "  bool   freeLP1,freeLP2,freeLP3,freeLP4,freeLP5,freeLP6,freeLP6\n";
      out << "  bool   freeLP7,freeLP8,freeLP9,freeLP10,freeLP11,freeLP12,freeLP15,freeLP16\n";
      out << "  nat(3) loadLP13\n";
      out << "  bool   okM1,okM2,okM4,okM5,okConv1,okConv2\n";
      
      // then ladles of recipe A
      out << "\% then ladles of recipe A\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  nat(5) posLadleA" << i << endl;
	  out << "  nat(3) stateLadleA" << i << endl;
	}
      
      // finally ladles of recipe B
      out << "\% finally ladles of recipe B\n";
      for (int i = 1; i <= b; i++)
	{
	  out << "  nat(5) posLadleB" << i << endl;
	  out << "  nat(3) stateLadleB" << i << endl;
	}
      
      
      
      out << "\n\nSYSTEM\n\n";
      
      out << " process:  sys\n\n";  

      ////////////////////////////////////
      // converter 1 process
      ////////////////////////////////////
  
      for (int i = 1; i <= a; i++)
	{
	  out << "  convert1LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP1" << ",okConv1\n";
	  out << "    pre: posLadleA" << i << " = 0 /\\ stateLadleA" << i << " = 0 /\\ freeLP1 /\\ okConv1\n";
	  out << "    eff: posLadleA" << i << "' = 1 /\\ stateLadleA" << i << "' = 1 /\\ ~freeLP1' /\\ okConv1'\n";
	  out << "    err: posLadleA" << i << "' = 0 /\\ stateLadleA" << i << "' = 0 /\\ freeLP1' /\\ ~okConv1'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  convert1LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP1" << ",okConv1\n";
	  out << "    pre: posLadleB" << i << " = 0 /\\ stateLadleB" << i << " = 0 /\\ freeLP1 /\\ okConv1\n";
	  out << "    eff: posLadleB" << i << "' = 1 /\\ stateLadleB" << i << "' = 1 /\\ ~freeLP1' /\\ okConv1'\n";
	  out << "    err: posLadleB" << i << "' = 0 /\\ stateLadleB" << i << "' = 0 /\\ freeLP1' /\\ ~okConv1'\n\n";
	}



      ////////////////////////////////////
      // converter 2 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  convert2LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP7" << ",okConv2\n";
	  out << "    pre: posLadleA" << i << " = 0 /\\ stateLadleA" << i << " = 0 /\\ freeLP7 /\\ okConv2\n";
	  out << "    eff: posLadleA" << i << "' = 7 /\\ stateLadleA" << i << "' = 1 /\\ ~freeLP7' /\\ okConv2'\n";
	  out << "    err: posLadleA" << i << "' = 0 /\\ stateLadleA" << i << "' = 0 /\\ freeLP7'  /\\ ~okConv2'\n\n";
	}
    
      for (int i = 1; i <= b; i++)
	{
	  out << "  convert2LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP7" << ",okConv2\n";
	  out << "    pre: posLadleB" << i << " = 0 /\\ stateLadleB" << i << " = 0 /\\ freeLP7 /\\ okConv2\n";
	  out << "    eff: posLadleB" << i << "' = 7 /\\ stateLadleB" << i << "' = 1 /\\ ~freeLP7' /\\ okConv2'\n";
	  out << "    err: posLadleB" << i << "' = 0 /\\ stateLadleB" << i << "' = 0 /\\ freeLP7' /\\ ~okConv2'\n\n";
	}


      ////////////////////////////////////
      // machine 1 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat1M1LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM1\n";
	  out << "    pre: posLadleA" << i << " = 2 /\\ stateLadleA" << i << " = 1 /\\ okM1\n";
	  out << "    eff: posLadleA" << i << "' = 2 /\\ stateLadleA" << i << "' = 2 /\\ okM1'\n";
	  out << "    err: posLadleA" << i << "' = 2 /\\ stateLadleA" << i << "' = 1 /\\ ~okM1'\n\n";
	}

      for (int i = 1; i <= b; i++)
	{
	  out << "  treat1M1LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM1\n";
	  out << "    pre: posLadleB" << i << " = 2 /\\ stateLadleB" << i << " = 1 /\\ okM1\n";
	  out << "    eff: posLadleB" << i << "' = 2 /\\ stateLadleB" << i << "' = 2 /\\ okM1'\n";
	  out << "    err: posLadleB" << i << "' = 2 /\\ stateLadleB" << i << "' = 1 /\\ ~okM1'\n\n";
	}
      

      ////////////////////////////////////
      // machine 4 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat1M4LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM4\n";
	  out << "    pre: posLadleA" << i << " = 8 /\\ stateLadleA" << i << " = 1 /\\ okM4\n";
	  out << "    eff: posLadleA" << i << "' = 8 /\\ stateLadleA" << i << "' = 2 /\\ okM4'\n";
	  out << "    err: posLadleA" << i << "' = 8 /\\ stateLadleA" << i << "' = 1 /\\ ~okM4'\n\n";
	}

      for (int i = 1; i <= b; i++)
	{
	  out << "  treat1M4LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM4\n";
	  out << "    pre: posLadleB" << i << " = 8 /\\ stateLadleB" << i << " = 1 /\\ okM4\n";
	  out << "    eff: posLadleB" << i << "' = 8 /\\ stateLadleB" << i << "' = 2 /\\ okM4'\n";
	  out << "    err: posLadleB" << i << "' = 8 /\\ stateLadleB" << i << "' = 1 /\\ ~okM4'\n\n";
	}
      
  


      ////////////////////////////////////
      // machine 2 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat2M2LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM2\n";
	  out << "    pre: posLadleA" << i << " = 4 /\\ stateLadleA" << i << " = 2 /\\ okM2\n";
	  out << "    eff: posLadleA" << i << "' = 4 /\\ stateLadleA" << i << "' = 3 /\\ okM2'\n";
	  out << "    err: posLadleA" << i << "' = 4 /\\ stateLadleA" << i << "' = 2 /\\ ~okM2'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat2M2LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM2\n";
	  out << "    pre: posLadleB" << i << " = 4 /\\ ( stateLadleB" << i << " = 2 \\/ stateLadleB" << i << " = 4) /\\ okM2\n";
	  out << "    eff: posLadleB" << i << "' = 4 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1 /\\ okM2'\n";
	  out << "    err: posLadleB" << i << "' = 4 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1 /\\ ~okM2'\n\n";
	}
      
      
      
      ////////////////////////////////////
      // machine 5 process
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  treat2M5LadleA" << i << endl;
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",okM5\n";
	  out << "    pre: posLadleA" << i << " = 10 /\\ stateLadleA" << i << " = 2 /\\ okM5\n";
	  out << "    eff: posLadleA" << i << "' = 10 /\\ stateLadleA" << i << "' = 3 /\\ okM5'\n";
	  out << "    err: posLadleA" << i << "' = 10 /\\ stateLadleA" << i << "' = 2 /\\ ~okM5'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat2M5LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",okM5\n";
	  out << "    pre: posLadleB" << i << " = 10 /\\ ( stateLadleB" << i << " = 2 \\/ stateLadleB" << i << " = 4) /\\ okM5\n";
	  out << "    eff: posLadleB" << i << "' = 10 /\\ stateLadleB" << i << "' = stateLadleB" << i << " + 1 /\\ okM5'\n";
	  out << "    err: posLadleB" << i << "' = 10 /\\ stateLadleB" << i << "' = stateLadleB" << i << " /\\ ~okM5'\n\n";
	}


      
      ////////////////////////////////////
      // machine 3 process
      ////////////////////////////////////
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  treat3M3LadleB" << i << endl;
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << endl;
	  out << "    pre: posLadleB" << i << " = 6 /\\ stateLadleB" << i << " = 3\n";
	  out << "    eff: posLadleB" << i << "' = 6 /\\ stateLadleB" << i << "' = 4\n\n";
	}
      



      ////////////////////////////////////
      // caster process
      ////////////////////////////////////
      
  
      out << "\% begin continuous casting (can only be done once in a batch)\n\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  startLadleA" << i << endl; 
	  out << "    mod: posLadleA" << i << ",castToken,freeLP15,freeLP16\n";
	  out << "    pre: posLadleA" << i << " = 15 /\\ stateLadleA" << i << " = 3 /\\ castToken\n";
	  out << "    eff: posLadleA" << i << "' = 16 /\\ freeLP15' /\\ ~freeLP16' /\\  ~castToken'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  startLadleB" << i << endl; 
	  out << "    mod: posLadleB" << i << ",castToken,freeLP15,freeLP16\n";
	  out << "    pre: posLadleB" << i << " = 15 /\\ stateLadleB" << i << " = 5 /\\ castToken\n";
	  out << "    eff: posLadleB" << i << "' = 16 /\\ freeLP15' /\\ ~freeLP16' /\\  ~castToken'\n\n";
	}
      

      out << "\% swap ladles in continuous casting\n\n";
      for (int i = 1; i <= a; i++)
	for (int j = 1; j <= a; j++)
	  if (i != j)
	    {
	      out << "  swapLadleA" << i << "A" << j << endl; 
	      out << "    mod: posLadleA" << i << ",posLadleA" << j << ",stateLadleA" << j <<endl;
	      out << "    pre: stateLadleA" << i << " = 3 /\\ posLadleA" << i << " = 15 /\\  posLadleA" << j << " = 16\n";
	      out << "    eff: posLadleA" << i << "' = 16 /\\  posLadleA" << j << "' = 15 /\\ stateLadleA" << j << "' = 4\n\n";
	    }
      
      for (int i = 1; i <= a; i++)
	for (int j = 1; j <= b; j++)
	  {
	    out << "  swapLadleA" << i << "B" << j << endl; 
	    out << "    mod: posLadleA" << i << ",posLadleB" << j << ",stateLadleB" << j <<endl;
	    out << "    pre: stateLadleA" << i << " = 3 /\\ posLadleA" << i << " = 15 /\\  posLadleB" << j << " = 16\n";
	    out << "    eff: posLadleA" << i << "' = 16 /\\  posLadleB" << j << "' = 15 /\\ stateLadleB" << j << "' = 6\n\n";
	  }
      


      for (int i = 1; i <= b; i++)
	for (int j = 1; j <= b; j++)
	  if (i != j)
	    {
	      out << "  swapLadleB" << i << "B" << j << endl; 
	      out << "    mod: posLadleB" << i << ",posLadleB" << j << ",stateLadleB" << j <<endl;
	      out << "    pre: stateLadleB" << i << " = 5 /\\ posLadleB" << i << " = 15 /\\  posLadleB" << j << " = 16\n";
	      out << "    eff: posLadleB" << i << "' = 16 /\\  posLadleB" << j << "' = 15 /\\ stateLadleB" << j << "' = 6\n\n";
	    }
      

      for (int i = 1; i <= b; i++)
	for (int j = 1; j <= a; j++)
	  {
	    out << "  swapLadleB" << i << "A" << j << endl; 
	    out << "    mod: posLadleB" << i << ",posLadleA" << j << ",stateLadleA" << j <<endl;
	    out << "    pre: stateLadleB" << i << " = 5 /\\ posLadleB" << i << " = 15 /\\  posLadleA" << j << " = 16\n";
	    out << "    eff: posLadleB" << i << "' = 16 /\\  posLadleA" << j << "' = 15 /\\ stateLadleA" << j << "' = 4\n\n";
	  }
      


      out << "\% end continuous casting \n\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  stopLadleA" << i << endl; 
	  out << "    mod: posLadleA" << i << ",stateLadleA" << i << ",freeLP15,freeLP16\n";
	  out << "    pre: posLadleA" << i << " = 16  /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ stateLadleA" << i << "' = 4 /\\ freeLP16' /\\ ~freeLP15'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  stopLadleB" << i << endl; 
	  out << "    mod: posLadleB" << i << ",stateLadleB" << i << ",freeLP15,freeLP16\n";
	  out << "    pre: posLadleB" << i << " = 16  /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ stateLadleB" << i << "' = 6 /\\ freeLP16' /\\ ~freeLP15'\n\n";
	}
      
      

      ////////////////////////////////////
      // track1 move processes
      ////////////////////////////////////

      for (int i = 1; i <= a; i++)
	{
	  
	  for (int pos = 1; pos <= 5; pos++)
	    {
	      out << "  rightTrk1At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos + 1 << " /\\ freeLP" 
	      << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
      
	  for (int pos = 2; pos <= 6; pos++)
	    {
	      out << "  leftTrk1At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}

      for (int i = 1; i <= b; i++)
	{	  
	  for (int pos = 1; pos <= 5; pos++)
	    {
	      out << "  rightTrk1At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 2; pos <= 6; pos++)
	    {
	      out << "  leftTrk1At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}
      

      
      ////////////////////////////////////
      // track2 move processes
      ////////////////////////////////////
      
      for (int i = 1; i <= a; i++)
	{
	  
	  for (int pos = 7; pos <= 9; pos++)
	    {
	      out << "  rightTrk2At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 8; pos <= 10; pos++)
	    {
	      out << "  leftTrk2At" << pos << "LadleA" << i << endl;
	      out << "    mod: posLadleA" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleA" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleA" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}
      
      for (int i = 1; i <= b; i++)
	{
	  
	  for (int pos = 7; pos <= 9; pos++)
	    {
	      out << "  rightTrk2At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos + 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos + 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos + 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos + 1 << "'\n\n";
	    }
	  
	  for (int pos = 8; pos <= 10; pos++)
	    {
	      out << "  leftTrk2At" << pos << "LadleB" << i << endl;
	      out << "    mod: posLadleB" << i << ", freeLP" << pos << ", freeLP" << pos - 1 << endl;
	      out << "    pre: posLadleB" << i << " = " << pos << " /\\ freeLP" << pos - 1 << endl; 
	      out << "    eff: posLadleB" << i << "' = " << pos - 1 << " /\\ freeLP" 
		  << pos << "' /\\ ~freeLP" << pos - 1 << "'\n\n";
	    }
	}





      ////////////////////////////////////
      // crane1 
      ////////////////////////////////////
      
      
      for (int pos = 4; pos >= 1; pos--)
	{
	  out << "  upC1At" << pos << endl;
	  out << "    mod: posCrane1, freeCP" << pos << ", freeCP" << pos - 1 << endl;
	  out << "    pre: posCrane1 = " << pos << " /\\ freeCP" << pos - 1 << endl; 
	  out << "    eff: posCrane1' = " << pos - 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos - 1 << "'\n\n";
	}
      
      for (int pos = 3; pos >= 0; pos--)
	{
	  out << "  downC1At" << pos << endl;
	  out << "    mod: posCrane1, freeCP" << pos << ", freeCP" << pos + 1 << endl;
	  out << "    pre: posCrane1 = " << pos << " /\\ freeCP" << pos + 1 << endl; 
	  out << "    eff: posCrane1' = " << pos + 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos + 1 << "'\n\n";
	}
      
      
      for (int i = 1; i <= a; i++)
	{
	  out << "  loadC1LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleA" << i << " = 3 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP3' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleA" << i << " = 9 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP9' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleA" << i << " = 13 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP11'\n\n";
	  
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC1LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleA" << i << " = 15 /\\ freeLP11\n";
	  out << "    eff: posLadleA" << i << "' = 11 /\\ freeLP15' /\\ ~freeLP11'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  loadC1LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleB" << i << " = 3 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP3' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleB" << i << " = 9 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP9' /\\ ~freeLP11'\n\n";
	  
	  out << "  loadC1LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleB" << i << " = 13 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP11'\n\n";
      
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC1LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleB" << i << " = 15 /\\ freeLP11\n";
	  out << "    eff: posLadleB" << i << "' = 11 /\\ freeLP15' /\\ ~freeLP11'\n\n";
	}
      


      for (int i = 1; i <= a; i++)
	{
	  out << "  unloadC1LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleA" << i << " = 11 /\\ freeLP3\n";
	  out << "    eff: posLadleA" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP11'\n\n";
	  
	  out << "  unloadC1LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleA" << i << " = 11 /\\ freeLP9\n";
	  out << "    eff: posLadleA" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleA" << i << " = 11 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleA" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleA" << i << "at3\n";
	  out << "    mod: posLadleA" << i << ",freeLP11\n";
	  out << "    pre: posCrane1 = 3 /\\ posLadleA" << i << " = 11\n";
	  out << "    eff: posLadleA" << i << "' = 14 /\\ freeLP11'\n\n";
      
	  out << "  unloadC1LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleA" << i << " = 11 /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP11'\n\n";
	}



      for (int i = 1; i <= b; i++)
	{
	  out << "  unloadC1LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP11\n";
	  out << "    pre: posCrane1 = 0 /\\ posLadleB" << i << " = 11 /\\ freeLP3\n";
	  out << "    eff: posLadleB" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP11\n";
	  out << "    pre: posCrane1 = 1 /\\ posLadleB" << i << " = 11 /\\ freeLP9\n";
	  out << "    eff: posLadleB" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP11'\n\n";


	  out << "  unloadC1LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP11\n";
	  out << "    pre: posCrane1 = 2 /\\ posLadleB" << i << " = 11 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleB" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP11'\n\n";

	  out << "  unloadC1LadleB" << i << "at3\n";
	  out << "    mod: posLadleB" << i << ",freeLP11\n";
	  out << "    pre: posCrane1 = 3 /\\ posLadleB" << i << " = 11\n";
	  out << "    eff: posLadleB" << i << "' = 14 /\\ freeLP11'\n\n";
      
	  out << "  unloadC1LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP11\n";
	  out << "    pre: posCrane1 = 4 /\\ posLadleB" << i << " = 11 /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP11'\n\n";
	}



      ////////////////////////////////////
      // crane2 
      ////////////////////////////////////
  

      for (int pos = 4; pos >= 1; pos--)
	{
	  out << "  upC2At" << pos << endl;
	  out << "    mod: posCrane2, freeCP" << pos << ", freeCP" << pos - 1 << endl;
	  out << "    pre: posCrane2 = " << pos << " /\\ freeCP" << pos - 1 << endl; 
	  out << "    eff: posCrane2' = " << pos - 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos - 1 << "'\n\n";
	}
      
      for (int pos = 3; pos >= 0; pos--)
	{
	  out << "  downC2At" << pos << endl;
	  out << "    mod: posCrane2, freeCP" << pos << ", freeCP" << pos + 1 << endl;
	  out << "    pre: posCrane2 = " << pos << " /\\ freeCP" << pos + 1 << endl; 
	  out << "    eff: posCrane2' = " << pos + 1 << " /\\ freeCP" << pos << "' /\\ ~freeCP" << pos + 1 << "'\n\n";
	}


      for (int i = 1; i <= a; i++)
	{
	  out << "  loadC2LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleA" << i << " = 3 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP3' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleA" << i << " = 9 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP9' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleA" << i << " = 13 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP12'\n\n";
	  
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC2LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleA" << i << " = 15 /\\ freeLP12\n";
	  out << "    eff: posLadleA" << i << "' = 12 /\\ freeLP15' /\\ ~freeLP12'\n\n";
	}
      
      for (int i = 1; i <= b; i++)
	{
	  out << "  loadC2LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleB" << i << " = 3 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP3' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleB" << i << " = 9 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP9' /\\ ~freeLP12'\n\n";
	  
	  out << "  loadC2LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleB" << i << " = 13 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ loadLP13' = loadLP13 - 1  /\\ ~freeLP12'\n\n";
      
	  out << "\% Obs. we assume ladles can't be taken from storage (crane pos 3) when first placed there\n\n";
	  out << "  loadC2LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleB" << i << " = 15 /\\ freeLP12\n";
	  out << "    eff: posLadleB" << i << "' = 12 /\\ freeLP15' /\\ ~freeLP12'\n\n";
	}



      for (int i = 1; i <= a; i++)
	{
	  out << "  unloadC2LadleA" << i << "at0\n";
	  out << "    mod: posLadleA" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleA" << i << " = 12 /\\ freeLP3\n";
	  out << "    eff: posLadleA" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at1\n";
	  out << "    mod: posLadleA" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleA" << i << " = 12 /\\ freeLP9\n";
	  out << "    eff: posLadleA" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at2\n";
	  out << "    mod: posLadleA" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleA" << i << " = 12 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleA" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at3\n";
	  out << "    mod: posLadleA" << i << ",freeLP12\n";
	  out << "    pre: posCrane2 = 3 /\\ posLadleA" << i << " = 12\n";
	  out << "    eff: posLadleA" << i << "' = 14 /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleA" << i << "at4\n";
	  out << "    mod: posLadleA" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleA" << i << " = 12 /\\ freeLP15\n";
	  out << "    eff: posLadleA" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP12'\n\n";
	}
      


      for (int i = 1; i <= b; i++)
	{
	  out << "  unloadC2LadleB" << i << "at0\n";
	  out << "    mod: posLadleB" << i << ",freeLP3,freeLP12\n";
	  out << "    pre: posCrane2 = 0 /\\ posLadleB" << i << " = 12 /\\ freeLP3\n";
	  out << "    eff: posLadleB" << i << "' = 3 /\\ ~freeLP3' /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleB" << i << "at1\n";
	  out << "    mod: posLadleB" << i << ",freeLP9,freeLP12\n";
	  out << "    pre: posCrane2 = 1 /\\ posLadleB" << i << " = 12 /\\ freeLP9\n";
	  out << "    eff: posLadleB" << i << "' = 9 /\\ ~freeLP9' /\\ freeLP12'\n\n";


	  out << "  unloadC2LadleB" << i << "at2\n";
	  out << "    mod: posLadleB" << i << ",loadLP13,freeLP12\n";
	  out << "    pre: posCrane2 = 2 /\\ posLadleB" << i << " = 12 /\\ loadLP13 < 5\n";
	  out << "    eff: posLadleB" << i << "' = 13 /\\ loadLP13' = loadLP13 + 1  /\\ freeLP12'\n\n";

	  out << "  unloadC2LadleB" << i << "at3\n";
	  out << "    mod: posLadleB" << i << ",freeLP12\n";
	  out << "    pre: posCrane2 = 3 /\\ posLadleB" << i << " = 12\n";
	  out << "    eff: posLadleB" << i << "' = 14 /\\ freeLP12'\n\n";
	  
	  out << "  unloadC2LadleB" << i << "at4\n";
	  out << "    mod: posLadleB" << i << ",freeLP15,freeLP12\n";
	  out << "    pre: posCrane2 = 4 /\\ posLadleB" << i << " = 12 /\\ freeLP15\n";
	  out << "    eff: posLadleB" << i << "' = 15 /\\ ~freeLP15' /\\ freeLP12'\n\n";
	}

  
      out << "\n\nINITIALLY\n";
      out << "  okM1 /\\ okM2 /\\ okM4 /\\ okM5 /\\ okConv1 /\\ okConv2 /\\\n";
      out << "  ~freeCP0 /\\ freeCP1 /\\ ~freeCP2 /\\ freeCP3 /\\ freeCP4 /\\\n";
      out << "  freeLP1 /\\ freeLP2 /\\ freeLP3 /\\ freeLP4 /\\ freeLP5 /\\ freeLP6 /\\\n";
      out << "  freeLP6 /\\ freeLP7 /\\ freeLP8 /\\ freeLP9 /\\ freeLP10 /\\ freeLP11 /\\\n";
      out << "  freeLP12 /\\ freeLP15 /\\ freeLP16 /\\\n";  
      out << "  loadLP13 = 0 /\\\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  posLadleA" << i <<  " = 0 /\\\n";
	  out << "  stateLadleA" << i << " = 0 /\\\n";
	}	
      for (int i = 1; i <= b; i++)
	{
	  out << "  posLadleB" << i <<  " = 0 /\\\n";
	  out << "  stateLadleB" << i << " = 0 /\\\n";
	}
      out << "  castToken /\\ posCrane1 = 0 /\\ posCrane2 =  2\n";
      
      
      out << "\n\nGOAL\n";
      for (int i = 1; i <= a; i++)
	{
	  out << "  posLadleA" << i <<  " = 14 /\\\n";
	  out << "  stateLadleA" << i << " = 4 /\\\n";
	}	
      for (int i = 1; i <= b-1; i++)
	{
	  out << "  posLadleB" << i <<  " = 14 /\\\n";
	  out << "  stateLadleB" << i << " = 6 /\\\n";
	}
      out << "  posLadleB" << b <<  " = 14 /\\\n";
      out << "  stateLadleB" << b << " = 6\n";
    }

  
  return 0;
}
