/////////////////////////////////////////////////////////////////////
// File  : main.cc   
// Desc. : extNADL Generator of the Power Supply Restauration 
//         problem of Thiebaux ECP-01 with no partial observability
//
//         Obs !!
//         Open : open switch: turn off current
//         Close: close switch: turn on current 
//  
//         This domain is the simple network considered PG
//        
//
// Author: Rune M. Jensen
// Date  : 10/22/02
/////////////////////////////////////////////////////////////////////


#include <vector>
#include <set>
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "set.hpp"


using namespace std; 


void  printusage() {
  printf("USAGE: PSR [-hafo]                                      \n");
  printf("       -h print usage                                   \n");
  printf("       -f <formatNum> (default 0)                       \n");
  printf("          0 : error                                     \n");
  printf("          1 : nonDet                                    \n");
  printf("          2 : errorStrong                               \n");
  printf("       -o FILE (default PSR)                            \n");
  printf("          it will be prefixed with a<a>                 \n");
} 


enum unitType  {CBt,SDt};

// current direction is positive if going from line 0 to 1
// slot 0 : left or lower slot
// slot 1 : right or upper slot 
struct unit {
  unitType type; 
  string  name;        // string name
  int     lineNum;      
  int     lineNo[2];   // line no in line slot
  int     childNum[2];  
  unit*   child[2][4]; // children in slot
  void initCB(string n,int l0,int cn0)
  { type = CBt; name = n; lineNum = 1; lineNo[0] = l0; 
  lineNo[1] = -1; childNum[0] = cn0; childNum[1] = -1; }

  void initSD(string n,int l0,int l1,int cn0,int cn1)
  { type = SDt; name = n; lineNum = 2; lineNo[0] = l0; 
  lineNo[1] = l1; childNum[0] = cn0; childNum[1] = cn1; }
};


struct searchState {
  set< pair<unit*,int>  >  frontierUnits; // unit pointer + slot to expand
  set< pair<unit*,bool>  > doneUnits;     // unit pointer + openInTree 
  set<int>                 lines;         // lines in tree
  searchState() {}; 
  searchState(const searchState&); 
  searchState& operator=(const searchState&);
};


searchState::searchState(const searchState& s) {
  frontierUnits = s.frontierUnits;
  doneUnits = s.doneUnits;
  lines = s.lines;
}



searchState& searchState::operator=(const searchState& s) {
  frontierUnits = s.frontierUnits;
  doneUnits = s.doneUnits;
  lines = s.lines;
  return *this;
}





bool operator<(const searchState& s1, const searchState& s2)  {
  if (s1.frontierUnits < s2.frontierUnits) return true;
  else if (s1.frontierUnits == s2.frontierUnits && 
	   s1.doneUnits < s2.doneUnits) return true;
  else if (s1.frontierUnits == s2.frontierUnits && 
	   s1.doneUnits == s2.doneUnits && 
	   s1.lines < s2.lines) return true;
  return false;
}
  



///////////////////////////////////////////////////
//
// Globals
//
///////////////////////////////////////////////////

unit CB[4];
unit SD[8];


///////////////////////////////////////////////////
//
// Aux. funcs
//
///////////////////////////////////////////////////




void fixPoint(set<searchState>& open,set<searchState>& closed) {
  
  while (open.size() > 0)
    {      
      // pick a search state to expand
      searchState state = (*open.begin());

      open.erase(state);
      closed.insert(state);
      
      // expand each frontier unit for new children 
      for (set< pair<unit*,int>  >::iterator spi = state.frontierUnits.begin();
	   spi != state.frontierUnits.end(); ++spi)
	{
	  pair<unit*,int> pp = *spi;
	  searchState  childState = state ;
      
	  unit* parent = pp.first;
	  int parentExpSlot = pp.second;
	  childState.frontierUnits.erase(pp);
	  
	  if (parent->type == CBt && parentExpSlot == -1)
	    {
	      // do nothing
	    }      
	  else
	    {      
	      // find out if expansion leads to cycle
	      bool cycle = false;
	      for (int i = 0; i < parent->childNum[parentExpSlot];i++)
		{
		  if (cycle) break;
		  for (set< pair<unit*,int>  >::iterator pi = childState.frontierUnits.begin();
		       pi != childState.frontierUnits.end(); ++pi)
		    if (pi->first == parent->child[parentExpSlot][i])
		      {
			cycle = true;
			break;
		      }
		  for (set< pair<unit*,bool>  >::iterator pi = childState.doneUnits.begin();
		       pi != childState.doneUnits.end(); ++pi)
		    if (pi->first == parent->child[parentExpSlot][i])
		      {
			cycle = true;
			break;
		      }
		}
	      
	      if (cycle)
		{
		  // do nothing
		}
	      else
		{
		  // add line to lines in tree
		  int lineNo = parent->lineNo[parentExpSlot];
		  childState.lines.insert(lineNo);
		  
		  for (int i = 0; i < parent->childNum[parentExpSlot]; i++)
		    {
		      unit* child = parent->child[parentExpSlot][i];
		      
		      // set expSlot
		      int childExpSlot;
			if (child->type == CBt)
			  childExpSlot = -1;
			else if (lineNo == child->lineNo[0])
			  childExpSlot = 1;
			else
			  childExpSlot = 0;
		      
		      // add to frontier
		      pair<unit*,int> childEntry(child,childExpSlot);
		      childState.frontierUnits.insert(childEntry);	      
		    }

		  pair<unit*,bool> parentEntry(parent,false); 
		  childState.doneUnits.insert(parentEntry);
		  open.insert(childState);
		}
	    }
	}
      
    }
}



////////////////////////////////////////////////////////
//  Error MODEL
///////////////////////////////////////////////////////


///////////////////////////////////////////
//  CB transitions
//////////////////////////////////////////



void writeCloseCBTran(ofstream& out, int CBno,searchState state)
{
  out << "    mod: okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;
  out << "    pre: okCB" << CBno << " /\\ ~onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << endl;
  out << "    err: ~okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << "\n\n";
}





void writeOpenCBTran(ofstream& out, int CBno,searchState state) {

  out << "    mod: okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okCB" << CBno << " /\\ onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else	
	out << " /\\ on" << ui->first->name;	
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << endl;

  out << "    err: ~okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << "\n\n";
}





///////////////////////////////////////////
//  SD transitions
//////////////////////////////////////////




void writeCloseSDTran(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ ~onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui =  state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << endl;

  out << "    err: ~okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << "\n\n";
}



void writeOpenSDTran(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << endl;

  out << "    err: ~okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << "\n\n";
}






















////////////////////////////////////////////////////////
//  Non det MODEL
///////////////////////////////////////////////////////


///////////////////////////////////////////
//  CB transitions
//////////////////////////////////////////



void writeCloseCBTranND(ofstream& out, int CBno,searchState state)
{
  out << "    mod: okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;
  out << "    pre: okCB" << CBno << " /\\ ~onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: (okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << ")" << endl;
  out << "    \\/  ( ~okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " )\n\n";
}





void writeOpenCBTranND(ofstream& out, int CBno,searchState state) {

  out << "    mod: okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okCB" << CBno << " /\\ onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else	
	out << " /\\ on" << ui->first->name;	
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: (okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << ")" << endl;

  out << "    \\/ (~okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << ")\n\n";
}





///////////////////////////////////////////
//  SD transitions
//////////////////////////////////////////




void writeCloseSDTranND(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ ~onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui =  state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: (okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << ")" << endl;

  out << "    \\/ (~okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << ")\n\n";
}



void writeOpenSDTranND(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: (okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << ")" << endl;

  out << "    \\/ (~okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << ")\n\n";
}









////////////////////////////////////////////////////////
//  Error Strong MODEL (01/20/03)
///////////////////////////////////////////////////////


///////////////////////////////////////////
//  CB transitions
//////////////////////////////////////////



void writeCloseCBTranES(ofstream& out, int CBno,searchState state)
{
  out << "    mod: e,okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;
  out << "    pre: okCB" << CBno << " /\\ ~onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: e = 0 -> ( (okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << " /\\ e' = 0)" << endl;
  out << "    \\/  ( ~okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " /\\ e' = 1 )) , (okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << " /\\ e' = 1)\n\n";
}





void writeOpenCBTranES(ofstream& out, int CBno,searchState state) {

  out << "    mod: e,okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okCB" << CBno << " /\\ onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else	
	out << " /\\ on" << ui->first->name;	
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: e = 0 -> ( (okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " /\\ e' = 0 )" << endl;

  out << "    \\/ (~okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << " /\\ e' = 1) ), (okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " /\\ e' = 1) \n\n";
}





///////////////////////////////////////////
//  SD transitions
//////////////////////////////////////////




void writeCloseSDTranES(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: e,okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ ~onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui =  state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: e = 0 -> ( (okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << " /\\ e' = 0 )" << endl;

  out << "    \\/ (~okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " /\\ e' = 1) ), (okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << " /\\ e' = 1 ) \n\n";
}



void writeOpenSDTranES(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: e,okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: e = 0 -> ( (okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " /\\ e' = 0 )" << endl;

  out << "    \\/ (~okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << " /\\ e' = 1 ) ), (okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << " /\\ e' = 1 ) \n\n";
}
















///////////////////////////////////////////////////
//
// Main function
//
///////////////////////////////////////////////////
int main(int argc, char **argv) {
  
  // heuristic type
  int format = 0;

  char outName[512] = "PSR";
  char finalOutName[512];


  /*** Scan command line ***/
  for (int ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'o': 
	strcpy(outName,argv[++ind]);
	break;
      case 'f': 
	format = atoi(argv[++ind]);
	break;
      case 'h': 
	printusage();
	exit(1);
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }



  // define circuit breakers
  int CBnum = 3;
  CB[1].initCB("CB1",1,1);
  CB[2].initCB("CB2",5,1);
  CB[3].initCB("CB3",7,1);

  // define switching devices
  int SDnum = 7;
  SD[1].initSD("SD1",3,4,1,2);           
  SD[2].initSD("SD2",6,4,2,2);	   
  SD[3].initSD("SD3",7,4,1,2);	   
  SD[4].initSD("SD4",5,6,1,2);	   
  SD[5].initSD("SD5",2,3,2,1);	   
  SD[6].initSD("SD6",1,2,1,2);	   
  SD[7].initSD("SD7",2,6,2,2);	   

  // init CBs
  CB[1].child[0][0] = &SD[6];

  CB[2].child[0][0] = &SD[4];

  CB[3].child[0][0] = &SD[3];

  // init SDs
  SD[1].child[0][0] = &SD[5];
  SD[1].child[1][0] = &SD[2];
  SD[1].child[1][1] = &SD[3];

  SD[2].child[0][0] = &SD[4];
  SD[2].child[0][1] = &SD[7];
  SD[2].child[1][0] = &SD[1];
  SD[2].child[1][1] = &SD[3];

  SD[3].child[0][0] = &CB[3];
  SD[3].child[1][0] = &SD[2];
  SD[3].child[1][1] = &SD[1];

  SD[4].child[0][0] = &CB[2];
  SD[4].child[1][0] = &SD[2];
  SD[4].child[1][1] = &SD[7];

  SD[5].child[0][0] = &SD[6];
  SD[5].child[0][1] = &SD[7];
  SD[5].child[1][0] = &SD[1];

  SD[6].child[0][0] = &CB[1];
  SD[6].child[1][0] = &SD[5];
  SD[6].child[1][1] = &SD[7];

  SD[7].child[0][0] = &SD[5];
  SD[7].child[0][1] = &SD[6];
  SD[7].child[1][0] = &SD[4];
  SD[7].child[1][1] = &SD[2];

  
  sprintf(finalOutName,"%s.nadl",outName);      
  ofstream  out(finalOutName,ios::out);

  if (format == 0)
    {
      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   PSR simple problem in format " << format << endl;
      out << "\%         For details see Thiebaux & Cordier ECP-01\n";
      out << "\%         and Bertoli et al. ECAI-02\n"; 
      out << "\% Date:   02\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      

      // circuit breakers      
      for (int i = 1; i <= CBnum; i++) 
	out << "  bool   okCB" << i << ",onCB" << i << endl;

      // switching devices      
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okSD" << i << ",onSD" << i << endl;

      // lines
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okL" << i << ",onL" << i <<  endl;


      
      out << "\n\nSYSTEM\n\n";
      
      out << " process:  sys\n\n";  


      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open circuit breaker\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= CBnum; i++)
	{
	  {
	    // init search State
	    searchState init;
	    
	    pair<unit*,int>  key(&CB[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);
	    
	    
	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeCB" << i << endl;	  		  
		writeCloseCBTran(out,i,*ssi);
		out << "  openCB" << i << endl;	  		  
		writeOpenCBTran(out,i,*ssi);
	      }
	  }
	}
	  
      
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open switching device\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= SDnum; i++)
	{
	  out << "  closeSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ ~onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: okSD" << i << "' /\\ onSD" << i << "'\n";
	  out << "    err: ~okSD" << i << "' /\\ ~onSD" << i << "'\n\n";


	  out << "  openSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: okSD" << i << "' /\\ ~onSD" << i << "'\n";
	  out << "    err: ~okSD" << i << "' /\\ onSD" << i << "'\n\n";

	
	  {	    
	    out << "\%   Feed in positive direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],1);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTran(out,i,*ssi,0);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTran(out,i,*ssi,0);
	      }	    
	  }
	  
	  	  
	  {
	    out << "\%   Feed in negative direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTran(out,i,*ssi,1);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTran(out,i,*ssi,1);
	      }	    
	  }
	  
	}
      
      out << "Initially\n";
      out << "  ~onL1 /\\ okL1 /\\\n";
      out << "  ~onL2 /\\ okL2 /\\\n";
      out << "  ~onL3 /\\ okL3 /\\\n";
      out << "  ~onL4 /\\ okL4 /\\\n";
      out << "  ~onL5 /\\ okL5 /\\\n";
      out << "  ~onL6 /\\ okL6 /\\\n";
      out << "  ~onL7 /\\ okL7 /\\\n";
      out << "  ~onSD1 /\\ okSD1 /\\\n";
      out << "  ~onSD2 /\\ okSD2 /\\\n";
      out << "  ~onSD3 /\\ okSD3 /\\\n";
      out << "  ~onSD4 /\\ okSD4 /\\\n";
      out << "  ~onSD5 /\\ okSD5 /\\\n";
      out << "  ~onSD6 /\\ okSD6 /\\\n";
      out << "  ~onSD7 /\\ okSD7 /\\\n";      
      out << "  ~onCB1 /\\ okCB1 /\\\n";
      out << "  ~onCB2 /\\ okCB2 /\\\n";
      out << "  ~onCB3 /\\ okCB3\n\n";
      
      out << "Goal\n";
      out << "\% Activity constraint: All non-broken lines must be on\n";
      for (int i = 1; i <= 7; i++)
	out << "  (onL" << i << " <=> okL" << i << ") /\\\n";

      out << "\% Soundness constraint 1): all lines must be fed by some CB\n";
      out << "  (onL1 <=> ( onCB1  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD7 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD7 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 )  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD5 /\\ onL3 /\\ onSD1 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 ) )) /\\ \n";  

      out << "  (onL2 <=> ( ( onSD6 /\\ onL2 /\\ onCB1 )  \\/\n";  
      out << "              ( onSD7 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2  )  \\/\n";  
      out << "              ( onSD7 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 )  \\/\n";  
      out << "              ( onSD5 /\\ onL3 /\\ onSD1 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 ) ))  /\\\n";  

      out << "  (onL3 <=> ( ( onSD5 /\\ onL2  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD5 /\\ onL2  /\\ onSD7  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD1 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD1 /\\ onL4  /\\ onSD3  /\\ onL7 /\\ onCB3 ) )) /\\\n";  
 
      out << "  (onL4 <=> ( ( onSD1 /\\ onL3 /\\ onSD5 /\\ onL2  /\\ onCB1)  \\/\n";  
      out << "              ( onSD2 /\\ onL6 /\\ onSD7 /\\ onL2 /\\ onSD6 /\\ onL1 /\\ onCB1 )  \\/\n";  
      out << "              ( onSD2 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD3 /\\ onL7 /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL5 <=> ( onCB2  \\/\n";  
      out << "              ( onSD4 /\\ onL6 /\\ onSD7 /\\ onL2 /\\ onSD6  /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD4 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3  /\\ onL7  /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL6 <=> ( ( onSD4 /\\ onL5  /\\ onCB2 )  \\/\n";  
      out << "              ( onSD7 /\\ onL2 /\\ onSD6  /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD2 /\\ onL4 /\\ onSD3  /\\ onL7  /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL7 <=> ( onCB3  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD7 /\\ onL2  /\\ onSD6 /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD1  /\\ onL3 /\\ onSD5 /\\ onL2  /\\  onSD6 /\\ onL1  /\\ onCB1 ) )) /\\ \n";  

      out << "\% Soundness constraint 2): no feed cycles allowed\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ ~onSD2 /\\ onL4 /\\ onSD1 /\\ onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ ~onSD1 /\\ onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ ~onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ onSD5 /\\ onL2 /\\ ~onSD7) /\\\n";
      

      out << "\% Soundness constraint 3): no CB2CB connections\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD7 /\\ onL6  /\\ onSD4 /\\ onL5 /\\ onCB2) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD5  /\\ onL3 /\\ onSD1 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD5  /\\ onL3 /\\ onSD1 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD7  /\\ onL6 /\\ onSD2 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB2 /\\ onL5  /\\ onSD4  /\\ onL6 /\\ onSD2  /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";

      out << "\% Soundness constraint 4): at most one more thing broken\n";


out << " ( ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( ~okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    ~okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    ~okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    ~okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    ~okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    ~okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    ~okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    ~okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    ~okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    ~okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    ~okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    ~okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    ~okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    ~okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    ~okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    ~okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    ~okCB3 ))    \n";	  



    }
  else if (format == 1)
    {
      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   PSR simple problem in format " << format << endl;
      out << "\%         For details see Thiebaux & Cordier ECP-01\n";
      out << "\%         and Bertoli et al. ECAI-02\n"; 
      out << "\% Date:   02\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      

      // circuit breakers      
      for (int i = 1; i <= CBnum; i++) 
	out << "  bool   okCB" << i << ",onCB" << i << endl;

      // switching devices      
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okSD" << i << ",onSD" << i << endl;

      // lines
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okL" << i << ",onL" << i <<  endl;


      
      out << "\n\nSYSTEM\n\n";
      
      out << " process:  sys\n\n";  


      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open circuit breaker\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= CBnum; i++)
	{
	  {
	    // init search State
	    searchState init;
	    
	    pair<unit*,int>  key(&CB[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);
	    
	    
	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeCB" << i << endl;	  		  
		writeCloseCBTranND(out,i,*ssi);
		out << "  openCB" << i << endl;	  		  
		writeOpenCBTranND(out,i,*ssi);
	      }
	  }
	}
	  
      
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open switching device\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= SDnum; i++)
	{
	  out << "  closeSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ ~onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: (okSD" << i << "' /\\ onSD" << i << "')\n";
	  out << "    \\/ (~okSD" << i << "' /\\ ~onSD" << i << "')\n\n";


	  out << "  openSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: (okSD" << i << "' /\\ ~onSD" << i << "')\n";
	  out << "    \\/ (~okSD" << i << "' /\\ onSD" << i << "')\n\n";

	
	  {	    
	    out << "\%   Feed in positive direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],1);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTranND(out,i,*ssi,0);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTranND(out,i,*ssi,0);
	      }	    
	  }
	  
	  	  
	  {
	    out << "\%   Feed in negative direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTranND(out,i,*ssi,1);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTranND(out,i,*ssi,1);
	      }	    
	  }
	  
	}
      
      out << "Initially\n";
      out << "  ~onL1 /\\ okL1 /\\\n";
      out << "  ~onL2 /\\ okL2 /\\\n";
      out << "  ~onL3 /\\ okL3 /\\\n";
      out << "  ~onL4 /\\ okL4 /\\\n";
      out << "  ~onL5 /\\ okL5 /\\\n";
      out << "  ~onL6 /\\ okL6 /\\\n";
      out << "  ~onL7 /\\ okL7 /\\\n";
      out << "  ~onSD1 /\\ okSD1 /\\\n";
      out << "  ~onSD2 /\\ okSD2 /\\\n";
      out << "  ~onSD3 /\\ okSD3 /\\\n";
      out << "  ~onSD4 /\\ okSD4 /\\\n";
      out << "  ~onSD5 /\\ okSD5 /\\\n";
      out << "  ~onSD6 /\\ okSD6 /\\\n";
      out << "  ~onSD7 /\\ okSD7 /\\\n";      
      out << "  ~onCB1 /\\ okCB1 /\\\n";
      out << "  ~onCB2 /\\ okCB2 /\\\n";
      out << "  ~onCB3 /\\ okCB3\n\n";
      
      out << "Goal\n";
      out << "\% Activity constraint: All non-broken lines must be on\n";
      for (int i = 1; i <= 7; i++)
	out << "  (onL" << i << " <=> okL" << i << ") /\\\n";

      out << "\% Soundness constraint 1): all lines must be fed by some CB\n";
      out << "  (onL1 <=> ( onCB1  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD7 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD7 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 )  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD5 /\\ onL3 /\\ onSD1 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 ) )) /\\ \n";  

      out << "  (onL2 <=> ( ( onSD6 /\\ onL2 /\\ onCB1 )  \\/\n";  
      out << "              ( onSD7 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2  )  \\/\n";  
      out << "              ( onSD7 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 )  \\/\n";  
      out << "              ( onSD5 /\\ onL3 /\\ onSD1 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 ) ))  /\\\n";  

      out << "  (onL3 <=> ( ( onSD5 /\\ onL2  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD5 /\\ onL2  /\\ onSD7  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD1 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD1 /\\ onL4  /\\ onSD3  /\\ onL7 /\\ onCB3 ) )) /\\\n";  
 
      out << "  (onL4 <=> ( ( onSD1 /\\ onL3 /\\ onSD5 /\\ onL2  /\\ onCB1)  \\/\n";  
      out << "              ( onSD2 /\\ onL6 /\\ onSD7 /\\ onL2 /\\ onSD6 /\\ onL1 /\\ onCB1 )  \\/\n";  
      out << "              ( onSD2 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD3 /\\ onL7 /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL5 <=> ( onCB2  \\/\n";  
      out << "              ( onSD4 /\\ onL6 /\\ onSD7 /\\ onL2 /\\ onSD6  /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD4 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3  /\\ onL7  /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL6 <=> ( ( onSD4 /\\ onL5  /\\ onCB2 )  \\/\n";  
      out << "              ( onSD7 /\\ onL2 /\\ onSD6  /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD2 /\\ onL4 /\\ onSD3  /\\ onL7  /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL7 <=> ( onCB3  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD7 /\\ onL2  /\\ onSD6 /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD1  /\\ onL3 /\\ onSD5 /\\ onL2  /\\  onSD6 /\\ onL1  /\\ onCB1 ) )) /\\ \n";  

      out << "\% Soundness constraint 2): no feed cycles allowed\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ ~onSD2 /\\ onL4 /\\ onSD1 /\\ onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ ~onSD1 /\\ onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ ~onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ onSD5 /\\ onL2 /\\ ~onSD7) /\\\n";
      

      out << "\% Soundness constraint 3): no CB2CB connections\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD7 /\\ onL6  /\\ onSD4 /\\ onL5 /\\ onCB2) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD5  /\\ onL3 /\\ onSD1 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD5  /\\ onL3 /\\ onSD1 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD7  /\\ onL6 /\\ onSD2 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB2 /\\ onL5  /\\ onSD4  /\\ onL6 /\\ onSD2  /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";

      out << "\% Soundness constraint 4): at most one more thing broken\n";


out << " ( ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( ~okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    ~okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    ~okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    ~okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    ~okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    ~okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    ~okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    ~okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    ~okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    ~okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    ~okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    ~okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    ~okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    ~okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    ~okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    ~okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    ~okCB3 ))    \n";	  
    }























  else if (format == 2)
    {
      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   PSR simple problem in format " << format << endl;
      out << "\%         Optimal fault tolerance using strong planning\n";    
      out << "\%         For details about PSR see Thiebaux & Cordier ECP-01\n";
      out << "\%         and Bertoli et al. ECAI-02\n"; 
      out << "\% Date:   03\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      
      // error counter
      out << "  nat(1) e\n";

      // circuit breakers      
      for (int i = 1; i <= CBnum; i++) 
	out << "  bool   okCB" << i << ",onCB" << i << endl;

      // switching devices      
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okSD" << i << ",onSD" << i << endl;

      // lines
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okL" << i << ",onL" << i <<  endl;


      
      out << "\n\nSYSTEM\n\n";
      
      out << " process:  sys\n\n";  


      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open circuit breaker\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= CBnum; i++)
	{
	  {
	    // init search State
	    searchState init;
	    
	    pair<unit*,int>  key(&CB[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);
	    
	    
	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeCB" << i << endl;	  		  
		writeCloseCBTranES(out,i,*ssi);
		out << "  openCB" << i << endl;	  		  
		writeOpenCBTranES(out,i,*ssi);
	      }
	  }
	}
	  
      
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open switching device\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= SDnum; i++)
	{
	  out << "  closeSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: e,okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ ~onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: e = 0 -> ( (okSD" << i << "' /\\ onSD" << i << "' /\\ e' = 0 )\n";
	  out << "    \\/ (~okSD" << i << "' /\\ ~onSD" << i << "' /\\ e' = 1) ), (okSD" << i << "' /\\ onSD" << i << "' /\\ e' = 1 ) \n\n";


	  out << "  openSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: e,okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: e = 0 -> ( (okSD" << i << "' /\\ ~onSD" << i << "' /\\ e' = 0)\n";
	  out << "    \\/ (~okSD" << i << "' /\\ onSD" << i << "' /\\ e' = 1) ), (okSD" << i << "' /\\ ~onSD" << i << "' /\\ e' = 1) \n\n";

	
	  {	    
	    out << "\%   Feed in positive direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],1);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTranES(out,i,*ssi,0);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTranES(out,i,*ssi,0);
	      }	    
	  }
	  
	  	  
	  {
	    out << "\%   Feed in negative direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTranES(out,i,*ssi,1);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTranES(out,i,*ssi,1);
	      }	    
	  }
	  
	}
      
      out << "Initially\n";
      out << "  e = 0 /\\\n";
      out << "  ~onL1 /\\ okL1 /\\\n";
      out << "  ~onL2 /\\ okL2 /\\\n";
      out << "  ~onL3 /\\ okL3 /\\\n";
      out << "  ~onL4 /\\ okL4 /\\\n";
      out << "  ~onL5 /\\ okL5 /\\\n";
      out << "  ~onL6 /\\ okL6 /\\\n";
      out << "  ~onL7 /\\ okL7 /\\\n";
      out << "  ~onSD1 /\\ okSD1 /\\\n";
      out << "  ~onSD2 /\\ okSD2 /\\\n";
      out << "  ~onSD3 /\\ okSD3 /\\\n";
      out << "  ~onSD4 /\\ okSD4 /\\\n";
      out << "  ~onSD5 /\\ okSD5 /\\\n";
      out << "  ~onSD6 /\\ okSD6 /\\\n";
      out << "  ~onSD7 /\\ okSD7 /\\\n";      
      out << "  ~onCB1 /\\ okCB1 /\\\n";
      out << "  ~onCB2 /\\ okCB2 /\\\n";
      out << "  ~onCB3 /\\ okCB3\n\n";
      
      out << "Goal\n";
      out << "\% Activity constraint: All non-broken lines must be on\n";
      for (int i = 1; i <= 7; i++)
	out << "  (onL" << i << " <=> okL" << i << ") /\\\n";

      out << "\% Soundness constraint 1): all lines must be fed by some CB\n";
      out << "  (onL1 <=> ( onCB1  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD7 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD7 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 )  \\/\n";  
      out << "              (onSD6 /\\ onL2 /\\ onSD5 /\\ onL3 /\\ onSD1 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 ) )) /\\ \n";  

      out << "  (onL2 <=> ( ( onSD6 /\\ onL2 /\\ onCB1 )  \\/\n";  
      out << "              ( onSD7 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2  )  \\/\n";  
      out << "              ( onSD7 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 )  \\/\n";  
      out << "              ( onSD5 /\\ onL3 /\\ onSD1 /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3 ) ))  /\\\n";  

      out << "  (onL3 <=> ( ( onSD5 /\\ onL2  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD5 /\\ onL2  /\\ onSD7  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD1 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD1 /\\ onL4  /\\ onSD3  /\\ onL7 /\\ onCB3 ) )) /\\\n";  
 
      out << "  (onL4 <=> ( ( onSD1 /\\ onL3 /\\ onSD5 /\\ onL2  /\\ onCB1)  \\/\n";  
      out << "              ( onSD2 /\\ onL6 /\\ onSD7 /\\ onL2 /\\ onSD6 /\\ onL1 /\\ onCB1 )  \\/\n";  
      out << "              ( onSD2 /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD3 /\\ onL7 /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL5 <=> ( onCB2  \\/\n";  
      out << "              ( onSD4 /\\ onL6 /\\ onSD7 /\\ onL2 /\\ onSD6  /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD4 /\\ onL6 /\\ onSD2 /\\ onL4 /\\ onSD3  /\\ onL7  /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL6 <=> ( ( onSD4 /\\ onL5  /\\ onCB2 )  \\/\n";  
      out << "              ( onSD7 /\\ onL2 /\\ onSD6  /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD2 /\\ onL4 /\\ onSD3  /\\ onL7  /\\ onCB3 ) )) /\\ \n";  


      out << "  (onL7 <=> ( onCB3  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD4 /\\ onL5 /\\ onCB2 )  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD2  /\\ onL6 /\\ onSD7 /\\ onL2  /\\ onSD6 /\\ onL1  /\\ onCB1 )  \\/\n";  
      out << "              ( onSD3 /\\ onL4  /\\ onSD1  /\\ onL3 /\\ onSD5 /\\ onL2  /\\  onSD6 /\\ onL1  /\\ onCB1 ) )) /\\ \n";  

      out << "\% Soundness constraint 2): no feed cycles allowed\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ ~onSD2 /\\ onL4 /\\ onSD1 /\\ onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ ~onSD1 /\\ onL3 /\\ onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ ~onSD5 /\\ onL2 /\\ onSD7) /\\\n";
      out << "  ~(onL6 /\\ onSD2 /\\ onL4 /\\ onSD1 /\\  onL3 /\\ onSD5 /\\ onL2 /\\ ~onSD7) /\\\n";
      

      out << "\% Soundness constraint 3): no CB2CB connections\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD7 /\\ onL6  /\\ onSD4 /\\ onL5 /\\ onCB2) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD5  /\\ onL3 /\\ onSD1 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD5  /\\ onL3 /\\ onSD1 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB1 /\\ onL1  /\\ onSD6  /\\ onL2 /\\ onSD7  /\\ onL6 /\\ onSD2 /\\ onL4  /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";
      out << "  ~(onCB2 /\\ onL5  /\\ onSD4  /\\ onL6 /\\ onSD2  /\\ onL4 /\\ onSD3 /\\ onL7 /\\ onCB3) /\\\n";

      out << "\% Soundness constraint 4): at most one more thing broken\n";


out << " ( ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( ~okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    ~okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    ~okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    ~okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    ~okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    ~okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    ~okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    ~okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    ~okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    ~okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    ~okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    ~okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    ~okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    ~okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    ~okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    ~okCB2 /\\	   \n";
out << "    okCB3 ) \\/    \n";	  

out << "   ( okL1 /\\       \n";
out << "    okL2 /\\	   \n";
out << "    okL3 /\\	   \n";
out << "    okL4 /\\	   \n";
out << "    okL5 /\\	   \n";
out << "    okL6 /\\	   \n";
out << "    okL7 /\\	   \n";
out << "    okSD1 /\\	   \n";
out << "    okSD2 /\\	   \n";
out << "    okSD3 /\\	   \n";
out << "    okSD4 /\\	   \n";
out << "    okSD5 /\\	   \n";
out << "    okSD6 /\\	   \n";
out << "    okSD7 /\\	   \n";
out << "    okCB1 /\\	   \n";
out << "    okCB2 /\\	   \n";
out << "    ~okCB3 ))    \n";	  
    }
















  return 0;
}
