/////////////////////////////////////////////////////////////////////
// File  : main.cc   
// Desc. : extNADL Generator of the Power Supply Restauration 
//         problem of Thiebaux ECP-01 with no partial observability
//
//         Obs !!
//         Open : open switch: turn off current
//         Close: close switch: turn on current 
//  
//         This domain is the simple network considered PG
//        
//
// Author: Rune M. Jensen
// Date  : 10/22/02
/////////////////////////////////////////////////////////////////////


#include <vector>
#include <set>
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "set.hpp"


using namespace std; 


void  printusage() {
  printf("USAGE: PSR [-hafo]                                      \n");
  printf("       -h print usage                                   \n");
  printf("       -a NUM (default 2)                               \n");
  printf("          number of failed lines at most 7              \n");
  printf("       -f <formatNum> (default 0)                       \n");
  printf("          0 : non-deterministic                         \n");
  printf("       -o FILE (default PSR)                            \n");
  printf("          it will be prefixed with a<a>                 \n");
} 


enum unitType  {CBt,SDt};

// current direction is positive if going from line 0 to 1
// slot 0 : left or lower slot
// slot 1 : right or upper slot 
struct unit {
  unitType type; 
  string  name;        // string name
  int     lineNum;      
  int     lineNo[2];   // line no in line slot
  int     childNum[2];  
  unit*   child[2][4]; // children in slot
  void initCB(string n,int l0,int cn0)
  { type = CBt; name = n; lineNum = 1; lineNo[0] = l0; 
  lineNo[1] = -1; childNum[0] = cn0; childNum[1] = -1; }

  void initSD(string n,int l0,int l1,int cn0,int cn1)
  { type = SDt; name = n; lineNum = 2; lineNo[0] = l0; 
  lineNo[1] = l1; childNum[0] = cn0; childNum[1] = cn1; }
};


struct searchState {
  set< pair<unit*,int>  >  frontierUnits; // unit pointer + slot to expand
  set< pair<unit*,bool>  > doneUnits;     // unit pointer + openInTree 
  set<int>                 lines;         // lines in tree
  searchState() {}; 
  searchState(const searchState&); 
  searchState& operator=(const searchState&);
  void print();
};





searchState::searchState(const searchState& s) {
  frontierUnits = s.frontierUnits;
  doneUnits = s.doneUnits;
  lines = s.lines;
}


void searchState::print() {
  cout << " [F =";
  for (set< pair<unit*,int>  >::iterator fi = frontierUnits.begin();
	 fi != frontierUnits.end(); ++fi)
    cout << " (" << fi->first->name << "," << fi->second << ")";
  cout << " D =";
  for (set< pair<unit*,bool>  >::iterator di = doneUnits.begin();
	 di != doneUnits.end(); ++di)
    cout << " " << di->first->name;
  cout << "]";
}

  
searchState& searchState::operator=(const searchState& s) {
  frontierUnits = s.frontierUnits;
  doneUnits = s.doneUnits;
  lines = s.lines;
  return *this;
}





bool operator<(const searchState& s1, const searchState& s2)  {
  if (s1.frontierUnits < s2.frontierUnits) return true;
  else if (s1.frontierUnits == s2.frontierUnits && 
	   s1.doneUnits < s2.doneUnits) return true;
  else if (s1.frontierUnits == s2.frontierUnits && 
	   s1.doneUnits == s2.doneUnits && 
	   s1.lines < s2.lines) return true;
  return false;
}
  



///////////////////////////////////////////////////
//
// Globals
//
///////////////////////////////////////////////////

unit CB[8];
unit SD[27];


///////////////////////////////////////////////////
//
// Aux. funcs
//
///////////////////////////////////////////////////




void fixPoint(set<searchState>& open,set<searchState>& closed) {

  
  // debug
  int it = 0;

  while (open.size() > 0)
    {      
      it++;
      cout << "it =" << it << endl;
      /*
      //debug
      it++;
      cout << "it=" << it << " Open:\n"; 
      for (set<searchState>::iterator ssi = open.begin(); ssi != open.end(); ++ssi)
	{
	  searchState s = *ssi;
	  s.print();
	}
      cout << "\nClosed:\n"; 
      for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	{
	  searchState s = *ssi;
	  s.print();
	}
      cout << "\n\n\n";
      */


      // pick a search state to expand
      searchState state = (*open.begin());

      open.erase(state);
      closed.insert(state);
      
      // expand each frontier unit for new children 
      for (set< pair<unit*,int>  >::iterator spi = state.frontierUnits.begin();
	   spi != state.frontierUnits.end(); ++spi)
	{
	  pair<unit*,int> pp = *spi;
	  searchState  childState = state ;
      
	  unit* parent = pp.first;
	  int parentExpSlot = pp.second;
	  childState.frontierUnits.erase(pp);
	  
	  if (parent->type == CBt && parentExpSlot == -1)
	    {
	      // do nothing
	    }      
	  else
	    {      
	      // find out if expansion leads to cycle
	      bool cycle = false;
	      for (int i = 0; i < parent->childNum[parentExpSlot];i++)
		{
		  if (cycle) break;
		  for (set< pair<unit*,int>  >::iterator pi = childState.frontierUnits.begin();
		       pi != childState.frontierUnits.end(); ++pi)
		    if (pi->first == parent->child[parentExpSlot][i])
		      {
			cycle = true;
			break;
		      }
		  for (set< pair<unit*,bool>  >::iterator pi = childState.doneUnits.begin();
		       pi != childState.doneUnits.end(); ++pi)
		    if (pi->first == parent->child[parentExpSlot][i])
		      {
			cycle = true;
			break;
		      }
		}
	      
	      if (cycle)
		{
		  // do nothing
		}
	      else
		{
		  // add line to lines in tree
		  int lineNo = parent->lineNo[parentExpSlot];
		  childState.lines.insert(lineNo);
		  
		  for (int i = 0; i < parent->childNum[parentExpSlot]; i++)
		    {
		      unit* child = parent->child[parentExpSlot][i];
		      
		      // set expSlot
		      int childExpSlot;
			if (child->type == CBt)
			  childExpSlot = -1;
			else if (lineNo == child->lineNo[0])
			  childExpSlot = 1;
			else
			  childExpSlot = 0;
		      
		      // add to frontier
		      pair<unit*,int> childEntry(child,childExpSlot);
		      childState.frontierUnits.insert(childEntry);	      
		    }

		  pair<unit*,bool> parentEntry(parent,false); 
		  childState.doneUnits.insert(parentEntry);
		  open.insert(childState);
		}
	    }
	}
      
    }
}






///////////////////////////////////////////
//  CB transitions
//////////////////////////////////////////



void writeCloseCBTran(ofstream& out, int CBno,searchState state)
{
  out << "    mod: okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;
  out << "    pre: okCB" << CBno << " /\\ ~onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << endl;
  out << "    err: ~okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << "\n\n";
}





void writeOpenCBTran(ofstream& out, int CBno,searchState state) {

  out << "    mod: okCB" << CBno << ",onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okCB" << CBno << " /\\ onCB" << CBno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &CB[CBno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else	
	out << " /\\ on" << ui->first->name;	
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okCB" << CBno << "' /\\ ~onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << endl;

  out << "    err: ~okCB" << CBno << "' /\\ onCB" << CBno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << "\n\n";
}





///////////////////////////////////////////
//  SD transitions
//////////////////////////////////////////




void writeCloseSDTran(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ ~onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << " /\\ okL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui =  state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << endl;

  out << "    err: ~okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << "\n\n";
}



void writeOpenSDTran(ofstream& out, int SDno,searchState state,int dir)
{
  out << "    mod: okSD" << SDno << ",onSD" << SDno;
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << ",onL" << (*si);
  out << endl;

  out << "    pre: okSD" << SDno << " /\\ onSD" << SDno << " /\\ onL" << SD[SDno].lineNo[dir];
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si);
  for (set< pair<unit*,bool> >::iterator ui = state.doneUnits.begin(); ui != state.doneUnits.end(); ++ui)
    if (!(ui->first == &SD[SDno]))
      if (ui->second)
	out << " /\\ ~on" << ui->first->name;
      else
	out << " /\\ on" << ui->first->name;
  for (set< pair<unit*,int> >::iterator ui = state.frontierUnits.begin(); ui != state.frontierUnits.end(); ++ui)
      out << " /\\ ~on" << ui->first->name;
  out << endl;

  out << "    eff: okSD" << SDno << "' /\\ ~onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ ~onL" << (*si) << "'";
  out << endl;

  out << "    err: ~okSD" << SDno << "' /\\ onSD" << SDno << "'";
  for (set<int>::iterator si = state.lines.begin(); si != state.lines.end(); ++si)
    out << " /\\ onL" << (*si) << "'";
  out << "\n\n";
}





















///////////////////////////////////////////////////
//
// Main function
//
///////////////////////////////////////////////////
int main(int argc, char **argv) {
  
  // heuristic type
  int a = 2;
  int format = 0;

  char outName[512] = "PSR";
  char finalOutName[512];


  /*** Scan command line ***/
  for (int ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'a': 
	a = atoi(argv[++ind]);
	break;
      case 'o': 
	strcpy(outName,argv[++ind]);
	break;
      case 'f': 
	format = atoi(argv[++ind]);
	break;
      case 'h': 
	printusage();
	exit(1);
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }


  // define circuit breakers
  int CBnum = 7;
  CB[1].initCB("CB1",1,2);
  CB[2].initCB("CB2",12,1);
  CB[3].initCB("CB3",26,1);
  CB[4].initCB("CB4",22,1);
  CB[5].initCB("CB5",6,1);
  CB[6].initCB("CB6",8,1);
  CB[7].initCB("CB7",17,2);

  // define switching devices
  int SDnum = 26;
  SD[1].initSD("SD1",1,2,2,2);           
  SD[2].initSD("SD2",2,3,2,1);	   
  SD[3].initSD("SD3",3,4,1,1);	   
  SD[4].initSD("SD4",4,5,1,2);	   
  SD[5].initSD("SD5",6,5,1,2);	   
  SD[6].initSD("SD6",2,7,2,2);	   
  SD[7].initSD("SD7",7,8,2,1);	   
  SD[8].initSD("SD8",9,7,1,2);	   
  SD[9].initSD("SD9",10,9,2,1);	   
  SD[10].initSD("SD10",11,10,1,2);	   
  SD[11].initSD("SD11",12,11,1,1);	   
  SD[12].initSD("SD12",10,13,2,3);	   
  SD[13].initSD("SD13",13,16,3,2);	   
  SD[14].initSD("SD14",16,17,2,2);	   
  SD[15].initSD("SD15",19,16,1,2);	   
  SD[16].initSD("SD16",20,19,1,1);	   
  SD[17].initSD("SD17",21,20,2,1);	   
  SD[18].initSD("SD18",22,21,1,2);	   
  SD[19].initSD("SD19",21,23,2,1);	   
  SD[20].initSD("SD20",23,24,1,1);	   
  SD[21].initSD("SD21",24,5,1,2);	   
  SD[22].initSD("SD22",13,14,3,0);	   
  SD[23].initSD("SD23",25,13,1,3);	   
  SD[24].initSD("SD24",26,25,1,1);	   
  SD[25].initSD("SD25",18,17,0,2);	   
  SD[26].initSD("SD26",1,15,2,0);	   

  // init CBs
  CB[1].child[0][0] = &SD[1];
  CB[1].child[0][1] = &SD[26];

  CB[2].child[0][0] = &SD[11];

  CB[3].child[0][0] = &SD[24];

  CB[4].child[0][0] = &SD[18];

  CB[5].child[0][0] = &SD[5];

  CB[6].child[0][0] = &SD[7];

  CB[7].child[0][0] = &SD[25];
  CB[7].child[0][1] = &SD[14];

  // init SDs
  SD[1].child[0][0] = &CB[1];
  SD[1].child[0][1] = &SD[26];
  SD[1].child[1][0] = &SD[6];
  SD[1].child[1][1] = &SD[2];

  SD[2].child[0][0] = &SD[1];
  SD[2].child[0][1] = &SD[6];
  SD[2].child[1][0] = &SD[3];

  SD[3].child[0][0] = &SD[2];
  SD[3].child[1][0] = &SD[4];

  SD[4].child[0][0] = &SD[3];
  SD[4].child[1][0] = &SD[21];
  SD[4].child[1][1] = &SD[5];

  SD[5].child[0][0] = &CB[5];
  SD[5].child[1][0] = &SD[21];
  SD[5].child[1][1] = &SD[4];

  SD[6].child[0][0] = &SD[1];
  SD[6].child[0][1] = &SD[2];
  SD[6].child[1][0] = &SD[8];
  SD[6].child[1][1] = &SD[7];

  SD[7].child[0][0] = &SD[6];
  SD[7].child[0][1] = &SD[8];
  SD[7].child[1][0] = &CB[6];

  SD[8].child[0][0] = &SD[9];
  SD[8].child[1][0] = &SD[6];
  SD[8].child[1][1] = &SD[7];

  SD[9].child[0][0] = &SD[10];
  SD[9].child[0][1] = &SD[12];
  SD[9].child[1][0] = &SD[8];

  SD[10].child[0][0] = &SD[11];
  SD[10].child[1][0] = &SD[12];
  SD[10].child[1][1] = &SD[9];
	   
  SD[11].child[0][0] = &CB[2];
  SD[11].child[1][0] = &SD[10];

  SD[12].child[0][0] = &SD[10];
  SD[12].child[0][1] = &SD[9];
  SD[12].child[1][0] = &SD[23];
  SD[12].child[1][1] = &SD[22];
  SD[12].child[1][2] = &SD[13];

  SD[13].child[0][0] = &SD[12];
  SD[13].child[0][1] = &SD[22];
  SD[13].child[0][2] = &SD[23];
  SD[13].child[1][0] = &SD[15];
  SD[13].child[1][1] = &SD[14];

  SD[14].child[0][0] = &SD[15];
  SD[14].child[0][1] = &SD[13];
  SD[14].child[1][0] = &SD[25];
  SD[14].child[1][1] = &CB[7];

  SD[15].child[0][0] = &SD[16];
  SD[15].child[1][0] = &SD[13];
  SD[15].child[1][1] = &SD[14];

  SD[16].child[0][0] = &SD[17];
  SD[16].child[1][0] = &SD[15];

  SD[17].child[0][0] = &SD[18];
  SD[17].child[0][1] = &SD[19];
  SD[17].child[1][0] = &SD[16];

  SD[18].child[0][0] = &CB[4];
  SD[18].child[1][0] = &SD[19];
  SD[18].child[1][1] = &SD[17];
	   
  SD[19].child[0][0] = &SD[18];
  SD[19].child[0][1] = &SD[17];
  SD[19].child[1][0] = &SD[16];

  SD[20].child[0][0] = &SD[19];
  SD[20].child[1][0] = &SD[21];

  SD[21].child[0][0] = &SD[20];
  SD[21].child[1][0] = &SD[4];
  SD[21].child[1][1] = &SD[5];

  SD[22].child[0][0] = &SD[23];
  SD[22].child[0][1] = &SD[12];
  SD[22].child[0][2] = &SD[13];

  SD[23].child[0][0] = &SD[24];
  SD[23].child[1][0] = &SD[12];
  SD[23].child[1][1] = &SD[13];
  SD[23].child[1][2] = &SD[22];

  SD[24].child[0][0] = &CB[3];
  SD[24].child[1][0] = &SD[23];

  SD[25].child[1][0] = &SD[14];
  SD[25].child[1][1] = &CB[7];

  SD[26].child[0][0] = &CB[1];
  SD[26].child[0][1] = &SD[1];


  
  sprintf(finalOutName,"a%d%s.nadl",a,outName);      
  ofstream  out(finalOutName,ios::out);

  if (format == 0)
    {
      out << "\% File:   " << finalOutName << endl;
      out << "\% Desc:   PSR simple problem in format " << format << " with " << a << endl;
      out << "\%         failed lines. For details see Thiebaux & Cordier ECP-01\n";
      out << "\%         and Bertoli et al. ECAI-02\n"; 
      out << "\% Date:   02\n";
      out << "\% Auth:   Rune M. Jensen CS, CMU\n";
      
      out << "\nVARIABLES\n";
      

      // circuit breakers      
      for (int i = 1; i <= CBnum; i++) 
	out << "  bool   okCB" << i << ",onCB" << i << endl;

      // switching devices      
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okSD" << i << ",onSD" << i << endl;

      // lines
      for (int i = 1; i <= SDnum; i++) 
	out << "  bool   okL" << i << ",onL" << i <<  endl;


      
      out << "\n\nSYSTEM\n\n";
      
      out << " process:  sys\n\n";  


      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open circuit breaker\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= CBnum; i++)
	{
	  {
	    // init search State
	    searchState init;
	    
	    pair<unit*,int>  key(&CB[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);
	    
	    
	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeCB" << i << endl;	  		  
		writeCloseCBTran(out,i,*ssi);
		out << "  openCB" << i << endl;	  		  
		writeOpenCBTran(out,i,*ssi);
	      }
	  }
	}
	  
      
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n";
      out << "\% close/open switching device\n";
      out << "\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\n\n";
      for (int i = 1; i <= SDnum; i++)
	{
	  out << "  closeSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ ~onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: okSD" << i << "' /\\ onSD" << i << "'\n";
	  out << "    err: ~okSD" << i << "' /\\ ~onSD" << i << "'\n\n";


	  out << "  openSD" << i << endl;	  
	  
	  out << "\%   No feed to SD\n";
	  
	  out << "    mod: okSD" << i << ",onSD" << i << endl;
	  out << "    pre: okSD" << i << " /\\ onSD" << i
	      << " /\\ ~onL" << SD[i].lineNo[0] << " /\\ ~onL" << SD[i].lineNo[1] << endl; 
	  out << "    eff: okSD" << i << "' /\\ ~onSD" << i << "'\n";
	  out << "    err: ~okSD" << i << "' /\\ onSD" << i << "'\n\n";

	
	  {	    
	    out << "\%   Feed in positive direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],1);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTran(out,i,*ssi,0);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTran(out,i,*ssi,0);
	      }	    
	  }
	  
	  	  
	  {
	    out << "\%   Feed in negative direction\n";
	    searchState init;

	    pair<unit*,int>  key(&SD[i],0);
 	    init.frontierUnits.insert(key);
	    set<searchState> open;
	    set<searchState> closed;
	    open.insert(init);

	    // find fixed point
	    fixPoint(open,closed);
	    closed.erase(init);

	    for (set<searchState>::iterator ssi = closed.begin(); ssi != closed.end(); ++ssi)
	      {
		out << "  closeSD" << i << endl;	  		  
		writeCloseSDTran(out,i,*ssi,1);
		out << "  openSD" << i << endl;	  		  
		writeOpenSDTran(out,i,*ssi,1);
	      }	    
	  }
	  
	}

      out << "Initially\n";
      out << "  onL1 /\\ okL1 /\\\n";
      out << "  onL2 /\\ okL2 /\\\n";
      out << "  onL3 /\\ " << (a > 0 ? "~" : "") << "okL3 /\\\n";
      out << "  onL4 /\\ " << (a > 4 ? "~" : "") << "okL4 /\\\n";
      out << "  ~onL5 /\\ okL5 /\\\n";
      out << "  ~onL6 /\\ okL6 /\\\n";
      out << "  ~onL7 /\\ okL7 /\\\n";
      out << "  ~onL8 /\\ okL8 /\\\n";
      out << "  ~onL9 /\\ okL9 /\\\n";
      out << "  onL10 /\\ okL10 /\\\n";
      out << "  ~onL11 /\\ okL11 /\\\n";
      out << "  ~onL12 /\\ " << (a > 6 ? "~" : "") << "okL12 /\\\n";
      out << "  onL13 /\\ okL13 /\\\n";
      out << "  onL14 /\\ " << (a > 5 ? "~" : "") << "okL14 /\\\n";
      out << "  ~onL15 /\\ " << (a > 1 ? "~" : "") << "okL15 /\\\n";
      out << "  ~onL16 /\\ okL16 /\\\n";
      out << "  ~onL17 /\\ okL17 /\\\n";
      out << "  ~onL18 /\\ okL18 /\\\n";
      out << "  onL19 /\\ okL19 /\\\n";
      out << "  onL20 /\\ okL20 /\\\n";
      out << "  onL21 /\\ " << (a > 3 ? "~" : "") << "okL21 /\\\n";
      out << "  onL22 /\\ okL22 /\\\n";
      out << "  ~onL23 /\\ okL23 /\\\n";
      out << "  ~onL24 /\\ okL24 /\\\n";
      out << "  onL25 /\\ " << (a > 2 ? "~" : "") << "okL25 /\\\n";
      out << "  onL26 /\\ okL26 /\\\n";

      out << "  onSD1 /\\ okSD1 /\\\n";
      out << "  onSD2 /\\ okSD2 /\\\n";
      out << "  onSD3 /\\ okSD3 /\\\n";
      out << "  ~onSD4 /\\ okSD4 /\\\n";
      out << "  ~onSD5 /\\ okSD5 /\\\n";
      out << "  ~onSD6 /\\ okSD6 /\\\n";
      out << "  ~onSD7 /\\ okSD7 /\\\n";
      out << "  ~onSD8 /\\ okSD8 /\\\n";
      out << "  onSD9 /\\ okSD9 /\\\n";
      out << "  ~onSD10 /\\ okSD10 /\\\n";
      out << "  ~onSD11 /\\ okSD11 /\\\n";
      out << "  onSD12 /\\ okSD12 /\\\n";
      out << "  ~onSD13 /\\ okSD13 /\\\n";
      out << "  ~onSD14 /\\ okSD14 /\\\n";
      out << "  ~onSD15 /\\ okSD15 /\\\n";
      out << "  onSD16 /\\ okSD16 /\\\n";
      out << "  onSD17 /\\ okSD17 /\\\n";
      out << "  onSD18 /\\ okSD18 /\\\n";
      out << "  ~onSD19 /\\ okSD19 /\\\n";
      out << "  ~onSD20 /\\ okSD20 /\\\n";
      out << "  ~onSD21 /\\ okSD21 /\\\n";
      out << "  onSD22 /\\ okSD22 /\\\n";
      out << "  onSD23 /\\ okSD23 /\\\n";
      out << "  onSD24 /\\ okSD24 /\\\n";
      out << "  ~onSD25 /\\ okSD25 /\\\n";
      out << "  onSD26 /\\ okSD26 /\\\n";
      
      out << "  onCB1 /\\ okCB1 /\\\n";
      out << "  ~onCB2 /\\ okCB2 /\\\n";
      out << "  onCB3 /\\ okCB3 /\\\n";
      out << "  onCB4 /\\ okCB4 /\\\n";
      out << "  ~onCB5 /\\ okCB5 /\\\n";
      out << "  ~onCB6 /\\ okCB6 /\\\n";
      out << "  ~onCB7 /\\ okCB7\n\n";
      
      out << "Goal\n";
      for (int i = 1; i <= 26; i++)
	if (i < 26)
	  out << "  (onL" << i << " <=> okL" << i << ") /\\\n";
	else
	  out << "  (onL" << i << " <=> okL" << i << ")\n";      
    }
  return 0;
}
