/****************************************************** 
 * File  : main.cc   
 * Desc. : extNADL Generator of 8-15-24-35 puzzles for 
 *         BDDSearch version 0.6, for FAULTTOLERANT 
 *         Guided search (FT file) and a version for fault toleraint planning 
 *         via strong planning (FTS file)
 *         (including guiding heuristic)
 * 
 *       
 *         Positions and goal state
 *
 *       Y
 *
 *         ----------     
 *       2 |  |  |  |    
 *         | 1| 2| 3|    
 *         ----------   
 *       1 |  |  |  |    
 *         | 4| 5| 6|   
 *         ----------   
 *       0 |  |  |  |    
 *         | 7| 8|  |   
 *         ----------   
 *           0  1  2  X   
 *
 *
 *         This domain is generated for fault tolerant planning
 *         Thus any action may fail, in which case no move is made
 *
 * Author: Rune M. Jensen
 * Date  : 10/23/02
 ******************************************************/

#include <vector>
#include <set>
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


using namespace std; 


void  printusage() {
  printf("USAGE: Npuzzle [-hglos]\n");
  printf("       -h print usage                                   \n");
  printf("       -n NUM                                           \n");
  printf("          size of N-puzzle                              \n");
  printf("            8                                           \n");
  printf("           15 (default)                                 \n");
  printf("           24                                           \n");
  printf("           35                                           \n");
  printf("       -g NUM                                           \n");
  printf("          different heuristics                          \n");
  printf("          1: Sum of Manhattan distances (default)       \n");
  printf("       -l NUM                                           \n");
  printf("          nuumber of random permutations of goal state  \n");
  printf("          to make initial state (default 10)            \n");
  printf("       -s NUM                                           \n");
  printf("          seed for random generator                     \n");
  printf("          (default 130271)                              \n");
  printf("       -o FILE                                          \n");
  printf("          name of outfile (it will be prefixed with the \n");
  printf("          solution length and postpixed with [n]FT.nadl)\n");
  printf("          and [n]FTS.nadl)                              \n");
  printf("                                                        \n");
  printf("Creates a random NADL problem of the tile-puzzle with   \n");
  printf("the European goal state and the given solution bound.   \n");
  printf("H is for backward search an thus is the heuristic       \n");
  printf("distance to the initial state (NOT THE GOAL STATE)      \n");
} 


int myAbs(int v) {
  if (v < 0) 
    return -v;
  else
    return v;
}

//IN
// x,y    : board position 
// j      : tile number of tile occupying that position
// size   : size of puzzle (3,4,5)
//OUT
// the manhattan distance of the tile to it goal position
int manhattanDist(int x,int y,int xi,int yi) {
  
  return ( myAbs(x - xi) + myAbs(y - yi) );
} 


// compute delta h for various non-det outcomes
int dhUp(int x,int y,int t,int* initX,int* initY) {
  int hOld = manhattanDist(x,y+1,initX[t],initY[t]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]);
  return hNew - hOld;
}

int dhDown(int x,int y,int t,int* initX,int* initY) {
  int hOld = manhattanDist(x,y-1,initX[t],initY[t]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]);
  return hNew - hOld;
}


int dhRight(int x,int y,int t,int* initX,int* initY) {
  int hOld = manhattanDist(x+1,y,initX[t],initY[t]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]);
  return hNew - hOld;
}


int dhLeft(int x,int y,int t,int* initX,int* initY) {
  int hOld = manhattanDist(x-1,y,initX[t],initY[t]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]);
  return hNew - hOld;
}
  


int dhUpRight(int x,int y,int t,int tR,int* initX,int* initY) {
  int hOld = manhattanDist(x,y+1,initX[t],initY[t]) + manhattanDist(x+1,y+1,initX[tR],initY[tR]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]) + manhattanDist(x,y+1,initX[tR],initY[tR]);
  return hNew - hOld;
}



int dhUpLeft(int x,int y,int t,int tL,int* initX,int* initY) {
  int hOld = manhattanDist(x,y+1,initX[t],initY[t]) + manhattanDist(x-1,y+1,initX[tL],initY[tL]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]) + manhattanDist(x,y+1,initX[tL],initY[tL]);
  return hNew - hOld;
}

int dhDownRight(int x,int y,int t,int tR,int* initX,int* initY) {
  int hOld = manhattanDist(x,y-1,initX[t],initY[t]) + manhattanDist(x+1,y-1,initX[tR],initY[tR]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]) + manhattanDist(x,y-1,initX[tR],initY[tR]);
  return hNew - hOld;
}

int dhDownLeft(int x,int y,int t,int tL,int* initX,int* initY) {
  int hOld = manhattanDist(x,y-1,initX[t],initY[t]) + manhattanDist(x-1,y-1,initX[tL],initY[tL]);
  int hNew = manhattanDist(x,y,initX[t],initY[t]) + manhattanDist(x,y-1,initX[tL],initY[tL]);
  return hNew - hOld;
}



///////////////////////////////////////////////////
//
// Main function
//
///////////////////////////////////////////////////
int main(int argc, char **argv) {
  
  // heuristic type
  int n = 15;
  int size = 4;
  int heuristic = 1;
  int solutionLengthBound = 10;
  int seed = 130271;

  char outName[512] = "Puzzle";
  char finalOutName1[512];
  char finalOutName2[512];


  /*** Scan command line ***/
  for (int ind = 1; ind < argc; ind++) {
    
    /*** Parse switches ***/
    if (argv[ind][0] == '-') {
      switch (argv[ind][1]) {  
      case 'n': 
	n = atoi(argv[++ind]);
	size = int(sqrt(n+1));
	break;
      case 'g': 
	heuristic = atoi(argv[++ind]);
	break;
      case 'l': 
	solutionLengthBound = atoi(argv[++ind]);
	break;
      case 's': 
	seed = atoi(argv[++ind]);
	break;
      case 'o': 
	strcpy(outName,argv[++ind]);
	break;
      case 'h': 
	printusage();
	exit(1);
	break;
      default : printf("main.cc main : Unknown switch '%c'\nexiting\n", argv[ind][1]);
	exit(1);
	break;
      }
    }
  }


  // 1) Compute the initial state by a randomly moving the empty space 
  //    solutionLengthBound times
  
  // enter goal state
  int* goalX = new int[n+2];
  int* goalY = new int[n+2];

  // pos encodes init state
  int **pos = new int*[size];  
  for (int x = 0; x < size; x++)
    pos[x] =  new int[size];

  int tile = n + 1;
  for (int y = 0; y < size; y++)
    for (int x = size-1; x >= 0; x--)
      {
	pos[x][y] = tile;
	goalX[tile] = x;
	goalY[tile] = y;
	tile--;
      }

  pos[size - 1][0] = 0;
  goalX[0] = size - 1;
  goalY[0] = 0;

  int zeroTileX = size - 1;
  int zeroTileY = 0;
  int lastMove = 1;
  set<int> moves;

  srand(seed);

  // randomly move empty tile around
  for (int i = 0; i < solutionLengthBound; i++)
    {
      // 0 : up
      // 1 : down
      // 2 : right
      // 3 : left
      moves.clear();
      
      if (zeroTileY < size - 1)
	moves.insert(0);
      if (zeroTileY > 0)
	moves.insert(1);
      if (zeroTileX < size - 1)
	moves.insert(2);
      if (zeroTileX > 0)
	moves.insert(3);

      moves.erase(lastMove);
      
      // pick a random move
      int shuffle = rand() % moves.size();
      set<int>::iterator si = moves.begin();
      for (int s = 0; s < shuffle; s++)
	++si;
      
      int move = *si;

      lastMove = move;

      // perform state change
      switch (move) 
	{
	case 0: // up
	  pos[zeroTileX][zeroTileY] = pos[zeroTileX][zeroTileY+1];
	  pos[zeroTileX][zeroTileY+1] = 0; 
	  zeroTileY++;
	  break;
	case 1: // down
	  pos[zeroTileX][zeroTileY] = pos[zeroTileX][zeroTileY-1];
	  pos[zeroTileX][zeroTileY-1] = 0; 
	  zeroTileY--;
	  break;
	case 2: // right
	  pos[zeroTileX][zeroTileY] = pos[zeroTileX+1][zeroTileY];
	  pos[zeroTileX+1][zeroTileY] = 0; 
	  zeroTileX++;
	  break;
	case 3: // down
	  pos[zeroTileX][zeroTileY] = pos[zeroTileX-1][zeroTileY];
	  pos[zeroTileX-1][zeroTileY] = 0; 
	  zeroTileX--;
	  break;
	}
    }
  
  int* initX = new int[n + 1];
  int* initY = new int[n + 1];
  for (int x = 0; x < size; x++)
    for (int y = 0; y < size; y++)
      {
	initX[pos[x][y]] = x;
	initY[pos[x][y]] = y;
      }
    

  // find heuristic value of init
  // sum of Manhattan distances to the goal 
  int Hgoal = 0;
  for (int i = 1; i < n + 1; i++)
    {
      Hgoal += myAbs(goalX[i] - initX[i]);
      Hgoal += myAbs(goalY[i] - initY[i]);
    }
  
  sprintf(finalOutName1,"%d%s%dFTS.nadl",solutionLengthBound,outName,n);      
  sprintf(finalOutName2,"%d%s%dFT.nadl",solutionLengthBound,outName,n);      
  ofstream  out1(finalOutName1,ios::out);
  ofstream  out2(finalOutName2,ios::out);
 












  ////////////////////////////////////////////////
  // Writing FTS (fault tolerant via strong file)
  ////////////////////////////////////////////////
  
  out1 << "\% File:   " << finalOutName1 << endl;
  out1 << "\% Desc:   " << n << "-Puzzle problem with solution length upper bound of " 
      <<  solutionLengthBound << endl;
  out1 << "\%         Initial state:                                 \n"; 
  out1 << "\%     \n"; 
  for (int y = size - 1; y >= 0; y--)
    {
      out1 << "\%         ";
      for (int x = 0; x < size; x++)
	out1 << pos[x][y] << " ";
      out1 << endl;
    }
  out1 << "\%         \n"; 
  out1 << "\%         This encoding tracks the tile of each position\n"; 
  out1 << "\%         It's a fault tol. nondet. version where each action\n"; 
  out1 << "\%         may fail (in which case no action takes place)\n"; 
  out1 << "\% Date:   02\n";
  out1 << "\% Auth:   Rune M. Jensen CS, CMU\n";
  
  out1 << "\nVARIABLES\n";

  out1 << "  nat(1) e\n";
  out1 << "  nat(" << int(ceil(log(size*size)/log(2))) << ") ";
    for (int x = 0; x < size; x++)
      for (int y = 0; y < size; y++)
	if (x == 0 && y == 0)
	  out1 << "P" << x << y;
	else
	  out1 << ",P" << x << y;  
      
  
  
  out1 << "\n\nSYSTEM\n\n";
  
  // make all up moves
  out1 << " agt:  sys\n\n";  

  out1 << "  Up\n";
  // go through each position
  for (int x = 0; x < size; x++)
    for (int y = 0; y < (size - 1); y++)
      // go through each tile in turn
      for (int t = 1; t < (n + 1); t++)
	{
	  if (-dhUp(x,y,t,initX,initY) > -2)
	    {
	      // transitions with no horizontal effect
	      out1 << "     dg: 1\n";
	      out1 << "     dh: " << -dhUp(x,y,t,initX,initY) << "\n";
	      out1 << "    mod: e,P" << x << y << ",P" << x << y+1 << endl;
	      out1 << "    pre: P" << x << y << " = 0 /\\ P" << x << y+1 << " = " << t << endl; 
	      out1 << "    eff: e = 0 -> ( P" << x << y << "' = " << t << " /\\ P" << x << y+1 << "' = 0 /\\ e' = 0 \\/\n";
	      out1 << "         P" << x << y << "' = 0 /\\ P" << x << y+1 << "' = " << t << " /\\ e' = 1),\n";
	      out1 << "         P" << x << y << "' = " << t << " /\\ P" << x << y+1 << "' = 0 /\\ e' = 1 \n\n"; 
	    }
	}


   
  
  // make all down moves
  out1 << "  Down\n";    
  for (int x = 0; x < size; x++)
    for (int y = 1; y < size; y++)
      // go through each tile in turn
      for (int t = 1; t < n+1; t++)
	if (-dhDown(x,y,t,initX,initY) > -2)
	{
	  // transitions with no horizontal effect
	  out1 << "     dg: 1\n";
	  out1 << "     dh: " << -dhDown(x,y,t,initX,initY) << "\n";
	  out1 << "    mod: e,P" << x << y << ",P" << x << y-1 << endl;
	  out1 << "    pre: P" << x << y << " = 0 /\\ P" << x << y-1 << " = " << t << endl; 
	  out1 << "    eff: e = 0 -> ( P" << x << y << "' = " << t << " /\\ P" << x << y-1 << "' = 0 /\\ e' = 0 \\/\n";
	  out1 << "         P" << x << y << "' = 0 /\\ P" << x << y-1 << "' = " << t << " /\\ e' = 1 ),\n"; 
	  out1 << "         P" << x << y << "' = " << t << " /\\ P" << x << y-1 << "' = 0 /\\ e' = 1 \n\n";
	}  
	  
    

    
  // make all right moves
  out1 << "  Right\n";  
  for (int x = 0; x < size - 1; x++)
    for (int y = 0; y < size; y++)
      // go through each tile in turn
      for (int t = 1; t < n + 1; t++)
	{
	  out1 << "     dg: 1\n";
	  out1 << "     dh: " << -dhRight(x,y,t,initX,initY) << "\n";
	  out1 << "    mod: e,P" << x << y << ",P" << x+1 << y << endl;
	  out1 << "    pre: P" << x << y << " = 0 /\\ P" << x+1 << y << " = " << t << endl; 
          out1 << "    eff: e = 0 -> ( P" << x << y << "' = "<< t << " /\\ P" << x+1 << y << "' = 0 /\\ e' = 0 \\/\n";
	  out1 << "         P" << x << y << "' = 0 /\\ P" << x+1 << y << "' = " << t << " /\\ e' = 1 ),\n"; 
	  out1 << "         P" << x << y << "' = "<< t << " /\\ P" << x+1 << y << "' = 0 /\\ e' = 1 \n\n";
	}
  
  

  // make all left moves
  out1 << "  Left\n";  
  for (int x = 1; x < size; x++)
    for (int y = 0; y < size; y++)
      // go through each tile in turn
      for (int t = 1; t < n + 1; t++)
	{
	  out1 << "     dg: 1\n";	    
	  out1 << "     dh: " << -dhLeft(x,y,t,initX,initY) << "\n";
	  out1 << "    mod: e,P" << x << y << ",P" << x-1 << y << endl;
	  out1 << "    pre: P" << x << y << " = 0 /\\ P" << x-1 << y << " = " << t << endl; 
          out1 << "    eff: e = 0 -> ( P" << x << y << "' = "<< t << " /\\ P" << x-1 << y << "' = 0 /\\ e' = 0 \\/\n";
	  out1 << "         P" << x << y << "' = 0 /\\ P" << x-1 << y << "' = " << t << " /\\ e' = 1 ),\n"; 
	  out1 << "         P" << x << y << "' = "<< t << " /\\ P" << x-1 << y << "' = 0 /\\ e' = 1 \n\n";
	}
  

 
	  
  out1 << "\n\nINITIALLY\n";
  // give positions init tiling
  for (int i = 0; i < n + 1; i++)
    if (i == 0)
      out1 << " e = 0 /\\ P" << initX[i]  << initY[i] << " = " << i << endl;
    else
      out1 << " /\\ P" << initX[i]  << initY[i] << " = " << i << endl;
  
  
  out1 << "\n\nGOAL\n";
  // give positions goal tiling
  for (int i = 0; i < n + 1; i++)
    if (i == 0)
      out1 << " P" << goalX[i]  << goalY[i] << " = " << i << endl;
    else
      out1 << " /\\ P" << goalX[i]  << goalY[i] << " = " << i << endl;
  out1 << " heu: " << Hgoal << endl;





















  ////////////////////////////////////////////////
  // Writing FT (specialized fault tolerant alg.)
  ////////////////////////////////////////////////


  out2 << "\% File:   " << finalOutName2 << endl;
  out2 << "\% Desc:   " << n << "-Puzzle problem with solution length upper bound of " 
      <<  solutionLengthBound << endl;
  out2 << "\%         Initial state:                                 \n"; 
  out2 << "\%     \n"; 
  for (int y = size - 1; y >= 0; y--)
    {
      out2 << "\%         ";
      for (int x = 0; x < size; x++)
	out2 << pos[x][y] << " ";
      out2 << endl;
    }
  out2 << "\%         \n"; 
  out2 << "\%         This encoding tracks the tile of each position\n"; 
  out2 << "\%         It's a fault tol. nondet. version where each action\n"; 
  out2 << "\%         may fail (in which case no action takes place)\n"; 
  out2 << "\% Date:   02\n";
  out2 << "\% Auth:   Rune M. Jensen CS, CMU\n";
  
  out2 << "\nVARIABLES\n";
  
  out2 << "  nat(" << int(ceil(log(size*size)/log(2))) << ") ";
    for (int x = 0; x < size; x++)
      for (int y = 0; y < size; y++)
	if (x == 0 && y == 0)
	  out2 << "P" << x << y;
	else
	  out2 << ",P" << x << y;  
      
  
  
  out2 << "\n\nSYSTEM\n\n";
  
  // make all up moves
  out2 << " agt:  sys\n\n";  

  out2 << "  Up\n";
  // go through each position
  for (int x = 0; x < size; x++)
    for (int y = 0; y < (size - 1); y++)
      // go through each tile in turn
      for (int t = 1; t < (n + 1); t++)
	{
	  if (-dhUp(x,y,t,initX,initY) > -2)
	    {
	      // transitions with no horizontal effect
	      out2 << "     dg: 1\n";
	      out2 << "     dh: " << -dhUp(x,y,t,initX,initY) << "\n";
	      out2 << "    mod: P" << x << y << ",P" << x << y+1 << endl;
	      out2 << "    pre: P" << x << y << " = 0 /\\ P" << x << y+1 << " = " << t << endl; 
	      out2 << "    eff: P" << x << y << "' = " << t <<" /\\ P" << x << y+1 << "' = 0\n";
	      out2 << "    err: P" << x << y << "' = 0 /\\ P" << x << y+1 << "' = " << t << "\n\n"; 
	    }
	}


   
  
  // make all down moves
  out2 << "  Down\n";    
  for (int x = 0; x < size; x++)
    for (int y = 1; y < size; y++)
      // go through each tile in turn
      for (int t = 1; t < n+1; t++)
	if (-dhDown(x,y,t,initX,initY) > -2)
	{
	  // transitions with no horizontal effect
	  out2 << "     dg: 1\n";
	  out2 << "     dh: " << -dhDown(x,y,t,initX,initY) << "\n";
	  out2 << "    mod: P" << x << y << ",P" << x << y-1 << endl;
	  out2 << "    pre: P" << x << y << " = 0 /\\ P" << x << y-1 << " = " << t << endl; 
	  out2 << "    eff: P" << x << y << "' = " << t << " /\\ P" << x << y-1 << "' = 0\n";
	  out2 << "    err: P" << x << y << "' = 0 /\\ P" << x << y-1 << "' = " << t << "\n\n"; 
	}  
	  
    

    
  // make all right moves
  out2 << "  Right\n";  
  for (int x = 0; x < size - 1; x++)
    for (int y = 0; y < size; y++)
      // go through each tile in turn
      for (int t = 1; t < n + 1; t++)
	{
	  out2 << "     dg: 1\n";
	  out2 << "     dh: " << -dhRight(x,y,t,initX,initY) << "\n";
	  out2 << "    mod: P" << x << y << ",P" << x+1 << y << endl;
	  out2 << "    pre: P" << x << y << " = 0 /\\ P" << x+1 << y << " = " << t << endl; 
          out2 << "    eff: P" << x << y << "' = "<< t << " /\\ P" << x+1 << y << "' = 0\n";
	  out2 << "    err: P" << x << y << "' = 0 /\\ P" << x+1 << y << "' = " << t << "\n\n"; 
	}
  
  

  // make all left moves
  out2 << "  Left\n";  
  for (int x = 1; x < size; x++)
    for (int y = 0; y < size; y++)
      // go through each tile in turn
      for (int t = 1; t < n + 1; t++)
	{
	  out2 << "     dg: 1\n";	    
	  out2 << "     dh: " << -dhLeft(x,y,t,initX,initY) << "\n";
	  out2 << "    mod: P" << x << y << ",P" << x-1 << y << endl;
	  out2 << "    pre: P" << x << y << " = 0 /\\ P" << x-1 << y << " = " << t << endl; 
          out2 << "    eff: P" << x << y << "' = "<< t << " /\\ P" << x-1 << y << "' = 0\n";
	  out2 << "    err: P" << x << y << "' = 0 /\\ P" << x-1 << y << "' = " << t << "\n\n"; 
	}
  

 
	  
  out2 << "\n\nINITIALLY\n";
  // give positions init tiling
  for (int i = 0; i < n + 1; i++)
    if (i == 0)
      out2 << " P" << initX[i]  << initY[i] << " = " << i << endl;
    else
      out2 << " /\\ P" << initX[i]  << initY[i] << " = " << i << endl;
  
  
  out2 << "\n\nGOAL\n";
  // give positions goal tiling
  for (int i = 0; i < n + 1; i++)
    if (i == 0)
      out2 << " P" << goalX[i]  << goalY[i] << " = " << i << endl;
    else
      out2 << " /\\ P" << goalX[i]  << goalY[i] << " = " << i << endl;
  out2 << " heu: " << Hgoal << endl;






  
  return 0;
}
