/****************************************************** 
 * File  : main.cc   
 * Desc. : translation of placed ISPD98-IBM benchmarks
 *         from the labyrinth router
 *         http://er.cs.ucla.edu/~kastner/labyrinth/
 * Author: Rune M. Jensen
 * Date  : 8/20/01
 ******************************************************/

#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <string.h>
#include <vector>
#include <string.h>
#include <map>
#include <set>

using namespace std;

struct Pin {
  int x;
  int y;
  Pin(int xi, int yi) {x = xi; y = yi;}
};

struct Net {
  vector<Pin> pin;
  Net(vector<Pin> &pini) {pin = pini;}
};
  



// Usage
// translator net_input_file  x0 y0 colnum tracknum netnum seed channel_output_file
//   x0, y0   : lower left corner of channel
//   colnum   : number of columns in channel
//   tracknum : number of tracks in channel
//   netnum   : number of nets in channel chosen randomly
//   seed     : seed for random pick of nets 
int main(int argc, char *argv[]) {

  if (argc < 9)
    {
      cout << "Usage\n";
      cout << " translator net_input_file x0 y0 colnum tracknum netnum channel_output_file\n";
      cout << "  x0, y0   : lower left corner of channel\n";
      cout << "  colnum   : number of columns in channel\n";
      cout << "  tracknum : number of tracks in channel\n";
      cout << "  netnum   : number of nets in channel chosen randomly\n";
      cout << "  seed     : seed for random number generator\n";
      return -1;
    }

  ifstream input;

  input.open(argv[1], ios::in);
  if(!input)
    {
      cout << argv[1] << " could not be opened.  Exiting" << endl;
      return -1;
    }
  
  //Read in the row and columns.  grid column row
  int x,y,numNets,verticalCapacity,horizontalCapacity;
  char buffer[80];
  input.getline(buffer, 80);
  sscanf(buffer, "grid %d %d", &x, &y);
  cout << "Grid size " << x << " x " << y << endl;
  input.getline(buffer, 80);
  sscanf(buffer, "vertical capacity %d",&verticalCapacity);
  cout << "vertical capacity " << verticalCapacity << endl;
  input.getline(buffer, 80);
  sscanf(buffer, "horizontal capacity %d",&horizontalCapacity);
  cout << "horizontal capacity " << horizontalCapacity << endl;

  
  //Read in num nets and process nets and their pins
  input.getline(buffer, 80);
  sscanf(buffer, "num net %d", &numNets);
  vector<Net> net;
  for(int i = 0; i < numNets; i++)
    {
      string netName;
      int netID,numPins;
      input >> netName;
      input.getline(buffer, 80);
      sscanf(buffer, "%d %d", &netID, &numPins);
      vector<Pin> pin;
      pin.clear();
      for(int j = 0; j < numPins; j++)
	{
	  int pinX, pinY;
	  input.getline(buffer, 80);
	  sscanf(buffer, "%d %d", &pinX, &pinY);
	  if(pinX < 0 || pinX >= x || pinY < -1 || pinY >= y)
	      cout << "removing invalid pin " << pinX << "," << pinY << endl;
	  else
	      pin.push_back(Pin(pinX, pinY));
	}
      net.push_back(pin);
    }
  cout << "done getting input" << endl; 
  cout << net.size() << " initial nets" << endl;

  // read channel data
  int x0 = atoi(argv[2]);
  int y0 = atoi(argv[3]);
  int colnum = atoi(argv[4]);
  int tracknum = atoi(argv[5]);
  int netnum = atoi(argv[6]);
  int seed = atoi(argv[7]);

  cout << "Extracting channel with lower left corner [" << x0 << "," << y0 << "]\n";
  cout << "colnum=" << colnum << " tracknum=" << tracknum << " netnum=" << netnum << endl;

  int xMin = x0;
  int xMax = x0 + colnum - 1;
  int yMin = y0;
  int yMax = y0 + tracknum - 1;


  // put relevant nets into map;
  map< pair<int,int> , int > netmap;

  // find nets that overlap with channel region
  set<int> netInChannel;
  for (int i = 0; i < numNets; i++)
    {
      int pinsInChannel = 0;
      for (int j = 0; j < net[i].pin.size(); j++)
	if (  net[i].pin[j].x >= xMin && net[i].pin[j].x <= xMax &&
	      net[i].pin[j].y >= yMin && net[i].pin[j].y <= yMax )
	  pinsInChannel++;
      if (pinsInChannel > 1)
	netInChannel.insert(i);
    }

  // pick netnum nets from the overlapping set
  // take only nets not with pins not overlapping any other net
  // chosen for the channel
  int idPicked = 0;
  srand(seed);

  while (idPicked < netnum)
    {
      // pick a random net
      int elem = rand() % netInChannel.size();
      set<int>::iterator n = netInChannel.begin();
      for (int i = 0; i < elem; i++)
	++n;
      
      // check if net has pins already in the netmap if so
      // discard net
      bool discard = false;
      for (int j = 0; j < net[*n].pin.size(); j++)
	{
	  pair<int,int> key(net[*n].pin[j].x - x0,net[*n].pin[j].y - y0);
	  if (netmap.count(key)) discard = true;
	}
      
      if (discard)
	netInChannel.erase(*n);
      else
	{
	  // insert net	  
	  idPicked++;
	  cout << "Adding pins from net " << *n << " (net " << idPicked << " in channel)\n";	  
	  for (int j = 0; j < net[*n].pin.size(); j++)
	    {
	      if (  net[*n].pin[j].x >= xMin && net[*n].pin[j].x <= xMax &&
		    net[*n].pin[j].y >= yMin && net[*n].pin[j].y <= yMax )
		{
		  
		  // add pin to channel
		  pair<int,int> key(net[*n].pin[j].x - x0,net[*n].pin[j].y - y0);
		  cout << "  added pin " << j << " [" << net[*n].pin[j].x << "," 
		       << net[*n].pin[j].y << "] (in channel [" << key.first << "," 
		       << key.second << "])\n";
		  netmap[key] = idPicked;
		}
	    }
	  // remove net from netInChannel set
	netInChannel.erase(*n);	  
	}
    }


  // write channel file
  ofstream output;

  output.open(argv[8], ios::out);
  if(!output)
    {
      cout << argv[8] << " could not be opened.  Exiting" << endl;
      return -1;
    }
  
  output << colnum << " " << tracknum << " " << netnum << "\n\n";
  cout << colnum << " " << tracknum << " " << netnum << "\n\n";
  
  for (int i = 0; i < colnum; i++)
    {
      for (int j = 0; j < tracknum; j++)
	{
	  pair<int,int> key(i,j);
	  if (netmap.count(key))
	    {
	      output << " " << netmap[key];
	      cout << " " << netmap[key];
	    }
	  else
	    {
	      output << " 0";
	      cout << " 0";
	    }
	}
      output << endl;
      cout << endl;
    }
  return 0;
}
 





