function y = c2f (left, right, window_size, attention, max_disp, low_res_width, pixel, shrink, scale)

% C2f -- Compute stereo disparity using a simple coarse to fine normalized SAD
% algorithm.  Disparity is defined as LEFT-RIGHT, only positive disparities
% will be checked.  A bisymmetric window (truncated at the available image area
% and the input window_size) is used for all comparisons.
%
%	left  -- Left input image.  Must have same dimensions as:
%	right -- Right input image (same size as left image)
%	window_size -- size of the SSD window [5]
%	attention -- Number of the row on which to focus attention [0] (none)
%	max_disp -- Maximum integer disparity at each level [4*window_size]
%	low_res_width -- Width of the smallest image [4*window_size]
%	pixel_col -- Generate profiles for pixel (attention,pixel_col) [0]
%	shrink -- Filter to decimate the input image ['halve']
%	scale [internal parameter] -- Used to save disparity space
%
%  Make sure you type:   global SPACE SCALES  on the command line if you want
%  to focus attention on the disparity space of a particular row

global SPACE SCALES	%%%  These will hold either the Disparity Space for
			%%%  the row given in    attention,   or the evolution
			%%%  of the evaluation functino at   pixel   on that
			%%%  row.

if nargin < 2,
	error ('You must supply a left and right image');
end;
[  vlen  hlen ] = size (left);
[ rvlen rhlen ] = size (right);
if (vlen ~= rvlen | hlen ~= rhlen),
	error ('Left and right images must have the same dimensions!');
end,
if nargin < 3,
	window_size = 5;
end;
if nargin < 4,
	attention = 0;
end;
if nargin < 5,
	max_disp = window_size * 4;
end;
if nargin < 6,
	low_res_width = window_size * 4;
end;
if nargin < 7,
	pixel = 0;
end;
if nargin < 8,
	shrink = 'halve';
end;
if nargin < 9,
	scale = 1;
end;

if (pixel > 0),
    if (attention < 1),
	error ('c2f:  Need to specify attention ROW as well as pixel COL');
    end;
end;

%%%
%%%  If we're saving results at a certain row, make sure we allocate enough
%%%  storage for the disparity space
%%%

if (attention > 0 & scale == 1),
  total_scales = floor(log(hlen/low_res_width)/log(2));
  if (pixel < 1),
    total_disps = max_disp * 2^total_scales;
    ['Allocating global SPACE and SCALES matrices with ' ...
	 num2str(total_disps) ' integer disparities']
    SPACE = zeros (total_disps, hlen);
    SCALES = zeros (total_disps, hlen);
  else
    SPACE = zeros (total_scales * max_disp, total_scales);
    SCALES = zeros(total_scales * max_disp, total_scales);
  end;
end;

%%%
%%% Find the answer at the lowest resolution
%%%

if (vlen / 2 > low_res_width | hlen / 2 > low_res_width),

%%%  Keep attention properly focused at lower resolutions

	newatt = attention;
	newpix = pixel;
	if vlen > 1,
	    newatt = ceil (attention / 2);
	end;
	if hlen > 1,
	    newpix = ceil (pixel / 2);
	end;

%%%  Halve   will generate images that have ceil(length/2) length and height

	prior = c2f (feval (shrink, left), feval (shrink, right), ...
		window_size, newatt, max_disp, low_res_width, newpix, ...
		shrink, scale*2);
else

%%%  We're at the lowest resolution; initialize the disparities to point
%%%  to the middle of the window at the next lower resolution

	prior = floor (max_disp/4+.5) * ones (ceil(vlen/2), ceil(hlen/2));
end;

tic				%%%  Turn on the timer for this scale

%%%
%%%  Display the current image size and focus of attention
%%%

if attention > 0,
  if (pixel < 1),
    ['Running on ' num2str(vlen) 'x' num2str(hlen)  ...
	' image (only looking at row ' num2str(attention) ')']
  else
    ['Running on ' num2str(vlen) 'x' num2str(hlen)  ...
	' image (only looking at pixel (' num2str(attention) ',' ...
	num2str(pixel) ')']
  end;
else
  ['Running on ' num2str(vlen) 'x' num2str(hlen) ' image']
end;

vwin = window_size;		%%%  These will grow or shrink as required
hwin = window_size;		%%%  by the available image area

%%%  Allocate storage for results at this level.

y=zeros(vlen,hlen);

%%%  We assume the images came from cameras that have parallel optical axes,
%%%  or that the images have been warped so that corresponding pixels are
%%%  guaranteed to lie on same-numbered rows, with pixels in the right image
%%%  always lying to the LEFT of their correspondents in the left image.
%%%  Don't bother searching over adjacent rows.

%%%  Iterate over the parts of the left image that include the complete
%%%  correlation window.   row   will be the BOTTOM of the window area,
%%%   col   will be the LEFT of the window area.

rowstart = 1;
rowend = vlen;
if (attention > 0),
  rowstart = attention;
  rowend = rowstart;
end;

colstart = 1;
colend = hlen;
if (pixel > 0),
  colstart = pixel;
  colend = pixel;
end;

%%%  Centerrow will point to the center row of the current window.  Compute the
%%%  appropriate bounds for a symmetric window, given the desired window height
%%%  and the dimensions of the image.

for centerrow = rowstart:rowend,
  this_vwin = min ([2 * centerrow - 1, vwin, 2 * (vlen - centerrow) + 1]);
  row = centerrow - floor(this_vwin/2);

  for centercol = colstart:colend;
    this_hwin = min ([2 * centercol - 1, hwin, 2 * (hlen - centercol) + 1]);
    col = centercol - floor (this_hwin/2);

%%%  Now   row and col   point to the lower border of the rectangular window
%%%  in the left image, which is   this_vwin   tall and   this_hwin   wide.
%%%  This window is guaranteed to fit within the actual images.

%%%  Use the results of the coarser level to restrict this search.  Offset
%%%  will point to the LEFT edge of the window area specified from the earlier
%%%  level.  This is *not* truncated at image borders (yet).

    pr = prior(ceil(centerrow/2),ceil(centercol/2))*2;
    offset = col - pr - floor(max_disp/2);

%%%  Trim the bounds to fit in the current image, on the correct side of the
%%%  current pixel.  Remember that both   rightmost and leftmost   point to
%%%  the *lower bottom corner* of the window candidates, *not* the center.

    rightmost = min (col, offset + max_disp-1);
    leftmost  = max (1, offset);

    if rightmost >= leftmost,

%%%  Create storage for the error terms at this pixel

      reslen = rightmost-(leftmost-1);
      res = zeros(reslen,1);

%%%  Check all the candidates.  Put the rightmost one (i.e., minimum
%%%  disparity) first, so that the minimum index selected below is biased
%%%  toward small disparities (given a list of equals, matlab's "min"
%%%  returns the lowest-numbered index).

      j = 1;

      for disp = rightmost:-1:leftmost,

%%%  Although we've computed the appropriate window size for the left image,
%%%  it's possible this disparity will take us out of bounds in the right
%%%  image.  Compensate.

	right_hwin = min ([2 * disp - 1, this_hwin, 2 * (hlen - disp) + 1]);
	loff = this_hwin-right_hwin;

	res(j) = sum(sum(abs( ...
		   left(row:row+this_vwin-1,col+loff:col+loff+right_hwin-1) ...
		- right(row:row+this_vwin-1,disp:disp+right_hwin-1)))) / ...
		(this_vwin * right_hwin);

	if (attention > 0),
	 thisdisp = col-(rightmost-(j-1));
	 if (pixel < 1),

%%%  Save all results into disparity space

	  SPACE(thisdisp*scale+1:(thisdisp+1)*scale, ...
		(centercol-1)*scale+1:centercol*scale) = res(j) * ...
		ones (scale,scale);
	  SCALES(thisdisp*scale+1:(thisdisp+1)*scale, ...
		(centercol-1)*scale+1:centercol*scale) = scale * ...
		ones (scale,scale);
	 else
	  SPACE(thisdisp*scale+1:(thisdisp+1)*scale, log(scale)/log(2)+1) ...
		= res(j) * ones (scale,1);
	  SCALES(thisdisp*scale+1:(thisdisp+1)*scale, log(scale)/log(2)+1) ...
		= scale * ones(scale,1);
	 end;
	end;
	
	j = j+1;
      end;

%%%  Pick the best one.  If several error terms are the same, pick the index
%%%  closest to the guess from the coarser level.  centercol-pr  is the
%%%  column number of the prior guess; subtract   leftmost  to get the
%%%  corresponding index number for res().

%%%  How?  Make a list prefering indices near the prior result:
%%%		abs( [1 2 3 ... reslen] - index(pr) )
%%%  This will be zero at the prior index.  Now mask out all the candidates
%%%  with more than minimum error by adding   reslen  to them:
%%%		reslen * (res > val)

      val = min(res);
      [val2 index] = min (reslen * (res > val) + abs( [1:reslen]' - ...
			      ((max(centercol-pr,leftmost)-leftmost)+1)));
      y(centerrow,centercol) = col-(rightmost-(index-1));

    end;
  end;
end;
toc
if (attention > 0 & scale == 1),
  if (pixel < 1),

%%%  Up to now, coarse scales have had higher numbers than fine scales.  The
%%%  next line flips that around.

    SCALES = (max(max(SCALES)) + 1 - SCALES) .* (SCALES > 0);
  else
    [a b] = size (SPACE);

%%%  Shift evaluation functions to enforce smaller value at finer scales

    for i=1:b-1,

%%%  m gets the fine MAX minus the coarse MIN

      m = max (SPACE(:,i)) - ...
	  min ((SPACE(:,i+1) == 0) .* max(SPACE(:,i+1)) + SPACE(:,i+1));
      SPACE(:,i+1) = (SPACE(:,i+1) > 0) .* (SPACE(:,i+1) + m);
    end;

%%%  Flow the coarse results over into fine columns

    for i=b-1:-1:1,
      SCALES(:,i)= (SCALES(:,i) == 0) .* SCALES(:,i+1) + SCALES(:,i);
      SPACE(:,i) = min (((SPACE(:,i) == 0) .* SPACE(:,i+1)) + SPACE(:,i), ...
			SPACE(:,i+1));
    end;
    mi = min(min(SPACE));
    ma = max(max(SPACE)) * 1.1;
    for i=1:b,
      subplot(b,1,1+b-i);
      plot([0:(a-1)],SPACE(:,i));

%%%  Adds dots to show the granularity of the coarse the fine search

      hold on;
      col = 1;
      while col <= a,
	width = SCALES(col,i);
	if width == 2^(i-1),

%%%  For an evolutionary plot that includes dots from all the coarser levels,
%%%  use these four lines instead of "if width == 2^(i-1)," :
%	if width >= 2^(i-1),
%	  if rem(col, width) ~= 1 & width > 1,
%	    width = width / 2;
%	  end;

	  index = floor((col+width-1) / width) * width - width/2;
	  plot (index-.5, SPACE(index,i),'o');
	  col = col + width;
	else
	  col = col + 1;
	end;
      end;
      hold off;

      ylabel('Error');
      if (i==1),
	xlabel ('Disparity');
      end;
      set(gca,'YLim', [0 ma]);
      set(gca,'YTick', []);
    end;
  end;
end;
end;

% This software provided by Mark Maimone (mwm@cmu.edu) as part of his Ph.D.
% thesis.  For more info, please see http://www.cs.cmu.edu/~mwm/thesis
%
% Release Date: 9 October 1997
%
% 	Carnegie Mellon University and the individuals associated with this
% project assume no liability for the fitness or performance of this software
% or any damages resulting from its use.  It's free, you get what you pay
% for...
