/* voyVAPI.c -- support for Voyetra's VAPI driver */

/* Copyright 1993 Intelligistics, Inc. */

/*****************************************************************************
* Change Log
*   Date    | who : Change
*-----------+-----------------------------------------------------------------
* 29-Jul-93 | GWL :
*****************************************************************************/

/*
	voyVAPI.c replaces Leigh Smith's mpu.c but retains its
	communications protocol via midifns.c.  Output is
	straightforward, using Voyetra's CAPI-like protocol, VAPI.
	VAPI consists of a resident driver, e.g., VAPImpu.com or
	VAPIvp11.com, with Voyetra's cVAPIl.obj (for large model).

	The idea of input is to simulate an MPU-401 in uart mode,
	by modifying MPUisr to service data from VAPI.  This
	hack works adequately in spite of the redundant buffers
	but does use Voyetra's interrupt structure.

	Code adapted from Voyetra's thru.c

	externals satisfied:

	mOpen()			set up interrupt service
	mClose()		undo the above
	mPutData()		send byte
	mPutCmd()		send command (no distinction to MIDIator)
	xbuff			external sysex buffer, CMT standard
	xbufhead
	xbuftail
	xbufmask
	buff			MIDI data buffer
	buffhead
	bufftail
	midi_flush()		empty out buffers
	check_midi()		check for input data

	Use option -trace to see the progress of linkage.

	Code here includes Roger's new MIDI filter.
*/

#include "switches.h"
#include <stdlib.h>     /* for ANSI malloc */
#include "cext.h"
#include <stdio.h>      /* for NULL */
#include <signal.h>     /* for ^C trapping */
#include <dos.h>        /* for FP_SEG, FP_OFF */
#include <bios.h>       /* for GETTIMEOFDAY */
#include "mpu.h"        /* for MPU-401 constants */
#include "midibuff.h"
#include "midicode.h"
#include "userio.h"	/* for abort_flag */
#include "midierr.h"	/* error flags */
#include <conio.h>
#include "voyVAPI.h" 	/* for constants */
#include "VAPI.h"	/* Voyetra's VAPI.h */

#ifdef MICROSOFT	/* for Microsoft C V5.1 and above */
#define DISABLEINTS     _disable
#define ENABLEINTS      _enable
#define INTERRUPT	interrupt far
#define GETVECTOR       _dos_getvect
#define SETVECTOR       _dos_setvect
#define GETTIMEOFDAY(a) _bios_timeofday(_TIME_GETCLOCK, &(a))
#else           	/* for Turbo C V2.0 and above */
#define DISABLEINTS     disable
#define ENABLEINTS      enable
#define INTERRUPT       interrupt
#define GETVECTOR       getvect
#define SETVECTOR       setvect
#define GETTIMEOFDAY(a) (a = biostime(0, 0))
#endif

#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

#define DBG_RECTIME     0x7d4	/* record time assign debug */
#define DBG_ISRBYTE     0x7d6	/* where to dump the ISR byte debug */
#define DBG_UNH_SYS     0x7d8	/* unhandled sys-ex debug */
#define DBG_RETDATA	0x7da	/* returned data debug */
#define DBG_CLK2HST	0x7dc	/* clock to host called debug */

#define debug(a, b, c)

#define MPU_ACK		0xFE
#define	IRQ9MASK	(2)

static void MPUisr(unsigned);	/* MPUisr called with MIDIator data */
static void (*ClockToHostFn)(void);	/* Ptr to fn called each MPU Clock */
static int  PlayFini = FALSE;	/* TRUE when the play data is at the end */
static int  UartMode = TRUE;	/* TRUE when MPU is in UART mode */
static int  CmdAck;		/* MPU-401 command acknowledge flag */
static int  ExpectingDataAfterCmd; /* Flag indicating MPU will return a byte */
static byte ReturnedData;	/* The byte MPU has returned */ 
static unsigned long RecordTime; /* No of MPU ticks since mRecord called */
static int  verbose=FALSE;	/* TRUE = show where MIDIator is */

byte huge *xbuff = NULL;	/* application-supplied sysex buffer */
long xbufmask;			/* mask for circular buffer */
long xbufhead = 0;      	/* buffer head and tail offsets */
long xbuftail = 0;
static long old_xbuftail;

boolean xbuf_flush = false;	/* says to flush remainder of sysex message */
MidiMsg SysexHeader;	/* saves 1st 3 bytes of sysex messages for buffer */
byte SysexCounter = 0;	/* index into SysexHeader */

/* midi input buffer */
/* data buffer, declared long to get 32-bit alignment: */
long buff[BUFF_SIZE/4]; 
int buffhead = 0;    		/* buffer head and tail pointers */
int bufftail = 0;

private MidiMsg CurrentMsg, RealTimeMsg;
private int data_len;
private int data_rcv;
private boolean firstbyte;

extern boolean ctrlFilter;
extern boolean exclFilter;
extern boolean realFilter;

extern int musictrace;

#define set_error(bit) midi_error_flags |= (1 << bit);

/*	VAPI data areas */

int far VPdata_trap(void);
int	VPin_port=0;		/* MIDI input port, here only one */
int	VPout_port=0;		/* MIDI port, here only one */
int	VP_counter=1;		/* single element array for the timer */

#define MIDI_BUF_SIZE	64
int	VPin_buf[MIDI_BUF_SIZE];
int	VPout_buf[MIDI_BUF_SIZE];

#define VAPI_STACK_SIZE 256
char	VP_stack[VAPI_STACK_SIZE];

/***************
* LOCAL TABLES *
***************/

private byte VoiceTable[8] = {2,2,2,2,1,1,2,0};
/* note: length entry for EOX MUST BE ZERO: */
private byte CommonTable[8] = {0,1,2,1,0,0,0,0};

/* midi_flush -- empty out buffers */
/**/
void midi_flush()
{
    buffhead = 0;
    bufftail = 0;
    xbufhead = 0;
    xbuftail = 0;
    xbuf_flush = true;	/* in case sysex continuation messages are still coming */
}

/*
 * #define dbmax 32
 * int dbb[dbmax];
 * char * dbs[dbmax];
 * int dbx = 0;
 * int dbh = 0;
 */



/*
    Get all available data from VAPI
*/
boolean check_midi(void)
{
    int	inword;

    if (buffhead == bufftail) return false;
    else return true;
}

/*
    Put out data to VAPI
*/
void mPutData(byte dat)
{
    v1_putw(VPout_port, dat);
}


/*
    Commands go to VAPI in same way as data, no acks
    However, process as if an MPU in uart mode
*/

void mPutCmd(byte cmd)
{
    v1_putw(VPout_port, cmd);
}


/* PutMidiMsg -- copy complete message from interrupt to buffer */
/**/
static void PutMidiMsg(MidiMsg msg)
{
    register byte *cmt_data = ((byte *) buff) + bufftail;

    /* filter out control changes */
    if (ctrlFilter) {
	register int data1 = databyte1(msg);
	register int hibits = statusbyte(msg) & 0xF0;
	if (hibits == 0xD0 ||   /* Chan Pressure */
	    hibits == 0xE0 ||       /* Pitch Bend */
	    hibits == 0xA0 ||       /* Poly Pressure */
	    ((hibits == 0xB0) &&    /* Control change (don't count switches) */
	     ((data1 < 64) || (data1 > 121)))) {
	    /* CONTROL MESSAGE HAS BEEN FILTERED */
	    return;
	}
    }
    if (realFilter) {
	if (statusbyte(msg) >= 0xF8) return;
    }
    if (exclFilter) {
	if (statusbyte(msg) == MIDI_SYSEX) return;
    }
    *((long *) cmt_data) = msg.l;
    bufftail = (bufftail + 4) & BUFF_MASK;
    if (bufftail == buffhead) {
	/* filled buffer faster than client emptied it */
	set_error(BUFFOVFL);
    }
}


/*
** Function to close down VAPI interrupt service, return FALSE if
** unable to close routine, if signal complains. If err is non-zero,
** print the corresponding error message returned by mOpen.
*/
int mClose(int err)
{
    int i;
    int	vret=0;
    static char *errmsgs[] = {	/* ensure error codes align with the msgs */
	"Can\'t set up an abnormal end trap\n",
	"Unable to open link to VAPI\n",
	"Unable to close link to VAPI\n",
	"Unable to delay closing\n"
    };
	
    if (!err) {

	/* shut off the output */
	vret=v1_txclose(VPout_port);
	if (vret) {
	    gprintf(ERROR, "VAPI Tx Close code -> %d\n", vret);
	    err=VAPI_BAD_CLOSE;
	}

	/* remove the trap for rx */
	vret=v1_trap(VN_Rx0 + VPin_port, 0, 1, 0L);
	if (vret) {
	    gprintf(ERROR, "VAPI close Rx trap code -> %d\n", vret);
	    err=VAPI_BAD_CLOSE;
	}

	/* shut off the input */
	vret=v1_rxclose(VPin_port);
	if (vret) {
	    gprintf(ERROR, "VAPI Rx Close code -> %d\n", vret);
	    err=VAPI_BAD_CLOSE;
	}

	if (!err) {
	    /* delay timer -- variable is one element array */
	    vret = v1_tmaopen(&VP_counter, 1);
	    if (vret) {
		gprintf(ERROR, "VAPI Timer A open ret -> %d\n", vret);
		err=VAPI_BAD_TIMER;
	    }

	    /* start timer A with a 10 ms period */
	    /* (units are 2us.) */
	    vret = v1_tmastart(10 * 500);
	    if (vret) {
		gprintf(ERROR, "VAPI Timer A open ret -> %d\n", vret);
		err=VAPI_BAD_TIMER;
	    }

	    /* wait for 100 msec to make sure all data sent */
	    if (!err)
		for (VP_counter=10; VP_counter > 0; );

	    /* close */
	    if (vret = v1_close()) {
		gprintf(ERROR, "VAPI close code -> %d\n", vret);
		err=VAPI_BAD_CLOSE;
	    }
	}
    }

    if (verbose) gprintf(TRANS,"VAPI Terminating\n");

    if(err > 0 && err <= (sizeof(errmsgs) / sizeof(char *)))
	gprintf(ERROR, "%s", errmsgs[err - 1]);
    else if (err)
	gprintf(ERROR, "mClose err %d\n",err);
    if(signal(SIGINT, SIG_DFL) == SIG_ERR)
	return(FALSE);
    if (err) return FALSE;
    return TRUE;
}


/*
** If a Control-C or Abort interrupt occurs, exit gracefully,
** disabling the interrupt if it had been assigned.
*/
static void AbEnd(int i)
{
    abort_flag = true;
}


/*
** Routine to prime the internal variables and attempt to establish
** communications with the VAPI.
** Returns an error number for the different error messages if unable to
** communicate with the VAPI, or 0 if everything went ok.
** BaseAddr and IrqNo are holdover from mpu, ignored.
*/
int mOpen(unsigned int BaseAddr, int IrqNo)
{
    VXPI_STATS	driver_stats;	/* parameters re VAPI.com */
    char far *	drv_str;	/* pointer to driver i.d. */
    int		vret=0;		/* return code */

    verbose=cl_switch("trace");

    setstatus(CurrentMsg, 0);
    firstbyte = TRUE;

    /* SIGINT handler is set up by USERIO.C.  It causes a cu_cleanup
     * that in turn calls mClose, so we shouldn't override the handler
     * here.  On the other hand, I don't set up a handler for SIGABRT
     * elsewhere, so I'm leaving in this handler.  Depending on how
     * and when SIGABRT can be raised, it is probably a bug to call
     * AbEnd and not do other cleanup as well.
     */
    /* if(signal(SIGINT, AbEnd) == SIG_ERR)
	return(ABEND_ERR); */
    if(signal(SIGABRT, AbEnd) == SIG_ERR)
	return(ABEND_ERR);

    /* check if VAPI installed */
    if ((drv_str = vapi_id()) == NULL) {
	gprintf(ERROR, "no VAPI driver found\n");
	return (VAPI_BAD_OPEN);
    }

    /* make sure it is really VAPI down there */
    if (vxpi_id() != ('V' << 8 | '1')) {
	return (VAPI_BAD_OPEN);
    }

    if (verbose)
	gprintf(TRANS,"Found %Fs\n",drv_str);	/* driver id string */

    /* fill structure with hardware stats */
    vxpi_stats(&driver_stats);

    /* Check if the input port specified is greater than what
       the hardware supports */
    if (VPin_port >= driver_stats.inputs) {
	gprintf(ERROR, "Input port %d is not valid.\n", VPin_port + 1);
	return (VAPI_BAD_OPEN);
	}

    /* If the output port specified is greater than what
       the hardware supports and there is no sapi driver on that port */
    if ((VPout_port >= driver_stats.outputs)
	&& ((drv_str = vxpi_sound_enquire(VPout_port)) == NULL)) {
	gprintf(ERROR, "Output port %d is not valid.\n", VPout_port + 1);
	return (VAPI_BAD_OPEN);
	}
    else if (verbose)
	gprintf(TRANS,"Output port is %Fs\n",drv_str);

    /* Enable MIDI input if the board does not support both in and out */
    if (!driver_stats.simul_inout) {
	vxpi_output_enable(0);	/* Enable MIDI input */
	/* In this case, check if board has digital output */
	/* I.e., if the out_port is a sapi port */
	if (!vxpi_sound_enquire(VPout_port)) { /* if not sapi */
	    if (verbose) {
		gprintf(TRANS, "MIDI output is disabled.\n");
		}
	    }
	}

    /* open driver */
    vret = v1_open(0, 0, VP_stack, VAPI_STACK_SIZE);
    if (vret) {
	gprintf(ERROR, "VAPI open code -> %d\n", vret);
	return (VAPI_BAD_OPEN);
	}

    /* open MIDI out */
    vret = v1_txopen(VPout_port, VPout_buf, MIDI_BUF_SIZE);
    if (vret) {
	gprintf(ERROR, "VAPI Tx open code -> %d\n", vret);
	return (VAPI_BAD_OPEN);
	}

    /* open MIDI in */
    vret = v1_rxopen(VPin_port, VPin_buf, MIDI_BUF_SIZE);
    if (vret) {
	gprintf(ERROR, "VAPI Rx open code -> %d\n", vret);
	return (VAPI_BAD_OPEN);
	}

    /* set a trap for MIDI input */
    /* make it highest priority */
    /* call VPdata_trap() when there is input */
    {
    // #include <dos.h>
    // struct SREGS sr;
    // void prt_es(int (far *)());

    // printf("before: VPdata_trap %p\n",&VPdata_trap);
    // segread(&sr);
    // printf("before: ES %4x CS %4x SS %4x DS %4x\n",sr.es,sr.cs,sr.ss,sr.ds);
    // prt_es(&VPdata_trap);
    vret = v1_trap(VN_Rx0 + VPin_port, 0, 1, &VPdata_trap);
    // printf("after:  VPdata_trap %p\n",&VPdata_trap);
    // segread(&sr);
    // printf("after:  ES %4x CS %4x SS %4x DS %4x\n",sr.es,sr.cs,sr.ss,sr.ds);
    if (vret) {
	gprintf(ERROR, "VAPI open Rx trap code -> %d\n", vret);
	return (VAPI_BAD_OPEN);
	}
    return (VAPI_INITED);			/* return no error */
    }
}

// void prt_es(int (far *p)())
// {
//     struct SREGS sr;
//     segread(sr);
//     printf("within: VPdata_trap %p\n",p);
//     printf("within: ES %4x CS %4x SS %4x DS %4x\n",sr.es,sr.cs,sr.ss,sr.ds);
// }


/* ----- Routines used by MPUisr, not to be used by other programs ------ */

/* VPdata_trap(void)
 * Called by VAPI every time there is MIDI input
 */
int far VPdata_trap(void)
{
    int	data;

    // v1_msk(~0);
    while (1)
	{
	data = v1_get(VPin_port);
	if (data >= 0) {
	    MPUisr ( ((unsigned)data) & 0x00ff );
	    }
	/* a value less than 0 means empty or error */
	else if (data==V1E_MFIFOEM) break;
	else {
	    gprintf(ERROR, "MIDI data error %d\n",data);
	    break;
	    }
	}
    // v1_msk(0);

    /* tell VAPI we want another trap when another byte comes in */
    return 1;
}

static void UndoSysexMsg()
{
    xbuftail = old_xbuftail;
}

static boolean PutSysExByte(byte value)
{
    if ((SYSEX_ERR & midi_error_flags) ||
	!xbuff || exclFilter) return FALSE;
    /* the 1st 3 bytes will be saved to the normal midi buffer */
    if (SysexCounter < 3) {
	if (SysexCounter == 0) old_xbuftail = xbuftail;
	SysexHeader.b[SysexCounter++] = value;
    }
    xbuff[xbuftail++] = value;
    xbuftail &= xbufmask;
    if (xbuftail == xbufhead) {
	set_error(SYSEXOVFL);
	UndoSysexMsg();
	return FALSE;
    }
    return TRUE;
}


/************************************************************
*			    MidiParse
*
* Input : A midi byte.
* Output: none
* Return: none
* Effect:
***************************************************************/

private void MidiParse(int c)
{
#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

   /* REALTIME MESSAGES */
    if ((c & MIDI_REALTIME) == MIDI_REALTIME) {
	setstatus(RealTimeMsg, c);
	PutMidiMsg(RealTimeMsg);
	return;
    }

    /* STATUS MESSAGES */
    if (c & MIDI_STATUS_BIT) {
	if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* END OF EXCLUSIVE */
	    if (PutSysExByte(MIDI_EOX))
		PutMidiMsg(SysexHeader);
	    firstbyte = TRUE;
	}
	setstatus(CurrentMsg, c);
	data_rcv = 0;
	if ((c & MIDI_COMMON) == MIDI_COMMON) {
	    data_len = CommonTable[c & LOWBITS];
	    if (data_len == 0) {
		if (c != MIDI_EOX) {
		    if (c != MIDI_SYSEX) PutMidiMsg(CurrentMsg);
		} else setstatus(CurrentMsg, 0); /* make sure EOX is ignored */
	    }
	} else data_len = VoiceTable[(c & MIDI_COMMON) >> 4];
	return;
    }

    /* EXCLUSIVE DATA */
    if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* PUT IN EXCL BUFFER */
	if (firstbyte) {
	    xbuf_flush = false;
	    PutSysExByte(MIDI_SYSEX);
	    firstbyte = FALSE;
	}
	PutSysExByte((byte) c);
	return;
    }

    /* VOICE MESSAGE DATA */
    if (statusbyte(CurrentMsg)) { /* use zero to mean "status unknown" */
	setdata(CurrentMsg, ++data_rcv, c);
	if (data_rcv >= data_len) {
	    PutMidiMsg(CurrentMsg);
	    midimsg(CurrentMsg) &= CLEAR_DATA;
	    data_rcv = 0;
	}
    }
}


/*
** Interrupt service routine for data from MPU-401.
** refer algorithm pp13 MPU-401 TRM.
**
** Modified to service data from MIDIator
*/
void MPUisr(unsigned inword)
{
    static byte MPUisr_dat;

    MPUisr_dat=inword;
    debug(MPUisr_dat, 2, DBG_ISRBYTE);

    /* if we are in UART mode, always for MIDIator */

    if (UartMode) {
	MidiParse(MPUisr_dat);	/* parse the data */
    } else if (MPUisr_dat == MPU_ACK) { /* o.w. interpret mode */
	CmdAck = TRUE;		/* command has been acknowledged */
	if (ExpectingDataAfterCmd) {	/* there is data following */
	    ExpectingDataAfterCmd = FALSE;
	    debug(ReturnedData, 3, DBG_RETDATA);
	}
    }
}

