/* mpuSB.c -- support for Creative Sound Blaster Normal MPU mode */

/* Copyright 1996 Intelligistics, Inc. */

/*****************************************************************************
* Change Log
*   Date    | who : Change
*-----------+-----------------------------------------------------------------
* 03-Jun-96 | GWL :
*****************************************************************************/

/*
** These functions are the interupt handling routines to process interrupt
** requests for a Sound Blaster operating in "Normal" mode ("SB-MPU");
** presumably MPU.C (q.v.) works for the latest SB 16's that have
** direct MPU-401 lookalike registers.
** 
** The routines are: (see mpu.h for each function's prototype)
**
** mOpen()      - Setup the Interupt Service Routine (ISR) handling SB-MPU
** mClose()     - Disable the ISR for the SB-MPU, optionally print errors
** mGetData()   - Retrieve a byte from the SB-MPU interface read port
** mPutData()   - Send a data byte to the SB-MPU
** mPutCmd()    - Send a command to the SB-MPU
** mPutGetCmd() - Send a command and return the returned data byte.
**
** mOpen should be called before any of the other routines here are called,
** and mClose should be called before the program finishes.
**
** This source file contains non standard code and functions for the
** MicroSoft C V5.1 and Turbo C V1.5 compilers and versions above.
**
*/
/*
** Modifications by Roger Dannenberg:
**
** In Leigh's design, sysex messages must be emptied from the input buffer
** as fast as they come in; however, the CMT program ExGet
** waits for all data to be received before writing data to disk.  In
** general, it can be hard for a machine to keep up with full-speed
** MIDI sysex dumps, and these can far exceed 64K.  We could always
** allocate a huge buffer for all MIDI data, or we could save sysex
** messages into a separate buffer that is provided by the application
** program.  Other CMT implementations for small machines do the latter,
** so that's what I've done for DOS.  Sysex messages are ignored until
** the application provides a buffer, which will be declared as HUGE so
** that the 64K segment size is not a limitation.  As a signal that a
** sysex message has arrived, the first 4 bytes are placed in the ordinary
** MIDI message buffer.  When the application sees the 1st 4 bytes, it
** should retrieve data from the buffer.  (This is enforced in the midifns.c
** level, but not here.)  Since sysex buffers must be huge to be greater
** than 64K, I do not reuse the existing ring buffer structures.
**
** Error codes in midifns.h are used to set bits in midi_error_flags
** when errors are detected.  The important errors are BUFOVFL and
** SYSEXOVFL.
*/
/* Mods by GWL
**
** Information herein is taken from Crative Labs "Hardware Programming
** Reference" manual in the "Developer Kit for Sound Blaster (TM) Series",
** 2nd Edition ("HPRM").

** It is very much a hack designed nevertheless to work on the entire series,
** and as such bypasses the more direct MPU-lookalike registers in the later
** SB's (SB 16 ff).

** The SoundBlaster series is supposed to maintain a simple "Normal" mode of
** MIDI I/O using the DSP registers, in a protocol similar to but not identical
** to the MPU-401 in UART mode.  In particular, a significant difference from
** the MPU-401 is the requirement to send to the interface a "MIDI output"
** command before each MIDI data or command.

** Also, the SB-MPU must be reset properly before being used.

** The conventions for definingthe base register and IRQ are retained in this
** module, although the "search for IRQ" has been deleted.
** Defaults are: BaseAddr=220; MPU_SB_IRQ==7.

** The user is cautioned to be careful: MPUisr does not chain to any other
** device that may share the MPU's IRQ, and there is only minimal checking
** for valid values.
**
** Anticipating other drivers, check_midi() is moved here from midifns.c
*/

#include "switches.h"
#include <stdlib.h>     /* for ANSI malloc */
#include "cext.h"
#include <stdio.h>      /* for NULL */
#include <signal.h>     /* for ^C trapping */
#include <dos.h>        /* for FP_SEG, FP_OFF */
#include <bios.h>       /* for GETTIMEOFDAY */
#include "mpusb.h"      /* for Sound Blaster constants */
#include "mpu.h"        /* for SB-MPU constants */
#include "midibuff.h"
#include "midicode.h"
#include "userio.h"     /* for abort_flag */
#include "midierr.h"    /* error flags */
#pragma check_stack (off)  /* stack checking reports errors calling DOS */

#ifdef MICROSOFT     /* for Microsoft C V5.1 and above */
#define DISABLEINTS     _disable
#define ENABLEINTS      _enable
#define INTERRUPT       interrupt far
#define GETVECTOR       _dos_getvect
#define SETVECTOR       _dos_setvect
#define GETTIMEOFDAY(a) _bios_timeofday(_TIME_GETCLOCK, &(a))
#else                /* for Turbo C V2.0 and above */
#define DISABLEINTS     disable
#define ENABLEINTS      enable
#define INTERRUPT       interrupt
#define GETVECTOR       getvect
#define SETVECTOR       setvect
#define GETTIMEOFDAY(a) (a = biostime(0, 0))
#endif

#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

#undef DEBUG                            /* cancel userio defn. */
#define PP if(TRUE)
#define DEBUG           FALSE           /* TRUE to write to screen RAM for debugging */
#define DBG_SCRNSEG 0xb800              /* segment of diag screen (CGA at present) */
#define DBG_RECTIME     0x7d4           /* record time assign debug */
#define DBG_ISRBYTE     0x7d6           /* where to dump the ISR byte debug */
#define DBG_UNH_SYS     0x7d8           /* unhandled sys-ex debug */
#define DBG_RETDATA 0x7da               /* returned data debug */
#define DBG_CLK2HST 0x7dc               /* clock to host called debug */

#if DEBUG
static char far *screen;
/*
** A useful debugging operation, prints a character on the screen
** to indicate where the Program Counter is in the ISR.
** a is character to display, b is screen attribute, c is address on screen
*/
#define debug(a, b, c)  { FP_SEG(screen) = DBG_SCRNSEG;\
	FP_OFF(screen) = (c);\
	*screen = (a);\
	FP_OFF(screen) = (c)+1;\
	*screen = (b); }
#else
#define debug(a, b, c)
#endif

#define MPU_ACK         0xFE
#define IRQ9MASK        (2)


/* address of original ISR, NULL if not assigned, to indicate no interrupt */
/* has been set up */
static void (INTERRUPT *Old_Intr)() = NULL;

static void INTERRUPT MPUisr(void);     /* prototype of our interrupt */
/* interrupt will run on a dedicated stack */
#define mystacksize 512
static unsigned oldss, oldsp, mystack[mystacksize];
static void (*ClockToHostFn)(void);     /* Ptr to fn called each MPU Clock */
static int  Intr,           /* interrupt used by SB-MPU */
    UartMode = FALSE,       /* TRUE when MPU is in UART mode */
    CmdAck,                 /* MPU-401 command acknowledge flag */
    ExpectingDataAfterCmd,  /* Flag indicating MPU will return a byte */
    IsAT,                   /* Flag set for AT */
    trace;                  /* true if -trace switch entered (to show IRQ) */
static byte ReturnedData,   /* The byte MPU has returned */
    pic_mask,               /* mask to enable/disable the interrupt */
    pic_irq_save,           /* save status of IRQ n in 1st PIC (any PC) */
    AT_irq9_save;           /* save status of IRQ 9 in 2nd PIC (AT only) */
static unsigned int

    ResetPort,              /* = BaseAddr + 0x6 */
    ReadStatusPort,         /* = BaseAddr + 0xe */
    ReadDataPort,           /* = BaseAddr + 0xa */
    WriteStatusPort,        /* = BaseAddr + 0xc */
    WriteDataPort,          /* = BaseAddr + 0xc */

    pic_IMR;                     /* The port of the PIC mask register */

byte huge *xbuff = NULL; /* application-supplied sysex buffer */
long xbufmask;           /* mask for circular buffer */
long xbufhead = 0;       /* buffer head and tail offsets */
long xbuftail = 0;
static long old_xbuftail;

boolean xbuf_flush = false;   /* says to flush remainder of sysex message */
MidiMsg SysexHeader;    /* saves 1st 3 bytes of sysex messages for buffer */
byte SysexCounter = 0;  /* index into SysexHeader */

/* midi input buffer */
/* data buffer, declared long to get 32-bit alignment: */
long buff[BUFF_SIZE/4]; 
int buffhead = 0;     /* buffer head and tail pointers */
int bufftail = 0;

private MidiMsg CurrentMsg, RealTimeMsg;
private int data_len;
private int data_rcv;
private boolean firstbyte;

extern boolean ctrlFilter;
extern boolean exclFilter;
extern boolean realFilter;

extern int musictrace;

#define set_error(bit) midi_error_flags |= (1 << bit);

/***************
* LOCAL TABLES *
***************/

private byte VoiceTable[8] = {2,2,2,2,1,1,2,0};
/* note: length entry for EOX MUST BE ZERO: */
private byte CommonTable[8] = {0,1,2,1,0,0,0,0};

/* check_midi -- return true if there's midi data to process */
/**/
public boolean check_midi()
{
	if (buffhead == bufftail) return false;
	else return true;
}

/* midi_flush -- empty out buffers */
/**/
void midi_flush()
{
    buffhead = 0;
    bufftail = 0;
    xbufhead = 0;
    xbuftail = 0;
    xbuf_flush = true;  /* in case sysex continuation messages are still coming */
}

/*
 * #define dbmax 32
 * int dbb[dbmax];
 * char * dbs[dbmax];
 * int dbx = 0;
 * int dbh = 0;
 */

/* ---------------------- SB-MPU Interface Routines ----------------------- */

/*
** Retrieve a byte from the SB-MPU interface DataPort
*/
byte mGetData(void)
{
    while( ! (inp(ReadStatusPort) & DOUTR_SB) )
	;                        /* wait until SB-MPU has data */
    return(inp(ReadDataPort));
}



/*
** Put out a single byte to SB-MPU
*/
void mPutByte(byte dat)
{
    while( inp(WriteStatusPort) & DOUTR_SB );
		;     /* wait until SB-MPU ready to send */
    outp(WriteDataPort, dat);
}



/*
** Put out data to SB-MPU
*/
void mPutData(byte dat)
{
#if 0
    static long prev_time=0L;
    long cur_time;

    cur_time=gettime();
    printf("%10ld %10ld\n",cur_time, cur_time-prev_time);
    prev_time=cur_time;
#endif

    if ( UartMode)

	    /* SB in Uart mode */
    {
	mPutByte(dat);
    }
    else
	    /* SB in Normal mode */
    {
	mPutByte(MPU_SB_INPUT_MIDI);	/* turn off input */
	mPutByte(MPU_SB_OUTPUT_MIDI);	/* send output */
	mPutByte(dat);
	mPutByte(MPU_SB_INPUT_MIDI);	/* restore input */
    }
}



/*
** Send a command and return a byte from the SB-MPU. As this routine calls
** mPutCmd, don't call this from within MPUisr either. Because getting the
** record counter value will reset it, we should add this record counter value
** to RecordTime to keep it accurate.
*/
byte mPutGetCmd(byte cmd)
{
    ExpectingDataAfterCmd = TRUE;
    mPutCmd(cmd);
    while (ExpectingDataAfterCmd)
		;
    return ReturnedData;
}



int NotCmdAck() { return !CmdAck; } /* used to defeat optimizer, see below */



/*
** Put out commands to the SB-MPU, waiting until the command is acknowledged
** DON'T CALL THIS FUNCTION FROM WITHIN MPUisr! - It waits for MPUisr
** to reset CmdAck, calling mPutCmd via MPUisr could cause some tail chasing.
** Don't use this function to retrieve data from the MPU (commands 0xA0-0xAF)
** Use mPutGetCmd() above.
*
* This function busy waits on the interrupt to set CmdAck.  Under the
* default Microsoft compiler options, the compiler "over"optimizes the
* loop by not doing the test!  I'm sure this could be fixed by disabling
* certain optimizations, but I decided to move the test into a routine
* to (hopefully) defeat any optimization attempts.  If small routines were
* in-lined by the compiler, an optimizer could still lead you astray.
*/

extern int mPutCmd_Loop_Count = 0; /* see note above */

void mPutCmd(byte cmd)
{
#if 1
    return;
#endif
    if(cmd != MPURESET && UartMode) {
	return;
    }
    CmdAck = FALSE; /* reset the acknowledge flag */
    mPutData(cmd);

    if(cmd == MPURESET && UartMode) /* if a UartMode Reset is being sent */
	UartMode = FALSE;     /* no longer in UartMode, don't check for */
    else                      /* an ACK in this one case */
	while (NotCmdAck());  /* wait until command recvd is acknowledged */
    if(cmd == UARTMODE)
	UartMode = TRUE;
}



/* PutMidiMsg -- copy complete message from interrupt to buffer */
/**/
static void PutMidiMsg(MidiMsg msg)
{
    register byte *cmt_data = ((byte *) buff) + bufftail;

    /* filter out control changes */
    if (ctrlFilter) {
	register int data1 = databyte1(msg);
	register int hibits = statusbyte(msg) & 0xF0;
	if (hibits == 0xD0 ||   /* Chan Pressure */
	    hibits == 0xE0 ||       /* Pitch Bend */
	    hibits == 0xA0 ||       /* Poly Pressure */
	    ((hibits == 0xB0) &&    /* Control change (don't count switches) */
	     ((data1 < 64) || (data1 > 121)))) {
	    /* CONTROL MESSAGE HAS BEEN FILTERED */
	    return;
	}
    }
    if (realFilter) {
	if (statusbyte(msg) >= 0xF8) return;
    }
    if (exclFilter) {
	if (statusbyte(msg) == MIDI_SYSEX) return;
    }
    *((long *) cmt_data) = msg.l;
    bufftail = (bufftail + 4) & BUFF_MASK;
    if (bufftail == buffhead) {
	/* filled buffer faster than client emptied it */
	set_error(BUFFOVFL);
    }
}



/* ----------------- ISR Initialise and Close down routines ----------------- */

/*
** Function to close down the interrupt service routine, return FALSE if
** unable to close routine, if signal complains. If err is non-zero,
** print the corresponding error message returned by mOpen. The interrupt
** is only replaced if it was actually changed.
*/
int mClose(int err)
{
    int i;
    static char *errmsgs[] = { /* ensure error codes align with the msgs */
	"Can\'t set up an abnormal end trap\n",
	"SoundBlaster will not reset - it may not be connected\n",
	"SoundBlaster will not acknowledge commands\n",
	"SoundBlaster is not installed on an expected interrupt\n",
	"Can\'t allocate enough memory for SoundBlaster operation",
	"Illegal IRQ request",
	"Illegal base address request"
    };
#if 0
    printf("mClose %d\n",err);
#endif
    if ( !UARTMODE)
	mPutByte(MPU_SB_INPUT_MIDI);
    SB_DSP_reset();
    if (Old_Intr != NULL) {
	DISABLEINTS();

	/* restore irq status bit in pic */
	/* printf("\n  pic: old %02x save %02x ",inp(pic_IMR),pic_irq_save); */
	outp(pic_IMR, (inp(pic_IMR) & ~pic_mask) | pic_irq_save);
	/* printf("new %02x\n",inp(pic_IMR)); */

	/* restore AT 2nd pic IRQ 9 * /
	if (IsAT)
	/* printf("ATpic: old %02x save %02x ",inp(PICIMR_AT),AT_irq9_save); */
	    outp(PICIMR_AT, (inp(PICIMR_AT) & ~IRQ9MASK) | AT_irq9_save);
	/* printf("new %02x\n",inp(PICIMR_AT)); */
	ENABLEINTS();
	SETVECTOR(Intr, Old_Intr);
	Old_Intr = NULL;    /* allow for multiple mOpen and mClose */
    }

    if (err > 0 && err <= (sizeof(errmsgs) / sizeof(char *)))
	gprintf(ERROR, errmsgs[err - 1]);
    else if (err)
	gprintf(ERROR, "mClose err %d\n",err);
    if (signal(SIGINT, SIG_DFL) == SIG_ERR)
	return FALSE;
#if 0
    printf("mClose OK\n");
#endif
    return TRUE;
}

/*
** If a Control-C or Abort interrupt occurs, exit gracefully,
** disabling the interrupt if it had been assigned.
*/
static void AbEnd(int i)
{
    abort_flag = true;
}

/*
** Function to use the time of day clock to check to see if a timeout
** has occured. Returns TRUE if the TimeOutPeriod has expired, FALSE
** if still within time. TimeOutPeriod is in system clock ticks, each
** 54.9 milliseconds.
*/
static int TimeOut(long StartTime, long TimeOutPeriod)
{
    long CurrentTime;

    GETTIMEOFDAY(CurrentTime);
    return CurrentTime - StartTime > TimeOutPeriod;
}

/*
** Check if we have a PC/AT architecture, by checking for the second PIC.
*/
static int UsingATpic(void)
{
    byte save_pic;
    int pic_there;

    DISABLEINTS();
    save_pic = inp(PICIMR_AT);
    outp(PICIMR_AT, 0xAA);
    pic_there = inp(PICIMR_AT) == 0xAA;
    outp(PICIMR_AT, save_pic);
    ENABLEINTS();
    return pic_there;
}

/*
** Test the given interrupt number to be connected to the SB-MPU.
** Returns TRUE if the interrupt is connected, FALSE if not.
*/
static int TestIntr(int Intr, int UsingAT)
{
   byte Old_IMR;
   long StartTime;

    /* set up MPUisr in interrupt vector */
    /* PP printf("TestIntr\n");  */
    Old_Intr = _dos_getvect(((unsigned) Intr));
    /* Old_Intr = GETVECTOR(((unsigned) Intr)); /* save old address of ISR */
    SETVECTOR(Intr, MPUisr);            /* put in new address */

    /* calculate IRQ's mask and status */
    DISABLEINTS();
    if (UsingAT && Intr == 0x0A) {      /* if an AT with IRQ-2 */
	pic_IMR = PICIMR_AT;            /* program second PIC */
	pic_mask = IRQ9MASK;
    } else {                            /* an XT or an AT not on IRQ-2 */
	pic_IMR = PICIMR;
	pic_mask = (1 << (Intr - 8));
    }
    Old_IMR = inp(pic_IMR);
    pic_irq_save = Old_IMR & pic_mask;
    outp(pic_IMR, Old_IMR & ~pic_mask); /* unmask the irq */
    ENABLEINTS();
    /* printf("pic %02x now %02x\n",pic_IMR,inp(pic_IMR)); */
    /*
     * If MPUUART then this is a "dumb" MPU compatible, so you
     * cannot use a metronome command to get a response. Just exit now
     */
    if ( ! UartMode )
    {
	mPutByte(MPU_SB_INPUT_MIDI);
	return TRUE;
    }

    /* SB in Uart interrupt mode */

    mPutByte(MPU_SB_UART_IRUPT);

    /*
    ** Send a metronome off command (reasonably benign), and see if we get
    ** a response from the ISR within a timeout period to determine if the
    ** interrupt is correct
    */
#if 0
    CmdAck = FALSE;                     /* reset the acknowledge flag */
    mPutData(METROOFF);
    GETTIMEOFDAY(StartTime);
    while (!CmdAck && !TimeOut(StartTime, TIMEUP))
	;                  /* wait for ISR to respond or timeout */
    if (!CmdAck) {
	inp(ReadDataPort);         /* throw away the acknowledge sent */
	DISABLEINTS();
	outp(pic_IMR, Old_IMR);
	ENABLEINTS();
	if (musictrace)
	    printf("Didn't find on Interrupt %02X\n", Intr);
	SETVECTOR(Intr, Old_Intr);      /* put back old address */
	return FALSE;         /* we didn't find the ISR on this int */
    }
#endif
    if (musictrace)
	printf("Found on Interrupt %02X\n", Intr);
    return TRUE;               /* we've found the ISR */
}



/*
   Routine to reset DSP, per HPRM pp 2-2 ff
*/

int SB_DSP_reset (void)
{
    long StartTime;
    int NoOfResets = 2;
    int input_value;

    do {
	/* send "1" and wait 3 microseconds */

	outp(ResetPort, 1);
	GETTIMEOFDAY(StartTime);
	while ( !TimeOut (StartTime, 1L) ); /* at least 1 tick, 55 ms */

	/* send "0" and wait for 0xaa (in ca. 100 microseconds */

	outp(ResetPort, 0);
	GETTIMEOFDAY(StartTime);
	while ( ! (inp(ReadStatusPort) & DINR_SB) &&
	    ! TimeOut(StartTime, TIMEUP) ) ;
	input_value=inp(ReadDataPort);
#if 0
	printf("reset returns %x\n",input_value);
#endif
	if (input_value==0xaa) return (TRUE);
    } while (--NoOfResets);

    return (FALSE);
}



/*
** Routine to prime the internal variables and attempt to establish
** communications with the SB-MPU and set up the interrupt to MPUisr.
** Returns an error number for the different error messages if unable to
** communicate with the MPU, and 0 if everything went ok.
** BaseAddr gives the base address of the MPU interface, IrqNo the interrupt
** number to use.
*/
int mOpen(unsigned int BaseAddr, int IrqNo)
{
    int TimedOut = FALSE, NoOfResets = 2;
    long StartTime;

    setstatus(CurrentMsg, 0);
    firstbyte = TRUE;

    /* validate parameters for SB */

    if (BaseAddr==MPUBASEADDR) BaseAddr=MPU_SB_BASE;
    if (IrqNo==SEARCHIRQ) IrqNo=MPU_SB_IRQ;

    if (IrqNo!=2 && IrqNo!=5 && IrqNo!=7 && IrqNo!=10)
	return (MPUIRQ_ERR);
    if (BaseAddr & 0xf00 != 0x200) return (MPUBASE_ERR);

#if 0
    printf("base %x irq %x\n",BaseAddr, IrqNo);
#endif

    ResetPort       = BaseAddr + 0x6;
    ReadStatusPort  = BaseAddr + 0xe;
    ReadDataPort    = BaseAddr + 0xa;
    WriteStatusPort = BaseAddr + 0xc;
    WriteDataPort   = BaseAddr + 0xc;

    if (getenv("MPUUART"))
	UartMode = TRUE;
    else
	UartMode = FALSE;
    if (musictrace)
	printf("UartMode %s\n",(UartMode)?"on":"off");

    /* SIGINT handler is set up by USERIO.C.  It causes a cu_cleanup
     * that in turn calls mClose, so we shouldn't override the handler
     * here.  On the other hand, I don't set up a handler for SIGABRT
     * elsewhere, so I'm leaving in this handler.  Depending on how
     * and when SIGABRT can be raised, it is probably a bug to call
     * AbEnd and not do other cleanup as well.
     */
    /* if(signal(SIGINT, AbEnd) == SIG_ERR)
	return(ABEND_ERR); */
    if (signal(SIGABRT, AbEnd) == SIG_ERR)
	return ABEND_ERR;

    /* if it's an AT, disable IRQ9 in 2nd PIC */

    IsAT = UsingATpic();
#if DEBUG
    puts(IsAT ? "Running on an AT" : "Running on an XT");
    puts("setting up ISR");
#endif
    DISABLEINTS();
    if (IsAT)
    {
	AT_irq9_save = inp(PICIMR_AT);
	outp(PICIMR_AT, AT_irq9_save | IRQ9MASK);
	AT_irq9_save &= IRQ9MASK ;
    }
    ENABLEINTS();

    /* see if MPU_SB allows a reset within a timeout period */

    if ( ! SB_DSP_reset() )
	return RESET_ERR;
    /*
    ** Send a metronome off command (reasonably benign), and see if we get a
    ** response from the MPU within a timeout period to determine if there is
    ** SB-MPU connected
    */
#if 0
    if ( UartMode )
    {
#if 1
	puts("sending metronome off");
#endif
	mPutByte(METROOFF);
	GETTIMEOFDAY(StartTime);
	if (mGetData() != MPU_ACK)
	    return(ACK_ERR);
    }
#endif

    /* try setting up the ISR for requested interrupt */

    if (musictrace)
	printf("About to search for IRQ %02x\n",IrqNo);
    if (TestIntr(Intr=(IrqNo+8), IsAT)) /* if only on a single interrupt */
	    return(MPUINITED);
    return INT_ERR;
}

static void UndoSysexMsg()
{
    xbuftail = old_xbuftail;
}

static boolean PutSysExByte(byte value)
{
    if ((SYSEX_ERR & midi_error_flags) ||
	!xbuff || exclFilter) return FALSE;
    /* the 1st 3 bytes will be saved to the normal midi buffer */
    if (SysexCounter < 3) {
	if (SysexCounter == 0) old_xbuftail = xbuftail;
	SysexHeader.b[SysexCounter++] = value;
    }
    xbuff[xbuftail++] = value;
    xbuftail &= xbufmask;
    if (xbuftail == xbufhead) {
	set_error(SYSEXOVFL);
	UndoSysexMsg();
	return FALSE;
    }
    return TRUE;
}


/************************************************************
*                           MidiParse
*
* Input : A midi byte.
* Output: none
* Return: none
* Effect:
***************************************************************/

private void MidiParse(int c)
{
#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

   /* REALTIME MESSAGES */
    if ((c & MIDI_REALTIME) == MIDI_REALTIME) {
	setstatus(RealTimeMsg, c);
	PutMidiMsg(RealTimeMsg);
	return;
    }

    /* STATUS MESSAGES */
    if (c & MIDI_STATUS_BIT) {
	if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* END OF EXCLUSIVE */
	    if (PutSysExByte(MIDI_EOX))
		PutMidiMsg(SysexHeader);
	    firstbyte = TRUE;
	}
	setstatus(CurrentMsg, c);
	data_rcv = 0;
	if ((c & MIDI_COMMON) == MIDI_COMMON) {
	    data_len = CommonTable[c & LOWBITS];
	    if (data_len == 0) {
		if (c != MIDI_EOX) {
		    if (c != MIDI_SYSEX) PutMidiMsg(CurrentMsg);
		} else setstatus(CurrentMsg, 0); /* make sure EOX is ignored */
	    }
	} else data_len = VoiceTable[(c & MIDI_COMMON) >> 4];
	return;
    }

    /* EXCLUSIVE DATA */
    if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* PUT IN EXCL BUFFER */
	if (firstbyte) {
	    xbuf_flush = false;
	    PutSysExByte(MIDI_SYSEX);
	    firstbyte = FALSE;
	}
	PutSysExByte((byte) c);
	return;
    }

    /* VOICE MESSAGE DATA */
    if (statusbyte(CurrentMsg)) { /* use zero to mean "status unknown" */
	setdata(CurrentMsg, ++data_rcv, c);
	if (data_rcv >= data_len) {
	    PutMidiMsg(CurrentMsg);
	    midimsg(CurrentMsg) &= CLEAR_DATA;
	    data_rcv = 0;
	}
    }
}


/*
** Interrupt service routine for data from SB-MPU.
*/
static void INTERRUPT MPUisr(void)
{
    /* this is static for 2 reasons: keep the interrupt stack small,
     *  and we want MPUisr_dat to be addressable after we swap stacks.
     */
    static byte MPUisr_dat;
/* This code moves the stack to an area known to be large enough.
   For unknown reasons, this doesn't work with Quick C, so we just
   hope that the system stack on which interrupts are processed is
   big enough.  This is probably true with recent versions (5.0) of DOS.
 */
#ifndef _QC
    _asm {
	mov ax,ss
	mov oldss,ax
	mov ax,sp
	mov oldsp,ax
	push ds
	pop ss
	mov sp,OFFSET mystack+(mystacksize*2)-2
    }
#endif
    ENABLEINTS();
    MPUisr_dat = mGetData();
    MidiParse(MPUisr_dat);  /* parse the data */
    debug(MPUisr_dat, 2, DBG_ISRBYTE);
#if 0
    if (UartMode) {             /* if we are in UART mode, */
	MidiParse(MPUisr_dat);  /* parse the data */
    } else if (MPUisr_dat == MPU_ACK) { /* o.w. interpret mode */
	CmdAck = TRUE;                  /* command has been acknowledged */
	if (ExpectingDataAfterCmd) {    /* there is data following */
	    ReturnedData = mGetData();  /* get the next byte sent */
	    ExpectingDataAfterCmd = FALSE;
	    debug(ReturnedData, 3, DBG_RETDATA);
	}
    }
#endif
    DISABLEINTS();              /* prevent an interrupt for EOI */
    outp(PICCNTRL, EOI);        /* nonspecific End Of Interrupt to 8259 */
#ifndef _QC
    _asm {
	mov ax,oldss
	push ax
	pop ss
	mov sp,oldsp
    }
#endif
    /* returning will reenable the interrupt */
}
/* ====== */
