/*
** MPU.H Defines constants to access the Roland MPU-401 MIDI Processing Unit
** running on an IBM PC with the MIF-IPC interface card. Enabling MIDI Musical
** Synthesisers to be controlled. This file is used by nearly every other
** source file in the MPU library. It should be included before all others and
** before any code or data declations.
**				Leigh Smith, 26/5/88
*/

#ifndef FALSE
#define FALSE 0					/* boolean constants */
#define TRUE !(FALSE)
#endif

/* Hardware details of interface */
#define MPUBASEADDR		0x0330	/* Address of MPU-401 by default */
#define SEARCHIRQ		(-1)	/* Passed as IrqNo to cause IRQ searching */
#define DSR				0x80	/* Data Set Ready, active low */
#define DRR				0x40	/* Data Recieve Ready, active low */
#define PICCNTRL		0x0020	/* Programmable Interrupt Controller address */
#define PICIMR			0x0021	/* PIC interrupt mask register */
#define PICCNTRL_AT		0x00A0	/* AT Programmable Interrupt Controller address */
#define PICIMR_AT		0x00A1	/* AT PIC interrupt mask register */
#define EOI				0x20	/* Non-specific End Of Interrupt */
#define TIMEUP			3L		/* Counter value used in mOpen for time outs */
#define TRACKS			8		/* Number of tracks in sequencer */
#define PASSAGE_SIZE	512		/* Length of passage of MIDI play data */
#define RECORD_SIZE		32767	/* Length of passage of MIDI record data */

#define MINTR			0x0A	/* first interrupt to try MPU on: IRQ 2 */
#define MINTRLAST		0x0F	/* last interrupt to try using:   IRQ 7 */

/* error codes set back by mOpen */
#define MPUINITED		0		/* MPU properly installed */
#define ABEND_ERR		1		/* MPU can't setup an abnormal end trap */
#define RESET_ERR		2		/* MPU won't reset */
#define ACK_ERR			3		/* MPU won't acknowledge a command */
#define INT_ERR			4		/* MPU not using an expected interrupt */
#define ALLOC_ERR		5		/* MPU couldn't allocate passage buffers */
#define MPUIRQ_ERR		6		/* Illegal IRQ request */
#define MPUBASE_ERR		7		/* Illegal base addr request */

/* bit encoded field values for mRecord and mPlay I/O formats */
#define RAWMPU_FORMAT	0x0000	/* Straight from MPU record and play fns */
#define MSGPAD3			0x0001	/* pad messages to 3 bytes, expand run status */
#define MSGPAD4			0x0002	/* pad messages to 4 bytes, expand run status */
#define NO_TIMING		0x0004	/* timing information is not in data */
#define ALLOW_RT_CLOCK	0x0008	/* real time clock messages in the data */
#define NON_MIDI_FORMAT	0x0010	/* don't interpret the input as MIDI messages */

/* Macros for producing MIDI status codes */
#define NOTEOFF(chan)	(0x80 + chan)
#define NOTEON(chan)	(0x90 + chan)
#define NOTEAFT_TOUCH	0xA0
#define CONT_CHANGE		0xB0
#define PROG_CHANGE		0xC0
#define CHANAFT_TOUCH	0xD0
#define PITCH_WHEEL		0xE0
#define SOX				0xF0	/* Start of Exclusive message */
#define SONG_POSITION	0xF2
#define SONG_SELECT		0xF3
#define TUNE_REQUEST	0xF6
#define EOX				0xF7	/* End of Exclusive value */
#define RT_CLOCK		0xF8	/* Real Time messages */
#define RT_START		0xFA
#define RT_CONTINUE		0xFB
#define RT_STOP			0xFC
#define ACTIVE_SENSING	0xFE
#define SYSTEM_RESET	0xFF

#define SEQUENTIAL		0x01	/* Some SysEx manufacturers codes */
#define BIG_BRIAR		0x02
#define OCTAVE_PLATEAU	0x03
#define MOOG			0x04
#define PASSPORT		0x05
#define LEXICON			0x06
#define ENSONIQ			0x0F
#define OBERHEIM		0x10
#define BON_TEMPI		0x20
#define SIEL			0x21
#define KAWAI			0x40
#define ROLAND			0x41
#define KORG			0x42
#define YAMAHA			0x43

/*
** Often used MPU-401 commands sent to mPutCmd. See MPU-401 TRM for details of
** command actions.
*/
#define STOP_MIDI_RT	0x01
#define START_MIDI_RT	0x02
#define CONT_MIDI_RT	0x03
#define STOPPLAY		0x05
#define STARTPLAY		0x0A
#define STOPREC			0x11
#define STOPOVERDUB		0x15
#define RECORDSTANDBY	0x20
#define STARTREC		0x22
#define OVERDUBSTANDBY	0x28
#define STARTOVERDUB	0x2A

#define NOALLNOTESOFF	0x30
#define NOREALTIME		0x32
#define ALLTHRUOFF		0x33
#define TIMEBYTEON		0x34
#define MODEMSGSON		0x35
#define EXCLTHRUON		0x37
#define COMMONTOHOSTON	0x38
#define REALTIMETOHOST	0x39
#define UARTMODE		0x3F

#define INTERNCLOCK		0x80
#define FSKCLOCK		0x81
#define MIDICLOCK		0x82
#define METROON 		0x83
#define METROOFF		0x84
#define METROACC		0x85
#define BENDEROFF		0x86
#define BENDERON		0x87
#define THRUOFF 		0x88
#define THRUON  		0x89
#define DATASTOPOFF		0x8A
#define DATASTOPON		0x8B
#define SENDMEOFF		0x8C
#define SENDMEON		0x8D
#define CONDUCTOFF		0x8E
#define CONDUCTON		0x8F
#define RTAFFOFF		0x90
#define RTAFFON 		0x91
#define FSK2INT			0x92
#define FSK2MIDI		0x93
#define CLOCKOFF		0x94
#define CLOCKON 		0x95
#define EXCLUSIVOFF		0x96
#define EXCLUSIVON		0x97

#define GETPLAYCNT_1	0xA0
#define GETPLAYCNT_2	0xA1
#define GETPLAYCNT_3	0xA2
#define GETPLAYCNT_4	0xA3
#define GETPLAYCNT_5	0xA4
#define GETPLAYCNT_6	0xA5
#define GETPLAYCNT_7	0xA6
#define GETPLAYCNT_8	0xA7

#define GETRECORDCNT	0xAB
#define GETMPUVER		0xAC
#define GETMPUREV		0xAD
#define GETTEMPO		0xAF

#define CLRPLAYCNT		0xB8
#define CLRPLAYTBL		0xB9
#define CLRRECCNT		0xBA
#define TIMEBASE48		0xC2
#define TIMEBASE72		0xC3
#define TIMEBASE96		0xC4
#define TIMEBASE120		0xC5
#define TIMEBASE144		0xC6
#define TIMEBASE168		0xC7
#define TIMEBASE192		0xC8

#define SENDDATA_1		0xD0
#define SENDDATA_2		0xD1
#define SENDDATA_3		0xD2
#define SENDDATA_4		0xD3
#define SENDDATA_5		0xD4
#define SENDDATA_6		0xD5
#define SENDDATA_7		0xD6
#define SENDDATA_8		0xD7

#define SENDSYS 		0xDF
#define TEMPOSET		0xE0
#define RELTEMPO		0xE1
#define GRADUATION 		0xE2
#define MIDIMETRO		0xE4
#define METROMEAS		0xE6
#define INTCLKHOST		0xE7
#define ACTIVETRK		0xEC
#define SENDPLAYCNT		0xED
#define ACTIVCHN1_8		0xEE
#define ACTIVCHN9_F		0xEF
#define MPURESET		0xFF



/*
** definitions of MPU Marks returned. See MPU TRM section 3 for details.
*/
#define MPU_NOP			0xF8
#define MEASURE_END		0xF9
#define DATA_END		0xFC

extern byte huge *xbuff;	/* application-supplied sysex buffer */
extern long xbufmask;	    /* mask for circular buffer */
extern long xbufhead;       /* buffer head and tail offsets */
extern long xbuftail;
extern int midi_error;

boolean check_midi(void);

byte mGetData(void);
void mPutCmd(byte);
byte mPutGetCmd(byte);
void mPutData(byte);
int  mOpen(unsigned int, int);
int  mClose(int);
void mRecord(unsigned int);
int  mRecording(void);
byte mRecByte(void);
unsigned long TimeSinceRecord(void);
void mPlay(void);
void mPlayByte(int, byte);
int  mPlaying(int);
int  mStopPlay(void);
byte *mPlayBuffer(int);
void mConduct(void);
int  mCondCount(void);
void mClockToHost(void (*)(void));


/*************************
* MIDI MESSAGE TYPE *
*************************/

typedef union { 
	unsigned long l;
	byte b[4];
} MidiMsg;

/***************************
* MIDI MESSAGE MACROS *
***************************/

#define clearmsg(x) (x).l=0
#define copymsg(x,y) (y).l=(x).l
#define validmsg(l) ( ((l) & 0x80808000L) == 0x80000000L )
#define voicemsg(x,a) ( ((x).b[0] & 0xF0) == (a) )

#define setstatus(x,a) (x).b[0]=(a)
#define setdata(x,i,a) (x).b[(i)]=(a)
#define setdata1(x,a) (x).b[1]=(a)
#define setdata2(x,a) (x).b[2]=(a)

#define midimsg(x) (x).l
#define statusbyte(x) (x).b[0]
#define databyte(x,i) (x).b[(i)]
#define databyte1(x) (x).b[1]
#define databyte2(x) (x).b[2]

