/* Midi interface code for UNIX_MACH, including RTMach */

#include "switches.h"
#include "stdio.h"
#include "cext.h"
#include "userio.h"
#include "midifns.h" /* declares eventwait() */

#include <midistruct.h>
#include <mach/mig_errors.h>
#include "io.h"
static mach_port_t ascii_notify_port = MACH_PORT_NULL;
static mach_port_t portset = MACH_PORT_NULL;
char a_in;
int a_in_flag=0;
int i_am_running = 1;
static char ascii_thread_port_name[512];
#ifdef RTMach
itc_mutex_t a_mutex;
itc_condition_t a_cond, a_in_cond;
#define A_LOCK() itc_mutex_lock(&a_mutex)
#define A_UNLOCK() itc_mutex_unlock(&a_mutex)
#else /* RTMach */
struct mutex a_mutex;
struct condition a_cond, a_in_cond;
#define A_LOCK() mutex_lock(&a_mutex)
#define A_UNLOCK() mutex_unlock(&a_mutex)
#endif /* RTMach */


#define MAXMS 512

static void ascii_eat_msg(InHeadP, OutHeadP)
  mach_msg_header_t *InHeadP, *OutHeadP;
{
    mig_reply_header_t *OutP = (mig_reply_header_t *)OutHeadP;
    mach_msg_header_t *InP = InHeadP;

    OutP->Head.msgh_bits = MACH_MSGH_BITS(
				MACH_MSGH_BITS_REMOTE(InP->msgh_bits), 0);
    OutP->Head.msgh_size = sizeof *OutP;
    OutP->Head.msgh_remote_port = InP->msgh_remote_port;
    OutP->Head.msgh_local_port = MACH_PORT_NULL;
    OutP->Head.msgh_seqno = 0;
    OutP->Head.msgh_id = InP->msgh_id + 100;
    OutP->RetCodeType.msgt_name = MACH_MSG_TYPE_INTEGER_32;
    OutP->RetCodeType.msgt_size = 32;
    OutP->RetCodeType.msgt_number = 1;
    OutP->RetCodeType.msgt_inline = TRUE;
    OutP->RetCodeType.msgt_longform = FALSE;
    OutP->RetCodeType.msgt_deallocate = FALSE;
    OutP->RetCodeType.msgt_unused = 0;
    OutP->RetCode = KERN_SUCCESS;
}
#define REQUEST_SIZE MAXMS + sizeof(mach_msg_header_t)


/* eventwait -- wait for event or timeout */
/*
 * wait for an IPC indicating either a MIDI or an ascii event
 * note that events are not actually fully handled here, just sent
 * to the appropriate buffering mechanism
 * the caller of eventwait is responsible for dealing with events
 */
void eventwait(timeout)
long timeout;
{
    static char bufReq[REQUEST_SIZE], bufRep[REQUEST_SIZE];
    mig_reply_header_t *bufRequest, *bufReply, *bufTemp;
    mach_msg_return_t ret;
    long ntimeout;

    bufRequest = (mig_reply_header_t *)&bufReq;
    bufReply = (mig_reply_header_t *)&bufRep;
    ntimeout = timeout - gettime(); /* convert to millisecond delay */
    while (ntimeout > 0) {
	    /* wait for a mach message, or a timeout */
	    ret = mach_msg(&bufRequest->Head,  /* msg */
		       MACH_RCV_MSG|MACH_RCV_TIMEOUT, /* option */
		       0,                             /* send_size */
		       MAXMS,                         /* rcv_size */
		       portset,                       /* rcv_name */
		       ntimeout,                      /* timeout */
		       MACH_PORT_NULL);               /* notify */
	if (ret == MACH_MSG_SUCCESS) {
	    /* we have a valid mach message */
	    if (bufRequest->Head.msgh_local_port == MI_RCV_PORT(midiconn)) {
		/* it's a MIDI event, dispatch it to the appropriate server */
		midi_reply_server(&bufRequest->Head, &bufReply->Head);
	    } else if (bufRequest->Head.msgh_local_port == ascii_notify_port) {
		/* it's an ascii event, clean up message state */
		ascii_eat_msg(&bufRequest->Head, &bufReply->Head);
	    }
	    if ((bufReply->RetCode != KERN_SUCCESS) && 
		(bufReply->RetCode != MIG_NO_REPLY)) {
		bufRequest->Head.msgh_local_port = MACH_PORT_NULL;
		mach_msg_destroy(&bufRequest->Head);
	    }
	    if ((bufReply->RetCode == KERN_SUCCESS) || 
		(bufReply->RetCode != MIG_NO_REPLY)) {
		if (bufReply->Head.msgh_local_port == MACH_PORT_NULL) {
		    mach_msg_destroy(&bufRequest->Head);
		} else {
		    bufTemp = bufRequest;
		    bufRequest = bufReply;
		    bufReply = bufTemp;

		    /* send the response */
		    ret = mach_msg(&bufRequest->Head,
			((MACH_MSGH_BITS_REMOTE(bufRequest->Head.msgh_bits) ==
			  MACH_MSG_TYPE_MOVE_SEND_ONCE) ? 
			 MACH_SEND_MSG|MACH_RCV_MSG : 
			 MACH_SEND_MSG|MACH_SEND_TIMEOUT|MACH_RCV_MSG),
			bufRequest->Head.msgh_size, MAXMS, portset,
			0, MACH_PORT_NULL);
		}
	    }
	}
	return;
    }
    return;
}


#ifdef UNIX_MACH
/*
 * indicate that threads should terminate themselves,
 * wake any threads blocking on ascii conditions
 */
static void mach_cleanup(void *garbage)
{
	i_am_running = 0;
#ifdef RTMach
	itc_condition_broadcast(&a_cond);
	itc_condition_broadcast(&a_in_cond);
#else /* RTMach */
	condition_broadcast(&a_cond);
	condition_broadcast(&a_in_cond);
#endif /* RTMach */
}


/* ascii_send_notify -- send ascii notification IPC for eventwait() */
/**/
static void ascii_send_notify(mach_port_t port, int data)
{
    typedef struct {
	mach_msg_header_t Head;
	mach_msg_type_t dataType;
	int data;
    } Request;
    union {
	Request In;
    } Mess;
    Request *InP = &Mess.In;

    InP->dataType.msgt_name = MACH_MSG_TYPE_INTEGER_32;
    InP->dataType.msgt_size = 32;
    InP->dataType.msgt_number = 1;
    InP->dataType.msgt_inline = TRUE;
    InP->dataType.msgt_longform = FALSE;
    InP->dataType.msgt_deallocate = FALSE;
    InP->dataType.msgt_unused = 0;
    InP->data = data;
    InP->Head.msgh_bits = MACH_MSGH_BITS(MACH_MSG_TYPE_COPY_SEND,0);
    InP->Head.msgh_remote_port = port;
    InP->Head.msgh_local_port = MACH_PORT_NULL;
    InP->Head.msgh_seqno = 0;
    InP->Head.msgh_id = 45000;
    mach_msg(&InP->Head, MACH_SEND_MSG|MACH_MSG_OPTION_NONE, 
	sizeof(Request), 0, MACH_PORT_NULL, MACH_MSG_TIMEOUT_NONE, 
	MACH_PORT_NULL);
}

/* ascii_thread -- async. ascii input support */
/*
 * loop continuously, reading characters, make them
 * available to other threads
 * send an IPC to inform eventwait() that an ascii event
 * has occurred
 *
 * signal a condition so wait_ascii and friends know
 * whan a character has come in
 */
static void ascii_thread(int garbage)
{
    kern_return_t ret;
    mach_port_t port;
    int in;

    ret = netname_look_up(name_server_port, "*", 
			  ascii_thread_port_name, &port);
    if (ret != KERN_SUCCESS) {
	gprintf(FATAL, "could not relookup ascii port (%s)\n", 
		mach_error_string(ret));
	EXIT(1);
    }
    do {
	in = fgetc(stdin);
	if (!i_am_running) {
#ifdef RTMach
	    itc_thread_exit(0);
#else /* RTMach */
	    cthread_exit(0);
#endif /* RTMach */
	}
	A_LOCK();
	while(a_in_flag) {
#ifdef RTMach
	    itc_condition_wait(&a_cond, &a_mutex);
#else /* RTMach */
	    condition_wait(&a_cond, &a_mutex);
#endif /* RTMach */
	}
	a_in_flag=1;
	a_in = (char)in;
	_UNLOCK();
#ifdef RTMach
	itc_condition_broadcast(&a_in_cond);
#else /* RTMach */
	condition_broadcast(&a_in_cond);
#endif /* RTMach */
	ascii_send_notify(port, in);
    } while (i_am_running);
#ifdef RTMach
    itc_thread_exit(0);
#else /* RTMach */
    cthread_exit(0);
#endif /* RTMach */
}
#endif /* UNIX_MACH */


/* mach_midi_init -- mach-specific setup */
/**/
void mach_midi_init()
{
#ifdef UNIX_MACH
    static test_opts opts;
    kern_return_t ret;
#endif

#ifdef UNIX_MACH
    /* we want to handle input ourselves so eventwait can know when 
     * MIDI input arrives */
    mi_get_default_opts(&opts);
    opts.flags |= OPT_NO_AUTO_INPUT;
    midiconn = mi_open((char *)&opts);
    ret = mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_RECEIVE,
			     &ascii_notify_port);
    if (ret != KERN_SUCCESS) {
	gprintf(FATAL, "could not allocate ascii notification port (%s)\n", 
		mach_error_string(ret));
	    EXIT(1);
    }
    sprintf(ascii_thread_port_name, "CMT ascii input thread for %d", getpid());
    ret = netname_check_in(name_server_port, ascii_thread_port_name,
			   MACH_PORT_NULL, ascii_notify_port);
    if (ret != KERN_SUCCESS) {
	gprintf(FATAL, "could not register ascii input port (%s)\n",
		mach_error_string(ret));
	EXIT(1);
    }
    /* create a portset for eventwait to use */
    ret = mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_PORT_SET,
			     &portset);
    if (ret != KERN_SUCCESS) {
	gprintf(FATAL, "could not allocate notification portset (%s)\n",
		mach_error_string(ret));
	EXIT(1);
    }
    /* listen for ascii events */
    ret = mach_port_move_member(mach_task_self(), ascii_notify_port, portset);
    if (ret != KERN_SUCCESS) {
	gprintf(FATAL, 
		"could not put ascii port into notification portset (%s)\n",
		mach_error_string(ret));
	EXIT(1);
    }
    if (midiconn) {
	/* also listen for MIDI events */
	ret = mach_port_move_member(mach_task_self(), MI_RCV_PORT(midiconn),
				    portset);
	if (ret != KERN_SUCCESS) {
	    gprintf(FATAL, 
		    "could not put MIDI port into notification portset (%s)\n",
		     mach_error_string(ret));
	    EXIT(1);
	}
    }
    cu_register(mach_cleanup, (void *)midiconn);
#ifdef RTMach
    itc_mutex_init(&a_mutex);
    itc_condition_init(&a_cond);
    itc_condition_init(&a_in_cond);
    /* run the ascii input thread */
    fixpri_rtthread_fork(ascii_thread, 0, 12);
#else /* RTMach */
    mutex_init(&a_mutex);
    condition_init(&a_cond);
    condition_init(&a_in_cond);
    /* run the ascii input thread */
    cthread_detach(cthread_fork(ascii_thread, 0));
#endif /* RTMach */
    cu_register((cu_fn_type) mi_close, (void *) midiconn);
}
