/* dummyif.c -- support for no interface (allows, e.g., ceating a .mid)

/* Copyright 1995 Intelligistics, Inc. */

/*****************************************************************************
* Change Log
*   Date    | who : Change
*-----------+-----------------------------------------------------------------
* 16-May-93 |
* 10-Dec-95 | GWL : first draft
*****************************************************************************/

/*
	dummif.c replaces Leigh Smith's mpu.c but retains its
	communications protocol via midifns.c.  Output is
	totally suppressed but otherwise the music processes run.
	There is no input supported at this time.

	externals satisfied:

	mOpen()			set up interrupt service
	mClose()		undo the above
	mPutData()		acknowledge the request to send a byte
	mPutCmd()		acknowledge the request to send command
	xbuff			external sysex buffer, CMT standard
	xbufhead
	xbuftail
	xbufmask
	buff			MIDI data buffer
	buffhead
	bufftail
	midi_flush()		empty out buffers
	check_midi()		input all available data from MIDIator

	midifns.c calls check_midi to scoop in data and notify re the presence
	thereof by setting buffhead != bufftail.

	Use option -trace to see the progress of linkage.

	Code here includes Roger's new MIDI filter.
*/

#include "switches.h"
#include <stdlib.h>     /* for ANSI malloc */
#include "cext.h"
#include <stdio.h>      /* for NULL */
#include <signal.h>     /* for ^C trapping */
#include <dos.h>        /* for FP_SEG, FP_OFF */
#include <bios.h>       /* for GETTIMEOFDAY */
#include "mpu.h"        /* for MPU-401 constants */
#include "midibuff.h"
#include "midicode.h"
#include "userio.h"	/* for abort_flag */
#include "midierr.h"	/* error flags */
#include <conio.h>
#include "dummyif.h"

#ifdef MICROSOFT	/* for Microsoft C V5.1 and above */
#define DISABLEINTS     _disable
#define ENABLEINTS      _enable
#define INTERRUPT	interrupt far
#define GETVECTOR       _dos_getvect
#define SETVECTOR       _dos_setvect
#define GETTIMEOFDAY(a) _bios_timeofday(_TIME_GETCLOCK, &(a))
#else           	/* for Turbo C V2.0 and above */
#define DISABLEINTS     disable
#define ENABLEINTS      enable
#define INTERRUPT       interrupt
#define GETVECTOR       getvect
#define SETVECTOR       setvect
#define GETTIMEOFDAY(a) (a = biostime(0, 0))
#endif

#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

#define DBG_RECTIME     0x7d4	/* record time assign debug */
#define DBG_ISRBYTE     0x7d6	/* where to dump the ISR byte debug */
#define DBG_UNH_SYS     0x7d8	/* unhandled sys-ex debug */
#define DBG_RETDATA	0x7da	/* returned data debug */
#define DBG_CLK2HST	0x7dc	/* clock to host called debug */

#define debug(a, b, c)

#define MPU_ACK		0xFE
#define	IRQ9MASK	(2)

static void MPUisr(unsigned);	/* MPUisr called with MIDIator data */
static void (*ClockToHostFn)(void);	/* Ptr to fn called each MPU Clock */
static int  PlayFini = FALSE;	/* TRUE when the play data is at the end */
static int  UartMode = TRUE;	/* TRUE when MPU is in UART mode */
static int  CmdAck;		/* MPU-401 command acknowledge flag */
static int  ExpectingDataAfterCmd; /* Flag indicating MPU will return a byte */
static byte ReturnedData;	/* The byte MPU has returned */ 
static unsigned long RecordTime; /* No of MPU ticks since mRecord called */
static int  verbose=FALSE;	/* TRUE = show where MIDIator is */

byte huge *xbuff = NULL;	/* application-supplied sysex buffer */
long xbufmask;			/* mask for circular buffer */
long xbufhead = 0;      	/* buffer head and tail offsets */
long xbuftail = 0;
static long old_xbuftail;

boolean xbuf_flush = false;	/* says to flush remainder of sysex message */
MidiMsg SysexHeader;	/* saves 1st 3 bytes of sysex messages for buffer */
byte SysexCounter = 0;	/* index into SysexHeader */

/* midi input buffer */
/* data buffer, declared long to get 32-bit alignment: */
long buff[BUFF_SIZE/4]; 
int buffhead = 0;    		/* buffer head and tail pointers */
int bufftail = 0;

private MidiMsg CurrentMsg, RealTimeMsg;
private int data_len;
private int data_rcv;
private boolean firstbyte;

extern boolean ctrlFilter;
extern boolean exclFilter;
extern boolean realFilter;

extern int musictrace;

#define set_error(bit) midi_error_flags |= (1 << bit);

/*	MS-101 data areas */

byte	MSport=0;		/* MIDI port, on MS-101, only one, =0 */
int	MSclkfrq;
int	MScport=1;		/* COMx index */
int	MScomp=7;		/* CPU speed compensation factor */
int	MSclock=0;
int	MSmodel=101;		/* MS model, fixed here at 101 */
int	MSclkmin=392;		/* MS-114 uses different values */
int	MSclkmax=612;

/***************
* LOCAL TABLES *
***************/

private byte VoiceTable[8] = {2,2,2,2,1,1,2,0};
/* note: length entry for EOX MUST BE ZERO: */
private byte CommonTable[8] = {0,1,2,1,0,0,0,0};

/* midi_flush -- empty out buffers */
/**/
void midi_flush()
{
    buffhead = 0;
    bufftail = 0;
    xbufhead = 0;
    xbuftail = 0;
    xbuf_flush = true;	/* in case sysex continuation messages are still coming */
}

/*
 * #define dbmax 32
 * int dbb[dbmax];
 * char * dbs[dbmax];
 * int dbx = 0;
 * int dbh = 0;
 */



/*
    For now, no input data
*/
boolean check_midi(void)
{
    return false;
}

/*
    Dummy output data
*/
void mPutData(byte dat)
{
}


/*
    Dummy output cmmand
*/

void mPutCmd(byte cmd)
{
}


/* PutMidiMsg -- copy complete message from interrupt to buffer */
/**/
static void PutMidiMsg(MidiMsg msg)
{
    register byte *cmt_data = ((byte *) buff) + bufftail;

    /* filter out control changes */
    if (ctrlFilter) {
	register int data1 = databyte1(msg);
	register int hibits = statusbyte(msg) & 0xF0;
	if (hibits == 0xD0 ||   /* Chan Pressure */
	    hibits == 0xE0 ||       /* Pitch Bend */
	    hibits == 0xA0 ||       /* Poly Pressure */
	    ((hibits == 0xB0) &&    /* Control change (don't count switches) */
	     ((data1 < 64) || (data1 > 121)))) {
	    /* CONTROL MESSAGE HAS BEEN FILTERED */
	    return;
	}
    }
    if (realFilter) {
	if (statusbyte(msg) >= 0xF8) return;
    }
    if (exclFilter) {
	if (statusbyte(msg) == MIDI_SYSEX) return;
    }
    *((long *) cmt_data) = msg.l;
    bufftail = (bufftail + 4) & BUFF_MASK;
    if (bufftail == buffhead) {
	/* filled buffer faster than client emptied it */
	set_error(BUFFOVFL);
    }
}


/*
    Dummy close music processing
*/
int mClose(int err)
{
    if (verbose) printf("End music processing\n");
    return TRUE;
}


/*
** If a Control-C or Abort interrupt occurs, exit gracefully,
** disabling the interrupt if it had been assigned.
*/
static void AbEnd(int i)
{
    abort_flag = true;
}


/*
    Dummy open music processing
*/
int mOpen(unsigned int BaseAddr, int IrqNo)
{
    int		n;		/* working index */

    verbose=cl_switch("trace");

    setstatus(CurrentMsg, 0);
    firstbyte = TRUE;

    /* SIGINT handler is set up by USERIO.C.  It causes a cu_cleanup
     * that in turn calls mClose, so we shouldn't override the handler
     * here.  On the other hand, I don't set up a handler for SIGABRT
     * elsewhere, so I'm leaving in this handler.  Depending on how
     * and when SIGABRT can be raised, it is probably a bug to call
     * AbEnd and not do other cleanup as well.
     */
    /* if(signal(SIGINT, AbEnd) == SIG_ERR)
	return(ABEND_ERR); */
    if(signal(SIGABRT, AbEnd) == SIG_ERR)
	return(ABEND_ERR);

    if (verbose) printf("Begin music processing\n");
    return(DUMMY_OK);		/* code for OK */
}

/* ----- Routines used by MPUisr, not to be used by other programs ------ */

static void UndoSysexMsg()
{
    xbuftail = old_xbuftail;
}

static boolean PutSysExByte(byte value)
{
    return TRUE;
}


/************************************************************
*			    MidiParse
*
* Input : A midi byte.
* Output: none
* Return: none
* Effect:
***************************************************************/

private void MidiParse(int c)
{
#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

   /* REALTIME MESSAGES */
    if ((c & MIDI_REALTIME) == MIDI_REALTIME) {
	setstatus(RealTimeMsg, c);
	PutMidiMsg(RealTimeMsg);
	return;
    }

    /* STATUS MESSAGES */
    if (c & MIDI_STATUS_BIT) {
	if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* END OF EXCLUSIVE */
	    if (PutSysExByte(MIDI_EOX))
		PutMidiMsg(SysexHeader);
	    firstbyte = TRUE;
	}
	setstatus(CurrentMsg, c);
	data_rcv = 0;
	if ((c & MIDI_COMMON) == MIDI_COMMON) {
	    data_len = CommonTable[c & LOWBITS];
	    if (data_len == 0) {
		if (c != MIDI_EOX) {
		    if (c != MIDI_SYSEX) PutMidiMsg(CurrentMsg);
		} else setstatus(CurrentMsg, 0); /* make sure EOX is ignored */
	    }
	} else data_len = VoiceTable[(c & MIDI_COMMON) >> 4];
	return;
    }

    /* EXCLUSIVE DATA */
    if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* PUT IN EXCL BUFFER */
	if (firstbyte) {
	    xbuf_flush = false;
	    PutSysExByte(MIDI_SYSEX);
	    firstbyte = FALSE;
	}
	PutSysExByte((byte) c);
	return;
    }

    /* VOICE MESSAGE DATA */
    if (statusbyte(CurrentMsg)) { /* use zero to mean "status unknown" */
	setdata(CurrentMsg, ++data_rcv, c);
	if (data_rcv >= data_len) {
	    PutMidiMsg(CurrentMsg);
	    midimsg(CurrentMsg) &= CLEAR_DATA;
	    data_rcv = 0;
	}
    }
}


/*
** Interrupt service routine for data from MPU-401.
** refer algorithm pp13 MPU-401 TRM.
**
** Modified to service data from MIDIator
*/
void MPUisr(unsigned inword)
{
    static byte MPUisr_dat;

    MPUisr_dat=inword;
    debug(MPUisr_dat, 2, DBG_ISRBYTE);

    /* if we are in UART mode, always for MIDIator */

    if (UartMode) {
	MidiParse(MPUisr_dat);	/* parse the data */
    } else if (MPUisr_dat == MPU_ACK) { /* o.w. interpret mode */
	CmdAck = TRUE;		/* command has been acknowledged */
	if (ExpectingDataAfterCmd) {	/* there is data following */
	    ExpectingDataAfterCmd = FALSE;
	    debug(ReturnedData, 3, DBG_RETDATA);
	}
    }
}
