/* camdmidi.c -- Amiga specific MIDI interface code */

#include "switches.h"
#include "midi/camd.h"
#include "clib/camd_protos.h"
#ifdef AZTEC

/* note: azt_camd_pragmas.h was produced by running MAPFD on the
 * lib/camd_lib.fd file included in the CAMD disk from Commodore.
 * The "CamdClient" calls are manually removed from 
 */

#include "functions.h"
#include "pragmas/azt_camd_pragmas.h"

#else
#include "amiga.h"
#include "pragmas/lat_camd_pragmas.h"
#endif

#include "camdmidi.h"
#include "ctype.h"

#include "cext.h"
#include "stdio.h"
#include "midicode.h"
#include "cmdline.h"
#include "userio.h"
#include "midifns.h"

public struct CamdBase *CamdBase;
struct MidiNode *cmt_mi;
public ULONG timeoffset = 0L;
ULONG event_mask = 0L;
public MidiMsg cmt_msg;
public ULONG *camdtime;
public UBYTE amigaerrflags;

#define SIZE 1024L
ULONG CreateMidiTagArray[] = {CMA_Alarm, 1L, TAG_END, 0L};


/* amiga_midi_init -- set up some Amiga specifics for midi */
/**/
void amiga_midi_init()
{
    char *s;
    ULONG port_mask = 0L;
    
    CamdBase = (struct CamdBase *)OpenLibrary(CamdName, CamdVersion);
    cu_register((cu_fn_type) close_camd, CamdBase);
    if (!(cmt_mi = CreateMidiA(SIZE, (struct TagItem *) CreateMidiTagArray)))
/* CMA_Alarm, 1L, TAG_END))) */
	EXIT(1);
    cu_register(delete_midi, cmt_mi);
    SetMidiErrFilter(cmt_mi, CMEF_All);
    if (s = cl_option("inport")) {
	register char c;
	while (c = *s++) {
	    if (isupper(c)) c = tolower(c);
	    if (c >= '0' && c <= '9') {
		port_mask |= 1L << (c - '0');
	    } else if (c >= 'a' && c <= 'f') {
		port_mask |= 1L << (c + 10 - 'a');
	    }
	}
	gprintf(TRANS, "Input port mask %lx\n", port_mask);
    } else port_mask = 1L << CMP_In(0);
    SetMidiFilters(cmt_mi, port_mask,
		   (CMF_Note | CMF_Prog | CMF_CtrlSwitch | CMF_Mode),
		   0xFFFFL /* all chans */);

    if (s = cl_option("outport")) {
	register ULONG port_number;
	port_number = atoi(s);
	if (strlen(s) == 1) {
	    register char c = *s;
	    if (isupper(c)) c = tolower(c);
	    if (c >= 'a' && c <= 'f') {
		port_number = c + 10 - 'a';
	    }
	}
	SetMidiPort(cmt_mi, port_number);
	gprintf(TRANS, "Output to port %ld\n", port_number);
    }
    camdtime = MidiTime();
}


/* close_camd -- a cleanup routine */
/**/
void close_camd()
{
    /* CloseLibrary has special register passing conventions
     * the compiler handles according to a pragma -- I assume
     * we should not call it via a function pointer from cu_cleanup
     * so use cu_register(close_camd, anything) to do it.
     */
    CloseLibrary(CamdBase);
}


/* delete_midi -- a cleanup procedure for the Amiga */
/**/
void delete_midi(void *obj)
{
    DeleteMidi(cmt_mi);
}


/* eventwait -- see midifns.c for documentation */
/**/
void eventwait(timeout)
  long timeout;
{
    if (timeout >= 0) {
	long alarm_time = timeoffset + (timeout >> 1);
	/* NOTE: SetMidiAlarm returns true only if alarm will arrive later.
	   Otherwise, alarm is not set because the time has expired.  In
	   this case, don't wait for anything!
	 */
	if (!SetMidiAlarm(cmt_mi, alarm_time)) {
	    return;
	}
    }
    Wait(/*event_mask*/ 0xFFFFFFFF);
}


/* midi_write -- see midifns.c for documentation */
/**/
void midi_write(int n, int port, unsigned char c1, unsigned char c2, unsigned char c3)
{
    setstatus(cmt_msg, c1);
    setdata1(cmt_msg, c2);
    setdata2(cmt_msg, c3);
    PutMidi(cmt_mi, &cmt_msg);
    clearmsg(cmt_msg);
    midi_write_trace(n, port, c1, c2, c3);
}


void set_error_flags(UBYTE amigaerrflags)
{
    if (amigaerrflags & CMEF_MsgErr)
	midi_error_flags |= (1<<MSGERR);
    if (amigaerrflags & CMEF_BufferFull)
	midi_error_flags |= (1<<BUFFOVFL);
    if (amigaerrflags & CMEF_SysExFull)
	midi_error_flags |= (1<<SYSEXOVFL);
    if (amigaerrflags & (CMEF_ParseMem | CMEF_SysExTooBig))
	midi_error_flags |= (1<<MEMERR);
    if (amigaerrflags & (CMEF_RecvErr | CMEF_RecvOverflow))
	midi_error_flags |= (1<<RECVERR);
}
