/* tuning.c -- a scale definition program */

/*****************************************************************************
*           Change Log
*  Date     | Change
*-----------+-----------------------------------------------------------------
* 16-Jun-86 | Created changelog
* 12-Aug-86 | Fixed an off-by-12 indexing bug, use userio.h to open file
*****************************************************************************/

#include "stdio.h"
#include <math.h>
#include "cext.h"
#include "pitch.h"
#include "userio.h"
#include "cmdline.h"

#ifdef THINK_C
#include "console.h"
#endif

#define MINPITCH 0
#define MAXPITCH 127

/****************************************************************************
*       Routines local to this module
****************************************************************************/
private char    getcommand();
private void    getoctave();
private void    getonepitch();
private void    getrange();
private void    readpitch();
private void    writepitch();
private void    ratio_to_cents();

int debug = false; /* used in userio.c, declared in moxc.c, but we're not using moxc.c */

/****************************************************************************
*                               getcommand
* Outputs:
*       returns char: character typed by user
* Effect:
*       prompts for command and gets it
****************************************************************************/

private char getcommand()
{
    char result;
    char trash[100];

    gprintf(TRANS, "\nr - enter a range of values\n");
    gprintf(TRANS, "o - enter one octave (that will be generalized)\n");
    gprintf(TRANS, "p - enter one pitch (you choose)\n");
    gprintf(TRANS, "q - quit\n\n");
    gprintf(TRANS, "Command >> ");

    while ((result = ggetchar()) == '\n') ;
    ggets(trash);
    if (abort_flag) result = 'x';       /* force another request */
    return result;
}

/****************************************************************************
*                               getoctave
* Outputs:
*       pitch_table pit_tab[]: array of pitch data
* Effect:
*       prompts user for one octave of data, then transposes to get others
****************************************************************************/

private void getoctave(pit_tab)
pitch_table pit_tab[];
{
    float ratio[12], num, denom;
    int pitch[12], bend[12], base, key, i, j, count, choice;
    char inputline[100];

    gprintf(TRANS, "\n\nYou will input the information for one octave and that\n");
    gprintf(TRANS, "  will be generalized for all octaves.\n");
    gprintf(TRANS, "\nWould you like to enter the information as\n");
    gprintf(TRANS, "   (1)  frequency ratios of notes to a tonic\n");
    gprintf(TRANS, "or (2)  semitones (integer) + pitchbend (in cents)\n\n");
    gprintf(TRANS, " >> ");
    ggets(inputline);
    sscanf(inputline, "%d", &choice);
    if (choice == 1) {
	gprintf(TRANS, "Please enter ratio x/y as   x y   where x and y are floating point numbers.\n\n");
	gprintf(TRANS, "Note 0 (tonic)  ratio is  1/1  (1 1)\n");
	ratio[0] = 1.0;
	for (i = 0; i < 12; i++) {
	    gprintf(TRANS, "note %d ratio >> ", i);
	    ggets(inputline);
	    if (sscanf(inputline, "%f %f", &num, &denom) != 2) {
		num = 1;
		denom = 1;
	    }
	    ratio[i] = num/denom;
	    gprintf(TRANS, "ratio[%d] = %f\n",i,ratio[i]);
	}
	ratio_to_cents(ratio, bend, pitch);
    } else {
	gprintf(TRANS, "\n\nFor the given note number, enter the desired pitch\n");
	gprintf(TRANS, "and the amount of pitchbend in cents based on given tonic\n");
	gprintf(TRANS, "\n1 unit of pitch = 1 semitone -- the tonic (note 0) has pitch 0 and bend 0.\n");
	gprintf(TRANS, "   Bend range =  -100 .. 100 cents,  0 for no pitch bend\n");
	gprintf(TRANS, "     (100 cents = 1 semitone)\n");
	gprintf(TRANS, "\n\n");
	bend[0] = pitch[0] = 0;
	for (i = 0; i <12; i++) {
	    gprintf(TRANS, "Octave note %d\n", i);
	    gprintf(TRANS, "   pitch >> ");
	    ggets(inputline);
	    if (sscanf(inputline, "%d", &pitch[i]) != 1) pitch[i] = 60;
	    gprintf(TRANS, "   bend >> ");
	    ggets(inputline);
	    if (sscanf(inputline, "%d", &bend[i]) != 1) bend[i] = 0;
	}
    }
    gprintf(TRANS, "\nWhat note would you like your octave to start on?\n\n");
    gprintf(TRANS, "C   C#  D   D#  E   F   F#  G   G#  A   A#  B\n");
    gprintf(TRANS, "0   1   2   3   4   5   6   7   8   9   10  11\n");
    gprintf(TRANS, " >> ");
    ggets(inputline);
    if (sscanf(inputline, "%d", &key) != 1) key = 0;
    for (i = 0; i < 12; i++) {
	base = i + key;
	if (base > 11) {
	    base -= 12; key -=12;
	}
	/* 0 <= base <= 11 */
	/* remember that pitches range from MINPITCH to MAXPITCH: */
	for (count = MINPITCH/12, j = base+MINPITCH; j <= MAXPITCH; j+=12) {
	    pit_tab[j-MINPITCH].ppitch = key + pitch[i] + count*12;
	    pit_tab[j-MINPITCH].pbend = bend[i];
	    count++;
	}
    }
}

/****************************************************************************
*                               getonepitch
* Inputs:
*       pitch_table pit_tab[]: table to be modified
* Outputs:
*       pitch_table pit_tab[]: table with modifications is returned
* Effect: 
*       prompts user for a note to change, gets data, modifies table
****************************************************************************/

private void getonepitch(pit_tab)
pitch_table pit_tab[128];
{
    int  pitch = 0;
    int  bend = 0;
    int index, finished = 0;
    char inputline[100];

    while (!finished & !abort_flag) {
	gprintf(TRANS, "Note range is 0..127.  Which note would you like to enter ?");
	gprintf(TRANS, "\n (Q to quit)\n");
	gprintf(TRANS, "Note >> ");
	ggets(inputline);
	if (sscanf(inputline, "%d", &index) != 1) {
	    finished = true;
	} else {
	    gprintf(TRANS, "\n\nFor the given pitch number, enter pitch paramaters\n");
	    gprintf(TRANS, "   Bend range =  -100 .. 100 cents,  0 for no pitch bend\n");
	    gprintf(TRANS, "     (100 cents = 1 semitone)\n");
	    gprintf(TRANS, "   Pitch range = 0 .. 127, 60 is middle C\n\n\n");

	    gprintf(TRANS, "Adagio note %d\n", index);
	    gprintf(TRANS, "   pitch >> ");
	    ggets(inputline);
	    if (sscanf(inputline, "%d", &pitch) != 1) pitch = 60;
	    gprintf(TRANS, "   bend >> ");
	    ggets(inputline);
	    if (sscanf(inputline, "%d", &bend) != 1) bend = 0;
	    pit_tab[index - MINPITCH].pbend = bend;
	    pit_tab[index - MINPITCH].ppitch = pitch;
	}
    }
}

/****************************************************************************
*                               getrange
* Inputs:
*       pitch_table pit_tab[128]: a table of pitch info
* Outputs:
*       pitch_table pit_tab[128]: modified table of pitch info
* Effect: 
*       asks user for a range, then prompts for pitch data within that
*       range
****************************************************************************/

private void getrange(pit_tab)
    pitch_table pit_tab[128];
{
    int i,low,high,bend,pitch;
    char inputline[100];

    gprintf(TRANS, "Adagio note range is 0..127\nWhat range of notes would you like to enter ?\n");
    gprintf(TRANS, "From >> ");
    ggets(inputline);
    if (sscanf(inputline, "%d", &low) != 1) low = 0;;
    gprintf(TRANS, "  To >> ");
    ggets(inputline);
    if (sscanf(inputline, "%d", &high) != 1) high = 127;

    gprintf(TRANS, "\n\nFor the given Adagio note number, enter the desired pitch\n");
    gprintf(TRANS, "and the amount of pitchbend in cents\n");
    gprintf(TRANS, "   Bend range =  -100 .. 100,  '0' for no pitch bend\n");
    gprintf(TRANS, "     (100 cents = 1 semitone)\n");
    gprintf(TRANS, "   Pitch range = 0 .. 127, 60 is middle C\n");
    gprintf(TRANS, "\n\n");
    for (i = low; i <= high; i++)
    {
	gprintf(TRANS, "Adagio Note %d\n", i);
	gprintf(TRANS, "   pitch >> ");
	ggets(inputline);
	if (sscanf(inputline, "%d", &pitch) != 1) pitch = 60;
	gprintf(TRANS, "   bend >> ");
	ggets(inputline);
	if (sscanf(inputline, "%d", &bend) != 1) bend = 0;
	pit_tab[i - MINPITCH].pbend = bend;
	pit_tab[i - MINPITCH].ppitch = pitch;
    }
}

/****************************************************************************
*                               main
* Inputs:
*       int argc: number of command line arguments
*       char *argv[]: vector of command line arguments
* Effect: main program
****************************************************************************/

void main(argc, argv)
    int  argc;
    char *argv[];
{
    int i, done = 0;
    FILE *fp;
    char *s;    /* the filename argument, if any */
    char filename[100];
    pitch_table pit_tab[128];
#ifdef THINK_C
    /* Macintosh needs help to provide Unix-style command line */
    argc = ccommand(&argv);
#endif
    io_init();
    if (!cl_init(argv, argc)) {
	gprintf(TRANS, "Type anything to exit...");
#ifdef  DOS
	wait_ascii();
#else
	ggetchar();
#endif
	EXIT(1);
    }

    /* get output file: */
	if (s = cl_arg(1)) {
	    strcpy(filename, s);
	    if (s = cl_arg(2))
		gprintf(ERROR, "Extra (filename?) argument ignored: %s\n", s);
	} else strcpy(filename, "");
	fp = fileopen(filename, "tun", "w", "Name of tuning file");
	if (!fp) goto finish;

    for (i = MINPITCH; i <= MAXPITCH; i++) { /* initialize pit_tab */
	pit_tab[i - MINPITCH].pbend = 0;
	pit_tab[i - MINPITCH].ppitch = i;
    }

    gprintf(TRANS, "You will now create a pitch file.\n\n");
    while (!done & !abort_flag) {
	switch (getcommand()) {
	    case 'r': getrange(pit_tab); break;
	    case 'p': getonepitch(pit_tab); break;
	    case 'o': getoctave(pit_tab); break;
	    case 'q': done = 1; break;
	    default: break;
	}
	if (abort_flag == BREAK_LEVEL) abort_flag = false;
    }
    if (abort_flag != ABORT_LEVEL) writepitch(fp, pit_tab);
 finish:
    EXIT(0);
}

/****************************************************************************
*                               readpitch
* Inputs:
*       FILE *fp: file from which to read
* Outputs:
*       pitch_table pit_tab[128]: this table is initialized by reading fp
****************************************************************************/

private void readpitch(fp, pit_tab)
FILE *fp;
pitch_table pit_tab[128];
{
    int i, j, pitch, bend;

    for ( j = MINPITCH; j <= MAXPITCH; j++)
    {
	fscanf(fp,"%d %d %d\n", &i, &pitch, &bend);
	pit_tab[i].ppitch = pitch;
	pit_tab[i].pbend = bend;
    }
}

/****************************************************************************
*                               writepitch
* Inputs:
*       FILE *fp: the file to write
*       pitch_table pit_tab[128]: the table to write
* Effect: 
*       writes pit_tab to file *fp
****************************************************************************/

void writepitch(fp, pit_tab)
    FILE *fp;
    pitch_table pit_tab[128];
{
    int i;
    boolean changed_flag = false;

    for ( i = MINPITCH; i <= MAXPITCH; i++) {
	int p = pit_tab[i-MINPITCH].ppitch;
	int b = pit_tab[i-MINPITCH].pbend;
	while (p < 0) { p += 12; changed_flag = true; }
	while (p > 127) { p -= 12; changed_flag = true; }
	if (b <= -100) b = -100;
	if (b >= 100) b = 100;
	fprintf(fp,"%d %d %d\n", i, p, b);
    }
    if (changed_flag) {
	gprintf(TRANS, "Warning: one or more pitches shifted by octave(s) to\n");
	gprintf(TRANS, "    fit within range 0-127\n");
    }
}

/****************************************************************************
*                               ratio_to_cents
* Inputs:
*       float ratio[]: an octave of rations to convert
* Outputs:
*       int bend[]: an octave of pitch bends
*       int pitch[]: an octave of pitches
* Effect:
*       converts 12 ratios to pitch + bend pairs
****************************************************************************/

private void ratio_to_cents(ratio, bend, pitch)
float ratio[];
int bend[], pitch[];
{
    int i;
    float cents;

    for (i = 0; i < 12; i++) {
	cents = 1200 * log(ratio[i])/log(2.0);
	pitch[i] = (int)(cents/100);
	bend[i] = (int)(cents - (pitch[i] * 100) + 0.5);
    }
}
