/* midiprt -- Expand MIDI file contents with delta times */

/* Copyright 1995 Intelligistics, Inc. */

/*****************************************************************************
*      Change Log
*    Date   | Change
*-----------+-----------------------------------------------------------------
* 01-Apr-95 | GWL : Created changelog
*****************************************************************************/

#include 	<stdio.h>
#include 	<string.h>
#include	<stdlib.h>

#define		TRUE		1
#define		FALSE		0
#define		MAXLINELEN	79

#define     SOX         0xf0
#define     SONGPOS     0xf2
#define     SONGSEL     0xf3
#define     TUNEREQ     0xf6
#define     EOX         0xf7
#define     CLOCK       0xf8
#define     START       0xfa
#define     CONTINUE    0xfb
#define     STOP        0xfc
#define     ACTSENSE    0xfe
#define     SYSRESET    0xff
#define     METAEVENT   0xff


FILE		*midi_file;
char		midi_file_name[MAXLINELEN];
FILE		*parm_file;
char		parm_file_name[MAXLINELEN];
FILE		*output_file;
char		output_file_name[MAXLINELEN];

short		show_metas;
short		show_meta_details;
short		show_messages;
short		show_message_details;
short		show_sysex;
short		show_sysex_details;
short		show_delta_times;
short		show_run_times;
short		show_current_status;

char		*prog="midiprt";
long		expect;
long		delta_time;
long		run_time;
short		data_per_status[]={2,2,2,2,1,1,2};
short		data_per_system[]={-2,-1,2,1,-1,-1,0,0,0,-1,0,0,0,-1,0,0};
short		cur_status;
short		run_status;
short		run_status_exists;
short		meta_needs_space;
short		status_needs_space;
long		run_status_length;
short		meta_type;
short		linelen;

void		chunk_header(short*);
void        chunk_track(void);
void		line_check(short);
void		line_new(void);
short		next_char(void);
short		next_short(void);
long        next_long(void);
long        next_3long(void);
long        next_varlong(void);
void		next_text(char*);
short		skip_text(long);
void		show_time(void);
void		show_status_nybble(short);
short		show_text(long);
short		show_dec(long);
short		show_hex(long);
void		show_meta(long);
short		show_meta_text(long);
short		show_meta_dec(long);
short		show_meta_hex(long);

char*		ffgets(char*,char*,short);
short		getyes(char*);
void		clean_name(char*);
void		exit_out(short);

void main(int argc,char **argv)
{
	short	num_tracks;

	printf("Parameter file name: ");
	if (!gets(parm_file_name))
	{
		printf("Unable to read parm_file_name\n");
		exit(1);
	}
	if (*parm_file_name)
	{
		if (!(parm_file=fopen(parm_file_name,"r")))
		{
			printf("Unable to open file %s\n",parm_file_name);
			exit(1);
		}
	}
	else
	{
		parm_file=stdin;
	}
	
	if (!ffgets("MIDI file name",midi_file_name,MAXLINELEN)) exit(1);
	clean_name(midi_file_name);
	if (!(midi_file=fopen(midi_file_name,"rb")))
	{
		printf("Unable to open MIDI file \"%s\"\n",midi_file_name);
		exit(1);
	}
	
	if (!ffgets("Output file name",output_file_name,MAXLINELEN)) exit(1);
	clean_name(output_file_name);
	if (*output_file_name)
	{
		if (!(output_file=fopen(output_file_name,"w")))
		{
			printf("Can't open output file \"%s\"\n",output_file_name);
			exit(1);
		}
	}
	else
	{
		output_file=stdout;
	}	

	if (!(show_messages=show_message_details=show_metas=show_meta_details=
			show_delta_times=show_run_times=show_current_status=
			show_sysex=show_sysex_details=
			getyes("Show all?")))
	{
		if (show_messages=show_message_details=getyes("Show messages?"))
			show_message_details=getyes("Show message details?");
			
		if (show_metas=show_meta_details=getyes("Show meta events?"))
			show_meta_details=getyes("Show meta details?");
	
		if (show_sysex=show_sysex_details=getyes("Show sysex occurrences?"))
			show_sysex_details=getyes("Show sysex details?");
		
		if (show_message_details || show_meta_details || show_sysex_details)
		{
			show_delta_times=getyes("Show delta times?");
			show_run_times=getyes("Show running times?");
		}
		if (show_messages)
			show_current_status=getyes("Expand running status?");

	}
				
	if (output_file==stdout) printf("\n");
	fprintf(output_file,"File: %s\n\n",midi_file_name);

	/* expect a header chunk */

	chunk_header(&num_tracks);

	/* and specified number of track chunks */

	while (num_tracks--)
		chunk_track();

	/* and EOF */

	(void)next_char();
	exit_out(0);
}

/**************************************************************************************
	analyze header chunk
**************************************************************************************/

void chunk_header(short *num_tracks)
{
	short	ppq, smpte_format, smpte_ticks;

	next_text("MThd");
	expect=4;
	expect=next_long();
	fprintf(output_file,"length: %ld: ",expect);
	if (expect<6)
	{
		fprintf(output_file,"not enough! ");
	}
	fprintf(output_file,"format %d ",next_short());
	fprintf(output_file,"#tracks %d ",*num_tracks=next_short());
	if ((ppq=next_short())>=0)
	{
		fprintf(output_file,"ppq %d",ppq);
	}
	else
	{
		smpte_format = -(ppq& 0xff00)>>8;
		smpte_ticks = ppq&0x00ff;
		fprintf(output_file,"SMPTE format %d ticks/frame %d",
			smpte_format, smpte_ticks);
	}
	if (expect)			/* skip any superfluous data */
	{
		fprintf(output_file," extra data");
		skip_text(expect);
	}
	fprintf(output_file,"\n\n");
}


/**************************************************************************************
	analyze a track chunk
**************************************************************************************/

void chunk_track(void)
{
	short	c;
	long    length;
	short	more;

	next_text("MTrk");
	expect=4;
	expect=next_long();
	fprintf(output_file,"length: %ld\n",expect);
	linelen=0;
	run_time=0;
	status_needs_space=meta_needs_space=run_status_exists=more=FALSE;
	while(expect>0)
	{
		run_time+=(delta_time=next_varlong());
		c=next_char();
		if (c==SOX) 		       /* Sysex */
		{
			length=next_varlong();
			run_status_exists=FALSE;
			if (show_sysex)	
			{
				if (more)
				{
					line_check(7);
					fprintf(output_file," no EOX");
				}
				show_time();
				line_check(10);
				fprintf(output_file," sex (%ld): ",length);
				if (show_sysex_details)
				{
					if (show_hex(length)!=EOX)
						more=TRUE;
					else
						more=FALSE;
					line_new();
				}
			}
			else skip_text(length);
		}
		else if (c==EOX)         /* Sysex or continuation */
		{
			run_status_exists=FALSE;
			length=next_varlong();
			if (show_sysex)
			{
				show_time();
				line_check(11);
				fprintf(output_file," csex (%ld): ",length);
				if (show_sysex_details)
				{
					if (show_hex(length)!=EOX)
						more=TRUE;
					else
						more=FALSE;
					line_new();
				}
			}
			else skip_text(length);
		}
		else if (c==METAEVENT)   /* Meta event */
		{
			meta_type=next_char();
			length=next_varlong();
			run_status_exists=FALSE;
			if (show_metas)
			{
				if (show_meta_details)
				{
					show_time();
					show_meta(length);
				}
				else
				{
					if (meta_needs_space)
					{
						line_check(1);
						if (linelen>1) fprintf(output_file," ");
						else linelen=0;
					}
					line_check(12);
					fprintf(output_file,"meta %x(%ld)",meta_type,length);
					status_needs_space=meta_needs_space=TRUE;
					skip_text(length);
				}
			}
			else skip_text(length);
		}
		else if (((cur_status=c)&0xf0)==0xf0)	/* system status are invalid */
		{
			line_new();
			fprintf(output_file," ??? %x",c);
			exit_out(1);
		}
		else
		{
			if (cur_status & 0x80)			/* new running status */
			{
				run_status=cur_status;
				run_status_length=data_per_status[(cur_status>>4)-8];
				run_status_exists=TRUE;
				if (show_messages)
				{
					if (show_message_details)
					{
						show_time();
						line_check(13);
						fprintf(output_file,"      %02X(%ld): ",
							run_status,run_status_length);
						show_meta_hex(run_status_length);
					}
					else
					{
						 show_status_nybble(cur_status);
						 skip_text(run_status_length);
					}
				}
				else skip_text(run_status_length);
			}
			else if (!run_status_exists)	/* missing, e.g., first in track */
			{
				line_new();
				fprintf(output_file,"No status\n");
				exit_out(1);
			}
			else					/* remaining byte(s) if any */
			{
				if (show_message_details)
				{
					show_time();
					line_check(3);
					fprintf(output_file," %02x",c);
					show_meta_hex(run_status_length-1);
				}
				else
				{
					if (show_current_status) show_status_nybble(run_status);
					skip_text(run_status_length-1);
				}
			}
		}
	}
	fprintf(output_file,"\n");
}


/* clean name of EOL characters */

void clean_name(char *test)
{
	while (*test)
	{
		if (*test=='\r') *test='\0';
		else if (*test=='\n') *test='\0';
		test++;
	}
}


/* close output and exit */

void exit_out(short n)
{
	if (*output_file_name) fclose(output_file);
	exit(n);
}


/* read and document response */

char *ffgets(char *prompt, char *target, short len)
{
	printf("%s ",prompt);
	if (!fgets(target, len, parm_file))
	{
		printf("Unable to read parameter file \"%s\"\n",parm_file_name);
		exit_out(1);
		return((char*)NULL);
	
	}
	if (parm_file!=stdin) printf("%s",target);
	return(target);
}


/* read and analyze y-n response */

short getyes(char *prompt)
{
	char	response[MAXLINELEN];
	
	if (!ffgets(prompt, response, MAXLINELEN)) exit_out(1);
	if (*response=='Y' || *response=='y') return (TRUE);
	else return (FALSE);
}
	


/* check for having reached end of line */

void line_check(short newlen)
{
	if ((linelen+=newlen)>=MAXLINELEN)
	{
		fprintf(output_file,"\n");
		linelen=newlen;
	}
}


/* new line if needed */

void line_new(void)
{
	if (linelen>0) line_check(MAXLINELEN);
	linelen=0;
	meta_needs_space=status_needs_space=FALSE;
}



/* return next char or indicate EOF */

short next_char(void)
{
	short	c;

	if (expect--)		/* !=0 = expect EOF */
	{
		if ((c=getc(midi_file))!=EOF) return(c);
		else
		{
			fprintf(output_file," premature EOF\n");
			exit_out(1);
		}
	}
	else
	{
		if ((c=getc(midi_file))==EOF) return(c);
		else
		{
			fprintf(output_file," unexpected EOF\n");
			exit_out(1);
		}
	}
}


/* next two characters are a binary short integer */

short next_short(void)
{
	short	number=0;
	short	length=2;

	while(length--)
		number=number*256+next_char();
	return (number);
}


/* next four characters are a binary long integer */

long next_long(void)
{
	long	number=0;
	short	length=4;

	while(length--)
		number=number*256+next_char();
	return (number);
}


/* next three characters are a binary long integer */

long next_3long(void)
{
	long    number=0;
	short	length=3;

	while(length--)
		number=number*256+next_char();
	return (number);
}


/* next text should match explicit text */

void next_text(char *stuff)
{
	expect=strlen(stuff);
	fprintf(output_file,"%s: ",stuff);
	while (*stuff && expect)
	{
		if (next_char()!=*stuff) break;
		stuff++;
	}
	if (*stuff)
	{
		fprintf(output_file,"couldn\'t find %c\n");
		exit_out(1);
	}
}


/* variable length long */

long next_varlong(void)
{
	long    value;
	short	c;
	
	if((value=next_char()) & 0x80)
	{
		value &= 0x7f;
		do
		{
			value= (value<<7) + ((c=next_char()) & 0x7f);
		} while (c&0x80);
	}
	return(value);
}



/* skip requested number of characters */

short skip_text(long num_skip)
{
	short	c=0;
	while (num_skip--) c=next_char();
	return(c);
}


/* time(s) of entry */


void show_time(void)
{
	line_new();
	if (show_run_times)
	{
		fprintf(output_file," %10ld",run_time);
		linelen+=11;
	}
	if (show_delta_times)
	{
		fprintf(output_file," %6ld",delta_time);
		linelen+=7;
	}
}


/* show status byte in stream */

void show_status_nybble(short status)
{
	if (status_needs_space)
	{
		line_check(1);
		if (linelen>1) fprintf(output_file," ");
		else linelen=0;
		status_needs_space=FALSE;
	}
	line_check(1);
	fprintf(output_file,"%X",status>>4);
	meta_needs_space=TRUE;
}



/* show meta event data */

void show_meta(long length)
{

	fprintf(output_file," meta %02x(%ld): ",meta_type,length);
	switch (meta_type)
	{
	case 0x00:
		fprintf(output_file," sequence number: %d",next_short());
		break;
	case 0x01:
		fprintf(output_file," text: ");
		show_meta_text(length);
		break;
	case 0x02:
		fprintf(output_file," copyright: ");
		show_meta_text(length);
		break;
	case 0x03:
		fprintf(output_file," track name: ");
		show_meta_text(length);
		break;
	case 0x04:
		fprintf(output_file," instrument: ");
		show_meta_text(length);
		break;
	case 0x05:
		fprintf(output_file," lyric: ");
		show_meta_text(length);
		break;
	case 0x06:
		fprintf(output_file," marker: ");
		show_meta_text(length);
		break;
	case 0x07:
		fprintf(output_file," cue point: ");
		show_meta_text(length);
		break;
	case 0x20:
		fprintf(output_file," channel prefix: %d",next_char());
		break;
	case 0x2f:
		fprintf(output_file," end of track");
		break;
	case 0x51:
		fprintf(output_file," set tempo: %ld",next_3long());
		break;
	case 0x54:
		fprintf(output_file," SMPTE offset:");
		show_meta_dec(length);
		break;
	case 0x58:
		fprintf(output_file," time signature:");
		show_meta_dec(length);
		break;
	case 0x59:
		fprintf(output_file," key signature:");
		show_meta_dec(length);
		break;
	case 0x7f:
		fprintf(output_file," sequencer-specific:");
		show_meta_hex(length);
		break;
	default:
		fprintf(output_file," unknown: ");
		show_meta_hex(length);
		break;
	}
	line_new();
}


/* show text */

short show_text(long length)
{
	short	c=0;

	while (length--)
	{
		line_check(1);
		fprintf(output_file,"%c",c=next_char());
	}
	return(c);
}


/* meta text */

short show_meta_text(long length)
{
	short	c=show_text(length);

	line_new();
	return(c);
}


/* show bytes in hex */

short show_hex(long length)
{
	short	c=0;

	while (length--)
	{
		line_check(3);
		fprintf(output_file," %02x",c=next_char());
	}
	return(c);
}


/* meta bytes in hex */

short show_meta_hex(long length)
{
	short	c=show_hex(length);

	line_new();
	return(c);
}


/* show bytes in decimal */

short show_dec(long length)
{
	short	c=0;

	while (length--)
	{
		line_check(3);
		fprintf(output_file," %d",c=next_char());
	}
	return(c);
}


/* meta bytes in decimal */

short show_meta_dec(long length)
{
	short	c=show_dec(length);

	line_new();
	return(c);
}


