/* mm.c -- midi monitor */
/* Copyright 1989 Carnegie Mellon University */

/*****************************************************************************
*       Change Log
*  Date | Change
*-----------+-----------------------------------------------------------------
*  7-Apr-86 | Created changelog
* 31-Jan-90 | GWL : use new cmdline
*  5-Apr-91 | JDW : Further changes
* 16-Feb-92 | GWL : eliminate label mmexit:; add error recovery
* 18-May-92 | GWL : continuous clocks, etc.
* 17-Jan-94 | GWL : option to display notes
*****************************************************************************/

#include "ctype.h"
#include "cext.h"
#include "stdio.h"
#include "userio.h"
#include "midicode.h"
#include "midifns.h"
#include "cmdline.h"

#define buffsize (16*1024L)
byte huge *midibuff = NULL;

#ifdef AMIGA
/* #include "camd.h"
   #include "camdext.h"
   #include "mididefs.h"
 */
#endif

#ifdef THINK_C
#include "console.h"
extern int sysexcount;
extern int sysexheadcount;
extern int sysexdone;
extern byte sysexfirst;
#endif

int debug = false;	/* never set, but referenced by userio.c */

boolean inited = false;     /* suppress printing during command line parsing */
boolean done = false;       /* when true, exit */
boolean notes = true;       /* show notes? */
boolean sliders = true;     /* show continuous controllers */
boolean bender = true;      /* record pitch bend etc.? */
boolean excldata = false;   /* record system exclusive data? (note: this is toggled
			     * during initialization, so default state is true)
			     */
boolean verbose = true;     /* show text representation? */
boolean realdata = true;    /* record real time messages? */
boolean clksencnt = true;   /* clock and active sense count on */
boolean chmode = true;      /* show channel mode messages */
boolean pgchanges = true;   /* show program changes */
boolean switches = true;    /* show controller switches */
boolean flush = false;		/* flush all pending MIDI data */

long clockcount = 0;        /* count of clocks */
long actsensecount = 0;     /* cout of active sensing bytes */
long notescount = 0;        /* #notes since last request */
long notestotal = 0;        /* total #notes */

char val_format[] = "    Val %d";

/*****************************************************************************
*    Imported variables
*****************************************************************************/

extern  int     abort_flag;
#ifdef  DOS
extern  int     buffhead;
extern  int     bufftail;
extern  int     interror;
#endif


/*****************************************************************************
*    Routines local to this module
*****************************************************************************/

private    void    mmexit();
private    void    output();
private    int     put_pitch();
private    void    showhelp();
private    void    showbytes();
private    void    showstatus(boolean);
private    void    doascii(char);


/****************************************************************************
*               main
* Effect: prompts for parameters, starts monitor
****************************************************************************/

void main(argc, argv)
  int argc;
  char *argv[];
{
    byte data[4];    /* midi input data buffer */
    char *argument;
    
#ifdef THINK_C
    /* Macintosh needs help to provide Unix-style command line */
    argc = ccommand(&argv);
#endif

    io_init();

    cl_syntax(midifns_syntax);
    if (!cl_init(argv, argc)) mmexit();
    musicinit();
    doascii('x');           /* default is x true */
    if (argument = cl_arg(1)) { /* first arg can change defaults */
	while (*argument) doascii(*argument++);
    }
    inited=true;            /* now can document changes */
    showhelp();
    midi_cont(bender || sliders);
    midi_real(realdata || clksencnt);
    gprintf(TRANS,"Midi Monitor ready.\n");

    while (!done) {
	char c;
	eventwait((long) -1);
	if (get_ascii(&c)) doascii(c);
	if (getbuf(false, data)) output(data);
    }
    EXIT(0);
}


/****************************************************************************
*               doascii
* Inputs:
*    char c: input character
* Effect: interpret to revise flags
****************************************************************************/

private void doascii(char c)
{
    if (isupper(c)) c = tolower(c);
    if (c == 'q' || abort_flag) done = true;
    else if (c == 'b') {
	bender = !bender;
	if (inited)
	    gprintf(TRANS, "\nPitch Bend, etc. %s\n", (bender ? "ON" : "OFF"));
	midi_cont(bender || sliders);
    } else if (c == 'c') {
	sliders = !sliders;
	if (inited)
	    gprintf(TRANS, "\nControl Change %s\n", (sliders ? "ON" : "OFF"));
	midi_cont(bender || sliders);
    } else if (c == 'w') {
	switches = !switches;
	if (inited)
	    gprintf(TRANS, "\nOn/Off Switch Control %s\n", (switches ? "ON" : "OFF"));
    } else if (c == 'h') {
	pgchanges = !pgchanges;
	if (inited)
	    gprintf(TRANS, "\nProgram Changes %s\n", (pgchanges ? "ON" : "OFF"));
    } else if (c == 'n') {
	notes = !notes;
	if (inited)
	    gprintf(TRANS, "\nNotes %s\n", (notes ? "ON" : "OFF"));
    } else if (c == 'x') {
	if (!midibuff) {
	    midibuff = midibuff_alloc(buffsize);
	    if (!midibuff) {
		gprintf(TRANS, "buffer allocation failed\n");
		gprintf(TRANS, "Type anything to continue ...");
		ggetchar();
		return;
	    }
	    if (!midi_buffer(midibuff, buffsize)) {
		gprintf(TRANS,"midi_buffer installation error\n");
		EXIT(1);
	    }
	}
	excldata = !excldata;
	if (inited)
	    gprintf(TRANS, "\nSystem Exclusive data %s\n", (excldata ? "ON" : "OFF"));
	exclusive(excldata);
    } else if (c == 'r') {
	realdata = !realdata;
	if (inited)
	    gprintf(TRANS, "\nReal Time messages %s\n", (realdata ? "ON" : "OFF"));
	midi_real(realdata || clksencnt);
    } else if (c == 'k') {
	clksencnt = !clksencnt;
	if (inited)
	    gprintf(TRANS, "\nClock and Active Sense Counting %s\n", (clksencnt ? "ON" : "OFF"));
	    midi_real(realdata || clksencnt);
	if (!clksencnt) clockcount = actsensecount = 0;
    } else if (c == 's') {
	if (clksencnt) {
	    if (inited)
		gprintf(TRANS, "\nClock Count %ld\nActive Sense Count %ld\n", 
			clockcount, actsensecount);
	} else if (inited) {
	    gprintf(TRANS, "\nClock Counting not on\n");
	}
    } else if (c == 't') {
	notestotal+=notescount;
	if (inited)
	    gprintf(TRANS, "\nThis Note Count %ld\nTotal Note Count %ld\n",
		    notescount, notestotal);
	notescount=0;
    } else if (c == 'v') {
	verbose = !verbose;
	if (inited)
	    gprintf(TRANS, "\nVerbose %s\n", (verbose ? "ON" : "OFF"));
    } else if (c == 'm') {
	chmode = !chmode;
	if (inited)
	    gprintf(TRANS, "\nChannel Mode Messages %s\n", (chmode ? "ON" : "OFF"));
    } else {
	if (inited) {
	    if (c == ' ') {
		byte data[4];
		while (getbuf(false, data)) ;	/* flush midi input */
		gprintf(TRANS, "...FLUSHED MIDI INPUT\n\n");
	    } else showhelp();
	}
    }
}



private void mmexit()
{
    gprintf(TRANS, "Type anything to exit...");
    wait_ascii();
    EXIT(1);
}


/****************************************************************************
*               output
* Inputs:
*    byte data[]: midi data buffer holding one command
* Effect: format and print  midi data
****************************************************************************/

char vel_format[] = "    Vel %d";

private void output(data)
byte data[];
{
    int command;    /* the current command */
    int chan;   /* the midi channel of the current event */
    int len;    /* used to get constant field width */

    command = data[0] & MIDI_CODE_MASK;
    chan = (data[0] & MIDI_CHN_MASK) + 1;

    if (command == MIDI_ON_NOTE && data[2] != 0) {
	notescount++;
	if (notes) {
	    showbytes(data, 3, verbose);
	    if (verbose) {
		gprintf(TRANS, "NoteOn  Chan %2d Key %3d ", chan, data[1]);
		len = put_pitch(data[1] - 12);
		gprintf(TRANS, vel_format + len, data[2]);
	    }
	}
    } else if ((command == MIDI_ON_NOTE /* && data[2] == 0 */ ||
	       command == MIDI_OFF_NOTE) && notes) {
	showbytes(data, 3, verbose);
	if (verbose) {
	    gprintf(TRANS, "NoteOff Chan %2d Key %3d ", chan, data[1]);
	    len = put_pitch(data[1] - 12);
	    gprintf(TRANS, vel_format + len, data[2]);
	}
    } else if (command == MIDI_CH_PROGRAM && pgchanges) {
	showbytes(data, 2, verbose);
	if (verbose) {
	    gprintf(TRANS, "  ProgChg Chan %2d Prog %2d", chan, data[1] + 1);
	}
    } else if (command == MIDI_CTRL) {
	       /* controls 121 (MIDI_RESET_CONTROLLER) to 127 are channel
		* mode messages. */
	if (data[1] < MIDI_RESET_CONTROLLERS) {
	    if ((sliders && (data[1] < 64)) || 
		(switches && (data[1] >= 64) && (data[1] <= 96))) {
		showbytes(data, 3, verbose);
		if (verbose) {
		    gprintf(TRANS, "CtrlChg Chan %2d Ctrl %2d Val %2d",
			    chan, data[1], data[2]);
		}
	    }
	} else /* channel mode */ if (chmode) {
	    showbytes(data, 3, verbose);
	    if (verbose) {
		switch (data[1]) {
		  case MIDI_RESET_CONTROLLERS:
		    gprintf(TRANS, "Reset All Controllers");
		    break;
		  case MIDI_LOCAL:
		    gprintf(TRANS, "LocCtrl Chan %2d %s",
			    chan, data[2] ? "On" : "Off");
		    break;
		  case MIDI_ALL_OFF:
		    gprintf(TRANS, "All Off Chan %2d", chan);
		    break;
		  case MIDI_OMNI_OFF:
		    gprintf(TRANS, "OmniOff Chan %2d", chan);
		    break;
		  case MIDI_OMNI_ON:
		    gprintf(TRANS, "Omni On Chan %2d", chan);
		    break;
		  case MIDI_MONO_ON:
		    gprintf(TRANS, "Mono On Chan %2d", chan);
		    if (data[2])
			gprintf(TRANS, " to %d received channels", data[2]);
		    else
			gprintf(TRANS, " to all received channels");
		    break;
		  case MIDI_POLY_ON:
		    gprintf(TRANS, "Poly On Chan %2d", chan);
		    break;
		}
	    }
	}
    } else if (command == MIDI_POLY_TOUCH && bender) {
	showbytes(data, 3, verbose);
	if (verbose) {
	    gprintf(TRANS, "P.Touch Chan %2d Key %2d ", chan, data[1]);
	    len = put_pitch(data[1] - 12);
	    gprintf(TRANS, val_format + len, data[2]);
	}
    } else if (command == MIDI_TOUCH && bender) {
	showbytes(data, 2, verbose);
	if (verbose) {
	    gprintf(TRANS, "  A.Touch Chan %2d Val %2d", chan, data[1]);
	}
    } else if (command == MIDI_BEND && bender) {
	showbytes(data, 3, verbose);
	if (verbose) {
	    gprintf(TRANS, "P.Bend  Chan %2d Val %2d", chan,
		    (data[1] + (data[2]<<7)));
	}
    } else if (data[0] == MIDI_SONG_POINTER) {
	showbytes(data, 3, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Song Position %d",
		    (data[1] + (data[2]<<7)));
	}
    } else if (data[0] == MIDI_SONG_SELECT) {
	showbytes(data, 2, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Song Select %d", data[1]);
	}
    } else if (data[0] == MIDI_TUNE_REQ) {
	showbytes(data, 1, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Tune Request");
	}
    } else if (data[0] == MIDI_Q_FRAME && realdata) {
	showbytes(data, 2, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Time Code Quarter Frame Type %d Values %d",
		    (data[1] & 0x70) >> 4, data[1] & 0xf);
	}
    } else if (data[0] == MIDI_SYSEX) {
#define sysex_max 16
	byte msg[sysex_max];
	int togo;
	while (togo = get_excl(msg, sysex_max)) {
	    showbytes(msg, togo, verbose);
	    /* don't fetch past EOX; we'll get a message in the MIDI
	     * buffer for the next sysex message.
	     */
	    if (msg[togo-1] == MIDI_EOX) break;
	}
	if (verbose) gprintf(TRANS, "System Exclusive");
    } else if (data[0] == MIDI_START && realdata) {
	showbytes(data, 1, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Start");
	}
    } else if (data[0] == MIDI_CONTINUE && realdata) {
	showbytes(data, 1, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Continue");
	}
    } else if (data[0] == MIDI_STOP && realdata) {
	showbytes(data, 1, verbose);
	if (verbose) {
	    gprintf(TRANS, "    Stop");
	}
    } else if (data[0] == MIDI_SYS_RESET && realdata) {
	showbytes(data, 1, verbose);
	if (verbose) {
	    gprintf(TRANS, "    System Reset");
	}
    } else if (data[0] == MIDI_TIME_CLOCK) {
	if (clksencnt) clockcount++;
	else if (realdata) {
	    showbytes(data, 1, verbose);
	    if (verbose) {
		gprintf(TRANS, "    Clock");
	    }
	}
    } else if (data[0] == MIDI_ACTIVE_SENSING) {
	if (clksencnt) actsensecount++;
	else if (realdata) {
	    showbytes(data, 1, verbose);
	    if (verbose) {
		gprintf(TRANS, "    Active Sensing");
	    }
	}
    } else showbytes(data, 3, verbose);
#ifdef MACINTOSH_OR_UNIX
    fflush(stdout);
#endif

}


/****************************************************************************
*               put_pitch
* Inputs:
*    int p: pitch number
* Effect: write out the pitch name for a given number
****************************************************************************/

private int put_pitch(p)
int p;
{
    char result[8];
    static char *ptos[] = {
	"c", "cs", "d", "ef", "e", "f", "fs", "g",
	"gs", "a", "bf", "b"    };
    /* p is now in the range -12 to 115 */
    p += 12;
    /* note octave correction below */
    sprintf(result, "%s%d", ptos[p % 12], (p / 12) - 1);
    gprintf(TRANS, result);
    return strlen(result);
}


/****************************************************************************
*               showbytes
* Effect: print hex data, precede with newline if asked
****************************************************************************/

char nib_to_hex[] = "0123456789ABCDEF";

private void showbytes(data, len, newline)
  register byte data[];
  int len;
  boolean newline;
{
    char result[80];
    register int resultx = 0;
    register int i;

    if (newline) result[resultx++] = '\n';
    for (i = 0; i < len; i++) {
	result[resultx++] = nib_to_hex[data[i] >> 4];
	result[resultx++] = nib_to_hex[data[i] & 0xF];
	if (resultx > 72) {
	    result[resultx++] = '.';
	    result[resultx++] = '.';
	    result[resultx++] = '.';
	    break;
	}
    }
    result[resultx++] = ' ';
    result[resultx] = 0; /* End of string */
    gprintf(TRANS, result);
}



/****************************************************************************
*               showhelp
* Effect: print help text
****************************************************************************/

private void showhelp()
{
    gprintf(TRANS, "\n");
    gprintf(TRANS, "   Item Reported  Range     Item Reported  Range\n");
    gprintf(TRANS, "   -------------  -----     -------------  -----\n");
    gprintf(TRANS, "   Channels       1 - 16    Programs       1 - 128\n");
    gprintf(TRANS, "   Controllers    0 - 127   After Touch    0 - 127\n");
    gprintf(TRANS, "   Loudness       0 - 127   Pitch Bend     0 - 16383, center = 8192\n");
    gprintf(TRANS, "   Pitches        0 - 127, 60 = c4 = middle C\n");
    gprintf(TRANS, " \n");
    gprintf(TRANS, "n toggles notes");
    showstatus(notes);
    gprintf(TRANS, "t displays noteon count since last t\n");
    gprintf(TRANS, "b toggles pitch bend, aftertouch");
    showstatus(bender);
    gprintf(TRANS, "c toggles continuous control");
    showstatus(sliders);
    gprintf(TRANS, "w toggles discrete switch controllers");
    showstatus(switches);
    gprintf(TRANS, "h toggles program changes");
    showstatus(pgchanges);
    gprintf(TRANS, "x toggles system exclusive");
    showstatus(excldata);
    gprintf(TRANS, "k toggles clock and sense counting only");
    showstatus(clksencnt);
    gprintf(TRANS, "r toggles other real time messages & SMPTE");
    showstatus(realdata);
    gprintf(TRANS, "s displays clock and sense count since last k\n");
    gprintf(TRANS, "m toggles channel mode messages");
    showstatus(chmode);
    gprintf(TRANS, "v toggles verbose text");
    showstatus(verbose);
    gprintf(TRANS, "q quits\n");
    gprintf(TRANS, "\n");
}

/****************************************************************************
*               showstatus
* Effect: print status of flag
****************************************************************************/

private void showstatus(boolean flag)
{
    gprintf(TRANS, ", now %s\n", flag ? "ON" : "OFF" );
}
