/* exget.c -- program to get system exclusive messages and write to file */
/* Copyright 1989 Carnegie Mellon University */

/*****************************************************************************
*       Change Log
*  Date | Change
*-----------+-----------------------------------------------------------------
* 13-Jun-86 | Created Change Log
*  5-Apr-91 | JDW-Further Changes
*  6-Mar-92 | GWL : ok for DOS
*  5-Aug-92 | RBD : major changes to use midifns.c's new get_excl.  This is
*           |  a more "safe" and portable interface
*****************************************************************************/

#include "ctype.h"
#include "cext.h"
#include "stdio.h"
#include "midifns.h"
#include "userio.h"
#include "midicode.h"
#include "cmdline.h"

#ifdef AMIGA
#include "midi/camd.h"
#endif

#ifdef THINK_C
#include "console.h"
#endif

/****************************************************************
* The original CMU CAMD driver for Amiga had a special bypass 
* or "shortcircuit" to allow slow processors to keep up with
* big sysex messages.  I'm keeping much of this code around
* for possible future use, but I do not plan to debug this
* option.
*****************************************************************/
#ifdef AMIGA_USE_SHORT_CIRCUIT
/* use the MidiSysExHook or ByteHook to bypass normal interface */
#define SHORTCIRCUIT
/* In the CMU version of CAMD, there was a MidiSysExHook function.
   This seems to be missing from the Commodore version, so we'll
   try using the MidiByteHook instead.  The following switch does it:
 */
#define USEBYTEHOOK
/* #define USESYSEXHOOK */
#endif

/* buffsize must be a power of 2! */
long buffsize = 16*1024L;
byte huge *midibuff;

int debug = false; /* used in userio.c, declared in moxc.c, but we're not using moxc.c */


/****************************************************************************
*
* System exclusive buffer variables.
*
****************************************************************************/

#ifdef DOS
extern byte *xbuff;     /* address of the user-supplied buffer */
extern int xbufmask;    /* mask for circular buffer address calculation */
extern int xbufhead;    /* head of exclusive buffer */
extern int xbuftail;    /* tail of exclusive buffer */
#endif


/*****************************************************************************
*    Routines local to this module
*****************************************************************************/

private void instruct();
#ifdef SHORTCIRCUIT
private boolean output();
#else
private void output();
#endif


/****************************************************************************
*               instruct
* Effect: prints instructions for this routine
****************************************************************************/

private void instruct()
{
    gprintf(TRANS,"This program will let you save MIDI system exclusive\n");
    gprintf(TRANS,"    messages to a file.  When you get the prompt:\n");
    gprintf(TRANS,"    \"Ready for your data...\", send one or more sysex\n");
    gprintf(TRANS,"    messages from your synthesizer.  See the CMU Midi\n");
    gprintf(TRANS,"    Toolkit Manual or your synthesizer operator's manual\n");
    gprintf(TRANS,"    for instructions on this.\n");
    gprintf(TRANS,"Use the exput program to send sysex messages recorded by\n");
    gprintf(TRANS,"    exget.\n");
}


#ifdef SHORTCIRCUIT
byte *msghead;  /* message pointer */
byte *oldmsghead;
byte *msgtail;


/****************************************************************************
*               my_sysex_handler
* Effect: interrupt handler for sys ex messages
****************************************************************************/

int mybuffhead = 0;
int mybufftail = 0;

extern struct MidiInterface *cmt_mi;

#ifdef USESYSEXHOOK
my_sysex_handler(c)
  long c;
#endif
#ifdef USEBYTEHOOK
UWORD receive_error;

my_byte_handler(struct hook *hook, 
		struct MidiNode *object, struct MidiByteHookMsg *message)
#endif
{
    ;
    #asm
    movem.l a4,-(sp)
    jsr _geta4#
    #endasm

    midibuff[mybufftail++]
#ifdef USESYSEXHOOK
	= c;
#endif
#ifdef USEBYTEHOOK
	= message->RecvData;
    receive_error |= message->RecvData;
#endif
    mybufftail &= (buffsize - 1);

    #asm
    movem.l (sp)+,a4
    #endasm
}
#endif /* SHORTCIRCUIT */


char no_data_msg[] = "No data!  Please start again.\n";

/****************************************************************************
*               main
* Effect: main program -- prompt user for information and get data
****************************************************************************/

void main(argc, argv)
  int argc;
  char *argv[];
{
    char key;
    FILE *fp;   /* the   output file */
    int done = false;    /* flag is true when user is done */
    char filename[100];    /* the output file name */
    char *s;    /* the file name in the command line */
    boolean binary_mode = false;
    char *size;
    byte msg[4];	
    int msg_count;
    long byte_count;

#ifdef THINK_C
    /* Macintosh needs help to provide Unix-style command line */
    argc = ccommand(&argv);
#endif
    io_init();
    cl_syntax(midifns_syntax);
    cl_syntax("help<s>Print help info;binary<s>Use binary file format;\
	size<o>Specifies minimum buffer size in bytes");
    if (!cl_init(argv, argc)) {
#ifndef MACINTOSH
	gprintf(TRANS, "Type anything to exit...");
	ggetchar();
#endif
	goto finish;
    }
    if (cl_switch("help")) instruct();
    if (cl_switch("binary")) binary_mode = true;
    size = cl_option("size");
    if (size != NULL) {
	int newsize = atoi(size);
	/* go up by powers of two until you exceed requested size */
	while (newsize > buffsize) buffsize += buffsize;
    }
    midibuff = midibuff_alloc(buffsize);
    if (!midibuff) {
	gprintf(TRANS, "buffer allocation failed\n");
#ifndef MACINTOSH
	gprintf(TRANS, "Type anything to exit...");
	ggetchar();
#endif
	goto finish;
    }   

    /* get input file name: */
    filename[0] = EOS;    /* empty string */
    if (s = cl_arg(1)) {
	strcpy(filename, s);
	if (s = cl_arg(2))
	    gprintf(ERROR, "Extra (filename?) argument ignored: %s\n", s);
    }
    musicinit();
    while (!done) {
#ifdef SHORTCIRCUT
	/* for overflow detection */
	midibuff[buffsize-1] = 0xff;
	midibuff[buffsize-2] = 0xff;
#endif

	if (askbool("Do you want instructions", false)) instruct();

	fp = fileopen(filename, "syx", (binary_mode ? "wb" : "w"),
		      "file name");
	if (!fp) goto finish;
	cu_register((cu_fn_type) fclose, fp);

#ifndef UNIX
#ifndef SHORTCIRCUIT
	/* note that this will flush all previous MIDI messages */
	if (!midi_buffer(midibuff, buffsize)) {
	    gprintf(TRANS,"midi_buffer installation error\n");
	    EXIT(1);
	}
#else
#ifdef USESYSEXHOOK
	if (!MidiSysExHook(cmt_mi, 0L, my_sysex_handler)) {
#else
	receive_error = 0;	/* clear error */
	if (!(MidiByteHook(0L, cmt_mi, my_byte_handler)) {
#endif /* USESYSEXHOOK */
	    gprintf(TRANS,"Hook installation error\n");
	    EXIT(1);
	}	    
#endif
#endif
	while (getkey(false) != -1) ;	/* clear ascii */
	exclusive(true);    /* accept exclusive data */
#ifdef SHORTCIRCUIT
	mybufftail = 0;
#endif
	gprintf(TRANS,"\nReady for your data. Type space when you are done...\n");

	done = false;
	while (!done) {
	    eventwait((long) -1);
	     if (get_ascii(&key)) {
		 done = ((key == ' ') || abort_flag);
	     }
#ifndef SHORTCIRCUIT
	    if (midi_error_flags & (1<<SYSEXOVFL)) {
#else
	    if (midibuff[buffsize-1] != 0xff || midibuff[buffsize-2] != 0xff) {
#endif
		gprintf(TRANS, "Buffer overflow, buffer size is %d.\n", 
			buffsize);
		gprintf(TRANS,
			"Use the -size xxx option to get a larger buffer\n");
		done = true;
	    }

#ifdef DOS
	    /* the following is not supported under Leigh's driver */
	    /* mpu_error_check();*/    /* look for buffer overflow */
#endif

#ifdef USEBYTEHOOK
	    if (receive_error & 0x8000) {
		gprintf(TRANS, "Midi data receive error.\n");
		receive_error = 0;
	    }
#endif
	}
#ifdef SHORTCIRCUIT
	msghead = midibuff;
	msgtail = midibuff + mybufftail;
	oldmsghead = msghead;
#endif
	byte_count = 0;
#ifdef SHORTCIRCUIT
	if (mybufftail == 0) {
	    gprintf(TRANS, no_data_msg);
	} else {
	    /* write messages */
	    while (output(fp, &msghead, msgtail, binary_mode, &byte_count)) {
		if (check_aborted()) break;
	    }
	    if (abort_flag) {
		gprintf(TRANS, "File write interrupted, maybe incomplete.\n");
		if (abort_flag) abort_flag = 0;
	    }
	}
#else
	msg_count = 0;
	while (getbuf(false, msg)) {
	    if (msg[0] == MIDI_SYSEX) {
		msg_count++;
		if (midi_error_flags) {
		    /* flush the pending sysex message */
		    while (get_excl(msg, 4)) ;
		} else {
		    output(fp, binary_mode, &byte_count);
		}
	    }
	}
	if (msg_count == 0) gprintf(TRANS, no_data_msg);
#endif
	cu_unregister(fp);
	fclose(fp);
	gprintf(TRANS,"Wrote %ld bytes\n", byte_count);
	/* now that we've flushed messages... */
	midi_error_flags = false;

	done = abort_flag || 
	       !askbool("Do you want to receive another file", true);
	filename[0] = EOS;    /* force prompt for new file name */
    }
  finish:
    EXIT(0);
}


/****************************************************************************
*                               output
* Inputs:
*    FILE *fp: the file to write
*    byte **msg: pointer to the message to write
*    byte *msg_tail: points to byte after last byte of recorded data
*    boolean binary_mode: selects binary vs. ascii file format
* Outputs:
**    byte **msg: is advanced to byte after the last byte of the message
*    returns true if there is more data to output
* Effect: 
*    write data file using recorded data
* Assumes:
*    fp is opened for writing
*    msg_tail > *msg
****************************************************************************/

#ifdef SHORTCIRCUIT
private boolean output(fp, msg, msg_tail, binary_mode, byte_count_ptr)
  FILE *fp;
  byte **msg;
  byte *msg_tail;
  boolean binary_mode;
  long *byte_count_ptr;
{
    int n;          /* counts bytes printed on one line */
    byte *this_msg = *msg;    /* local message pointer */

    /*
	gprintf(DEBUG,"*msg %x, this_msg %x, msg_tail %x\n",
	       *msg, this_msg, msg_tail);
	gprintf(DEBUG,"%d bytes to go.\n", msg_tail - this_msg);
    */

    excldesc(this_msg);

    if (binary_mode) {
	do {
	    putc(*this_msg++, fp);
	    *byte_count_ptr += 1;
	} while (this_msg < msg_tail && *(this_msg-1) != MIDI_EOX);
    } else {
	n = 0;
	do {                    /* always print first byte, then terminate */
	    fprintf(fp, "%2x ", *this_msg);     /* after printing MIDI_EOX */
	    if (n >= 15) {                      /* print 16 bytes per line */
		n = 0;
		fprintf(fp, "\n");
	    } else {
		n++;
	    }
	    this_msg++;
	    *byte_count_ptr += 1;
	} while (this_msg < msg_tail && *(this_msg-1) != MIDI_EOX) ;
	fprintf(fp, "\n\n");
    }
    *msg = this_msg;
    return (boolean)(this_msg < msg_tail);
}

#else
/*
 * this is the "normal" output function:
 */
#define excl_buf_len 256

private void output(fp, binary_mode, byte_count_ptr)
  FILE *fp;
  boolean binary_mode;
  long *byte_count_ptr;
{
    int n;          /* counts bytes printed on one line */
    byte this_msg[excl_buf_len];
    long togo = get_excl(this_msg, excl_buf_len);
    byte *ptr;
    boolean last_block;

    excldesc(this_msg);

    if (binary_mode) {
	do {
	    checksum(this_msg, togo);
	    last_block = (this_msg[togo-1] == MIDI_EOX);
	    *byte_count_ptr += togo;
	    ptr = this_msg;
	    while (togo--) putc(*ptr++, fp);
	} while (!last_block && (togo = get_excl(this_msg, excl_buf_len)));
    } else {
	n = 0;
	do {
	    checksum(this_msg, togo);
	    last_block = (this_msg[togo-1] == MIDI_EOX);
	    *byte_count_ptr += togo;
	    ptr = this_msg;
	    while (togo--) {
		fprintf(fp, "%2x ", *ptr++);
		if (n >= 15) {   /* print 16 bytes per line */
		    n = 0;
		    fprintf(fp, "\n");
		} else {
		    n++;
		}
	    }
	} while (!last_block && (togo = get_excl(this_msg, excl_buf_len)));
	fprintf(fp, "\n\n");
	checksum_end();
    }
}
#endif
