/* scan.c -- a simple scanner.  The interface is:

    global variables:
	line -- the current line of input from file
	token -- the current token scanned from line
	linex -- the current location in line, points after last thing scanned
	tokenx -- the current location in token, usually set by client, but
		used to generate error messages
	tokenid -- the value returned by issymbol(token)
	lineno -- the line number

    scan_init(fp) -- initialize and provide a file pointer
    fferror(string) -- print an error message with pointer to location
    issymbol() -- is token a symbol in ssymbol array?
    scan() -- get a token from line
    scan1() -- get a single character from line
    scanint() -- get an integer from token
    scansgnint() -- get a signed integer from token
    scansymb() -- get a symbol from token
 */


#include "switches.h"
#include "cext.h"
#include "stdio.h"
#include "userio.h"
#include "ctype.h"
#include "cmdline.h"
#include "scan.h"

boolean scan_debug = false;

FILE *scan_fp;

#ifdef LATTICE
private char *ssymbols[] = {{"NT"}, {"INIT"}, {"ANY"}, {"PITCH"},
			    {"PROG"}, {"CTRL"}, {"MAP"}, {"CALL"},
			    {"RET"}, {"GO"}, {"DLY"}, {"CHAN"},
			    {"NOOP"}, {"PASS"}, {"CLASS"}, {"REDO"},
			    {"SCORE"}, {"GATED"}, {"CYCLE"}, {"PRINT"}};
#else
private char *ssymbols[] = {"NT", "INIT", "ANY", "PITCH",
			    "PROG", "CTRL", "MAP", "CALL",
			    "RET", "GO", "DLY", "CHAN",
			    "NOOP", "PASS", "CLASS", "REDO",
			    "SCORE", "GATED", "CYCLE", "PRINT"};
#endif

#define linesize 100
char line[linesize];    /* the input line */
char token[linesize];    /* a token scanned from the input line */

int lineno;
int linex;
int tokenx;
int tokenid;

/****************************************************************************
*               marker
* Inputs:
*    int count: the number of characters to indent
* Effect: 
*    prints a carat (^) at the position specified on file stderr
****************************************************************************/

private void marker(count)
  int count;
{
    int i;
    char s[128];
    for (i = 0; i < (count - 1); s[i++] = ' ') /* */ ;
    s[count - 1] = '^';
    s[count] = '\0';
   gprintf(ERROR, "%s\n", s);
}


void scan_init(fp)
  FILE *fp;
{
    scan_fp = fp;       /* make it global to module */
    scan_debug = cl_switch("scan");
    lineno = 0;
    linex = 0;
    tokenx = 0;
    line[0] = '\n';
    line[1] = EOS;
}


/****************************************************************************
*               fferror
* Inputs:
*    char *s: an error message string
* Effect:
*    prints the line with the error
*    puts a cursor (^) at the error location
*    prints the error message (s)
* Implementation:
*    this routine prints a carat under the character that
*    was copied into token[tokenx].    E.g. if tokenx = 0, the
*    carat will point to the first character in the field.
****************************************************************************/

void fferror(s)
  char *s;
{
    gprintf(ERROR, "%3d | %s", lineno, line);
    marker(linex-strlen(token)+tokenx+2+6);
    gprintf(ERROR, "Error: %s.\n", s);
}

/****************************************************************************
*               scan
* Effect: 
*    skips over leading blanks
*    copies characters from start into token;
*    scanning stops on delimiter: one of space, tab, newline
****************************************************************************/

void scan()
{
    register char c;

    tokenx = 0;
    while (((c = line[linex]) == ' ') || (c == '\t') || (c == '\n')) {
	if (c == '\n') {
	    if (fgets(line, linesize, scan_fp) == NULL) goto file_end;
	    lineno++;
	    linex = 0;
	    /* quick and dirty comments: ignore line if '*' in col. 1 */
	    if (line[0] == '*') line[0] = '\n';
	} else linex++;
    }
    while ((c = line[linex]) != ' ' && c != '\n' && c != '\t') {
/*	if (islower(line[linex])) token[tokenx] = toupper(line[linex]);
	else */
	token[tokenx] = line[linex];
	tokenx++; 
	linex++;
    }
file_end:
    token[tokenx] = '\0';
    tokenx = 0;
    if (scan_debug) gprintf(TRANS, "|%s|", token);
}

/****************************************************************************
*               scan1
* Inputs:
*    char *start: the string to scan
* Effect: 
*    copies one char from linex into token, converting to upper case
****************************************************************************/


/****************************************************************************
*               scanint
* Outputs:
*    returns int: the scanned integer
* Effect:
*    scans an unsigned integer from token, starting at tokenx
*    tokenx is incremented to end of the integer
****************************************************************************/

long scanint()
{
    long i = 0;
    char c;
    while (c = token[tokenx]) {
	if (isdigit(c)) {
	    i = (i*10) + (c - '0');
	    tokenx++;
	} else return i;
    }
    return i;
}

long scansgnint()
{
    if (token[tokenx] == '-') {
	tokenx++;
	return -scanint();
    } else return scanint();
}


/****************************************************************************
*               issymbol
* Outputs: returns symbol number, or -1 if no match
****************************************************************************/

int issymbol()
{
    int i;

    token_upper(); /* convert token to upper case for comparison */
    for (tokenid = 0; tokenid < sym_n; tokenid++) {
	char *sym = ssymbols[tokenid];
	i = 0;
	while (true) {
	    if (token[i] != *sym) break;
	    if (*sym == 0) {
		if (scan_debug) gprintf(TRANS, " symb=%d ", tokenid);
		return tokenid;
	    }
	    sym++; 
	    i++;
	}
    }
    tokenid = -1;
    if (scan_debug) gprintf(TRANS, " symb=%d ", tokenid);
    return -1;
}


/* scanstr -- scan a string, including quotes */
/*
 * NOTE: token will have an initial and final dbl quote if scan was ok
 */
void scanstr()
{
    register char c;

    tokenx = 0;
    while (((c = line[linex]) == ' ') || (c == '\t') || (c == '\n')) {
	if (c == '\n') {
	    if (fgets(line, linesize, scan_fp) == NULL) goto file_end;
	    lineno++;
	    linex = 0;
	    /* quick and dirty comments: ignore line if '*' in col. 1 */
	    if (line[0] == '*') line[0] = '\n';
	} else linex++;
    }
    if (line[linex] != '"') return;
    else {
	token[tokenx++] = '"';
	linex++;
    }
    while ((c = line[linex]) != '"' && c != '\n') {
	token[tokenx] = line[linex];
	tokenx++; 
	linex++;
    }
    if (line[linex] == '"') {
	token[tokenx++] = '"';
	linex++;
    }
file_end:
    token[tokenx] = '\0';
    tokenx = 0;
    if (scan_debug) gprintf(TRANS, "|%s|", token);
}


/* token_upper -- convert token to upper case */
/*
 * NOTE: we don't do this automatically because filenames
 * would get upper cased and unix cares.
 */
void token_upper()
{
    char *p = token;
    while (*p != EOS) {
	if (islower(*p)) *p = toupper(*p);
	p++;
    }
}
