/* mapread -- map midi events into output sequences */

/* Documentation:

    A map translates input midi events into a sequence of output events.
The primary functions in the interface are
     map_note(chan, key, vel) -- map a note on or off (vel=0) to output
     map_select(char) -- select map indexed by ascii character char
     map_read(file) -- read in the map from an open FILE
Maps are stored in the array maps[26].
map_read loads a file with one or more maps and installs them in maps[].
A three-level hierarchy exists to translate note events into output events.
A map can specify an output sequence for EACH pitch or an output sequence
for ANY pitch.  Furthermore, a map can be channel specific or global, and 
this information is a property of the map.  The map_select() function looks
at the map to decide where to install it.
    An array active_maps[17] stores the global map (if any) in location 0
and channel specific maps in locations 1-16.  Each entry of active_maps
points to a character which names the current map.  The character is at
the head of a string that represents a stack of maps.  There is no map
is the string is empty.  Each map in turn consists of
an array of pointers to mapevt lists, each of which encodes a sequence
of output events.  Locations 0-35 encode three octaves of pitch-specific
output.  Location 36 is the event list to be called when the map is 
installed, and Location 37 is the non-pitch-specific sequence.
    When a NoteOn with key K on channel CH occurs, we look first in 
active_maps[CH] for a map, then look at entry (K mod 36) in the map for
a sequence.  If none is found, we look at entry 37 for a sequence.  If
none is found here, or if no map at all was found for channel CH, we
look at active_maps[0].  If there is a map here, we look at its entry
(K mod 36) and then at its entry 37 for a sequence.  The first sequence
found is output.  If no sequence is found, no output occurs.

(See crn.doc for syntax and semantics.)

*/


#include "switches.h"
#include "cext.h"
#include "stdio.h"
#include "midifns.h" /* to get time_type */
#include "timebase.h"
#include "moxc.h"    /* to get debug declared */
#include "userio.h"
#include "scan.h"
#include "cmdline.h"
#include "seq.h"
#include "seqread.h"
#include "mapper.h"

#ifdef UNIX
#include "ctype.h"
#endif

#ifdef THINK_C
#define index(s,c) strchr(s,c)
#endif
#ifdef LATTICE
#define index(s,c) strchr(s,c)
#endif
#ifdef DOS
#define index(s,c) strchr(s,c)
#endif

#ifdef AMIGA
#include "ctype.h"
/* we're already taking precautions, so inline version of toupper is ok: */
/* WARNING: don't use _toupper from AZTEC 5.0! */
#ifndef toupper
#define toupper(x) ((x)-'a'+'A')
#endif
/* #include "functions.h" */
/* #include "exec/exec.h" */
/* #include "cmtcmd.h" */
#endif

void mapevt_free_a_list();
void parse_command_list(mapevt_type *mapevt_ptr, boolean init_flag);
void parse_end();
char parse_letter();
#ifdef __STDC__
long parse_number(long mini, long maxi, char *what);
#else
long parse_number();
#endif
int parse_pitch();
char *parse_string();
int parse_vel();
long parse_dur();

map_type the_map;  /* this is the map we are parsing */

/****************************************************************************
*               map_read
* Inputs:
*    FILE *fp: input file
* Outputs:
*    returns char of the first map ID in the file
* Effect: 
*    parses map from input file and builds map data structure, installs map
*    in maps[]
****************************************************************************/

char map_read(fp)
  FILE *fp;
{
    char first_id = EOS;
    char map_id;
    scan_init(fp);

    /* this loop reads maps */
    scan();
    while (!(nullstring(token))) {
	the_map = map_create();
	if (!the_map) return 0;
	if (issymbol() != sym_map) {
	     fferror("MAP expected, skipping to MAP");
	    while ((!(nullstring(token))) && issymbol() != sym_map) {
		scan();
	    }
	    if (nullstring(token)) break;
	}
	scan();
	map_id = parse_letter() - 'A';
	if (!first_id) first_id = map_id + 'A';
	if (maps[map_id]) map_destroy(maps[map_id]);
	maps[map_id] = the_map;
	if (issymbol() == sym_chan) {
	    scan();
	    the_map->chan = parse_number(1L, 16L, "channel");
	    issymbol();         /* look up next token */
	} else the_map->chan = 0;
	if (tokenid == sym_init) {
	    scan();
	    parse_command_list(&(the_map->keymap[map_init]), true);
	}
	if (tokenid == sym_any) {
	    scan();
	    parse_command_list(&(the_map->keymap[map_any]), false);
	}
	while (tokenid == sym_pitch || tokenid == sym_class) {
	    int pitch_or_class = tokenid;
	    mapevt_type *mapevt_ptr;
	    mapevt_type *mapevt_ptr_1, *mapevt_ptr_2;
		int value;
	    scan();
	    value = parse_pitch();
	    value = value % 36;
	    mapevt_ptr = &(the_map->keymap[value]);
	    mapevt_free_a_list(mapevt_ptr);
	    parse_command_list(mapevt_ptr, false);
	    if (pitch_or_class == sym_class) { /* install it three more times */
		mapevt_ptr_1 = &(the_map->keymap[(value + 12) % 36]);
		mapevt_free_a_list(mapevt_ptr_1);
		mapevt_ptr_2 = &(the_map->keymap[(value + 24) % 36]);
		mapevt_free_a_list(mapevt_ptr_2);
		*mapevt_ptr_1 = *mapevt_ptr_2 = *mapevt_ptr;
	    }
	}
    }
    return first_id;
}


/* parse_loud -- parse a score loudness field */
/*
 * encoding: if mapevt_transpose_flag set, then this is an
 *   offset to be added to triggering event's velocity,
 *   otherwise this is a constant offset.
 */
int parse_loud()
{
    char c;
    int vel;
    token_upper();
    c = token[0];
    if (c == '+') {
	tokenx++;
	vel = parse_number(0L, 127L, "velocity offset");
	vel = vel >> 1;
    } else if (isdigit(c)) {
	vel = parse_number(0L, 127L, "velocity offset");
	vel = vel >> 1;
    } else if (c == '-') {
	tokenx++;
	vel = parse_number(0L, 127L, "velocity offset");
	vel = -(vel >> 1);
	vel &= ~mapevt_transpose_flag;
    } else if ((c == 'V') && (token[1] == '+')) {
	vel = parse_number(1L, 127L, "velocity offset");
	vel = (vel >> 1) | mapevt_transpose_flag;
    } else if ((c == 'V') && (token[1] == '-')) {
	vel = parse_number(1L, 127L, "velocity offset");
	vel = - (vel >> 1);
    }
    return vel;
}


/* parse_number -- parse any number field */
/**/
long parse_number(mini, maxi, what)
  long mini;
  long maxi;
  char *what;
{
    long i = scanint();
    if (token[tokenx] != EOS || i < mini || i > maxi) {
	char error_string[100];
	sprintf(error_string, "%s (%ld-%ld) expected", what, mini, maxi);
	fferror(error_string);
	i = mini;
    }
    scan();
    return i;
}

static short letter_to_step[] = {9, 11, 0, 2, 4, 5, 7};
			      /* A  B   C  D  E  F  G */

/* parse_pitch -- parse a pitch indication */
/**/
int parse_pitch()
{
    char c;
    int pitch = mapevt_transpose_flag;

    token_upper();
    c = token[0];
    if (c == '+') {
	tokenx++;
	pitch = parse_number(0L, 63L, "transposition");
	pitch |= mapevt_transpose_flag;
    } else if (c == '-') {
	tokenx++;
	pitch = parse_number(0L, 63L, "transposition");
	pitch = -pitch; /* this sets transposition flag too */
    } else if (isdigit(c)) {
	pitch = parse_number(0L, 127L, "pitch");
    } else if (index("ABCDEFG", c)) {
	pitch = parse_pitch_name("pitch");
    } else fferror("pitch expected");
    return pitch;
}


/* parse_pitch_name -- parse a pitch name such as FS5 */
/*
 * NOTE: pitch begins at tokenx.
 *       scan() called at end of parse.
 */
int parse_pitch_name(parse_what)
  char *parse_what;     /* what are we parsing? (for error msgs) */
{
    int pitch = 60 + letter_to_step[token[tokenx++] - 'A'];
    char c = token[tokenx];
    if (c == 'S') {
	pitch++;
	tokenx++;
    } else if (c == 'F') {
	pitch--;
	tokenx++;
    } else if (c == 'N') {
	tokenx++;
    }
    c = token[tokenx];
    if (isdigit(c)) {
	int octave = parse_number(0L, 9L, "octave");
	pitch = (pitch % 12) + (12 * (octave + 1));
    } else {
	parse_end(parse_what);
	scan();
    }
    return pitch;
}


/* parse_trans -- parse a transposition indication */
/*
 * encoding: if mapevt_transpose_flag (bit 7) clear, then
 *   remaining bits are a signed transposition.  If the bit
 *   is set, then the remaining bits are a negative (two's complement)
 *   offset to which the triggering pitch is added.
 */
int parse_trans()
{
    char c;
    int transp = 0;

    token_upper();
    c = token[0];
    if (isdigit(c)) {
	transp = parse_number(0L, 63L, "transposition");
    } else if (c == '+') {
	tokenx++;
	transp = parse_number(0L, 63L, "transposition");
    } else if (c == '-') {
	tokenx++;
	transp = parse_number(0L, 63L, "transposition");
	transp = -transp;       /* this sets transposition flag too */
	transp &= ~mapevt_transpose_flag;
    } else if (c == 'P' && token[1] == '-') {
	tokenx = 2;
	transp = -parse_pitch_name("transposition");
    } else fferror("transposition expected");
    return transp;
}


/* parse_command_list -- parse a list of commands */
/**/
void parse_command_list(
  mapevt_type *mapevt_ptr,
  boolean init_flag)  /* true if this is an INIT command list */
{
    boolean done = false;
    while (!done) {
	issymbol(); /* get the tokenid */
	if (tokenid == sym_nt) {
	    int chan, key, vel;
	    long dur;
	    scan();     /* and advance to next token */
	    chan = parse_number(1L, 16L, "channel");
	     key = parse_pitch();
	    vel = parse_vel();
	    dur = parse_dur();
/*          gprintf(TRANS, "got nt mapevt %d 0x%x 0x%x 0x%lx\n", chan, key, vel, 
			   dur); */
	    mapevt_insert_nt(mapevt_ptr, chan, key, vel, dur);
	} else if (tokenid == sym_prog) {
	    int chan, prog;
	    scan();
	    chan = parse_number(1L, 16L, "channel");
	    prog = parse_number(1L, 128L, "program");
/*          gprintf(TRANS, "got prog mapevt %d %d\n", chan, prog); */
	    mapevt_insert_prog(mapevt_ptr, chan, prog);
	} else if (tokenid == sym_ctrl) {
	    int chan, ctrl, value;
	    scan();
	    chan = parse_number(1L, 16L, "channel");
	    ctrl = parse_number(0L, 127L, "controller number");
	    value = parse_number(0L, 127L, "controller value");
/*          gprintf(TRANS, "got ctrl mapevt %d %d %d\n", chan, ctrl, value); */
	    mapevt_insert_ctrl(mapevt_ptr, chan, ctrl, value);
	} else if (tokenid == sym_dly) {
	    long dur;
	    scan();
	    dur = parse_number(0L, 32000L, "duration");
/*          gprintf(TRANS, "got dly mapevt %ld\n", dur); */
	    mapevt_insert_dly(mapevt_ptr, dur);
	} else if (tokenid == sym_call) {
	    char c;
	    scan();
	    c = parse_letter();
/*          gprintf(TRANS, "got call mapevt %c\n", c); */
	    mapevt_insert_call(mapevt_ptr, c);
	} else if (tokenid == sym_print) {
	    char *str;
	    str = parse_string();
	    mapevt_insert_print(mapevt_ptr, str);
	} else if (tokenid == sym_ret) {
	    scan();
/*          gprintf(TRANS, "got ret mapevt\n"); */
	    mapevt_insert_ret(mapevt_ptr, the_map->chan);
	} else if (tokenid == sym_go) {
	    char c;
	    scan();
	    c = parse_letter();
/*          gprintf(TRANS, "got go mapevt %c\n", c); */
	    mapevt_insert_go(mapevt_ptr, the_map->chan, c);
	} else if (tokenid == sym_noop) {
	    scan();
/*          gprintf(TRANS, "got noop mapevt\n"); */
	    mapevt_insert_noop(mapevt_ptr);
	} else if (tokenid == sym_pass) {
	    scan();
/*          gprintf(TRANS, "got pass mapevt\n"); */
	    if (init_flag) {
		fferror("PASS not allowed in INIT commands, PASS ignored");
	    } else mapevt_insert_pass(mapevt_ptr);
	} else if (tokenid == sym_redo) {
	    scan();
/*          gprintf(TRANS, "got redo mapevt\n"); */
	    if (init_flag) {
		fferror("REDO not allowed in INIT commands, REDO ignored");
	    } else mapevt_insert_redo(mapevt_ptr);
	} else if (tokenid == sym_score) {
	    int transp, vel, rate;
	    int score_id;
	    scan();
	    score_id = mapper_find_score(token);
	    scan();
	    transp = parse_trans();
	    vel = parse_loud();
	    rate = parse_number(1L, (long) mapevt_rate_mask, "score rate");
	    issymbol();
	    if (tokenid == sym_gated) {
		rate |= mapevt_gate_flag;
		scan();
		issymbol();
		if (tokenid == sym_cycle) {
		    rate |= mapevt_cycle_flag;
		    scan();
		}
	    }
	    mapevt_insert_score(mapevt_ptr, score_id, transp, vel, rate);
	} else done = true;
    }
}


/* parse_dur -- parse a duration specification */
long parse_dur()
{
    char c = token[0];
    long dur;
    if (c == '+' || c == '-') {
	tokenx = 1;
	dur = parse_number(-32000L, 32000L, "duration offset");
	if (c == '-') dur = -dur;
	dur = dur & duration_duration_field;
	dur |= duration_transpose_flag;
    } else if (c == '*') {
	tokenx++;
	dur = parse_number(1L, 32000L, "duration percent");
	dur = dur | duration_product_flag | duration_transpose_flag;
    } else if (isdigit(c)) {
	dur = parse_number(0L, 32000L, "duration");
    } else fferror("duration expected");
    return dur;
}


/* parse_end -- indicate end of field */
/**/
void parse_end(what)
  char *what;
{
    if (token[tokenx] != EOS) {
	char error_string[100];
	sprintf(error_string, "extra characters, %s expected", what);
	fferror(error_string);
    }
}


/* parse_letter -- parse a map identifier */
/**/
char parse_letter()
{
    char c = token[0];
    if (strlen(token) != 1 ||
	  !isalpha(c)) {
	fferror("Map ID letter expected");
	c = 'A';
    }
    scan();      /* read ahead */
    if (islower(c)) c = toupper(c);
    return c;
}


/* parse_string -- parse a quoted string, malloc space and copy */
/**/
char *parse_string()
{
    char *result = NULL;
    size_t len;
    int i;
    scanstr();
    if (token[0] == '"') {
	len = strlen(token) - 1;  /* strip off 2 quotes, add EOS */
	if (len == 0 || token[len] != '"') {
	    fferror("Close quote not found");
	    len++;
	}
	result = memget(len);
	for (i = 1; i < len; i++) result[i-1] = token[i];
	result[len - 1] = EOS;
    } else {
	fferror("Quoted string expected");
    }
    scan(); /* read ahead */
    return result;
}


/* parse_vel -- parse a velocity indication */
/**/
int parse_vel()
{
    char c = token[0];
    int vel;
    if (c == '+') {
	tokenx++;
	vel = parse_number(0L, 127L, "transposition");
	vel = (vel >> 1) | mapevt_transpose_flag;
    } else if (c == '-') {
	tokenx++;
	vel = parse_number(0L, 127L, "transposition");
	vel = -(vel >> 1);      /* this sets transposition flag too */
    } else if (isdigit(c)) {
	vel = parse_number(1L, 127L, "vel");
	vel = vel >> 1;
    }
    return vel;
}
