/*************************************************************************
 *
 *      evt.h -- event definitions
 *
 *      Copyright (C) 1987 John H. Maloney
 *      All rights reserved
 *
 *************************************************************************/

/* pitch:
 *      pitches are encoded as MIDI key numbers where "60" is middle-C
 *      NOTE: key numbers start with 1!
 */

#define MIN_KEY 1
#define MAX_KEY 128
#define NO_KEY (MAX_KEY + 1)

/* time:
 *  times are unsigned long numbers (ulong's) which indicate
 *  the number of 125th's of a second since the beginning of the
 *  piece. there are two contexts in which we measure time:
 *
 *    solo time is the elapsed time as measured in the solo score
 *    performance time is the elapsed time as measured
 *      in the accompaniment score
 *
 *  both solo and performance times are "virtual" times; a virtual clock maps
 *  between real time and solo/performance time. however, because an actual
 *  solo may not be played at a constant rate, some systems may
 *  keep a mapping between the "ideal" and "actual" solo time
 *  (developed during a rehearsal session). thus, the relationship
 *  between real time and solo time is actually:
 *
 *    real_time = RateMap(TimeMap(solo_time))
 *
 *  where RateMap is a function which adjusts the rate of the
 *  performance by tracking and ajusting to the soloist.
 *
 *  ETERNITY is a time greater than or equal to all other times.
 */

#define evt_to_vtime(dur) ((dur) << 3)
#define vtime_to_evt(dur) ((dur) >> 3)

#define ETERNITY 0xFFFFFFFFL

struct evt_note_struct {
    ushort pitchset;/* hint: other notes in cevt */
    byte pitch;    /* pitch (MIDI key number, middle C = 60) */
    byte filler;
};

struct evt_rest_struct {
    ulong rdur;     /* duration of long rest */
};

typedef struct evt_struct {
    ushort deltaTime;   /* the starting time of the next event
			 * is deltaTime after this
			 * event's starting time. if deltaTime equals:
			 *      LREST_FLAG then this is a long rest
			 *      FIRST_FLAG then this is the start sentinel
			 *      END_FLAG then this is the end sentinel
			 *      otherwise it is a note event. */
    union {
	struct evt_note_struct note;
	struct evt_rest_struct rest;
    } u;
} evt_node, *evt_type;

#define LREST_FLAG 0xFFFF
#define FIRST_FLAG 0xFFFE
#define LAST_FLAG 0xFFFD
#define MAX_DELTA_TIME 0xFFFC

#define evt_channel(e) ((e)->u.note.channel)
#define evt_dtime(e)        \
    ((ulong) (evt_is_note(e) ?    \
     (e)->deltaTime :   \
     (e)->u.rest.rdur))
#define evt_is_first(e) (((e)->deltaTime) == FIRST_FLAG)
#define evt_is_last(e) (((e)->deltaTime) == LAST_FLAG)
#define evt_is_long_rest(e) (((e)->deltaTime) == LREST_FLAG)
#define evt_is_note(e) (((e)->deltaTime) <= MAX_DELTA_TIME)
evt_type evt_make_evts(seq_type seq, int ch, int *cevts);
#define evt_pitch(e) ((e)->u.note.pitch)
void evt_show(evt_type event, evt_type end_event);
