/* conduct.c -- test score module and accompaniment */
/* Copyright 1989 Carnegie Mellon University */

/*****************************************************************************
*       Change Log
*  Date | Change
*-----------+-----------------------------------------------------------------
*****************************************************************************/

#include "stdio.h"
#include "switches.h"
#include "ctype.h"
#include "string.h"
#include "cext.h"
#include "userio.h"
#include "cmdline.h"
#include "midifns.h" /* to get time_type */
#include "timebase.h"
#include "seq.h"
#ifdef SMARTSEQ
#include "smartseq.h"
#endif
#include "moxc.h"
#include "seqread.h"
#include "midicode.h"
#include "evt.h"
#include "cevt.h"
#include "score.h"
#include "iter.h"
#include "sched.h"
#include "match.h"
#ifdef SLAVE
#include "slave.h"
#endif

#ifdef THINK_C
#include "console.h"
#endif

void midi_flush_input();
void play_it();
static void my_noteon_meth();

/****************************************************************************
*
*    variables shared with other modules
*
****************************************************************************/

extern boolean musictrace;      /* read by report_modes */
extern boolean miditrace;       /* read by report_modes */

/****************************************************************************
*
* macros and constants
*
****************************************************************************/

#define LINE_SIZE 100

/****************************************************************************
* Variables set by command line switches
****************************************************************************/

#ifdef DOS
extern int IsAT;
#endif

long solo_channel = 1;
boolean verbose = false;
#ifdef SLAVE
boolean slave = false;
#endif

/****************************************************************************
* Local Data 
****************************************************************************/

#ifdef SLAVE
char accomp_syntax[] = "solo<o>channel to be played by human;\
	verbose<s>print accomp debugging info;\
	slave<s>send sync messages to a slave sequencer";
#else
char accomp_syntax[] = "solo<o>channel to be played by human;\
	verbose<s>print accomp debugging info";
#endif

#ifdef SMARTSEQ
smartseq_type the_seq = NULL;
#else
seq_type the_seq = NULL;
#endif
private score_type the_score = NULL;
iter_type the_iter = NULL;
match_type the_match = NULL;
sched_type the_sched = NULL;
int stayalive_id = 0;
time_type a_start_time = 0;
boolean match_enable = true;
void (*default_noteon_meth)();
boolean audition_mode = false;

char *app_syntax = "print<s>Print score";

/****************************************************************************
*    Routines local to this module
****************************************************************************/

#ifdef SMARTSEQ
smartseq_type new_score();
#else
seq_type new_score();
#endif

#define ESCAPE 0x1B

/* asciievent -- ascii event handler */
/**/
void asciievent(char c)
{
    if (c == 'z') {
	if (the_seq->runflag) {
	    timebase_use(the_seq->timebase);
	    gprintf(TRANS, "Stopped at %ld\n", virttime / 10);
	}
	quit();
    } else if (c == ' ') {
	if (audition_mode) asciievent('z');
	else if (match_enable) match_event(the_match, 60);
    } else if (c == ESCAPE) {
	gprintf(TRANS, "All off!\n");
	alloff();
    } else if (c == '-' || c == '_') {
	time_type rate = seq_rate(the_seq) + 3;
	if (rate >= 2560) rate = 2560;
	seq_set_rate(the_seq, rate);
#ifdef SLAVE
	if (slave) slave_update(the_seq->timebase->virt_base, rate);
#endif
    } else if (c == '+' || c == '=') {
	time_type rate = seq_rate(the_seq) - 3;
	if (rate <= 0) rate = 1;
	seq_set_rate(the_seq, rate);
#ifdef SLAVE
	if (slave) slave_update(the_seq->timebase->virt_base, rate);
#endif
    }   
}


/* get_user_input -- get first character of user typein */
/**/
char get_user_input()
{
    char userinput[LINE_SIZE];
    ggets(userinput);
    if (isupper(*userinput)) *userinput = tolower(*userinput);
    return *userinput;
}


/* get_user_number -- get a number from the user */
/**/
boolean get_user_number(num, prompt, min_num, max_num, deflt)
  long *num;
  char *prompt;
  long min_num;
  long max_num;
  boolean deflt;
{
    long new;
    char userinput[LINE_SIZE];
    gprintf(TRANS, prompt, *num);
    ggets(userinput);
    /* I removed (byte *) coercion from userinput.  Why was it there? -RBD */
    while (sscanf(userinput, "%ld", &new) == 1) {
	if (new >= min_num && new <= max_num) {
	    *num = new;
	    return true;
	} else gprintf(TRANS, "out of range, ");
	ggets(userinput);
    }
    if (!deflt) gprintf(TRANS, "command cancelled.\n");
    return false;
}


/* midi_flush_input -- grab all queued input */
/**/
void midi_flush_input()
{
    byte midi_data[4];
    while (getbuf(false, midi_data)) ;
}


/* my_noteon_meth -- play a note with transformations, pitch 0 is a switch */
/**/
static void my_noteon_meth(seq, chan, pitch, vel)
  seq_type seq;
  int chan, pitch, vel;
{
    if (pitch == 0) {
	if (vel == 1) {         /* enable matching */
	    match_enable = true;
	    gprintf(TRANS, "match enable\n");
	} else if (vel == 2) {  /* disable matcher input */
	    match_enable = false;
	    gprintf(TRANS, "match disable\n");
	} else if (vel == 3) { /* set to nominal tempo */
	    seq_set_rate(seq, 256L);
	}
	return;
    }
    /* otherwise do the normal processing: */
    (*default_noteon_meth)(seq, chan, pitch, vel);
}


/****************************************************************************
*                   new_score
* Inputs:
*    char * defaultFile: the file name from command line or NULL
* Effect:
*    if defaultFile is not NULL, it is read
*    otherwise new_score prompts for file name and attempts to read that file
*    if the user doesn't supply a file name when prompted or if the
*    open operation fails, the NULL score is returned
****************************************************************************/

#ifdef SMARTSEQ
smartseq_type new_score(defaultFile)
#else
seq_type new_score(defaultFile)
#endif
  char *defaultFile;
{
#ifdef SMARTSEQ
    smartseq_type seq;  /* the new sequence */
#else
    seq_type seq;       /* the new sequence */
#endif
    char inFileName[LINE_SIZE];
    FILE *inFile = NULL;
    char *suffix;

    inFileName[0] = EOS;
    if (defaultFile != NULL)
	strcpy(inFileName, defaultFile);


    inFile = fileopen(inFileName, "mid", "rb", "input file");
    if (!inFile) return NULL;

#ifdef SMARTSEQ
    seq = smartseq_create();
#else
    seq = seq_create();
#endif
    default_noteon_meth = seq->noteon_fn;
    seq->noteon_fn = my_noteon_meth;
    suffix = strrchr(fileopen_name, '.');
    if (suffix && (strcmp(suffix, ".mid") == 0)) seq_read_smf(seq, inFile);
    else {
#ifndef UNIX
	/* oops - reopen as a text file */
	fclose(inFile);
	inFile = fopen(fileopen_name, "r");
#endif
	seq_read(seq, inFile);
    }
    return seq;
}

extern timebase_type default_base;

void stayalive(id)
{
    if (stayalive_id == id) {
	timebase_use(default_base);
	cause(1000, stayalive, id);
    }
}


void print_cevts()
{
    cevt_type prev, next;
    iter_set_next_cevt(next, the_iter, true);
    while (!evt_is_last(*(prev = iter_curr_cevt(the_iter)))) {
	gprintf(TRANS, "@%5ld ", iter_curr_time(the_iter));
	iter_set_next_cevt(next, the_iter, true);
	evt_show(*prev, *next);
    }
}


/* run the accompaniment */
/**/
void play_it(start_time, next_time, last_flag)
  time_type start_time;
  time_type next_time;
  boolean last_flag;
{
    gprintf(TRANS, "Type z to stop.\n");
    stayalive(++stayalive_id);
    seq_set_channel_mask(the_seq, ~(1 << (solo_channel - 1)));
    seq_play(the_seq);
    seq_set_rate(the_seq, 256L);
    seq_pause(the_seq, true);
    if (start_time > 0) {
	gprintf(TRANS, "Starting at %ld cs.\n", start_time/10);
	next_time = match_jump_to(the_match, start_time, &last_flag);
    }
    timebase_use(the_seq->timebase);
    cause(1000, creeper, the_sched);
/*    gprintf(TRANS, "the_seq duration is %ld\n", seq_duration(the_seq)); */
    if (verbose) gprintf(TRANS, "calling initial sched_match, next_time %ld\n",
			 next_time);
    sched_match(the_sched, 0, gettime(), start_time, last_flag, next_time);
    sched_flush(the_sched);
    midi_flush_input();
/*    gprintf(TRANS, "in play_it: vt %ld, rt %ld, now %ld\n",
		   the_seq->timebase->virt_base, the_seq->timebase->real_base,
		   real_to_virt(the_seq->timebase, gettime())); */
    moxcrun();
    seq_reset(the_seq);
    match_reset(the_match);
    match_enable = true;
    alloff();   /* make double sure all midi notes shut up */
}

/* play just the accompaniment */
/**/
void audition_it(time_type start_time)
{
    gprintf(TRANS, "Type z or space to stop.\n");
    audition_mode = true;
    stayalive(++stayalive_id);
    seq_set_channel_mask(the_seq, ~(1 << (solo_channel - 1)));
    seq_set_rate(the_seq, 256L);  /* rate setup is completed in seq_play */
    if (start_time != 0) {
	gprintf(TRANS, "Starting at %ld cs.\n", start_time/10);
	seq_start_time(the_seq, start_time);
    }
    seq_play(the_seq);
    /* gprintf(TRANS, "the_seq duration is %ld\n", seq_duration(the_seq)); */
    moxcrun();
    seq_reset(the_seq);
    match_reset(the_match);
    alloff();	/* make double sure all midi notes shut up */
    audition_mode = false;
}


/****************************************************************************
*                    main
* Effect: 
*   Start the accompaniment.
****************************************************************************/

void main(argc, argv)
  int argc;
  char *argv[];
{
    cevt_type next;
    time_type patience = 0;
    time_type next_time = 0;
    boolean last_flag = false;
    char *extras;
    
#ifdef THINK_C
    /* Macintosh needs help to provide Unix-style command line */
    argc = ccommand(&argv);
#endif

    cl_syntax(accomp_syntax);
    cl_syntax(sched_syntax);
    cl_syntax(match_syntax);
    if (moxcinit(argc, argv)) {
	seq_extensions();
	solo_channel = cl_int_option("solo", 1L);
	if ((solo_channel < 1) || (solo_channel > 16)) {
	    gprintf(TRANS, "-solo %d out of range, using 1\n", solo_channel);
	    solo_channel = 1;
	}
	verbose = cl_switch("verbose");
#ifdef SLAVE
	slave = cl_switch("slave");
#endif
	seq_print = cl_switch("print");
	the_seq = new_score(cl_arg(1));
	if (!the_seq) {
	    gprintf(TRANS, "No seq!\n");
	    goto endit;
	} else seq_register(the_seq);
	if (extras = cl_arg(2)) {
	    gprintf(ERROR, "Extra (filename?) argument ignored: %s\n",
		    extras);
	}
/*      gprintf(TRANS, "got the seq\n"); */
	the_score = score_convert(the_seq, (int) solo_channel);
	cu_register((cu_fn_type) score_destroy, the_score);
/*      gprintf(TRANS, "Converted the score\n"); */
/*      askbool("type to continue", true); */
/*      gprintf(TRANS, "Here is the Event Array:\n"); */
/*      askbool("type to continue", true); */
/*      score_show_evt_array(the_score); */
/*      askbool("type to continue", true); */
/*      gprintf(TRANS, "Here is the Cevt Array:\n"); */
/*      score_show_cevt_array(the_score); */
/*      gprintf(TRANS, "printed the result\n"); */

	the_iter = iter_create(the_score);
	cu_register((cu_fn_type) iter_destroy, the_iter);
/*      gprintf(TRANS, "Created the iterator, lets pull out the notes\n"); */
/*      print_cevts(); */
	the_iter->curr_cevt = the_iter->first_cevt;
  
	the_iter->curr_time = 0L;
	iter_set_next_cevt(next, the_iter, true);
	patience = cl_int_option("pat", patience);
	gprintf(TRANS, "Patience = %d\n", patience);
	the_sched = sched_create(the_seq, patience * 10);
	cu_register((cu_fn_type) sched_destroy, the_sched);
	the_match = match_create(the_iter, 0, the_sched);
	cu_register((cu_fn_type) match_destroy, the_match);
	while (true) {
	    gprintf(TRANS, 
		"<return> play, b set begin time, a audition, q quit : ");
	    switch (get_user_input()) {
	      case 0:
		play_it(a_start_time, next_time, last_flag);
		break;
	      case 'q':
		goto endit;
		break;
	      case 'a':
		audition_it(a_start_time);
		break;
	      case 'b':
		a_start_time /= 10;
		get_user_number((long *) &a_start_time,
			"Type starting time (in hundredths of seconds) [%d]: ",
			0L, 0xFFFFFL, true);
		a_start_time *= 10;
		next_time = match_jump_to(the_match, a_start_time, &last_flag);
		break;
	    }
	}
	; /* careful with a label after a while loop: */ ;
endit:
	gprintf(TRANS, "exiting now\n");
	;
    }
    EXIT(0);
}


/* keyup -- key up handler */
/**/
void keyup(int c, int k)
{
    /* insert key up actions here */
}


/* keydown -- key down handler */
/**/
void keydown(int c, int k, int v)
{
/*    gprintf(TRANS, "keydown: %d\n", k); */
    if (c == solo_channel && match_enable) match_event(the_match, k);
}


/* prgmchange -- program change handler */
/**/
void prgmchange(int chan, int value)
{
    /* insert program change actions here */
}


/* bendchange -- pitch bend handler */
/**/
void bendchange(int chan, int value)
{
    /* insert pitchbend actions here */
}


/* buttonchange -- amiga button handler */
/**/
void buttonchange(identifier, value)
{
    /* insert buttonchange actions here */
}


/* propchange -- amiga proportional controller handler */
/**/
void propchange(identifier, value)
{
    /* insert propchange actions here */
}


/* ctrlchange -- control change handler */
/**/
void ctrlchange(int chan, int ctrl, int value)
{
    /* insert control change actions here */
}


/* peddown -- pedal down handler */
/**/
void peddown(int c)
{
    /* insert pedal down actions here */
}


/* pedup -- pedal up handler */
/**/
void pedup(int c)
{
    /* insert pedal up actions here */
}


/* touchchange -- after touch handler */
/**/
void touchchange(int chan, int value)
{
    /* insert after touch actions here */
}


/* coda -- what to do at end of performance */
/*
 * NOTE: this is called just before closing down the midi interface.
 */
void coda(void)
{
}

void sysex()
{
}


void midievent(data)
  unsigned char data[4];
{
}
