/* Adagio compiler -- scan command line, call phase1, phase2 */
/* Copyright 1989 Carnegie Mellon University */

/*****************************************************************************
*       Change Log
*  Date | Change
*-----------+-----------------------------------------------------------------
* 12-Mar-86 | Created
* 28-May-86 | Added calls to cmdline.c for command line parsing
* 13-Oct-88 | JCD : Exclusive AMIGA version.
* 31-Jan-90 | GWL : LATTICE DOS version
* 30-Jun-90 | RBD : further changes
*  2-Apr-91 | JDW : further changes
*  1-Jul-91 | RBD : scores now start at virtual time 0, not 1 sec
* 16-Feb-92 | GWL : for LATTICE, add handlers and redefine rindex
*  7-May-94 | RBD & GWL: better support for Macintosh file types, e.g. 'Midi'
*****************************************************************************/

#include "switches.h"
#include "stdio.h"
#include "ctype.h"
#include "string.h"
#include "cext.h"
#include "userio.h"
#include "cmdline.h"
/* #include "handlers.h" */
#include "midifns.h" /* to get time_type */
#include "adagio.h"
#include "timebase.h"
#include "seq.h"
#include "moxc.h"
#include "seqread.h"
#include "midicode.h"


#ifdef THINK_C
#include "console.h"
#endif

extern timebase_type default_base;

extern int abort_flag;  /* control C or G*/

/****************************************************************************
*
*    variables exported to other modules
*
****************************************************************************/

char score_na[name_length];     /* this will be available to phase1 and 2 */
char rec_score_na[name_length]; /* name of record file */
boolean ctrlflag = false;       /* */

/****************************************************************************
*
*    variables shared with other modules
*
****************************************************************************/

extern boolean musictrace;      /* read by report_modes */
extern boolean miditrace;       /* read by report_modes */

/****************************************************************************
*
* macros and constants
*
****************************************************************************/

#define LINE_SIZE 100

/****************************************************************************
* Variables set by command line switches
****************************************************************************/

#ifdef DOS
extern int IsAT;
#endif
boolean script = false; /* if true, run non-interactively 
			 * (as if invoked from a script) */

/****************************************************************************
* Local Data 
****************************************************************************/

seq_type the_seq = NULL;
int adagio_mode;
FILE *pp_outfile;
boolean a_recording = false;
boolean midi_seq = false;   /* is this sequence from a std midi file? */

/* a_playing represents 3 phases: not playing, pre-play pause, playing */
#define A_RESET 0
#define A_PRE_PLAY 1
#define A_PLAYING 2
int a_playing = A_RESET;

boolean rec_mode_flag;
time_type a_start_time;
/* time_type pp_stop_time; */
int stayalive_flag = true;

/* midi clock data: */
time_type prevtime;     /* the actual time of previous clock */
extern time_type clock_ticksize; /* the ticksize (set by !clock commands) */
time_type midi_vtime;   /* the accumulated virtual time */
time_type midi_fraction;/* the fractional part of midi_vtime */
boolean midi_started;	 /* true between midi start and stop bytes */
boolean midi_stop_req;	/* used to respond to MIDI_STOP msg */
/* these two switches are declared in seq.c: */
extern boolean external_midi_clock;	/* set to enable external sync */
extern boolean suppress_midi_clock;	/* set to turn off clock output */
boolean write_switch = false;  /* default: file write switch off*/
char *app_syntax = "help<s>Show usages;print<s>Print score;\
		script<s>Play, record or transcribe once, then exit;\
		record<o>Enter record or transcribe mode on startup";

extern void seq_write_smf();

/****************************************************************************
*    Routines local to this module
****************************************************************************/

private void cmd_help();
private seq_type new_score();
private void play_cmd();
private void play_rec();
private void record_cmd();
private void report_activity();
private void report_modes();
private void report_stop();
private void score_end(seq_type seq);
private void showhelp();
private void stayalive(seq_type seq);
private void stop_cmd();
private void top_cmd();
private void transcribe_cmd();

/* asciievent -- ascii event handler */
/**/
void asciievent(char c)
{
    time_type seq_vt;

    if (a_playing != A_RESET) {
	if (c == ' ' || abort_flag) {
	    /* if we've actually started the score, compute the score 
	     * virtual time; otherwise, we're at a "negative" virtual
	     * time because the score hasn't even started. Just use 0,
	     * a special value which isn't printed.
	     */
	    if (abort_flag == BREAK_LEVEL) {
		/* treat BREAK (^G) as equivalent to SPACE in this case */
		abort_flag = 0;
	    }
	    if (a_playing == A_PLAYING) {
		seq_vt = real_to_virt(the_seq->timebase, eventtime);
		stop_cmd(STOP_REQ, seq_vt);
	    } else stop_cmd(STOP_REQ, 0L);
	} else if (!a_recording && c == 'p') {
	    stop_cmd(PAUSE, seq_pause(the_seq, true));
	} else {
	    gprintf(TRANS, "Type space to stop\n");
	}
    } else if (a_recording) { /* (and a_playing == A_RESET) */
	    if (c == ' ') stop_cmd(STOP_REQ, eventtime);
	    else gprintf(TRANS, "Type space to stop\n");
    }
}


/* call_seq_play -- call seq_play if in A_PRE_PLAY mode */
/*
 * NOTE: Adagio starts score 1 sec after you type RETURN.  This
 * period is called PRE_PLAY, and it is implemented by a cause()
 * of this routine.
 */
void call_seq_play()
{
    if (a_playing == A_PRE_PLAY) {
	seq_play(the_seq);
	/* the previous line will set the timebase according to the rate
	   of the sequence.  If external_midi_clock, we want to turn it back 
	   off. Note that we don't want to set_seq_rate to STOPRATE because
	   then if we disabled midi clocks, the seq wouldn't run.
	 */
	if (external_midi_clock) {
	    /* stop the seq so that it can be advanced by MIDI clocks */
	    set_rate(seq_timebase(the_seq), STOPRATE);
	}
	a_playing = A_PLAYING;
    }
    /* otherwise, assume score playing has been aborted */
}

/* get_user_input -- get first character of user typein */
/**/
char get_user_input()
{
#ifdef  LATTICE322
    char x; /* using this keeps the compiler happy */
#endif
    char userinput[LINE_SIZE];
    ggets(userinput);
    if (isupper(*userinput)) *userinput = tolower(*userinput);
    if (abort_flag) return (char)(abort_flag);
#ifdef  LATTICE322
    return (x=*userinput);
#else
    return *userinput;
#endif
}


/* get_user_number -- get a number from the user */
/**/
boolean get_user_number(num, prompt, min_num, max_num, deflt)
  long *num;
  char *prompt;
  long min_num;
  long max_num;
  boolean deflt;
{
    long new;
    char userinput[LINE_SIZE];

    /* as long as input is not aborted, keep asking for a number */
    while (!abort_flag) {
	gprintf(TRANS, prompt, *num);
	ggets(userinput);

	/* if you get a number in range, you're not aborted (^G or ^C
	 * would have set userinput to "") so you must be done -- return:
	 */
	if (sscanf((char *) userinput, "%ld", &new) == 1) {
	    if (new >= min_num && new <= max_num) {
		*num = new;
		return true;
	    } else gprintf(TRANS, "out of range, ");

	/* if you don't have a number, either 
	 *      you are aborted => fall out of while loop
	 *      or the user typed RETURN, but defaults not allowed => try again
	 *      or the user typed RETURN to get the default => return false
	 *      or the user typed something not a number => try again
	 * therefore, loop unless !aborted, empty input string, and deflt
	 */
	} else if ((!abort_flag) && (userinput[0] == EOS) && deflt) {
	    return false;
	}
    }

    /* input was aborted by ^G (BREAK) or ^C (ABORT)
     * BREAK means take the default if any; cancel the command otherwise */
    gprintf(TRANS, "\n");
    if (abort_flag == BREAK_LEVEL) {
	abort_flag = 0;
	if (!deflt) gprintf(TRANS, "command cancelled.\n");
    }
    return false;
}


/****************************************************************************
*                   new_score
* Inputs:
*    char * defaultFile: the file name from command line or NULL
* Effect:
*    if defaultFile is not NULL, it is read
*    otherwise new_score prompts for file name and attempts to read that file
*    if the user doesn't supply a file name when prompted or if the
*    open operation fails, the NULL score is returned.
*    Filenames ending in ".mid" are read as midi files.
****************************************************************************/

private seq_type new_score(defaultFile)
  char *defaultFile;
{
    seq_type seq;       /* the new sequence */
#ifdef MACINTOSH
    OSType ftype, fcreator;
#endif

    FILE *inFile = NULL;
    inFile = fileopen(defaultFile, "gio", "r", "input adagio file");
    
    if (!inFile) return NULL;
    
    seq = seq_create();
    if (seq) {
	char *suffix = strrchr(fileopen_name, '.');
	if ((suffix && (strcmp(suffix, ".mid") == 0))
#ifdef MACINTOSH
	    /* look at file type as well as extension on Macs: */
	    ||
	    (get_file_info(fileopen_name, &ftype, &fcreator) &&
	     (ftype == 'Midi'))
#endif
	   ) {
#ifndef UNIX
	    fclose(inFile);
	    inFile = fopen(fileopen_name, "rb");
#endif
	    seq_read_smf(seq, inFile);
	    midi_seq = true;
	} else {
	    seq_read(seq, inFile);
	    midi_seq = false;
	}
	a_start_time = 0;
    }
    fclose(inFile);

  
    return seq;
}


/****************************************************************************
*               play_rec
* Inputs:
*    ulong startTime: where to start playing
*    StreamPtr outStream: stream to record to, if recording.
* Effect:
*    plays the music, simultaneously recording if outStream isn't NULL
****************************************************************************/

private void play_rec(outfile)
  FILE *outfile;
{
    /* this is probably not strictly necessary... */
    eventtime = gettime();      /* capture when user really said "go" */
    set_rate(timebase, 2560L);  /* recompute real_base */

    clock_ticksize = (600L / 24L) << 16;        /* default ticksize for MM=100 */

    a_recording = (outfile != NULL);
    /* if recording wait for SPACE, if just playback, stop at end */
    if (a_recording) seq_at_end(the_seq, noop);
    else seq_at_end(the_seq, score_end);
    if (a_start_time == 0) {
	gprintf(TRANS, "Type <space> to stop, p to pause...\n");
    }

    midi_real(external_midi_clock);
    a_playing = A_PRE_PLAY;
    /* pause 1 second before starting score: */
    cause((delay_type) 100, call_seq_play);
    if (a_recording) stayalive(NULL);
    moxcrun();
}


/* continue_cmd -- */
/**/
void continue_cmd()
{
    while (wait_ascii() != ' ' && !abort_flag) ;
    gprintf(TRANS, "\ncontinued ...\n");
    a_playing = A_PLAYING;
    eventtime = gettime();
    seq_play(the_seq);
}


private void score_end(seq_type seq)
{
	stop_cmd(END_SCORE, virttime);
}

/* stop_cmd --  */
/*
 * NOTE: this may be called from score_end, in which case timebase is
 * the_seq->timebase, or from ascii_event, in which case timebase is
 * default_base.  Force timebase to default_base so that when we quit()
 * we end up with the default.  Otherwise, bad things happen.
 */
private void stop_cmd(pp_stop_reason, pp_stop_time)
  int pp_stop_reason;
  time_type pp_stop_time;
{
    midi_started = false;
    midi_stop_req = false;
    if (timebase != default_base) timebase_use(default_base);
    midi_real(false);

    if (a_recording) {
	moxcdone=true;
	a_recording = false;
	/* (rec_final closes the file) */
	rec_final(pp_outfile, rec_mode_flag);
    }

    /* if we are in the 1 sec preplay period, advance virtual time to
	flush out the pending call to call_seq_play()
     */
    if (a_playing == A_PRE_PLAY) {
	a_playing = A_RESET;    /* this is a signal to call_seq_play */
	set_virttime(timebase, (time_type) (virttime + 200));
	catchup();
    }
    a_playing = A_RESET;        /* in case we were in A_PLAY mode */
    if (pp_stop_reason != PAUSE) {
	if (the_seq) seq_reset(the_seq);
	/* in case a "stayalive" loop is running on timebase, flush it */
	stayalive_flag = false;
	set_virttime(timebase, (time_type) (virttime + 20000));
	catchup();
	stayalive_flag = true;
    }

    report_stop(pp_stop_time, pp_stop_reason);
    if (pp_stop_reason == PAUSE && !abort_flag) {
	gprintf(TRANS, "Type <space> to continue ... ");
	continue_cmd();
    } else {
	quit();
    }
}


private void report_realtime(ulong stop_time)
{
    if (seq_rate(the_seq) != 256) {
	long rate = ((100L << 8) + (seq_rate(the_seq) >> 1)) /
		    (long) seq_rate(the_seq);
	gprintf(TRANS,
		"(Due to rate of %ld percent, done at real time of %ld.)\n",
		rate, (stop_time * seq_rate(the_seq) + 1280) / 2560);
    }
}


/****************************************************************************
*               report_activity
* Effect:
*    reports currently playing notes or recently played notes in each voice
****************************************************************************/

/* RECENTLY determines how far back to look for "recent" notes to report
 * it is currently set to two seconds in miliseconds
 */
#define RECENTLY 2000

/* structure used by report_activity: */
struct ActivityRecord {
    boolean     played;
    boolean     reported;
    int line;
    ulong       start_time;
};

private void report_activity(stop_time)
  ulong stop_time;
{
    event_type score = seq_events(the_seq);
    struct ActivityRecord voice[MAX_CHANNELS + 1];
    /* voice[] keeps the state of each voice at stop_time
	     * (NOTE: voice[0] is unused) */
    event_type n;
    int i;

    /* initialize voice record array */
    for (i = 1; i <= MAX_CHANNELS; i++) {
	voice[i].played = voice[i].reported = false;
    }

    gprintf(TRANS, "At time %ld (hundreths of a second):\n",
	    (stop_time + 5) / 10);

    report_realtime(stop_time);

    if (midi_seq) return; /* don't report line nums, they don't exist */

    for (n = score; n != NULL; n = n->next) {
	if (is_note(n) &&
	    (n->ntime <= stop_time)) {
	    /* found a note which started before stop_time */
	    if ((n->ntime + (n->u.note.ndur >> 8)) >= stop_time) {
		/* report if it was playing at stop_time */
		gprintf(TRANS,"Voice %d was playing line %d\n",
		vc_voice(n->nvoice), n->nline);
		voice[vc_voice(n->nvoice) - 1].reported = true;
	    } 
	    else {
		/* otherwise remember it; it might be the last note played */
		voice[vc_voice(n->nvoice) - 1].played = true;
		voice[vc_voice(n->nvoice) - 1].start_time = n->ntime;
		voice[vc_voice(n->nvoice) - 1].line = n->nline;
	    }
	}
    } /* end of score traversal */

    for (i = 1; i <= MAX_CHANNELS; i++) {
	if ( voice[i].played &&
	    !voice[i].reported &&
	    (voice[i].start_time + RECENTLY >= stop_time))
	    gprintf(TRANS, "Voice %d recently played line %d\n",
	    i + 1, voice[i].line);
    }
}

private void report_channels()
{
    gprintf(TRANS, "Enabled: ");
    report_enabled_channels(the_seq);
    gprintf(TRANS, "\n");
}


/****************************************************************************
*               report_modes
* Effect:
*    report state of various operation modes
****************************************************************************/

private void report_modes()
{
    gprintf(TRANS, "\n");
    gprintf(TRANS, "MIDI byte trace (m option) %s\n",
    (miditrace ? "on" : "off"));
    gprintf(TRANS, "Music operation display (d option) %s\n\n",
    (musictrace ? "on" : "off"));
}


/****************************************************************************
*               report_stop
* Inputs:
*    ulong stop_time: the time that playing was stopped
*    int stop_reason: the reason that playing was stopped
* Effect:
*    announce stopped state and (under some conditions) tell
*    what was playing (or recently played) when we stopped
****************************************************************************/

private void report_stop(stop_time, stop_reason)
  ulong stop_time;
  int stop_reason;
{
    switch(stop_reason) {
      case STOP_MIDI:
	gprintf(TRANS, "      * * STOPPED * *\n");
	gprintf(TRANS, "Stopped via Midi\n");
	break;
      case STOP_REQ:
	gprintf(TRANS, "      * * STOPPED * *\n");
	gprintf(TRANS, "Stopped at user's request\n");
	break;
      case OUT_OF_MEMORY:
	gprintf(TRANS, "      * * STOPPED * *\n");
	gprintf(TRANS, "Out of memory for recording\n");
	break;
      case PAUSE:
	gprintf(TRANS, "Pause at %lu (hundredths)\n", (stop_time + 5) / 10);
	return;    /* quit early -- don't print activity */
      case END_SCORE:
	gprintf(TRANS, "Done at time %ld (hundreths of a second)\n",
	    (stop_time + 5) / 10);
	report_realtime(stop_time);
	return;
      default:
	/* never happens */
	gprintf(FATAL,
	"Implementation error (adagio.c): unknown stop reason.");
	return;
    }
    if (stop_time > 0 && the_seq) {
	report_activity(stop_time);
    }
}


/****************************************************************************
*               cmd_help
* Effect:
*    help for command interpreter
****************************************************************************/

private void cmd_help()
{
    gprintf(TRANS, "\n");
    gprintf(TRANS, "   q          quit\n");
    gprintf(TRANS, "   e          set channel enable\n");
    gprintf(TRANS, "   f          set channel mute\n");
    gprintf(TRANS, "   b          set begining time\n");
    gprintf(TRANS, "   n          new input file\n");
    gprintf(TRANS, "   c          toggle using external clock\n");
    gprintf(TRANS, "   r          change rate (play mode only)\n");
    gprintf(TRANS, "   t          set transposition (play mode only)\n");
    gprintf(TRANS, "   l          set loudness offset (play mode only)\n");
    gprintf(TRANS, "   m          toggle MIDI byte trace mode\n");
    gprintf(TRANS, "   d          toggle music operation trace mode\n");
    gprintf(TRANS, "   s          report operation mode\n");
    gprintf(TRANS, "   u          read a tuning file\n");
    gprintf(TRANS, "   w          write a MIDI file\n");
    gprintf(TRANS, "   a          write an Adagio file\n");
    gprintf(TRANS, "   ?,<other>  display this message\n");
    gprintf(TRANS, "\n");
}


/****************************************************************************
*                       default_response
* Inputs:
*
* Effect:
*
****************************************************************************/

void default_response(response)
  char response;
{
    char *defaultFile = NULL;
    FILE *outfile;

    switch (response) {
      case 'c':
	/* cycle through: Internal enabled, External enabled, 
	  * Internal suppressed.  The states in order are:
	 * Suppress External
	 *  false    false
	 *  false    true
	 *  true     false
	 */
	if (!suppress_midi_clock && !external_midi_clock) {
	    external_midi_clock = true;
	    gprintf(TRANS, "Adagio will sync to external MIDI clock messages.\n");
	} else if (external_midi_clock) {
	    external_midi_clock = false;
	    suppress_midi_clock = true;
	    gprintf(TRANS,
		"Adagio will NOT generate or sync to MIDI clock messages.\n");
	} else /* suppress_midi_clock is true */ {
	    suppress_midi_clock = false;
	    gprintf(TRANS, "Adagio will generate MIDI clock messages.\n");
	}
	break;

      case 'w':
	if (!the_seq) {
	    gprintf(TRANS, "No sequence to write!\n");
	    break;
	}
	outfile = fileopen(defaultFile, "mid", "wb", "standard midi file");
	if (outfile) {
#ifdef MACINTOSH
	    put_file_info(fileopen_name, 'Midi', 'CMTK');
#endif
	    seq_write_smf(the_seq, outfile);
	    fclose(outfile); 
	}
	break;
      case 'a':
	if (!the_seq) {
	    gprintf(TRANS, "No sequence to write!\n");
	    break;
	}
	outfile = fileopen(defaultFile, "gio", "w", "Adagio file");
	if (outfile) {
#ifdef MACINTOSH
	    put_file_info(fileopen_name, 'TEXT', 'CMTK');
#endif
	    seq_write(the_seq, outfile, !askbool("Use relative time", true));
	    fclose(outfile); 
	}
	break;
      case 'b': {
	time_type old_time = a_start_time;
	time_type st = a_start_time / 10L;
	if (get_user_number((long *)&st,
		"Type starting time (in hundredths of seconds) [%ld]: ",
		0L, 0xFFFFFL, true)) {
	    a_start_time = st * 10L;
	}
	if ((a_start_time < old_time) && the_seq) seq_reset(the_seq);
	break;
      }
      case 'e': {
	ulong chan = 0;
	report_channels();
	if (get_user_number((long *) &chan,
		"Enable what channel? (0 for all) : ", 0L, (long) MAX_CHANNELS, false)) {
	    if (chan == 0) chan = 0xFFFFFFFF;
	    else chan = 1 << (chan - 1);
	    seq_set_channel_mask(the_seq, seq_channel_mask(the_seq) | chan);
	}
	report_channels();
	break;
      }
      case 'f': {
	ulong chan = 0;
	report_channels();
	if (get_user_number((long *) &chan,
		"Disable what channel? (0 for all) : ", 0L, (long) MAX_CHANNELS, false)) {
	    if (chan == 0) chan = 0xFFFFFFFF;
	    else chan = 1 << (chan - 1);
	    seq_set_channel_mask(the_seq, seq_channel_mask(the_seq) & ~chan);
	}
	report_channels();
	break;
      }
      case 'm':
	tracemidi((boolean) !miditrace);
	report_modes();
	break;
      case 's':
	report_modes();
	break;
      case 'd':
	trace((boolean) !musictrace);
	report_modes();
	break;
      case 'u':
	read_tuning("");
	break;
      case '?':
      default:
	cmd_help();
	break;
    }
}


/****************************************************************************
*                               play_cmd
****************************************************************************/

private void play_cmd(file)
  char *file;
{
    while (!abort_flag) {
	char response;

	if (!the_seq) the_seq = new_score(file);

	if (!the_seq) {
	    gprintf(TRANS, "PLAY MODE: error encountered, no score to play!\n");
	    return;
	}
	if (a_start_time) seq_start_time(the_seq, a_start_time);
	if (!script) gprintf(TRANS, 
  "PLAY MODE: <return> to play, ? for help, q to quit, n for new score : ");
	
	switch (response = (script ? 0 : get_user_input())) {
	  case 0:
	    play_rec(NULL);
	    if (script) return;
	    break;
	  case BREAK_LEVEL:
	    abort_flag = 0;
	    /* no break here */
	  case ABORT_LEVEL:
	    return;
	  case 'n': 
	    if (the_seq) seq_free(the_seq);
	    the_seq = NULL;
	    file = NULL;
	    break;
	  case 'q':
	    /* NOTE: there's another return above */
	    return;
	  case 'r': {
	    /* divide with rounding: */
	    long prev = ((100L << 8) + (seq_rate(the_seq) >> 1)) /
			(long) seq_rate(the_seq);
	    if (get_user_number(&prev, "Type rate (percent) [%ld] : ",
				1L, 0x7FFFL, true)) {
		/* divide with rounding: */
		seq_set_rate(the_seq, ((100L << 8) + (prev >> 1)) / prev);
	    }
	    break;
	  }
	  case 't': {
	    long prev = seq_transpose(the_seq);
	    if (get_user_number(&prev, "Type transposition (semitones) [%ld] : ",
				-99L, 99L, true))
		seq_set_transpose(the_seq, (int) prev);
	    break;
	  }
	  case 'l': {
	    long prev = seq_loudness(the_seq);
	    if (get_user_number(&prev, "Type loudness offset [%ld] : ",
				-128L, 128L, true))
		seq_set_loudness(the_seq, (int) prev);
	    break;
	  }
	  default:
		default_response(response);
	    break;
	}
    }
}

/****************************************************************************
*                           transcribe_cmd
****************************************************************************/

private void transcribe_cmd(rec_file_name)
  char *rec_file_name;
{
    boolean do_pitch_bend = false;
    
    rec_mode_flag = false;
    if (the_seq) seq_free(the_seq);
    the_seq = NULL;

    while (!abort_flag) {
	char response;
	gprintf(TRANS,
  "TRANSCRIBE MODE: type <return> to transcribe, ? for help, q to quit : ");
	switch (response = get_user_input()) {
	  case 0:
	    do_pitch_bend =
		askbool("Enable recording of control change information", false);
	    if (!rec_init(do_pitch_bend)) {
		gprintf(TRANS,"Unable to continue. Type anything to exit.\n");
		ggetchar();
		/* it would be better perhaps to clean up and exit(): */
		abort_flag = ABORT_LEVEL;
		return;
	    }
	    pp_outfile = fileopen(rec_file_name, "gio", "w", "output file");
	    if (!pp_outfile) return;
	    gprintf(TRANS,"Type <space> to stop ...\n");
	    a_recording = true;
	    a_playing = A_RESET;
	    /* If this code is omitted, eventtime picks up some rather peculiar
		values somewhere the second time a transcribe is run.  I do not
		think that leads to a bug, but it made me nervous.  This code
		to call set_rate is also used in play_rec() above */
	    eventtime = gettime();    /* capture when user really said "go" */
	    set_rate(timebase, 2560L);  /* recompute real_base */
	    stayalive(NULL);
	    moxcrun();
	    break;
	  case BREAK_LEVEL:
	    abort_flag = 0;
	  case ABORT_LEVEL:
	  case 'q':
	    return;
	  default:
	    default_response(response);
	    break;
	}
    }
}

/****************************************************************************
*                               record_cmd
****************************************************************************/

private void record_cmd(rec_file_name, play_file_name)
  char *rec_file_name;
  char *play_file_name;
{
    boolean do_pitch_bend = false;

    rec_mode_flag = true;
    while (!abort_flag) {
	char response;
	
	if (!the_seq) the_seq = new_score(play_file_name);
	if (!the_seq) {
	    gprintf(TRANS,
		"RECORD MODE: error encountered, no score to play!\n");
	    return;
	}
	the_seq->transpose = 0;
	the_seq->loudness = 0;

	gprintf(TRANS,
	    "RECORD MODE: type <return> to record, ? for help, q to quit : ");
	switch (response = get_user_input())  {
	  case 0:
	    do_pitch_bend = 
		askbool("Enable recording of pitch bend information", false);

	    if (!rec_init(do_pitch_bend)) {
		gprintf(TRANS,"Unable to contine. Restart Adagio.\n");
		EXIT(1);
	    }
	    pp_outfile = fileopen(rec_file_name, "gio", "w", "output file");
	    if (!pp_outfile) return;
	    seq_at_end(the_seq, stayalive);
	    play_rec(pp_outfile);
	    break;
	  case BREAK_LEVEL:
	    abort_flag = 0;
	  case ABORT_LEVEL:
	  case 'q':
	    return;
	  case 'n': 
	    if (the_seq) seq_free(the_seq);
	    the_seq = NULL;
	    break;
	  default:
	    default_response(response);
	    break;
	}
    }
}


private void stayalive(seq_type seq)
{
    timebase_type prevbase = timebase;  /* save old one */
    timebase_use(default_base);
    if (stayalive_flag) {
	cause ((delay_type) 1000, stayalive);     
    }
    timebase_use(prevbase); /* restore old one */
}


/****************************************************************************
*                               top_cmd
* Effect: select a mode
****************************************************************************/

private void top_cmd()
{
    char *play_file_name = cl_arg(1);
    char *rec_file_name = cl_option("record");
    char *extras = cl_arg(2);

    if (extras) gprintf(ERROR, "Extra (filename?) argument ignored: %s\n", extras);

    /* debugging flag setup */
    seq_print = cl_switch("print");

    /* do we run interactive (!script) or interactive (script) */
    script = cl_switch("script");

    while (abort_flag != ABORT_LEVEL) {
	a_start_time = 0L; /* reset at the top to avoid confusion */
	if (rec_file_name && play_file_name) {
	    record_cmd(rec_file_name, play_file_name);
	} else if (rec_file_name) {
	    transcribe_cmd(rec_file_name);
	} else if (play_file_name) {
	    play_cmd(play_file_name);
	} else if (script) {
	    gprintf(ERROR,
		    "-script mode requires play file.");
#ifdef AMIGA
	    ggetchar(); /* hold up for acknowledgement before closing window */
#endif
#ifdef MAC
	    ggetchar(); /* hold up for acknowledgement before closing window */
#endif
	}
	play_file_name = NULL;
	rec_file_name = NULL;

	if (script) goto exit_top_cmd;

	gprintf(TRANS,"\n\t\t*** ADAGIO ***\n");
	gprintf(TRANS,"Type p=play, t=transcribe, r=record , q=quit : ");
	switch (get_user_input()) {
	  case 'p':
	    play_cmd(NULL);
	    break;
	  case 'r':
	    record_cmd(NULL, NULL);
	    break;
	  case 't':
	    transcribe_cmd(NULL);
	    break;
	  case BREAK_LEVEL:
	    abort_flag = 0;
	    break;
	  case ABORT_LEVEL:
	  case 'q':
	    goto exit_top_cmd;
	  default:
	    break;
	}
    }
    /* note: gotos are frowned upon, but this saves haveing two return
     * statements, each performing the same cleanup action
     */
exit_top_cmd:
    if (the_seq) seq_free(the_seq);
    the_seq = NULL;
}

/****************************************************************************
*                    main
* Effect: 
*    Gets adagio started.
****************************************************************************/

void main(argc, argv)
  int argc;
  char *argv[];
{
#ifdef THINK_C
    /* Macintosh needs help to provide Unix-style command line */
    argc = ccommand(&argv);
#endif
    if (moxcinit(argc, argv)) {
	if (cl_switch("help")) showhelp();
	mididecode = false;
	seq_extensions();
	top_cmd();
    }
    EXIT(0);
}

/* midi_stopper - called via cause from midievent to stop sequence */
/**/
void midi_stopper(seq_type a_seq, time_type vt)
{
    if (a_playing != A_RESET) {
	stop_cmd(STOP_MIDI, vt);
    }
}


void midievent(data)
  unsigned char data[4];
{
    register unsigned char status;
    /* first see if it is a time code: */
    if (((status = data[0]) >= MIDI_REALTIME)) {
	if (status == MIDI_TIME_CLOCK && midi_started) {
	    midi_fraction += clock_ticksize;
	    midi_vtime += (midi_fraction >> 16);
	    midi_fraction &= 0xFFFFL;
	    /* set time 5ms ahead for the following reasons:
		1. when we get a clock, we're 1 clock behind
		2. if the sender sends clock before a note, the
		   note will not get played until the next clock,
		   but with a slight advance, the note WILL get
		   played when the clock is received.
		3. Delays will account for a couple of ms anyway.
		4. 5ms is unlikely to be perceived as early
	     */
	    set_virttime(the_seq->timebase, midi_vtime + 5);
	    if (midi_stop_req) {
		/* seq may stop on its own, otherwise call midi_stopper
		   (at low priority) to make the stop.  We can't call
		   stop_cmd here, because we still have one clock to
		   go.  We can't call catchup, because if we stop during
		   catchup, the timebase will be reset - this is not
		   allowed during catchup().  Note that we must use
		   the default_base because the_seq's timebase only
		   advances when clocks arrive. The delay allows the_seq
		   to run first and catchup to midi_vtime+5.  It would
		   be best to use a delay of zero and low priority, but
		   priorities are not enforced across different timebases.
		 */
		cause(1L, midi_stopper, the_seq, midi_vtime + 5);
	    }
	} else if (status == MIDI_STOP) {
	    midi_stop_req = true;  /* request stop on next MIDI_CLOCK */
	} else if (status == MIDI_START) {
	    midi_fraction = 0L;
	    midi_started = true;
	    prevtime = 0L;
	    /* yes, we're assigning a negative number to an unsigned
	       long!  This will wrap around, but the same value gets
	       added back when the first MIDI_TIME_CLOCK arrives. */
	    midi_vtime = (unsigned long) -((long) (clock_ticksize >> 16));
	}
    } else if (a_recording) {
	time_type the_time;
	if (a_playing) {
	    timebase_use(seq_timebase(the_seq));
	    the_time = virttime;
	} else the_time = eventtime;
	if (!rec_event((long *) data, the_time)) {
	    stop_cmd(OUT_OF_MEMORY, the_time);
	}
    }
}

/****************************************************************************
*               showhelp
* Effect: print help text
****************************************************************************/

private void showhelp()
{
    gprintf(TRANS,"\n");
    gprintf(TRANS,"Using:\n");
    gprintf(TRANS,"     adagio <Enter>\n");
    gprintf(TRANS,"asks for mode:\n");
    gprintf(TRANS,"     p to play\n");
    gprintf(TRANS,"     t to transcribe\n");
    gprintf(TRANS,"     r to record (play and transcribe)\n");
#ifdef  DOS
    gprintf(TRANS,"Using:\n");
    gprintf(TRANS,"     adagio xxxxx <Enter>\n");
    gprintf(TRANS,"immediately plays file xxxxx [gio]\n");
#endif
    gprintf(TRANS,"\n");
}
