#!/usr/local/bin/perl5 -w
#
# Purpose:
#   To serve as a simple min-heap for objects -- $pri, $obj pairs
#
##########################################################################
#
# Package Header
#
##########################################################################

package Heap;
require Exporter;

use strict;

##########################################################################

BEGIN {
  use Exporter ();
  use Symbol;
  use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

  $VERSION     = 1.0;
  @ISA         = qw(Exporter);
  @EXPORT =  qw(
    new
    empty
    insert
    remove
    peek
    count
    reveal
    purge
    _upheap
    _downheap
    _swap
  );
  %EXPORT_TAGS = ();
  @EXPORT_OK   = ();

  return 1;
}


sub new($@) {
  my $class = shift;
  my $self  = +{};
  my @rest  = @_;

  $self->{"data"}  = +[];
  $self->{"ct"}    = 0;
  bless $self, $class;

  if (scalar(@rest) > 0) {
    $self->insert(@rest);
  }

  return $self;
}



sub empty($) {
  my $self = shift;

  $self->{"ct"}   = 0;
  $self->{"data"} = +[];

  return $self;
}


sub count($) {
  my $self = shift;

  return $self->{"ct"};
}


# $heap_obj->insert($pri0, $obj0, $pri1, $obj1, ...);
#
sub insert($@) {
  my $self = shift;
  my $ict  = scalar @_;

  (!($ict % 2)) || die "Odd no. of arguments passed to insert().";
  ($ict > 0) || return $self;

  if ($ict == 2) {
    my $pri = shift;
    my $obj = shift;

    push @{$self->{"data"}}, +[$pri, $obj];
    $self->{"ct"}++;

    $self->_upheap($self->{"ct"}-1);
  } else {
    push_loop: {
      my $pri = shift;
      my $obj = shift;

      defined($pri) || last push_loop;

      push @{$self->{"data"}}, +[$pri, $obj];
      redo push_loop;
    };
    my $idx = 0;
    my $max = 0;

    $self->{"ct"} += $ict/2;

    $max = $self->{"ct"} - 1;

    for ($idx=int(($max+1)/2); $idx >= 0; $idx--) {
      $self->_downheap($idx);
    }
  }

  return $self;
}



# Returns either undef or a ($pri, $obj) pair.
sub remove($) {
  my $self = shift;
  my $ct   = $self->{"ct"};

  if ($ct == 0) {
    return undef;
  }

  if ($ct == 1) {
    my $pair = shift @{$self->{"data"}};

    $self->{"ct"} = 0;
    return @$pair;
  }

  $ct   = $self->{"ct"};
  $self->_swap(0, $ct-1);

  $self->{"ct"} = $ct-1;

  my $pair = pop @{$self->{"data"}};
  $self->_downheap(0);

  return @{$pair};
}



# Same, but non-destructive.
sub peek($) {
  my $self = shift;

  return ($self->{"data"}->[0]);
}



sub _upheap($$) {
  my $self = shift;
  my $idx  = shift;
  my $ct   = $self->{"ct"};

  ($idx < $ct) || last;

  _upheap_loop:  {
    ($idx > 0) || last;

    my $par = int(($idx-1)/2);

    if ($self->{"data"}->[$par]->[0] >
    $self->{"data"}->[$idx]->[0]) {
      $self->_swap($idx, $par);
      $idx = $par;
      redo _upheap_loop;
    }
  }

  return $self;
}



# Return the contents of the array.
sub reveal($) {
  my $self = shift;

  return @{$self->{"data"}};
}



# Remove all data for which a provided function returns a non-zero
# value.  The function must take two values:  a priority, and the
# data item.
sub purge($$) {
  my $self  = shift;
  my $judge = shift;

  my $ct  = $self->{"ct"};
  my $idx = 0;

  purge_loop:
  while ($idx < $ct) {
    my ($pri, $obj) = @{$self->{"data"}->[$idx]};

    if (&$judge($pri, $obj)) {
      # Remove it.
      if ($idx == ($ct-1)) {
        # Special case:  last one.
        # Includes case of 'only'.
        $ct--;
        $self->{"ct"} = $ct;
        pop @{$self->{"data"}};
        last purge_loop;
      } else {
        $self->_swap($idx, $ct-1);
        pop @{$self->{"data"}};
        $ct--;
        $self->{"ct"} = $ct;
        $self->_upheap($idx);
        $self->_downheap($idx);
        $idx=0;
        redo purge_loop;
      }
    }
    $idx++;
  }
}


sub _downheap($$) {
  my $self = shift;
  my $idx  = shift;
  my $ct   = $self->{"ct"};

  _downheap_loop: {
    ($idx < $ct) || return;

    my $child_idx = 2*$idx + 1;  # left child_idx
    defined($child_idx) || die;
    defined($ct) || die;

    if ($child_idx >= $ct) {
      return;
    }

    if (($child_idx+1) < $ct) {
      if ($self->{"data"}->[$child_idx]->[0] >
      $self->{"data"}->[$child_idx+1]->[0]) {
        $child_idx++;
      }
    }

    if ($self->{"data"}->[$child_idx]->[0] <
    $self->{"data"}->[$idx]->[0]) {
      $self->_swap($child_idx, $idx);
      $idx = $child_idx;
      redo _downheap_loop;
    }
  }
}

sub _swap($$$) {
  my $self = shift;
  my $idx0 = shift;
  my $idx1 = shift;

  if ($idx0 == $idx1) {
    return $self;
  }

  my $obj0 = $self->{"data"}->[$idx0];
  $self->{"data"}->[$idx0] = $self->{"data"}->[$idx1];
  $self->{"data"}->[$idx1] = $obj0;

  return $self;
}

