/*
* Author:     Leejay Wu
* Started:    2001-Feb-01
* File:       wrapper.h
* Purpose:    To provide a very basic wrapper for handling
*             simple flat-files.
*
* Notes on file format:
*
*    Whitespace and commas are treated as separators.  Do NOT
*    use commas (or periods, for that matter) as group
*    separators -- 1,000,000 gets turned into three numbers!
*
*    Semicolons are treated as comment indicators (to the end
*       of the line).
*
*    Lines without any vectors are ignored.
*
*    Currently, all data MUST be numeric.
*
*    We currently stipulate that no line can exceed 1 MB in
*    size.  This limit is rather arbitrary, but is used so
*    that we don't have to read byte-by-byte but can instead
*    (safely) use fgets.  If you really, really want larger
*    (or smaller) buffers, change LINEBUFSIZE (in bytes) in
*    wrapper.cc and recompile.
*
***************************************************************
* Id:  $Id: wrapper.h,v 1.2 2003/09/01 15:21:28 lw2j Exp $
*
* $Log:	wrapper.h,v $
 * Revision 1.2  2003/09/01  15:21:28  lw2j
 * Indentified.
 * 
* Revision 1.1  2003/09/01  14:57:15  lw2j
* Initial revision
*
*
*/


#ifndef WRAPPER_H
#define WRAPPER_H

#include <stdio.h>
#include <stdlib.h>

#include "array.h"


class Wrapper {
  public:
  /* We neither close nor open the filehandle; we just use it.
  * However, we expect that nobody else is using the file handle
  * at the same time, or else that they call reset() after doing
  * so.
  *
  * We rewind the file upon receiving it.  Be warned.
  */

  Wrapper(FILE *inf = NULL);
  ~Wrapper(void);

  void          set_file(FILE *inf) {
    fh  = inf;
    reset();
  }

  FILE* get_file(void) { return fh; };

  /* err is set to 1 if the object requested is out of range,
  * 0 otherwise.
  */

  Array<double> get_object(long index, int &err);    /* 0-based */

  /* Same, but fills in the index for you, and stops at EOF */
  Array<double> get_next_object(long &index, int &err);

  /* Both these two return -1 if no file.  get_count() is SLOW
  * if the file hasn't been entirely read through yet.
  *
  * Dimensionality ignores masking.
  */
  int           get_dimensionality(void);
  long          get_count(void);


  /* See semantics below.  Masking is used to pretend that not all
  * columns exist exactly once.
  */
  Array<unsigned int> get_mask(void) { return mask; };
  void          set_mask(Array<unsigned int> new_mask) { mask=new_mask; };
  void          clear_mask(void) {
    mask.clear();
  }

  void          reset() {
    idx = 0;
    if (fh) {
      rewind(fh);
    }
  }

  protected:
  FILE *fh;
  int dimensionality;          /* should be negative if not yet
  * determined
  */
  long count;                  /* likewise */
  long idx;                    /* how many lines have we read into the
  * file? */
  Array<unsigned int> mask;    /* If empty, means use all columns;
  * otherwise, use only these columns,
  * in that order.  We allow
  * duplication.
  */
};


#endif
