/*
* Author:     Leejay Wu
* Started:    2001-Feb-01
* File:       tug.h
* Purpose:    A Tug-of-War-based correlation fractal
*             dimension estimation class...
*
*             Setting the parameters and then calling
*             FracDim is basically what's needed; see
*             the sample main.cc file for an example.
*
***************************************************************
* Id:  $Id: tug.h,v 1.2 2003/09/01 15:21:28 lw2j Exp $
*
* $Log:	tug.h,v $
 * Revision 1.2  2003/09/01  15:21:28  lw2j
 * Indentified.
 * 
* Revision 1.1  2003/09/01  14:57:15  lw2j
* Initial revision
*
*
*/

#ifndef TUG_H
#define TUG_H

#include <math.h>
#include <assert.h>

#include "array.h"
#include "wrapper.h"


class TugApprox {

  public:

  /* These defaults are pretty wide-ranging.  If you have
  * prior information about your ranges, narrow them.
  */

  TugApprox();
  ~TugApprox() {};


  /* Return the estimated fractal dimension, the points
  * found, and the parameters of the line.
  *
  * The return value is duplicated in a (the slope of
  * y=ax+b).
  *
  * All this function does is call compute_logs, then
  * trim_pseudo_flat, then robust_fit.  So if you want
  * to stop at any stage, you can simply call those
  * in order yourself.
  */
  double frac_dim(Wrapper &data,
  Array<double>& log_radii,
  Array<double>& log_counts,
  double &a,
  double &b,
  double &corr);


  /* The Wrapper object is the input; it must be a wrapper
  * with an already-open file handle of which we should be
  * the exclusive user.
  *
  * We fill in the log_radii and log_count vectors.  The
  * former has, in increasing order, the log_2 radii;
  * each count is a log_2 estimate that corresponds to
  * the matching radius.
  *
  * This does not yet fit the lines or estimate the slope.
  */
  int compute_logs(Wrapper &data,
  Array<double> &log_radii,
  Array<double> &log_counts);



  /* Trim pseudo-flat regions.  They're not quite flat
  * because of the randomness, of course.  We do this by
  * checking both extremes, looking for regions that are
  *
  * a) anchored at one extreme,
  * b) at least three points long,
  * c) score highly in terms of (SSQ error / length^2)
  *    with a horizontal line
  *
  * We insist on having at least *10* points beforehand.
  *
  * The only object parameter it uses is 'silent', so it
  * could be turned into a static method if desired.
  *
  * The xs are generally the log_radii; the ys,
  * the log_counts.  Likewise for the robust fitter.
  */

  void trim_pseudo_flat(Array<double> &xs,
  Array<double> &ys);



  /* Attempt ~robust line-fitting on the (hopefully
  * trimmed) log values.
  *
  * y=ax+b
  *
  * corr = correlation between x,y
  *
  * It doesn't produce any output, nor use any object
  * members, so it can be a static method.
  */
  static void robust_fit(Array<double> &xs,
  Array<double> &ys,
  double &a,
  double &b,
  double &corr);

  /* Beware overflow/underflow when using extreme radii
  * limits.
  *
  * Perhaps we should also parameterize the hash function
  * generation?
  *
  * Yes, these are meant to be public.  After all, they're
  * all parameters that can and should often be tweaked by
  * a user, and there's little sense in obscuring them
  * since they're all integral to the algorithm and not
  * going anywhere soon.
  *
  * The random seed should be changed to a known value
  * before every FracDim (or compute_logs, called by FD)
  * invocation if you care about reproducibility.
  */
  bool           silent;
  double         radius_min;
  double         radius_max;
  unsigned int   random_seed;
  unsigned int   radius_count;
  unsigned int   s1;   /* determines accuracy   */
  unsigned int   s2;   /* determines confidence */



  private:
  /* for quickly counting bits */
  int            parity_table[256];


};

#endif
