/* File:     main.cc
* Author:   Leejay Wu
* Purpose:  To serve as a sample program using the TugApprox
*           class.  All parameters are directly accessible.
*
***************************************************************
* $Id: main.cc,v 1.2 2003/09/01 15:21:28 lw2j Exp $
* $Log*
*/

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <array.h>


#ifdef __LINUX__
/* This should actually be checking for the existence of the
* GNU C Library, which supplies getopt_long.  Linux platforms
* tend to have this; others, no.  So we'll take that route,
* and if needed, define our own minimalist getopt_long (which
* will ONLY do what we need it to do, not implement
* everything specc'ed by the man page).  It won't even handle
* SHORT arguments, because this program doesn't use them.
*
* OTOH, if we're on a Linux box, we can safely use the GNU
* version; frankly, it's likely to be better.
*/

#include <getopt.h>

static struct option long_options[] = {
  {"verbose",     0, 0, 0},
  {"silent",      0, 0, 0},
  {"min",         1, 0, 0},
  {"max",         1, 0, 0},
  {"num",         1, 0, 0},
  {"s1",          1, 0, 0},
  {"s2",          1, 0, 0},
  {"rnd",         1, 0, 0},
  {"col",         0, 0, 0},    /* horrible, horrible hacks here */
  {0,0,0,0}
};

#else

static int     optind = 1;
static char   *optarg;

struct option {
  const char *name;
  int         has_arg;
  int        *flag;
  int         val;
};

static struct option long_options[] = {
  {"verbose",     0, 0, 0},
  {"silent",      0, 0, 0},
  {"min",         1, 0, 0},
  {"max",         1, 0, 0},
  {"num",         1, 0, 0},
  {"s1",          1, 0, 0},
  {"s2",          1, 0, 0},
  {"rnd",         1, 0, 0},
  {"col",         0, 0, 0},    /* horrible, horrible hacks here */
  {0,0,0,0}
};

/* It's a hacked-up version that doesn't even handle short args,
* because we don't use them.
*/
static int getopt_long(int argc,
char *const *argv,
const char *shortopts,
const struct option *longopts,
int *longind) {
  if (optind == argc) {
    /* no more args */
    return -1;
  }

  if ((argv[optind][0] != '-') || (argv[optind][1] != '-')) {
    /* it's not a long argument... are there any left? */
    int i=0;

    for (i=optind+1; i < argc; i++) {
      if ((argv[i][0] == '-') && (argv[i][1] == '-')) {
        /* Yes, there are.  So permute and re-run. */
        int permute_size    = argc-optind;
        int permute_amount  = i-optind;
        int j=0;

        char **argv_copy = (char**) malloc(sizeof(char*) * (permute_size));

        assert(argv_copy);

        for (j=0; j < permute_size; j++) {
          argv_copy[j] = argv[optind + ((j + permute_amount) % permute_size)];
        }

        memcpy((((char**)argv)+optind), argv_copy, sizeof(char*) * (permute_size));
        free(argv_copy);

        return(getopt_long(argc, argv, shortopts, longopts, longind));
      }
    }

    /* No, there aren't. */
    return -1;
  }

  /* So it is one... unless it is "--", which basically means that
  * nothing afterwards is an argument.
  */

  if (!strcmp(argv[optind], "--")) {
    optind++;  /* skip this one, it doesn't really exist */
    return -1;
  }

  /* So it is a REAL long argument.  Well, which one? */
  {
    int i=0;

    while (long_options[i].name) {
      if (!strcmp(long_options[i].name, argv[optind]+2)) {
        /* match! */

        if (long_options[i].has_arg) {
          /* ...and it requires an argument... */
          if (optind == (argc-1)) {
            /* ...and there isn't one! */
            return -1;
          } else {
            ++optind;
            optarg = argv[optind];
            ++optind;
            *longind = i;
            return 0;
          }
        } else {
          /* No argument required. */
          ++optind;
          *longind = i;
          return 0;
        }
      }
      i++;
    }

    /* No match... ?! */
    return -1;
  }
}

#endif


#include "wrapper.h"
#include "tug.h"


static void set_defaults(TugApprox &tug);
static void parse_args(int argc, char **argv, TugApprox &tug,
Wrapper &obj);
static void print_usage_and_exit(char *progname);
static void sanity_check(char* progname, TugApprox &tug, Wrapper &obj);


static bool verbose = false;


enum option_t {
  OPT_VERBOSE = 0,
  OPT_SILENT,
  OPT_MIN,
  OPT_MAX,
  OPT_NUM,
  OPT_S1,
  OPT_S2,
  OPT_RND,
  OPT_COL
};



/* These values may work for you -- but YMMV.
*
* The maximum distance could be estimated with a preprocessing
* pass.
*/
static void set_defaults(TugApprox& tug) {
  tug.radius_min = pow(2, -15);
  tug.radius_max = pow(2, 15);
  tug.radius_count = 31;
  tug.s1 = 16;
  tug.s2 = 16;
  tug.random_seed = 0;
}




/* changed to two hyphens before long options, to be closer with standards */
static void print_usage_and_exit(char *progname) {
  fprintf(stderr, "[%s] Usage:\n", progname);
  fprintf(stderr, "%s \n", progname);
  fprintf(stderr, "   [--verbose]    # show ALL log/log points\n");
  fprintf(stderr, "   [--silent]     # no 'progress' output from library\n");
  fprintf(stderr, "   [--min n]      # minimum radius;  pos. double [2^-15]\n");
  fprintf(stderr, "   [--max n]      # maximum radius;  pos. double [2^15]\n");
  fprintf(stderr, "   [--num n]      # radii;           pos. int [31]\n");
  fprintf(stderr, "   [--s1  n]      # accuracy;        pos. int [16]\n");
  fprintf(stderr, "   [--s2  n]      # confidence;      pos. int [16]\n");
  fprintf(stderr, "   [--rnd n]      # random seed;     0+   int [0]\n");
  fprintf(stderr, "   [--col <list>\n");
  fprintf(stderr, "                 # columns to use; list must be in\n");
  fprintf(stderr, "                 # square brackets.  Ranges may be\n");
  fprintf(stderr, "                 # specified via a..b.\n");
  fprintf(stderr, "   <filename>  # input filename\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "Only one filename can be specified.\n");
  exit(-1);
}



static void parse_args(int argc, char **argv, TugApprox &tug,
Wrapper &obj) {
  Array<unsigned int> mask;
  int   c            = '\0';
  unsigned int idx=0;


  while (1) {
    int option_index       = 0;

    c = getopt_long(argc, argv, "", long_options, &option_index);

    if (c==-1) {
      break;
    } else {
      if (c) { /* no short options */
        print_usage_and_exit(argv[0]);
      }

      switch(option_index) {
        case OPT_VERBOSE:
        verbose = true;
        break;
        case OPT_SILENT:
        tug.silent = true;
        break;
        case OPT_MIN:
        if (sscanf(optarg, "%lf", &(tug.radius_min)) != 1) {
          print_usage_and_exit(argv[0]);
        }
        break;
        case OPT_MAX:
        if (sscanf(optarg, "%lf", &(tug.radius_max)) != 1) {
          print_usage_and_exit(argv[0]);
        }
        break;
        case OPT_NUM:
        if (sscanf(optarg, "%u", &(tug.radius_count)) != 1) {
          print_usage_and_exit(argv[0]);
        }
        break;
        case OPT_S1:
        if (sscanf(optarg, "%u", &(tug.s1)) != 1) {
          print_usage_and_exit(argv[0]);
        }
        break;
        case OPT_S2:
        if (sscanf(optarg, "%u", &(tug.s2)) != 1) {
          print_usage_and_exit(argv[0]);
        }
        break;
        case OPT_RND:
        if (sscanf(optarg, "%d", &(tug.random_seed)) != 1) {
          print_usage_and_exit(argv[0]);
        }
        break;
        case OPT_COL:  /* uh-oh */
        idx=0;


        while (1) {
          unsigned int val;

          if (!(optind < argc)) {
            fprintf(stderr, "--col list not terminated with ].\n");
            print_usage_and_exit(argv[0]);
          }

          if (argv[optind][0] == '[') {
            argv[optind][0] = ' ';

            if (argv[optind][1] == '\0') {
              ++optind;
              continue;
            }
          }

          if (argv[optind][0] == ']') {
            break;
          }

          {
            /* Did the user specify a range, using a..b ? */
            unsigned int low;
            unsigned int high;

            if (sscanf(argv[optind], "%u..%u", &low, &high) == 2) {
              unsigned int j=low;

              for (j=low; j <= high; j++) {
                mask[idx++] = j;
              }

              if (strchr(argv[optind], ']')) {
                break;
              } else {
                ++optind;
                continue;
              }
            }
          }

          if ((sscanf(argv[optind], "%u", &val) != 1)) {
            fprintf(stderr, "Column indices must be positive integers.");
            print_usage_and_exit(argv[0]);
          }

          mask[idx++] = val;

          if (strchr(argv[optind], ']')) {
            break;
          }
          optind++;
        }

        optind++;
        obj.set_mask(mask);
        break;

        default: /* should be impossible */
        assert(0);
      }
    }
  }

  {
    /* One argument should be left -- a filename. */
    FILE *fh = NULL;

    if ((optind != (argc-1)) || (argc==1))  {
      /* nope, not exactly one */
      fprintf(stderr, "Exactly one filename must be specified.\n");
      print_usage_and_exit(argv[0]);
    }

    if (!(fh = fopen(argv[optind], "r"))) {
      fprintf(stderr, "File '%s' could not be opened.\n", argv[optind]);
      perror("Error message:  ");
      print_usage_and_exit(argv[0]);
    }

    obj.set_file(fh);
  }
}





/* Do some VERY basic constraint checking. */
static void sanity_check(char* progname, TugApprox &tug, Wrapper &obj) {
  bool flag = false;

  if (!(obj.get_file())) {
    flag = true;
    fprintf(stderr, "!!! Error:  no file was specified.\n");
  }

  if ((tug.radius_min <= 0) || (tug.radius_max <= 0)) {
    flag = true;
    fprintf(stderr, "!!! Error:  radii must be positive\n");
  }

  if (tug.radius_min >= tug.radius_max) {
    flag = true;
    fprintf(stderr, "!!! Error:  minimum radius must be smaller than maximum radius.\n");
  }

  if (tug.radius_count < 2) {
    flag = true;
    fprintf(stderr, "!!! Error:  at least two radii must be specified.\n");
  } else if (tug.radius_count < 10) {
    fprintf(stderr, "!   Warning:  at least ten radii are STRONGLY recommended.\n");
  }

  if ((tug.s1 <= 0) || (tug.s2 <= 0)) {
    flag = true;
    fprintf(stderr, "!!! Error:  both s1 and s2 must be positive.\n");
  }

  if (flag) {
    print_usage_and_exit(progname);
  }
}


int main(int argc, char **argv) {
  TugApprox tugger;
  Wrapper   obj;
  Array<double> log_radii;
  Array<double> log_counts;
  double        a,b,corr;

  set_defaults(tugger);
  parse_args(argc, argv, tugger, obj);
  sanity_check(argv[0], tugger, obj);

  if (!verbose) {
    unsigned int i=0;
    unsigned int count=0;

    tugger.frac_dim(obj, log_radii, log_counts, a, b, corr);
    count = log_radii.getCount();


    if (obj.get_count() < 1000) {
      /* Warning... few points.
      * This is an arbitrary threshold, but we might as well
      * provide a reminder that this is an approximate scheme.
      */
      printf("* Warning:  accuracy may be worse on smaller sets.\n");
      printf("* This method is, after all, a randomized algorithm\n");
      printf("* trying to estimate second-frequency moment with\n");
      printf("* small space and time constraints.  You may be\n");
      printf("* better off using a deterministic box-counting\n");
      printf("* algorithm if your set is so small the space and\n");
      printf("* time constraints would be minor anyways.\n\n");
      printf("\n");
    }

    if (!tugger.silent) {
      for (i=0; i < count; i++) {
        printf(";;; %f \t %f\n", log_radii[i], log_counts[i]);
      }
    }
  } else {
    unsigned int i=0;
    unsigned int count=0;

    tugger.compute_logs(obj, log_radii, log_counts);
    count = log_radii.getCount();

    if (obj.get_count() < 1000) {
      printf("* Warning:  accuracy may be worse on smaller sets.\n");
      printf("* This method is, after all, a randomized algorithm\n");
      printf("* trying to estimate second-frequency moment with\n");
      printf("* small space and time constraints.  You may be\n");
      printf("* better off using a deterministic box-counting\n");
      printf("* algorithm if your set is so small the space and\n");
      printf("* time constraints would be minor anyways.\n\n");
      printf("\n");
    }

    for (i=0; i < count; i++) {
      printf(";;; %f \t %f\n", log_radii[i], log_counts[i]);
    }

    tugger.trim_pseudo_flat(log_radii, log_counts);
    tugger.robust_fit(log_radii, log_counts, a, b, corr);
  }

  printf("Estimated fractal dimension:  %f\n", a);
  printf("y-intercept:                  %f\n", b);
  printf("Correlation:                  %f\n", corr);

  fclose(obj.get_file());

  return 0;
}
