#ifndef TLIST_H
#define TLIST_H

#ifndef NULL
#define NULL 0
#endif

#include <stdlib.h>                    /* for malloc() and free()*/
#include "adt.h"           /* common types and constants       */

#ifdef __cplusplus
extern "C" {
  #endif

  /* Type Declarations:
  */


  /* tListEntry ---------------------------------------------------------------
  The list is populated with objects of this type.  They contain pointers
  to the previous and next entries in the list, and a void pointer to the
  actual element of the entry.
  ---------------------------------------------------------------------------*/

  typedef struct tagListEntry {

    struct tagListEntry
    *prev,                             /* points to previous entry         */
    *next;                             /* points to next entry             */
    void* element;                       /* points to user-defined element   */

  } tListEntry;


  /* tList --------------------------------------------------------------------
  An object of this type is an instance of a doubly linked list.  It
  contains pointers to the head and tail entries of the list as well as
  the current pointer which points to the entry at which the next operation
  will begin.  It also maintains a count of the number of elements currently
  in the list (length).
  ---------------------------------------------------------------------------*/

  typedef struct tagList {

    tListEntry
    *head,                             /* points to first entry in list    */
    *tail,                             /* points to last entry in list     */
    *current;                          /* current traversal point          */
    unsigned length;                     /* number of elements in list       */

  } tList;


  /* Function Prototypes:
  */


  /* lstNew -------------------------------------------------------------------
  Allocate, initialize, and return a pointer to a new tList object.
  If allocation fails then return NULL.
  ---------------------------------------------------------------------------*/

  tList* lstNew(void);


  /* lstDelete ----------------------------------------------------------------
  Delete the entire list including the tList object.
  ---------------------------------------------------------------------------*/

  void lstDelete(tList* list, tDestructor destructor);


  /* lstFlush -----------------------------------------------------------------
  Delete all elements of the list and the corresponding tListEntry
  objects.  Do not delete the tList object.
  ---------------------------------------------------------------------------*/

  void lstFlush(tList* list, tDestructor destructor);


  /* lstRemove ----------------------------------------------------------------
  Delete the current list element and its corresponding tListEntry object.
  The element before the one deleted becomes current if current was not at
  the head of list, else the one after becomes current.
  ---------------------------------------------------------------------------*/

  void lstRemove(tList* list, tDestructor destructor);


  /* lstInsertBefore ----------------------------------------------------------
  Insert new_ele before the current element in list.  It becomes the
  new current element.  Return a pointer to the inserted element, or
  if allocation of the tListEntry object fails, return NULL and
  do not change current.
  ---------------------------------------------------------------------------*/

  void* lstInsertBefore(tList* list, void* new_ele);


  /* lstInsertAfter -----------------------------------------------------------
  Insert new_ele after the current element in list.  It becomes the
  new current element.  Return a pointer to the inserted element, or
  if allocation of the tListEntry object fails, return NULL and
  do not change current.
  ---------------------------------------------------------------------------*/

  void* lstInsertAfter(tList* list, void* new_ele);


  /* lstInsertInorder ---------------------------------------------------------
  Insert new_ele with respect to order by making comparisons through the
  function pointer compare.  Return a pointer to the inserted element and
  make it current.  If allocation of the tListEntry object fails,
  return NULL and do not change current.
  ---------------------------------------------------------------------------*/

  void* lstInsertInorder(tList* list, void* new_ele, tCmpFunc compare);


  /* lstAppend ----------------------------------------------------------------
  Append new_ele to the end of list.  It becomes the new current element.
  Return a pointer to the inserted element, or if allocation of the
  tListEntry object fails, return NULL and do not change current.
  ---------------------------------------------------------------------------*/

  void* lstAppend(tList* list, void* new_ele);


  /* lstPrepend (added by alwong@andrew.cmu.edu 3/7/2002) ---------------------
  Prepend new_ele to the beginning of list.
  It becomes the new current element.
  Return a pointer to the inserted element, or if allocation of the
  tListEntry object fails, return NULL.
  ---------------------------------------------------------------------------*/

  void* lstPrepend(tList* list, void* new_ele);


  /* lstContains --------------------------------------------------------------
  If list contains an element equivalent to ele, return a pointer to, and
  make current, the found element.  If no match is found return NULL and
  do not change current.
  ---------------------------------------------------------------------------*/

  void* lstContains(tList* list, void* ele, tCmpFunc compare);


  /* lstPrev ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the element before the current element.  If current points to the
  head of list, return NULL but do not change the current pointer of list.
  ---------------------------------------------------------------------------*/

  void* lstPrev(tList* list);


  /* lstNext ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the element after the current element.  If current points to the
  tail of list, return NULL but do not change the current pointer of list.
  ---------------------------------------------------------------------------*/

  void* lstNext(tList* list);


  /* lstHead ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the first element in list.
  ---------------------------------------------------------------------------*/

  void* lstHead(tList* list);


  /* lstTail ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the last element in list.
  ---------------------------------------------------------------------------*/

  void* lstTail(tList* list);


  /* lstCurrent ---------------------------------------------------------------
  Return the current element, or NULL if the list is empty.
  ---------------------------------------------------------------------------*/

  void* lstCurrent(tList* list);


  /* lstLength ----------------------------------------------------------------
  Return the number of elements in the list.
  ---------------------------------------------------------------------------*/

  unsigned lstLength(tList* list);


  /* lstMoveToTop -------------------------------------------------------------
  Make the current entry the first entry, without rotating.  Return the
  new head element.
  ---------------------------------------------------------------------------*/

  void* lstMoveToTop(tList* list);


  /* lstRotateToTop -------------------------------------------------------------
  Make the current entry the first entry by rotating.  Return the new
  head element.
  ---------------------------------------------------------------------------*/

  void* lstRotateToTop(tList* list);


  /* lstTraverse --------------------------------------------------------------
  Traverse the list forward, passing each element to the visit function.
  When visit is called, the entry whose element is passed is current.
  The visit function must return TR_HALT or TR_CONTINUE.
  ---------------------------------------------------------------------------*/

  void lstTraverse(tList* list, tVisitFunc visit);


  /* lstRevTraverse -----------------------------------------------------------
  Traverse the list backward, passing each element to the visit function.
  When visit is called, the entry whose element is passed is current.
  The visit function must return TR_HALT or TR_CONTINUE.
  ---------------------------------------------------------------------------*/

  void lstRevTraverse(tList* list, tVisitFunc visit);


  /* lstReplace ---------------------------------------------------------------
  Replace the current element with new_ele.
  ---------------------------------------------------------------------------*/

  void lstReplace(tList* list, void* new_ele, tDestructor destructor);


  /* lstIndex -----------------------------------------------------------------
  index must be in the range 1 to list->length inclusive, NULL is returned
  if it is not, else the index'th element is made current and returned.
  ---------------------------------------------------------------------------*/

  void* lstIndex(tList* list, unsigned index);


  #ifdef __cplusplus
}
#endif

#endif

/* end tList interface file */

