
#include "list.h"


/* Private Function Definition:
*/


/* new_entry ----------------------------------------------------------------
  Allocate, initialize, and return a pointer to a new tListEntry object.
  If allocation fails return NULL.  This function is local to this file.
  Allocation of the element that will go into the entry is strictly the
  responsibility of the user.  Deallocation of user-allocated elements is
  determined by the destructor member of the tList object.
---------------------------------------------------------------------------*/

tListEntry* new_entry(void* new_ele)
{
  tListEntry* ne;

  ne = (tListEntry*) malloc(sizeof(tListEntry));
  if(ne) {
    ne->element = new_ele;
    ne->prev = ne->next = NULL;
  }
  return ne;
}


/* Public Function Definitions:
*/


/* lstNew -------------------------------------------------------------------
  Allocate, initialize, and return a pointer to a new tList object.
  If allocation fails then return NULL.
---------------------------------------------------------------------------*/

tList* lstNew()
{
  tList* list;

  list = (tList*) malloc(sizeof(tList));
  if(list) {
    list->length = 0;
    list->head = list->tail = list->current = NULL;
  }
  return list;
}

/* lstDelete ----------------------------------------------------------------
  Delete the entire list including the tList object.
---------------------------------------------------------------------------*/

void lstDelete(tList* list, tDestructor destructor)
{
  if(list) {
    lstFlush(list, destructor);
    free(list);
  }
}

/* lstFlush -----------------------------------------------------------------
  Delete all elements of the list and the corresponding tListEntry
  objects.  Do not delete the tList object.
---------------------------------------------------------------------------*/

void lstFlush(tList* list, tDestructor destructor)
{
  tListEntry *next,
             *current;

  if(!list) return;
  next = current = list->head;
  while(next) {
    next = current->next;
    if(destructor != DESTRUCTOR_NONE) {
      if(destructor == DESTRUCTOR_FREE)
        free(current->element);
      else
        destructor(current->element);
    }
    free(current);
    current = next;
  }
  list->length = 0;
  list->head = list->tail = list->current = NULL;
}

/* lstRemove ----------------------------------------------------------------
  Delete the current list element and its corresponding tListEntry object.
  The element before the one deleted becomes current if current was not at
  the head of list, else the one after becomes current.
---------------------------------------------------------------------------*/

void lstRemove(tList* list, tDestructor destructor)
{
  tListEntry* old_current;

  if(!list || !list->length) return;
  old_current = list->current;
  /* disconnect current entry struct from list */
  if(old_current->prev)
    old_current->prev->next = old_current->next;
  if(old_current->next)
    old_current->next->prev = old_current->prev;
  /* update head, tail, and current pointers */
  if(list->length == 1)
    list->head = list->tail = list->current = NULL;
  else {
    if(old_current->prev) {
      list->current = old_current->prev;
      if(list->tail == old_current)
        list->tail = old_current->prev;
    }
    else list->head = list->current = old_current->next;
  }
  --(list->length);
  /* delete the disconnected entry */
  if(destructor != DESTRUCTOR_NONE) {
    if(destructor == DESTRUCTOR_FREE)
      free(old_current->element);
    else
      destructor(old_current->element);
  }
  free(old_current);
}

/* lstInsertBefore ----------------------------------------------------------
  Insert new_ele before the current element in list.  It becomes the
  new current element.  Return a pointer to the inserted element, or
  if allocation of the tListEntry object fails, return NULL.
---------------------------------------------------------------------------*/

void* lstInsertBefore(tList* list, void* new_ele)
{
  tListEntry* entry;

  if(!list) return NULL;
  entry = new_entry(new_ele);
  if(!entry) return NULL;

  if(list->length) {
    entry->prev = list->current->prev;
    entry->next = list->current;
    if(list->head == list->current)
      list->head = entry;
    list->current = entry;
    if(entry->prev)
      entry->prev->next = entry;
    entry->next->prev = entry;
  }
  else list->head = list->tail = list->current = entry;
  ++(list->length);
  return new_ele;
}

/* lstInsertAfter -----------------------------------------------------------
  Insert new_ele after the current element in list.  It becomes the
  new current element.  Return a pointer to the inserted element, or
  if allocation of the tListEntry object fails, return NULL and
  do not change current.
---------------------------------------------------------------------------*/

void* lstInsertAfter(tList* list, void* new_ele)
{
  tListEntry* entry;

  if(!list) return NULL;
  entry = new_entry(new_ele);
  if(!entry) return NULL;

  if(list->length) {
    entry->prev = list->current;
    entry->next = list->current->next;
    if(list->tail == list->current)
      list->tail = entry;
    list->current = entry;
    entry->prev->next = entry;
    if(entry->next)
      entry->next->prev = entry;
  }
  else list->head = list->tail = list->current = entry;
  ++(list->length);
  return new_ele;
}

/* lstInsertInorder ---------------------------------------------------------
  Insert new_ele with respect to order.  Return a pointer to the inserted
  element and make it current.  If allocation of the tListEntry object
  fails, return NULL and do not change current.
---------------------------------------------------------------------------*/

void* lstInsertInorder(tList* list, void* new_ele, tCmpFunc compare)
{
  if(!list || !compare) return NULL;
  list->current = list->head;
  if(list->length) {
    do {
      if(compare(new_ele, list->current->element) < 1)
        return lstInsertBefore(list, new_ele);
    } while(lstNext(list));
  }
  return lstAppend(list, new_ele);
}

/* lstAppend ----------------------------------------------------------------
  Append new_ele to the end of list.  It becomes the new current element.
  Return a pointer to the inserted element, or if allocation of the
  tListEntry object fails, return NULL.
---------------------------------------------------------------------------*/

void* lstAppend(tList* list, void* new_ele)
{
  tListEntry* entry;

  if(!list) return NULL;
  entry = new_entry(new_ele);
  if(!entry) return NULL;

  if(list->length) {
    list->tail->next = entry;
    entry->prev = list->tail;
    list->tail = list->current = entry;
  }
  else list->head = list->tail = list->current = entry;
  ++(list->length);
  return new_ele;
}

/* lstPrepend (added by alwong@andrew.cmu.edu 3/7/2002) ---------------------
  Prepend new_ele to the beginning of list.  
  It becomes the new current element.
  Return a pointer to the inserted element, or if allocation of the
  tListEntry object fails, return NULL.
---------------------------------------------------------------------------*/

void* lstPrepend(tList* list, void* new_ele)
{
  tListEntry* entry;

  if(!list) return NULL;
  entry = new_entry(new_ele);
  if(!entry) return NULL;

  if(list->length) {
    entry->next = list->head;
    list->head->prev = entry;
    list->head = list->current = entry;
  }
  else list->head = list->tail = list->current = entry;
  ++(list->length);
  return new_ele;
}

/* lstContains --------------------------------------------------------------
  If list contains an element equivalent to ele, return a pointer to, and
  make current, the found element.  If no match is found return NULL and
  do not change current.
---------------------------------------------------------------------------*/

void* lstContains(tList* list, void* ele, tCmpFunc compare)
{
  tListEntry* old_current;

  if(!list || !compare) return NULL;
  old_current = list->current;
  list->current = list->head;
  if(list->length) {
    do {
      if(compare(ele, list->current->element) == 0){
        return list->current->element;
      }
    } while(lstNext(list));
  }
  list->current = old_current;
  return NULL;
}

/* lstPrev ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the element before the current element.  If current points to the
  head of list, return NULL but do not change the current pointer of list.
---------------------------------------------------------------------------*/

void* lstPrev(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  if(list->current->prev == NULL) return NULL;
  list->current = list->current->prev;
  return list->current->element;
}

/* lstNext ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the element after the current element.  If current points to the
  tail of list, return NULL but do not change the current pointer of list.
---------------------------------------------------------------------------*/

void* lstNext(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  if(list->current->next == NULL) return NULL;
  list->current = list->current->next;
  return list->current->element;
}

/* lstHead ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the first element in list.
---------------------------------------------------------------------------*/

void* lstHead(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  list->current = list->head;
  return list->head->element;
}

/* lstTail ------------------------------------------------------------------
  Return a pointer to and update the current pointer of list with
  the last element in list.
---------------------------------------------------------------------------*/

void* lstTail(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  list->current = list->tail;
  return list->tail->element;
}

/* lstCurrent ---------------------------------------------------------------
  Return the current element, or NULL if the list is empty.
---------------------------------------------------------------------------*/

void* lstCurrent(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  return list->current->element;
}

/* lstLength ----------------------------------------------------------------
  Return the number of elements in the list.
---------------------------------------------------------------------------*/

unsigned lstLength(tList* list)
{
  if(!list) return 0;
  return list->length;
}

/* lstMoveToTop -------------------------------------------------------------
  Make the current entry the first entry, and return the new head element.
---------------------------------------------------------------------------*/

void* lstMoveToTop(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  if(list->current != list->head) {
    list->current->prev->next = list->current->next;
    if(list->current->next)
      list->current->next->prev = list->current->prev;
    else list->tail = list->tail->prev;
    list->current->prev = NULL;
    list->current->next = list->head;
    list->head->prev = list->current;
    list->head = list->current;
  }
  return list->head->element;
}

/* lstRotateToTop -------------------------------------------------------------
  Make the current entry the first entry, and return the new head element.
---------------------------------------------------------------------------*/

void* lstRotateToTop(tList* list)
{
  if(!list) return NULL;
  if(list->length == 0) return NULL;
  if(list->current != list->head) {
    /* connect old head to old tail */
    list->head->prev = list->tail;
    list->tail->next = list->head;
    /* make current into new head and its prev into new tail */
    list->head = list->current;
    list->tail = list->current->prev;
    list->head->prev = list->tail->next = NULL;
  }
  return list->head->element;
}

/* lstTraverse --------------------------------------------------------------
  Traverse the list forward, passing each element to the visit function.
  When visit is called, the entry whose element is passed is current.
  The visit function must return TR_HALT or TR_CONTINUE.
---------------------------------------------------------------------------*/

void lstTraverse(tList* list, tVisitFunc visit)
{
  if(!list || !visit) return;
  if(list->length == 0) return;
  list->current = list->head;
  do {
    if(visit(list->current->element) == TR_HALT)
      return;
  } while(lstNext(list));
}


/* lstRevTraverse -----------------------------------------------------------
  Traverse the list backward, passing each element to the visit function.
  When visit is called, the entry whose element is passed is current.
  The visit function must return TR_HALT or TR_CONTINUE.
---------------------------------------------------------------------------*/

void lstRevTraverse(tList* list, tVisitFunc visit)
{
  if(!list || !visit) return;
  if(list->length == 0) return;
  list->current = list->tail;
  do {
    if(visit(list->current->element) == TR_HALT)
      return;
  } while(lstPrev(list));
}


/* lstReplace ---------------------------------------------------------------
  Replace the current element with new_ele.
---------------------------------------------------------------------------*/

void lstReplace(tList* list, void* new_ele, tDestructor destructor)
{
  if(!list) return;
  if(list->length == 0) return;
  if(destructor != DESTRUCTOR_NONE) {
    if(destructor == DESTRUCTOR_FREE)
      free(list->current->element);
    else
      destructor(list->current->element);
  }
  list->current->element = new_ele;
}


/* cast i to unsigned for comparison, 10/30/2002 */
/* lstIndex -----------------------------------------------------------------
  index must be in the range 1 to list->length inclusive, NULL is returned
  if it is not, else the index'th element is made current and returned.
---------------------------------------------------------------------------*/

void* lstIndex(tList* list, unsigned index)
{
  int i;

  if(!list || !index) return NULL;
  if(index > list->length) return NULL;
  list->current = list->head;
  for(i = 1; (unsigned)i < index; ++i)
    list->current = list->current->next;
  return list->current->element;
}


/* end tList implementation file */
