// Leejay Wu


#include <iostream.h>
#include <assert.h>
#include <stdlib.h>

#include "array.h"

// simple constructor
template <class T>
Array<T>::Array(void):
fixedSize(false), highest(-1), capacity(0), arr(NULL)
{
  highest   = -1;
  capacity  = 0;
  fixedSize = false;
  arr       = NULL;
}


// simple constructor
template <class T>
Array<T>::Array(unsigned int size, bool fixed):
fixedSize(false), highest(-1), capacity(0), arr(NULL) {
  highest   = -1;
  capacity  = 0;
  arr       = NULL;

  assert(resize(size));

  fixedSize = fixed;
}


// Construction via duplication.
template <class T>
Array<T>::Array(const Array<T>& other):
fixedSize(false), highest(-1), capacity(0), arr(NULL) {
  if (!(resize(other.capacity))) {
    cerr << "Failed to resize for =: " << endl;
    cerr << (other.capacity) << endl;
    assert(0);
  }

  fixedSize = other.fixedSize;
  highest   = other.highest;

  for (int i=0; i <= highest; i++) {
    arr[i] = other.arr[i];
  }
}


// simple destructor
template <class T>
Array<T>::~Array(void) {
  if (arr) {
    delete [] arr;
  }
}

template <class T>
T Array<T>::ref(unsigned int idx) const {
  if (idx >= capacity) {
    assert(0);
  }
  return arr[idx];
}

// to allow as-natural use as possible
template <class T>
T& Array<T>::operator[](unsigned int idx) {
  if (idx >= capacity) {
    unsigned int newSize = (capacity ? (capacity * 2) : 8);
    while (newSize <= idx) newSize *= 2;

    if (!(resize(newSize))) {
      cerr << "Out-of-bounds [] usage on fixed-size array: " << endl;
      cerr << idx << " versus " << capacity << endl;
      assert(0);
    }
  }
  highest = (((int) idx) > highest) ? idx : ((int) highest);
  return arr[idx];
}


// ala linked list
template <class T>
void Array<T>::addItem(T& newItem) {
  if ((highest+1) >= ((int) capacity)) {
    unsigned int newSize = (capacity ? (2*capacity) : 8);
    if (!(resize(newSize))) {
      cerr << "Attempted to add item to filled fix-size array: " << endl;
      cerr << newSize << " versus " << capacity << endl;
      assert(0);
    }
  }

  assert(((int) capacity) > (highest+1));
  arr[++highest] = newItem;
}


// merging
template <class T>
void Array<T>::merge(const Array<T>& otherArray) {
  unsigned int addCount   = otherArray.getCount();
  unsigned int neededSize = addCount + (highest+1);
  unsigned int newSize    = capacity;
  unsigned int idx;

  while (neededSize >= newSize) {
    newSize *= 2;
  }

  if (!(resize(newSize))) {
    cerr << "Failed to resize for merge: " << endl;
    cerr << (newSize) << " versus " << capacity << endl;
    assert(0);
  }

  assert(capacity > (neededSize));

  for (idx=0; idx < addCount; idx++) {
    arr[++highest] = otherArray.ref(idx);
  }
}

// useful...
template <class T>
bool Array<T>::resize(unsigned int newSize) {
  if (newSize == capacity) {
    return true;
  }

  if (fixedSize) {
    return false;
  }

  highest = (highest <= ((int) (newSize-1))) ? highest : ((int) (newSize-1));
  unsigned int i;

  if (!newSize) {
    highest = -1;
  }
  T*  newArr  = newSize ? (new T[newSize]) : ((T*) NULL);

  if (capacity && newSize) {
    for (i=0; ((int) i) <= highest; i++) {
      newArr[i] = arr[i];
    }
  }

  capacity = newSize;
  delete [] arr;
  arr = newArr;

  return true;
}

// also useful
template <class T>
void Array<T>::swap(unsigned int idx1, unsigned int idx2) {
  if (idx1 == idx2) return;

  if (idx1 > idx2) {
    unsigned int idxSwap = idx1;
    idx1 = idx2;
    idx2 = idxSwap;
  }

  if (idx2 > capacity) {
    if (!(resize(idx2+1))) {
      cerr << "Failed to resize for swap [!]: " << endl;
      cerr << (idx2+1) << " versus " << capacity << endl;
      assert(0);
    }
  }

  T swapBox = arr[idx1];
  arr[idx1] = arr[idx2];
  arr[idx2] = swapBox;
}


// expensive!  preserves order
template <class T>
void Array<T>::remove(unsigned int idx) {
  unsigned int j;

  if (((int) idx) > highest) return;

  for (j=idx; ((int) j) < highest; j++) {
    swap(j, j+1);
  }

  // use the void constructor ?
  arr[highest] = T();

  highest--;
}

// not expensive. does NOT preserve order.
template <class T>
void Array<T>::remove2(unsigned int idx) {
  if (((int) idx) > highest) return;

  if (((int) idx) < highest) {
    swap(idx, (unsigned int) highest);
  }

  arr[highest] = T();

  highest--;
}


// Duplication via assignment.
template <class T>
void Array<T>::operator=(const Array<T>& other) {
  if (this == (&other)) {
    return;
  }

  if (arr != NULL) {
    delete [] arr;
  }
  arr      = NULL;
  capacity = 0;

  if (!(resize(other.capacity))) {
    cerr << "Failed to resize for =: " << endl;
    cerr << (other.capacity) << endl;
    assert(0);
  }

  fixedSize = other.fixedSize;
  highest   = other.highest;

  for (int i=0; i <= highest; i++) {
    arr[i] = other.arr[i];
  }
}



// Simple equality testing.
template <class T>
bool Array<T>::operator==(const Array<T>& other) const {
  if (highest != other.highest) {
    return false;
  }

  for (int i=0; i <= highest; i++) {
    if (!(arr[i] == other.arr[i])) {
      return false;
    }
  }

  return true;
}



// Simple inequality testing.
template <class T>
bool Array<T>::operator!=(const Array<T>& other) const {
  if (highest != other.highest) {
    return true;
  }

  for (int i=0; i <= highest; i++) {
    if (arr[i] != other.arr[i]) {
      return true;
    }
  }

  return false;
}


// Shifting.
template <class T>
Array<T> Array<T>::operator<<(unsigned int amount) {
  Array<T>      new_array;
  int  i=0;


  if (highest <= ((int) amount)) {
    return new_array;
  }

  new_array.resize(capacity - amount);

  /* don't do a straight memcpy, since constructors may
  * be needed
  */
  for (i=((int) amount); i <= highest; i++) {
    new_array[i - amount] = arr[i];
  }

  return new_array;
}



// Shifting.
template <class T>
Array<T> Array<T>::operator>>(unsigned int amount) {
  Array<T>      new_array;
  int  i=0;


  if ((new_array.highest + ((int) amount)) >= ((int) (new_array.capacity))) {
    assert(new_array.resize(capacity + amount));
  }

  /* don't do a straight memcpy, since constructors may
  * be needed
  */
  for (i=0; i <= highest; i++) {
    new_array[i + amount] = arr[i];
  }

  return new_array;
}


// Shifting.
template <class T>
Array<T>& Array<T>::operator<<=(unsigned int amount) {
  int  i=0;

  if (highest <= ((int) amount)) {
    highest = 0;
    return *this;
  }

  /* don't do a straight memcpy, since constructors may
  * be needed
  */
  for (i=((int) amount); i <= highest; i++) {
    arr[i - amount] = arr[i];
  }

  highest -= amount;
  /* we don't really care if this fails */
  resize(capacity - amount);

  return *this;
}



// Shifting.
template <class T>
Array<T>& Array<T>::operator>>=(unsigned int amount) {
  int  i=0;

  if ((highest + ((int) amount)) >= ((int) capacity)) {
    assert(resize(capacity + amount));
  }

  /* don't do a straight memcpy, since constructors may
  * be needed
  */
  for (i=highest; i >= 0; i--) {
    arr[i + amount] = arr[i];
  }

  return *this;
}



