: # Find Perl
eval 'exec perl5 -w -S $0 "$@"'
if 0;

# This is a simple demo script, and adds zero bits of
# information beyond the README and other files.  Since
# it serves no purpose but to give people a way to test
# in one command, even w/o reading anything else, this
# script has a few additional dependencies.
#
#
# - This file may be rife with Unixisms.
# - Gnuplot should be in the user's path.
# - Ghostview is required for display

# This is more of a generic post-install script than
# anything you'd want to take apart and include in your
# own code.


# Just making the include explicit.
use lib '.';

use strict;
use English;
use Symbol;
use FileHandle;
use Benchmark;

use mktemp;



# Certain phases (namely, the FracDim computation itself)
# may take a considerable amount of time.  Don't buffer
# STDOUT, because otherwise the delay may be considerable.
#
autoflush STDOUT 1;



# Phase 1 -- Permissions.
#
# This phase is a lot less complicated in the RUNME.pl for the
# Perl package, because we don't need to change the files -- just
# possibly set the execute bit on for TriGen.pl.
#
{
  # 0100 = user can execute
  # 0200 = user can write
  #
  # We fix ourselves in case we weren't shipped as writeable
  # but a user managed to invoke it via, say, ". RUNME.pl".
  # Just for future convenience.
  #


  print "*** Checking file permissions ***\n";

  if (!(-w ".")) {
    die "Fatal error -- cannot write to current directory.";
  }

  my @fix_files = (
    +[ "RUNME.pl",        0100 ],
    +[ "LineGen.pl",      0100 ],
    +[ "TriGen.pl",       0100 ],
    +[ "koch.pl",         0100 ],
    +[ "DiagGen.pl",      0100 ]
  );


  my $listref = undef;
  foreach $listref (@fix_files) {
    my $file = $listref->[0];
    my $goal = $listref->[1];

    print "Checking permissions on $file ($goal)... ";

    if (!(-e $file)) {
      die "Fatal error -- expected file $file does not exist.";
    }

    # Reuse stat info from the last test
    my ($dev,   $ino,   $mode,  $nlink,
    $uid,   $gid,   $rdev,  $size,
    $atime, $mtime, $ctime, $blksize,
    $blocks) = stat($file);

    if ($uid ne $EUID) {
      die "Fatal error -- '$file' owned by uid $uid.  EUID = $EUID.";
    }

    if (($mode & $goal) == $goal) {
      print "ok\n";
      next;
    } else {
      print "off... ";

      $mode |= $goal;
      if (!(chmod $mode, $file)) {
        die "Fatal error:  chmod() failed on $file:  $!";
      }

      print "(fixed)\n";
    }
  }
}




# Phase 2 -- Compile the C++ source.
#
# Note:  error detection here is going to be REALLY REALLY
# poor, since there are so many possible errors ranging
# from lack of permission to lack of disk space to, perhaps,
# lack of portability in the template code et al.
{
  my $make_fh = gensym();
  my $error   = undef;

  print "*** Compiling code ***\n";

  # Try 'gmake' first, since on many boxes that's the name for
  # GNU make.

  if (!open($make_fh, "gmake rebuild|")) {
    open($make_fh, "make rebuild|") || die "Failed to open (pipe) 'make':  $!";
  }

  {
    my $line = <$make_fh>;

    if (!defined($line)) {
      last;
    }

    # echo to user
    print "::: $line";

    if ($line =~ /Error/) {
      $error = 1;
    }

    redo;
  }

  close($make_fh) || die "Failed to close (pipe) 'make':  $!";

  if (defined($error)) {
    # Error(s) detected.  Abort.
    die "Errors detected during compilation.  Examine manually.";
  }

  if (!(-f "fractug")) {
    die "'fractug' binary not detected.  Aborting.";
  }

  # Make sure it's executable.
  chmod(0722, 'fractug');

  if (!(-x "fractug")) {
    # Weird.  At least, for a 'nix box.
    die "'fractug' binary not executable despite chmod call.";
  }
}


# Phase 3 -- Generating test data.
#
{
  print "*** Generating 2D diagonal line ***\n";

  if (system("./DiagGen.pl -n1000 -E2 > 2Ddiagonal.inp")) {
    die "Error running DiagGen.pl:  $!";
  }
}

my $twoDdiag_a = 0;   # slope
my $twoDdiag_b = 0;   # y-int
my $pair_fn    = undef;


# Phase 4 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;

  print "*** Invoking fractug *** \n";
  print "\n";
  print "Test set:  diagonal line, 2-D embedding, ";
  print "1000 pts.\n";
  print "\n";

  my $twoDdiag_fh = gensym();
  #my $cmd     = "./fractug --verbose 2Ddiagonal.inp";
  #this combination of s1 and s2 makes manual trimming necessary for this set
  my $cmd     = "./fractug --verbose --s1 30 --s2 5 --min 0.95 2Ddiagonal.inp";
  #my $cmd     = "./fractug --verbose --s1 30 --s2 5 2Ddiagonal.inp";
  # This part differs from the RUNME.pl for the Perl package,
  # because we're calling C++ code and parsing the output.


  my $t0 = new Benchmark;
  open($twoDdiag_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$twoDdiag_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($twoDdiag_fh) ||
  die "Unexpected fatal error closing fractug:  $!";
  my $t1 = new Benchmark;

  die unless defined($fd);
  die unless defined($corr);



  print "Fractal dimension reported:  $fd\n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # It's a randomized algorithm.  Since the TugApprox class does
  # not include its own RNG but uses a library implementation,
  # the exact value may vary between platforms.  Hrm.

  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $twoDdiag_a = $fd;
  $twoDdiag_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2 ** $log_r[$i]), " ", (2 ** $log_s[$i]), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 5 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# I've decided to plot it *without* the trimming (note the use
# of --verbose in the fractug invocation.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("2Ddiagonal.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"2Ddiagonal.eps\"\n";
  print $gnuplot_fh "set title \"Pair-count plot for 2Ddiagonal.inp\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"2Ddiagonal\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$twoDdiag_b) * (x**$twoDdiag_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"2Ddiagonal.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 6 -- Display

{
  print "*** Displaying resulting 2Ddiagonal.eps ***\n";

  system("gv 2Ddiagonal.eps") && die "Unexpected ghostview return value";
}



# Phase 7 -- Generating test data.
#
{
  print "*** Generating 3D diagonal line ***\n";

  if (system("./DiagGen.pl -n1000 -E3 > 3Ddiagonal.inp")) {
    die "Error running DiagGen.pl:  $!";
  }
}

my $threeDdiagonal_a = 0;   # slope
my $threeDdiagonal_b = 0;   # y-int

# Phase 8 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;


  print "*** Invoking fractug *** \n";
  print "\n";
  print "Test set:  diagonal line, 3-D embedding, ";
  print "1000 pts.\n";
  print "\n";

  my $threeDdiagonal_fh = gensym();

  # We take advantage of knowing that no two points will be more
  # than about one unit apart.
  #my $cmd     = "./fractug --verbose 3Ddiagonal.inp";
  my $cmd     = "./fractug --verbose --s1 30 --s2 5 3Ddiagonal.inp";


  my $t0 = new Benchmark;
  open($threeDdiagonal_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$threeDdiagonal_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($threeDdiagonal_fh) ||
  die "Unexpected fatal error closing fractug:  $!";

  die unless defined($fd);
  die unless defined($corr);

  my $t1 = new Benchmark;

  print "Fractal dimension reported:  $fd \n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # It's a randomized algorithm.  Since the TugApprox class does
  # not include its own RNG but uses a library implementation,
  # the exact value may vary between platforms.  Hrm.

  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $threeDdiagonal_a = $fd;
  $threeDdiagonal_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2** ($log_r[$i])), " ", (2 ** ($log_s[$i])), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 9 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# The data has not been trimmed.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("3Ddiagonal.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"3Ddiagonal.eps\"\n";
  print $gnuplot_fh "set title \"Pair-count plot for 3Ddiagonal.inp\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"3Ddiagonal\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$threeDdiagonal_b) * (x**$threeDdiagonal_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"3Ddiagonal.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 10 -- Display

{
  print "*** Displaying resulting 3Ddiagonal.eps ***\n";

  system("gv 3Ddiagonal.eps") && die "Unexpected ghostview return value";
}


# Phase 11 -- Generating test data.
#
{
  print "*** Generating 4D diagonal line ***\n";

  if (system("./DiagGen.pl -n1000 -E4 > 4Ddiagonal.inp")) {
    die "Error running DiagGen.pl:  $!";
  }
}

my $fourDdiagonal_a = 0;   # slope
my $fourDdiagonal_b = 0;   # y-int

# Phase 12 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;


  print "*** Invoking fractug *** \n";
  print "\n";
  print "Test set:  diagonal line, 4-D embedding, ";
  print "1000 pts.\n";
  print "\n";

  my $fourDdiagonal_fh = gensym();

  # We take advantage of knowing that no two points will be more
  # than about one unit apart.
  #my $cmd     = "./fractug --verbose 4Ddiagonal.inp";
  #this combination of s1 and s2 makes manual trimming necessary for this set
  my $cmd     = "./fractug --verbose --s1 30 --s2 5 --min 0.95 4Ddiagonal.inp";


  my $t0 = new Benchmark;
  open($fourDdiagonal_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$fourDdiagonal_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($fourDdiagonal_fh) ||
  die "Unexpected fatal error closing fractug:  $!";

  die unless defined($fd);
  die unless defined($corr);

  my $t1 = new Benchmark;

  print "Fractal dimension reported:  $fd \n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # It's a randomized algorithm.  Since the TugApprox class does
  # not include its own RNG but uses a library implementation,
  # the exact value may vary between platforms.  Hrm.

  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $fourDdiagonal_a = $fd;
  $fourDdiagonal_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2** ($log_r[$i])), " ", (2 ** ($log_s[$i])), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 13 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# The data has not been trimmed.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("4Ddiagonal.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"4Ddiagonal.eps\"\n";
  print $gnuplot_fh "set title \"Pair-count plot for 4Ddiagonal.inp\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"4Ddiagonal\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$fourDdiagonal_b) * (x**$fourDdiagonal_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"4Ddiagonal.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 14 -- Display

{
  print "*** Displaying resulting 4Ddiagonal.eps ***\n";

  system("gv 4Ddiagonal.eps") && die "Unexpected ghostview return value";
}


# Phase 15 -- Generating test data.
#
# Generating a Sierpinski data set, of 5,000 points
# only (for speed reasons.  Given that this script
# might be the first thing run, it behooves us to
# not use a terribly slow case).  5000 points should
# be done QUITE quickly.
{
  print "*** Generating Sierpinski.5000 ***\n";

  if (system("./TriGen.pl 5000 > Sierpinski.5000")) {
    die "Error running TriGen.pl:  $!";
  }
}


my $sier5000_a = 0;   # slope
my $sier5000_b = 0;   # y-int


# Phase 16 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;

  print "*** Invoking fractug *** \n";
  print "\n";
  print "Test set:  Sierpinski triangle, 2-D embedding, ";
  print "5000 pts.\n";
  print "\n";

  my $sier_fh = gensym();
  my $cmd     = "./fractug --verbose Sierpinski.5000";
  # This part differs from the RUNME.pl for the Perl package,
  # because we're calling C++ code and parsing the output.


  my $t0 = new Benchmark;
  open($sier_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$sier_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($sier_fh) ||
  die "Unexpected fatal error closing fractug:  $!";
  my $t1 = new Benchmark;

  die unless defined($fd);
  die unless defined($corr);



  print "Fractal dimension reported:  $fd (theoretical ~ 1.58)\n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # It's a randomized algorithm.  Since the TugApprox class does
  # not include its own RNG but uses a library implementation,
  # the exact value may vary between platforms.  Hrm.

  print "Expected estimated FD, on a Linux box:    1.596\n";
  print "Expected estimated FD, on a Solaris box:  1.665\n";

  # Ah, the joys of randomness.  So let's say... if it's not
  # between 1.5 and 1.7, which is a decent range, we reject.
  if (($fd < 1.5) || ($fd > 1.7)) {
    die "Fatal error -- expected the fd value to be within range [1.5,1.7].";
  }
  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $sier5000_a = $fd;
  $sier5000_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2 ** $log_r[$i]), " ", (2 ** $log_s[$i]), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 17 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# I've decided to plot it *without* the trimming (note the use
# of --verbose in the fractug invocation.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("sierpinski.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"sierpinski.eps\"\n";
  print $gnuplot_fh "set title \"Pair-count plot for Sierpinski.5000\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"Sierpinski\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$sier5000_b) * (x**$sier5000_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"sierpinski.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 18 -- Display

{
  print "*** Displaying resulting sierpinski.eps ***\n";

  system("gv sierpinski.eps") && die "Unexpected ghostview return value";
}



# Phase 19 -- Generating test data.
#
# Generating a koch snowflake using 7 iterations.
#
{
  print "*** Generating koch7.inp ***\n";

  if (system("./koch.pl -n7 > koch7.inp")) {
    die "Error running koch.pl:  $!";
  }
}

my $testpoints_a = 0;   # slope
my $testpoints_b = 0;   # y-int

# Phase 20 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;

  print "*** Invoking fractug *** \n";

  my $test_fh = gensym();
  my $cmd = "./fractug --verbose --s1 30 --s2 5 --num 19 --max 8 koch7.inp";
  # This part differs from the RUNME.pl for the Perl package,
  # because we're calling C++ code and parsing the output.


  my $t0 = new Benchmark;
  open($test_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$test_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($test_fh) ||
  die "Unexpected fatal error closing fractug:  $!";
  my $t1 = new Benchmark;

  die unless defined($fd);
  die unless defined($corr);



  print "Fractal dimension reported:  $fd\n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $testpoints_a = $fd;
  $testpoints_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2 ** $log_r[$i]), " ", (2 ** $log_s[$i]), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}

# Phase 21 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# The data has not been trimmed.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("koch.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"koch.eps\"\n";
  print $gnuplot_fh "set title \"Tug-of-War plot for Koch Snowflake\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"Koch Snowflake\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$testpoints_b) * (x**$testpoints_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"koch.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}

# Phase 22 -- Display

{
  print "*** Displaying resulting koch.eps ***\n";

  system("gv koch.eps") && die "Unexpected ghostview return value";
}


# Phase 23 -- Generating test data
#
# Montgomery County, MD
#
{
  print "*** Montgomery County, MD ***\n";
}

my $testpoints_a = 0;   # slope
my $testpoints_b = 0;   # y-int


# Phase 24 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;

  print "*** Invoking fractug *** \n";

  my $test_fh = gensym();
  my $cmd     = "./fractug --verbose --s1 30 --s2 5 --num 27 --min 1.0 --max 67108864 mgcounty.inp";
  # This part differs from the RUNME.pl for the Perl package,
  # because we're calling C++ code and parsing the output.


  my $t0 = new Benchmark;
  open($test_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$test_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($test_fh) ||
  die "Unexpected fatal error closing fractug:  $!";
  my $t1 = new Benchmark;

  die unless defined($fd);
  die unless defined($corr);



  print "Fractal dimension reported:  $fd\n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $testpoints_a = $fd;
  $testpoints_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2 ** $log_r[$i]), " ", (2 ** $log_s[$i]), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 25 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# I've decided to plot it *without* the trimming (note the use
# of --verbose in the fractug invocation.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("koch.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"mgcounty.eps\"\n";
  print $gnuplot_fh "set title \"Tug-of-War plot for Montgomery County, MD\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"Montgomery County, MD\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$testpoints_b) * (x**$testpoints_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"mgcounty.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 26 -- Display

{
  print "*** Displaying resulting mgcounty.eps ***\n";

  system("gv mgcounty.eps") && die "Unexpected ghostview return value";
}


# Phase 27 -- Generating test data
#
# Long Beach County, CA
#
{
  print "*** Long Beach County, CA ***\n";
}

my $testpoints_a = 0;   # slope
my $testpoints_b = 0;   # y-int


# Phase 28 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;

  print "*** Invoking fractug *** \n";

  my $test_fh = gensym();
  my $cmd = "./fractug --verbose --s1 30 --s2 5 --num 20 --max 16 longbeach.inp";
  # This part differs from the RUNME.pl for the Perl package,
  # because we're calling C++ code and parsing the output.


  my $t0 = new Benchmark;
  open($test_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$test_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($test_fh) ||
  die "Unexpected fatal error closing fractug:  $!";
  my $t1 = new Benchmark;

  die unless defined($fd);
  die unless defined($corr);



  print "Fractal dimension reported:  $fd\n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $testpoints_a = $fd;
  $testpoints_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2 ** $log_r[$i]), " ", (2 ** $log_s[$i]), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 29 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# I've decided to plot it *without* the trimming (note the use
# of --verbose in the fractug invocation.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("koch.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"longbeach.eps\"\n";
  print $gnuplot_fh "set title \"Tug-of-War plot for Long Beach County, CA\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"Long Beach County, CA\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$testpoints_b) * (x**$testpoints_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"longbeach.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 30 -- Display

{
  print "*** Displaying resulting longbeach.eps ***\n";

  system("gv longbeach.eps") && die "Unexpected ghostview return value";
}


# Phase 31 -- Generating test data.
#
# Generating a self-similar line (with gaps).
#
# Warning -- it's 50-dimensional and thus this takes ~8MB space or so.
#
{
  print "*** Generating Line.10000 ***\n";

  if (system("./LineGen.pl 10000 > Line.10000")) {
    die "Error running LineGen.pl:  $!";
  }
}


my $line10000_a = 0;   # slope
my $line10000_b = 0;   # y-int

# Phase 32 -- Invoke FracDim code.
#
{
  my $fd   = undef;
  my $corr = undef;
  my $intercept = undef;


  print "*** Invoking fractug *** \n";
  print "\n";
  print "Test set:  Self-similar line with large gaps, with an\n";
  print "embedding dimensionality of 50; 10k points.\n";

  my $line_fh = gensym();

  # We take advantage of knowing that no two points will be more
  # than about one unit apart.
  my $cmd     = "./fractug --verbose Line.10000 --min 0.001953 --max 1";


  my $t0 = new Benchmark;
  open($line_fh, "$cmd|") || die "Failed to invoke fractug:  $!";

  my @log_r = ();
  my @log_s = ();

  {
    {
      my $line = <$line_fh>;
      defined($line) || last;
      print "::: $line";

      if ($line =~ /^Estimated fractal dimension:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($fd));   # should only be once
        $fd = $1;
      } elsif ($line =~ /^Correlation:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($corr));   # should only be once
        $corr = $1;
      } elsif ($line =~ /^y-intercept:\s+(-?\d+\.?\d*)\s*$/) {
        die if (defined($intercept));
        $intercept = $1;
      } elsif ($line =~ /^;;;\s+(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*$/) {
        push @log_r, $1;
        push @log_s, $2;
      }
      redo;
    }
  }

  close($line_fh) ||
  die "Unexpected fatal error closing fractug:  $!";

  die unless defined($fd);
  die unless defined($corr);

  my $t1 = new Benchmark;

  print "Fractal dimension reported:  $fd (theoretical ~ 0.5)\n";
  print "Correlation:  $corr\n";
  print "Time elapsed:  ", timestr(timediff($t1, $t0)), "\n";


  # It's a randomized algorithm.  Since the TugApprox class does
  # not include its own RNG but uses a library implementation,
  # the exact value may vary between platforms.  Hrm.

  print "Expected estimated FD, on a Linux box:    0.509010\n";
  print "Expected estimated FD, on a Solaris box:  0.443260\n";


  # Perhaps 0.44 to 0.56 is acceptable.
  #
  # I'm not sure yet why the estimates are better on Linux.
  # *frown*
  #
  if (($fd < 0.44) || ($fd > 0.56)) {
    die "Fatal error -- expected the fd value to be within range [0.44,0.56].";
  }
  # Store this for Gnuplot to use.
  my $pair_fh = undef;
  my $ct = scalar @log_r;
  my $i  = undef;

  ($ct == (scalar @log_s)) ||
  die "Severe, unexpected logical error in FracTug!";
  ($pair_fh, $pair_fn) = mktemp();

  $line10000_a = $fd;
  $line10000_b = $intercept;

  # It's log data, but we'll use gnuplot's scale to make it
  # clear, so we undo the logs.
  for ($i=0; $i < $ct; $i++) {
    print $pair_fh (2** ($log_r[$i])), " ", (2 ** ($log_s[$i])), "\n";
  }

  close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 33 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# The data has not been trimmed.

{
  my $gnuplot_fh = gensym();

  print "*** Using GNUPLOT for log-log graph ***\n";

  if (!open($gnuplot_fh, "|gnuplot")) {
    die "Fatal error -- gnuplot not in path?  $!";
  }

  unlink("line.eps");

  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"line.eps\"\n";
  print $gnuplot_fh "set title \"Pair-count plot for Line.10000\"\n";
  print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
  print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
  print $gnuplot_fh "set logscale xy 2\n";
  print $gnuplot_fh "set key under\n";
  print $gnuplot_fh "plot \"$pair_fn\" title \"Line\" with linespoints\n";
  # y=(x^a) * (2^b)
  print $gnuplot_fh "replot ((2**$line10000_b) * (x**$line10000_a)) title \"Linear approximation\"\n";
  print $gnuplot_fh "set terminal postscript eps\n";
  print $gnuplot_fh "set output \"line.eps\"\n";
  print $gnuplot_fh "replot\n";
  close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 34 -- Display

{
  print "*** Displaying resulting line.eps ***\n";

  system("gv line.eps") && die "Unexpected ghostview return value";
}


# Phase 35 -- Final comments

{
  print "\n";
  print "Your installation appears to be in functioning\n";
  print "properly.  See the README for more information on\n";
  print "using this package.\n";
}


exit 0;

