/* File:     memorywrapper.cc
 * Purpose:  To implement the DataWrapper interface with an
 *           in-memory version -- the classic speed-for-space
 *           tradeoff.
 *
 * RCS:
 ************************************************************
 * $Id: MemoryWrapper.cc,v 1.5 2001/08/23 16:19:14 lw2j Exp $
 * $Log:	MemoryWrapper.cc,v $
// Revision 1.5  2001/08/23  16:19:14  lw2j
// Added the zero_translate option.
// 
// Revision 1.4  2001/08/22  17:20:18  lw2j
// Untabified.
// 
// Revision 1.3  2001/08/22  16:50:51  lw2j
// Added the two_tables method, which may give more speed
// when increasing the radius at the cost of storage.
// 
// Revision 1.2  2001/08/21  23:04:35  lw2j
// Fixed an assert(buf) that should have been an assert(dst), in
// MemoryWrapper::get_vector_common().
// 
 * Revision 1.1  2001/08/21 03:40:34  lw2j
 * Initial revision
 *
 *
 ************************************************************
 */


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include "MemoryWrapper.h"


/* A linked-list of double vectors.  Used so we can load in
 * a single pass -- vital for stdin / pipe support, since
 * there we can't seek.  The odd-looking structure type
 * name should prevent namespace collisions.
 */
struct MW_dbl_ll_t {
  MW_dbl_ll_t():dbl_arr(NULL), next(NULL) {};

  ~MW_dbl_ll_t() { 
    delete dbl_arr;
    /* delete next; */
  }

  double             *dbl_arr;
  struct MW_dbl_ll_t *next;
};




/* Single-pass load, using a linked list.  Consumes potentially
 * vast amounts of memory, but there aren't many pretty ways to
 * to stdin support (well, we could write to *disk*, and then
 * open that up, but...).
 */
int MemoryWrapper::load_file(FILE *fh_in) {
  char           line_buf[DataWrapper::MAX_LINE_SIZE];
  double*        dbl_array = NULL;
  unsigned int   number    = 0;
  unsigned int   index     = 0;
  unsigned int   i         = 0;
  struct MW_dbl_ll_t *head = NULL;
  struct MW_dbl_ll_t *tail = NULL;
  struct MW_dbl_ll_t *ptr   = NULL;

  if (buf) {
    delete [] buf;
    buf = NULL;
  }

  if (big_buf) {
    delete [] big_buf;
    big_buf = NULL;
  }

  if (minima) {
    delete [] minima;
    minima = false;
  }

  dimensionality = 0;
  cardinality    = 0;

  if (!fh_in) {
    errno = EINVAL;
    return 0;
  }

  /* Find count and dimensionality. */

  while (fgets(line_buf, DataWrapper::MAX_LINE_SIZE, fh_in)) {
    DataWrapper::process_string(line_buf);

    /* Only care if it's not truncated to nothingness. */
    if (line_buf[0] != '\0') {
      number = DataWrapper::find_doubles(dbl_array, line_buf, 0);
                                         

      if (number > 0) {
        /* Doubles found.  Count the vector. */
        cardinality++;

        /* Add to the linked list (shallow copy). */
        if (!head) {
          /* Empty list, so create one. */
          head = new MW_dbl_ll_t;
          assert(head);
          tail = head;
        } else {
          /* Add a new node. */
          tail->next = new MW_dbl_ll_t;
          tail       = tail->next;
          assert(tail);
        }

        tail->dbl_arr = dbl_array;
        tail->next    = NULL;
        
        
        if (!dimensionality) {
          /* And this is how many dimensions we'll expect from
           * all subsequent vectors.  We do NOT specify this
           * above, 'tho, because we want to be paranoid and
           * count all the doubles in every line just this 
           * once.
           */
          dimensionality = number; 

          /* Allocate minima vector, and copy into it. */
          minima = new double[dimensionality];
          assert(minima);
          memcpy((void*) minima, (void*) dbl_array, sizeof(double) *
                 dimensionality);
        } else {
          /* Do they match? */
          if (dimensionality != number) {
            errno = EINVAL;

            if (head) {
              ptr = head;
              
              while (ptr) {
                ptr = head->next;
                delete head;
                head = ptr;
              }
            };            

            if (minima) {
              delete [] minima;
              minima = NULL;
            }
            return 0;
          }
        }

        /* Update minima, if need be */
        for (i=0; i < dimensionality; i++) {
          minima[i] = (minima[i] <= dbl_array[i]) ?
            minima[i] : dbl_array[i];
        }
        

        /* Reset dbl_array for the next one. It's in the
         * list, so we'll do the deletions then.*/
        dbl_array = NULL;
      }
    }
  }

  /* Allocate big_buf. */
  big_buf = new double[dimensionality*cardinality];
  assert(big_buf);
    

  /* Copy from linked list to array. */
  index     = 0;
  ptr       = head;
  dbl_array = big_buf;

  if (!zero_translation) {
    for (index=0; index < cardinality; index++) {
      memcpy((void*) dbl_array,
             (void*) (ptr->dbl_arr),
             sizeof(double) * dimensionality);
      ptr        = ptr->next;
      dbl_array += dimensionality;
    }
  } else {
    /* translate it now, rather than every time we fetch */
    for (index=0; index < cardinality; index++) {
      for (i=0; i < dimensionality; i++) {
        dbl_array[i] = (ptr->dbl_arr)[i] - minima[i];
      }
      
      ptr        = ptr->next;
      dbl_array += dimensionality;
    }
  }

  /* Delete linked-list; done this way to avoid any stack
   * overflow issues (?).
   */
  if (head) {
    ptr = head;

    while (ptr) {
      ptr = head->next;
      delete head;
      head = ptr;
    }
  };

  /* For the non-dynamic version. */
  buf = new double[dimensionality];
  assert(buf);

  return 1;
}



/* We'll need to subtract the minima from every vector if the
 * new value is true and the old one is false, or add to every
 * vector if the new one is false and the old one is true.
 *
 * We translate in memory here to avoid having to do so with
 * ever fetch.
 */
void MemoryWrapper::set_zero_translation(bool new_zero_translation) {
  unsigned int item_idx=0;
  unsigned int dim_idx = 0;
  double      *bb_ptr  = big_buf;
  double      *mm_ptr  = minima;

  if (new_zero_translation == zero_translation) {
    /* no-op */
    return;
  }

  zero_translation = new_zero_translation;

  if ((!minima) || (!big_buf)) {
    return;
  }

  if (new_zero_translation) {
    /* wasn't subtracted, we need to do so */
    for (item_idx=0; item_idx < cardinality; item_idx++) {
      mm_ptr = minima;
      for (dim_idx=0; dim_idx < dimensionality; dim_idx++) {
        *bb_ptr -= *mm_ptr;
        bb_ptr++;
        mm_ptr++;
      }
    }
  } else {
    /* was subtracted, need to add it */
    for (item_idx=0; item_idx < cardinality; item_idx++) {
      mm_ptr = minima;
      for (dim_idx=0; dim_idx < dimensionality; dim_idx++) {
        *bb_ptr += *mm_ptr;
        bb_ptr++;
        mm_ptr++;
      }
    }
  }
}


/* Get a vector, by index.  It should be in memory, since
 * we really should have run 'load_file()' by now.
 */

double* MemoryWrapper::get_vector_common(unsigned int index,
                                       double* dst) {
  if ((!dimensionality) || (!big_buf)) {
    /* Something bogus -- either a very silly user, or a bug.
     */
    errno = ENOENT;
    return NULL;
  }

  if (index >= cardinality) {
    /* Out of range. */
    errno = EINVAL;
    return NULL;
  }

  if (!dst) {
    dst = new double[dimensionality];
    assert(dst);
  }

  memcpy((void*) dst,
         (void*) (&(big_buf[dimensionality * index])),
         sizeof(double) * dimensionality);

  return dst;
}

