//////////////////////////////////////////////////////////////////////////////////////////////////////
//
// QJuliaGPU -- Keenan Crane (kcrane@uiuc.edu)
//   4/17/2004
//
//
//    This program ray traces the quaternion Julia set in a fragment shader
//    using the sphere tracing method.  The program draws a fullscreen quad
//    where each fragment of the quad specifies a different ray.  These rays
//    are passed to the fragment shader which iteratively takes conservative
//    steps along a ray as determined by a distance estimator for the set.  The
//    rays will either stop when close to an isosurface of the distance
//    function (considered a hit), or leave the bounding sphere of the Julia
//    set.  If the ray is a hit, shading is performed by approximating the
//    gradient of the distance function and using this as a surface normal.
//
//    A more complete description of the sphere tracing method can be found in John Hart's paper,
//    "Ray Tracing Deterministic 3-D Fractals" (http://graphics.cs.uiuc.edu/~jch/papers/rtqjs.pdf).
//
//    Controls:
//
//         left mouse button:    rotate view
//         middle mouse button:  zoom in/out
//         right mouse button:   translate view
//         space:                toggle morph animation
//         s:                    toggle shadows on/off
//         r:                    reload shaders from disk
//         i/I:                  increment/decrement 1st imaginary component of Julia set constant
//         j/J:                  increment/decrement 2nd imaginary component of Julia set constant
//         k/K:                  increment/decrement 3rd imaginary component of Julia set constant
//         l/L:                  increment/decrement real component of Julia set constant
//         -/+:                  change number of iterations used to test convergence of a point
//         [/]:                  change precision of rendering
//
//    By default the program will shift through a random constants for the
//    Julia set within the cube [-1,1]^4.  Increasing the number of iterations
//    or the precision will increase the amount of detail seen in the
//    rendering.  The former more accurately determines whether a point is
//    included in the set, whereas the latter intersects an isosurface of the
//    distance function closer to the actual set.  Both of these parameters run
//    into precision or computation limits when set too high.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <iostream>
#include <GL\glut.h>
#include <Cg\cg.h>
#include <Cg\cgGL.h>

using namespace std;

////// Global state for the GUI //////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////

// constants ----------
#define M_PI 3.1415926f                      // where is PI defined on Windows??
const float fRandMax = 1.0f/(float)RAND_MAX; // (used to normalize random values)

// drawing state variables ----------
bool fullscreen = false,    // whether we're in fullscreen mode
     bAnimate   = true,     // whether we're randomly morphing among sets
     bShadows   = false;    // whether self-shadowing is being rendered

// shader variables ----------
CGcontext context;                         // context for shaders
CGprogram fragmentProgram = 0,             // ray tracing fragment program
          vertexProgram   = 0;             // ray tracing vertex program
CGprofile fragmentProfile, vertexProfile;  // shader profiles

// shader parameters -- these are handles to uniform parameters in the shader
CGparameter constant,             // constant specifying the Julia set
            eyeP,                 // eye location
            lightP,               // light location
            shadowP,              // flag for drawing shadows
            iterationsP,          // maximum number of iterations used to determine convergence
            epsilonP;             // precision of ray tracing

// Julia set parameters ----------
unsigned int maxIterations = 8;   // maximum number of iterations used to determine convergence
float epsilon = 0.003f;           // precision of ray tracing
float mu[4], mup[4], curMu[4];    // current constant parameters (used for morphing)
float morphTimer = 0.0f;          // morph interpolant parameter

// camera parameters ----------
float eye[4], lookAt[4], up[4], fov = 60.0f;
float N[3], T[3], B[3];  //normal, tangent, and binormal of the image plane
const int defaultWindowSize[2] = { 384, 384 };

// light parameters ----------
float light[4];    // point light location

// ======================|
// ugly GUI variables ---|
// ======================|

GLdouble lastRotation[16], curRotation[16];
float startPoint[3], rotationAxis[3], rotationAngle, spinAngle = 0.0f;
float spinRate      = 0.2f,
      zoomRate      = 0.01f,
      translateRate = 0.005f;
int firstDrag, maxSpinDelay = 250;
float lastZoom = 0.0f, zoom = 3.0f;
float lastTranslate[2] = {0.0f, 0.0f}, translate[2] = {0.0f, 0.0f};
int lastX, lastY;
int mouseButton;
bool spinOn = false;
float aspect;

//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////

// --------- getSpherePoint() -------------------------
//
// Map a point in the window to a point on the unit
// sphere (used for rotation).
//

void getSpherePoint( int x, int y, float* pt )
{
   int viewport[4];
   float mx, my, mz, r;

   glGetIntegerv( GL_VIEWPORT, viewport );

   mx = 2.0f * x / (float)viewport[2] - 1.0f;
   my = 2.0f * y / (float)viewport[3] - 1.0f;
   my = -my;

   r = sqrtf( mx*mx + my*my );
   if( r > 1 )
   {
      r = 1.0f / r;
      mx *= r;
      my *= r;
      mz = 0.0f;
   }
   else
   {
      mz = sqrtf( 1.0f - r );
   }

   r = 1.0f / sqrtf( mx*mx + my*my + mz*mz );
   mx *= r;
   my *= r;
   mz *= r;

   pt[0] = mx;
   pt[1] = my;
   pt[2] = mz;
}

// ----------- click() -----------------------
//
// Handle mouse clicks (changes the view).
//

void click( int button, int state, int x, int y )
{
   int tmp = x;
   x = y;
   y = tmp;

   mouseButton = button;

   if( state == GLUT_DOWN )
   {
      if( mouseButton == GLUT_LEFT_BUTTON )
      {
         getSpherePoint( x, y, startPoint );
         spinOn = false;
         firstDrag = clock();
         lastX = -1;
         lastY = -1;
      }
      else if( mouseButton == GLUT_MIDDLE_BUTTON )
      {
         lastX = x;
         lastY = y;
      }
      else if( mouseButton == GLUT_RIGHT_BUTTON )
      {
         lastX = x;
         lastY = y;
         lastTranslate[0] = translate[0];
         lastTranslate[1] = translate[1];
      }

      lastZoom = zoom;
   }
   else if( state == GLUT_UP )
   {
      for( int i=0; i<16; i++ )
         lastRotation[i] = curRotation[i];
      spinOn = true;

      if( lastX == -1 || lastY == -1 )
         spinAngle = 0.0f;
   }
}

// ------- spin() ----------------------------------------
//
// Rotate the modelview by the specified angle around the
// specified axis.

void spin( float theta, float* v )
{
   glMatrixMode( GL_MODELVIEW );
   glPushMatrix();
   glLoadIdentity();

   glRotatef( theta, v[0], v[1], v[2] );
   glMultMatrixd( lastRotation );
   glGetDoublev( GL_MODELVIEW_MATRIX, curRotation );

   glPopMatrix();
}

// --------- drag() -------------------------------------------
//
// Handle mouse movement while a button is pressed.  (Handles
// camera movement.)

void drag( int x, int y )
{
   int tmp = x;
   x = y;
   y = tmp;

   if( mouseButton == GLUT_LEFT_BUTTON )
   {
      float endPoint[3];
      getSpherePoint( x, y, endPoint );

      rotationAxis[0] = startPoint[1]*endPoint[2] - startPoint[2]*endPoint[1];
      rotationAxis[1] = startPoint[2]*endPoint[0] - startPoint[0]*endPoint[2];
      rotationAxis[2] = startPoint[0]*endPoint[1] - startPoint[1]*endPoint[0];

      float lastAngle = rotationAngle;
      rotationAngle = acosf( startPoint[0]*endPoint[0] +
            startPoint[1]*endPoint[1] +
            startPoint[2]*endPoint[2] );

      if( rotationAngle < 0.001f )
      {
         for( int i=0; i<16; i++ )
            curRotation[i] = lastRotation[i];
         return;
      }

      rotationAngle *= 180.0f / M_PI;

      spin( -rotationAngle, rotationAxis );

      float dx = (float)(lastX - x);
      float dy = (float)(lastY - y);

      if( clock() - firstDrag < maxSpinDelay )
         spinAngle = sqrt( dx*dx + dy*dy ) * spinRate;
      else
         spinAngle = 0.0f;

      lastX = x;
      lastY = y;
   }
   else if( mouseButton == GLUT_MIDDLE_BUTTON )
   {
      zoom = lastZoom + ( lastX - x ) * zoomRate;
	  if( zoom < 0.0f )
		  zoom = 0.0f;
   }
   else if( mouseButton == GLUT_RIGHT_BUTTON )
   {
      translate[0] = lastTranslate[0] + ( x - lastX ) * translateRate;
      translate[1] = lastTranslate[1] + ( lastY - y ) * translateRate;
   }
}


// --------- calculateView() ------------------
//
// Convert transformation matrix into a basis for the space of the image plane.
// This basis is used to generate the rays which are intersected with the
// Julia set.
//

void calculateView()
{
   // First apply the view transformations to the initial eye, look at, and
   // up.  These will be used later to determine the basis.

   float eyeStart[4] =    { 0.0f, 0.0f, 1.0f, 1.0f };    // eye starts on the unit sphere
   float lookatStart[4] = { 0.0f, 0.0f, 0.0f, 1.0f };    // initially look at the origin
   float upStart[4] =     { 0.0f, 1.0f, 0.0f, 0.0f };    // up is initially along the y-axis

   // translate the eye and look at points
   eyeStart[0] += translate[0];
   eyeStart[1] += translate[1];
   eyeStart[2] += zoom;
   lookatStart[0] += translate[0];
   lookatStart[1] += translate[1];
   lookatStart[2] += zoom;

   // rotate eye, lookat, and up by multiplying them with the current rotation matrix
   for( int i=0; i<4; i++ )
   {
      eye[i]    = 0.0f;
      lookAt[i] = 0.0f;
      up[i]     = 0.0f;

      for( int j=0; j<4; j++ )
      {
         eye[i]    += curRotation[i*4+j] * eyeStart[j];
         lookAt[i] += curRotation[i*4+j] * lookatStart[j];
         up[i]     += curRotation[i*4+j] * upStart[j];
      }
   }

   // Now we construct the basis:
   //
   //   N = (look at) - (eye)
   //   T = up
   //   B = N x T
   //
   
   float mag;

   // find and normalize N = (lookat - eye)
   for(int i=0; i<3; i++) N[i] = lookAt[i] - eye[i];
   mag = 1.0f / sqrt(N[0]*N[0] + N[1]*N[1] + N[2]*N[2]);
   for(int i=0; i<3; i++) N[i] *= mag;

   // find and normalize T = up
   for(int i=0; i<3; i++) T[i] = up[i];
   mag = 1.0f / sqrt(T[0]*T[0] + T[1]*T[1] + T[2]*T[2]);
   for(int i=0; i<3; i++) T[i] *= mag;

   // find B = N x T (already unit length)
   B[0] = N[1]*T[2] - N[2]*T[1];
   B[1] = N[2]*T[0] - N[0]*T[2];
   B[2] = N[0]*T[1] - N[1]*T[0];


   // we also use this basis to determine the light position
   // (move the light a little bit up and to the right of the eye).
   for( int i=0; i<3; i++ )
   {
      light[i] = eye[i] + B[i] * 0.5f;
      light[i] = eye[i] + T[i] * 0.5f;
   }
}

// ------ initializeCg() ------------------------------------
//
// Setup state required to run Cg shaders.  Namely, a context
// and profiles for fragment and vertex programs.
//

void initializeCg( void )
{
   context = cgCreateContext();

   if(!context)
   {
      cerr << "Error: could not create Cg context." << endl;
      return;
   }

   fragmentProfile = cgGLGetLatestProfile( CG_GL_FRAGMENT );
   if( fragmentProfile == CG_PROFILE_UNKNOWN )
   {
      cerr << "Error: unknown fragment profile." << endl;
      return;
   }

   vertexProfile = cgGLGetLatestProfile( CG_GL_VERTEX );
   if( vertexProfile == CG_PROFILE_UNKNOWN )
   {
      cerr << "Error: unknown vertex profile." << endl;
      return;
   }

   cgGLSetOptimalOptions( fragmentProfile );
   cgGLSetOptimalOptions( vertexProfile );
}

// -------- reloadShaders() ------------------------------------
//
// Reload the vertex and fragment shader from disk.  If the
// shaders don't compile, print a warning and use the previously
// compiled shader.  This function lets us modify the shaders
// while the program is running, and behaves nicely if we make a
// typo.
//

void reloadShaders( void )
{
   const char* args[] = { "-unroll", "none", 0 };
   const char* fragmentFn = "QJuliaFragment.cg";
   const char* vertexFn   = "QJuliaVertex.cg";
   CGprogram newProgram;

   newProgram = cgCreateProgramFromFile( context, CG_SOURCE, fragmentFn, fragmentProfile, "main", args );
   if( newProgram )
   {
      if( fragmentProgram )
         cgDestroyProgram( fragmentProgram );
      fragmentProgram = newProgram;
      cgGLLoadProgram( fragmentProgram );
   }
   else
   {
      CGerror error = cgGetError();
      cerr << "Error loading shader " << fragmentFn << ": " << cgGetErrorString( error ) << endl;
   }

   newProgram = cgCreateProgramFromFile( context, CG_SOURCE, vertexFn, vertexProfile, "main", args );
   if( newProgram )
   {
      if( vertexProgram )
         cgDestroyProgram( vertexProgram );
      vertexProgram = newProgram;
      cgGLLoadProgram( vertexProgram );
   }
   else
   {
      CGerror error = cgGetError();
      cerr << "Error loading shader " << vertexFn << ": " << cgGetErrorString( error ) << endl;
   }

   // get a handle to some fragment shader parameters
   constant    = cgGetNamedParameter( fragmentProgram, "mu"            );
   eyeP        = cgGetNamedParameter( fragmentProgram, "eye"           );
   lightP      = cgGetNamedParameter( fragmentProgram, "light"         );
   shadowP     = cgGetNamedParameter( fragmentProgram, "renderShadows" );
   iterationsP = cgGetNamedParameter( fragmentProgram, "maxIterations" );
   epsilonP    = cgGetNamedParameter( fragmentProgram, "epsilon"       );
}

// ----------- initialize() -----------------
//
// Setup some initial state.
//

void initialize( void )
{
   glClearColor( 0.2f, 0.2f, 0.3f, 0.0f );   // background color

   srand(time(NULL));  // seed the morph sequence

   // Setup Cg and load the shaders used for ray tracing.
   initializeCg();
   reloadShaders();

   // set the initial constant which specifies the Julia set
   mu[0] = 0.0f;
   mu[1] = 0.0f;
   mu[2] = 0.0f;
   mu[3] = 0.0f;

   // set the morph timer to 100% so on the next frame we generate a new
   // constant for the target Julia set and start morphing to it.
   morphTimer = 1.0f;

   // start the view rotation matrices with the identity matrix
   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();
   glGetDoublev( GL_MODELVIEW_MATRIX, lastRotation );
   glGetDoublev( GL_MODELVIEW_MATRIX, curRotation );
}

// ---------- getCurMu() ---------------------------------------------
//
// Get the interpolated constant for the current time (used for mophing
// between two Julia sets).
//

void getCurMu( float* cur, float t )
{
   for( int i=0; i<4; i++ )
      cur[i] = (1.0f - t)*mu[i] + t*mup[i];
}

// ----------- draw() -------------------------------------------------
//
// Draw is called every frame and is ultimately the thing that generates the
// ray traced image.  Draw generates rays by drawing a fullscreen quad where
// ray attributes are specified on the vertices.  These attributes are
// interpolated across the quad, generating a unique ray for every fragment on
// the screen.  These rays are then sent to the fragment processor which
// runs the ray-Julia intersection shader (found in QJuliaFragment.cg).
//

void draw()
{
   glClear (GL_COLOR_BUFFER_BIT);

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity ();
   
   // determine the basis for the image plane -- used to generate rays
   calculateView();

   // enable the vertex and fragment programs
   cgGLEnableProfile(vertexProfile);
   cgGLBindProgram(vertexProgram);
   cgGLEnableProfile(fragmentProfile);
   cgGLBindProgram(fragmentProgram);
   
   // determine the constant of the current Julia set being rendered and
   // pass it to the fragment program
   getCurMu( curMu, morphTimer );
   cgGLSetParameter4fv( constant, curMu );

   // send some other flags / parameters to the fragment program
   cgGLSetParameter1f( shadowP,     bShadows      );  // whether shadows are drawn
   cgGLSetParameter1f( iterationsP, maxIterations );  // max iterations for convergence test
   cgGLSetParameter1f( epsilonP,    epsilon       );  // precision of intersection
   cgGLSetParameter3f(eyeP, eye[0], eye[1], eye[2]);
   cgGLSetParameter3f(lightP, light[0], light[1], light[2]);

   // draw a fullscreen quad using the ray tracing fragment shader
   glBegin(GL_QUADS);

   // Rays are specified by determining the ray from the eye to each of the
   // corners of the image plane (fullscreen quad).  The image plane is
   // unit distance in front of the eye along the look at direction.  The
   // size of the image plane is determined by the field of view (fov) and
   // the aspect ratio of the current window.  To find the rays through a
   // corner, we calculate the world space position of the corner and
   // subtract the eye.  The position of a corner is found by adding
   // 1 times the look at direction, width times the tangent direction, and
   // height times the bitangent direction to the eye location.

   float beta = tan((fov*M_PI/180.0f)/2.0f);  //find height
   float alpha = beta*aspect;                 //find width

   // specify the ray origins with texture coordinates - in our case this
   // is always the same as the eye location
   glTexCoord4f(eye[0], eye[1], eye[2], 0);

   // specify ray directions with colors

   // upper left corner
   glColor4f(-alpha*T[0] - beta*B[0] + N[0],
             -alpha*T[1] - beta*B[1] + N[1],
             -alpha*T[2] - beta*B[2] + N[2], 0);
   glVertex2f(-1, -1);

   // upper right corner
   glColor4f( alpha*T[0] - beta*B[0] + N[0],
              alpha*T[1] - beta*B[1] + N[1],
              alpha*T[2] - beta*B[2] + N[2], 0);
   glVertex2f( 1, -1);

   // lower right corner
   glColor4f( alpha*T[0] + beta*B[0] + N[0],
              alpha*T[1] + beta*B[1] + N[1],
              alpha*T[2] + beta*B[2] + N[2], 0);
   glVertex2f( 1,  1);

   // lower left corner
   glColor4f(-alpha*T[0] + beta*B[0] + N[0],
             -alpha*T[1] + beta*B[1] + N[1],
             -alpha*T[2] + beta*B[2] + N[2], 0);
   glVertex2f(-1,  1);

   glEnd();

   // turn off the shaders
   cgGLDisableProfile(fragmentProfile);
   cgGLDisableProfile(vertexProfile);

   glutSwapBuffers();
}

void destruct( void )
{
   exit( 0 );
}

// --------- keyboard() ----------------------------------
//
// Handles keyboard input (mostly to set rendering options).
//

void keyboard( unsigned char key, int x, int y )
{
   const float stepSize = 0.05f;

   switch( key )
   {
      // toggle between fullscreen mode and windowed mode
      case 'f':
         if( fullscreen )
         {
            glutPositionWindow(50, 50);
            glutReshapeWindow( defaultWindowSize[0],
                               defaultWindowSize[1]);
            fullscreen = false;
         }
         else
         {
            glutFullScreen();	
            fullscreen = true;
         }
         break;

      // do any necessary deallocation and exit
      case 27:
         destruct();
         break;

      // toggle morphing on / off
      case ' ':
         if( bAnimate )
         {
            getCurMu( curMu, morphTimer );
            for( int i=0; i<4; i++ )
               mu[i] = curMu[i];
            morphTimer = 0.0f;
         }

         mup[0] = 2.0f*(rand()*fRandMax) - 1.0f;
         mup[1] = 2.0f*(rand()*fRandMax) - 1.0f;
         mup[2] = 2.0f*(rand()*fRandMax) - 1.0f;
         mup[3] = 2.0f*(rand()*fRandMax) - 1.0f;

         bAnimate = !bAnimate;
         break;

      // increase the maximum number of iterations used to determine convergence
      // (increases the detail of the rendering)
      case '+':
         maxIterations++;
         if( maxIterations > 20 )
            maxIterations = 20;
         break;

      // decrease the maximum number of iterations used to determine convergence
      // (decreases the detail of the rendering)
      case '-':
         maxIterations--;
         if( maxIterations < 1 )
            maxIterations = 1;
         break;

      // increase the rendering precision
      case '[':
         epsilon += 0.0001;
         break;

      // decrease the rendering precision
      case ']':
         epsilon -= 0.0001;
         if( epsilon < 0.0001 )
            epsilon = 0.0001;
         break;

      // toggle self-shadowing on / off
      case 's':
         bShadows = !bShadows;
         break;

      // i/I, j/J, k/K, and l/L explicitly change the components
      // of the quaternion Julia constant
      case 'L':
         mu[0] += stepSize; 
         break;

      case 'l':
         mu[0] -= stepSize; 
         break;

      case 'I':
         mu[1] += stepSize; 
         break;

      case 'i':
         mu[1] -= stepSize; 
         break;

      case 'J':
         mu[2] += stepSize; 
         break;

      case 'j':
         mu[2] -= stepSize; 
         break;

      case 'K':
         mu[3] += stepSize; 
         break;

      case 'k':
         mu[3] -= stepSize; 
         break;

      // reload shaders from disk
      case 'r':
         reloadShaders();
         break;

      default:
         break;
   }

   glutPostRedisplay();
}

// ---------- update() ----------------------
//
// Handle animation or other tasks performed every frame.
//

void update()
{
   // handle spinning the camera if the mouse was released quickly
   if( spinOn )
   {
      spin( -spinAngle, rotationAxis );
      for( int i=0; i<16; i++ )
         lastRotation[i] = curRotation[i];
   }

   // handle automatic morphing from one Julia set to another
   if( bAnimate )
   {
      morphTimer += 0.01f;

      if( morphTimer >= 1.0f )
      {
         morphTimer = 0.0f;

         mu[0] = mup[0];
         mu[1] = mup[1];
         mu[2] = mup[2];
         mu[3] = mup[3];

         mup[0] = 2.0f*rand()/float(RAND_MAX) - 1.0f;
         mup[1] = 2.0f*rand()/float(RAND_MAX) - 1.0f;
         mup[2] = 2.0f*rand()/float(RAND_MAX) - 1.0f;
         mup[3] = 2.0f*rand()/float(RAND_MAX) - 1.0f;
      }
   }		

   glutPostRedisplay();
}

// -------- resize() ---------------------
//
// Maintain relevant state when window is resized.
//

void resize( int width, int height )
{
   glViewport( 0, 0, width, height );

   aspect = width/(float)height;

   glMatrixMode (GL_PROJECTION);
   glLoadIdentity();
   glOrtho(-1, 1, -1, 1, 0, 100);

   glMatrixMode (GL_MODELVIEW);
   glLoadIdentity ();
}

// --------- main() -----------------------
//
// Set up GLUT and start the main loop.
//

int main( int argc, char *argv[] )
{
   glutInit( &argc, argv );
   glutInitWindowPosition( 50, 50 );
   glutInitWindowSize( defaultWindowSize[0], defaultWindowSize[1] );
   glutInitDisplayMode( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );
   glutCreateWindow( "GLUT" );
   glutIdleFunc( update );
   glutDisplayFunc( draw );
   glutKeyboardFunc( keyboard );
   glutReshapeFunc( resize );
   glutMouseFunc( click );
   glutMotionFunc( drag );
   initialize();
   glutMainLoop();

   return 0;
}

