/* Emulate a few 80387 instructions */

#include <stdio.h>
#include <signal.h>

typedef unsigned char u_char;

struct operands {int reg, base, index, scale, disp, op2; u_char *pc;};
extern struct operands *__modrmdec();

typedef struct {long l,h;} Double;
typedef struct {long h;} Float;

extern Double __float2double(), __long2double();
extern Float  __double2float();

Double __fpreg[8];
int __fpdepth = -1;

#define REG(n) ((n) == 4 ? (int)&stack[19] : stack[14 - (n)])

#define EADDR(ops) (ops->disp + \
		    (ops->base == -1 ? 0 : REG(ops->base)) + \
		    (ops->index == -1 ? 0 : REG(ops->index) * ops->scale))

#define CHECKPOP if(__fpdepth == -1) \
		    { fprintf(stderr, "fp stack underflow\n"); exit(1); }
#define CHECKPUSH if(__fpdepth == 7) \
		     { fprintf(stderr, "fp stack overflow\n"); exit(1); }

#define POP  __fpreg[__fpdepth--]
#define TOP  __fpreg[__fpdepth]
#define PUSH __fpreg[++__fpdepth]

#ifdef DEBUG
#define DODEBUG(m) fprintf(stderr, "%s: FP(0) is %08x_%08x\n", \
			   m, __fpreg[0].h, __fpreg[0].l);
#else
#define DODEBUG(m)
#endif

int __emulate(stack)
long *stack;
{
    u_char *instr;
    struct operands *ops;	
    long eaddr;

    instr = (u_char *)stack[16];
    ops = __modrmdec(instr+1);

    switch(instr[0])
    {
      case 0xd9:
	if(ops->reg == -1 && ops->op2 == 0)
	{
	    /* FLDS */
	    CHECKPUSH;
	    PUSH = __float2double(*(Float *)EADDR(ops));
	    DODEBUG("FLDS");
	}
	else if(ops->reg == -1 && ops->op2 == 2)
	{
	    /* FSTS */
	    CHECKPOP;
	    *(Float *)EADDR(ops) = __double2float(TOP);
	    DODEBUG("FSTS");
	}
	else if(ops->reg == -1 && ops->op2 == 3)
	{
	    /* FSTPS */
	    CHECKPOP;
	    *(Float *)EADDR(ops) = __double2float(POP);
	    DODEBUG("FSTPS");
#ifdef DEBUG
	    fprintf(stderr, "   stored %08x at %x\n",
		     *(long *)EADDR(ops), EADDR(ops));
#endif
	}
	else if(instr[1] == 0xe8)
	{
	    /* FLD1 */
	    CHECKPUSH;
	    PUSH = __long2double(0l);
	    DODEBUG("FLD1");
	}
	else if(instr[1] == 0xee)
	{
	    /* FLDZ */
	    CHECKPUSH;
	    PUSH = __long2double(1l);
	    DODEBUG("FLDZ");
	}
	else
	    return -1;
	break;

      case 0xdd:
	if(ops->reg == -1 && ops->op2 == 0)
	{
	    /* FLDL */
	    CHECKPUSH;
	    PUSH = *(Double *)EADDR(ops);
	    DODEBUG("FLDL");
	}
	else if(ops->reg == -1 && ops->op2 == 2)
	{
	    /* FSTL */
	    CHECKPOP;
	    *(Double *)EADDR(ops) = TOP;
	    DODEBUG("FSTL");
	}
	else if(ops->reg == -1 && ops->op2 == 3)
	{
	    /* FSTPL */
	    CHECKPOP;
	    *(Double *)EADDR(ops) = POP;
	    DODEBUG("FSTPL");
	}
	else if(ops->reg == 0 && ops->op2 == 3)
	{
	    /* FSTP ST(0) */
	    POP;
	    DODEBUG("FSTP ST(0)");
	}
	else
	    return -1;
	break;

      default:
	return -1;
    }

    stack[16] = (long)ops->pc;
    
    return 0;
}
