function [mu, mycputime] = flexible_kernel(m, vec_tr, T, constant_eigenvector_indices)
%
% Input:
%   m: number of eigenvectors
%   vec_tr: Lsize*m, each column is the training portion of an eigenvector (of the graph Laplacian)
%   T: Lsize * Lsize, target matrix
%   constant_eigenvector_indices: which eigenvectors in 1...m are constant?
% Output:
%   mu: the learned eigenvalues
%   mycputime: time to solve the SDP, in seconds
%
% we will solve the problem
%	max mu' q
%	s.t. mu' S mu <= 1
%	     mu >= 0
%	     mu_i >= mu_{i+1}
% where
%	q_i = (y' vec_i,tr)^2
%	S_ij = M'*M = (vec_j,tr' vec_i,tr)^2
%	M is a L^2 * m matrix, column i is K_i,tr vectorized.
%	The constraint mu' S mu <=1 is equiv. to ||M*mu||<=1
%	this cone constraint is more robust.
% The following code uses YALMIP/SeDuMi
%
%
% % Please note there is no warranty.  In fact it is not even a software, but
% merely research code.  Feel free to modify it for your work. 
% Xiaojin Zhu, zhuxj@cs.cmu.edu
% 2004


Lsize = size(vec_tr, 1);

for i=1:m, M(:,i)=reshape(vec_tr(:,i)*vec_tr(:,i)', Lsize^2, 1); end

q = M' * reshape(T, Lsize^2, 1);

% do not impose order constraint on constant eigenvector
for i=1:m-1, 
    if (~(ismember(i, constant_eigenvector_indices) || ismember(i+1, constant_eigenvector_indices)))
	order(i,i)=1; order(i,i+1)=-1;
    end
end

mu = sdpvar(m, 1);

F = set(-mu <= 0) + set(cone(M*mu,1)) + set(-order * mu <= 0);

start_time = cputime;
solvesdp(F, -mu'*q, sdpsettings('solver', 'sedumi'));
mycputime = cputime - start_time;

clear M;

% get the coefficents from YALMIP
mu = double(mu);
